﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace NetMonitor.AmfFormat
{
	abstract class AmfData
	{
		public AmfData()
		{
		}
		public abstract byte Type { get; }
		public virtual string GetStrValue() { throw new NotImplementedException(); }
		public virtual int GetIntValue() { throw new NotImplementedException(); }
		public virtual bool GetBooleanValue() { throw new NotImplementedException(); }
		public virtual DateTime GetDateTimeValue() { throw new NotImplementedException(); }
		
		public abstract void Write(StreamWriter writer, int tab,int max_tab);
	}

	class AmfNumberData : AmfData
	{
		public AmfNumberData(double d)
		{
			this.d = d;
		}
		public readonly double d;
		public override byte Type { get { return 0; } }
		public override int GetIntValue()
		{
			return (int)d;
		}
		public override bool GetBooleanValue()
		{
			return d != 0.0;
		}
		public override string ToString()
		{
			return d.ToString();
		}
		public override void Write(StreamWriter writer, int tab, int max_tab)
		{
			writer.WriteLine("num " + d.ToString());
		}
	}
	class AmfBooleanData : AmfData
	{
		public AmfBooleanData(bool b)
		{
			this.b = b;
		}
		public readonly bool b;
		public override byte Type { get { return 1; } }
		public override int GetIntValue()
		{
			return b ? 1 : 0;
		}
		public override bool GetBooleanValue()
		{
			return b;
		}
		public override string ToString()
		{
			return b.ToString();
		}
		public override void Write(StreamWriter writer, int tab, int max_tab)
		{
			writer.WriteLine("bool " + b.ToString());
		}
	}

	class AmfStringData : AmfData
	{
		public AmfStringData(string s)
		{
			this.s= s;
		}
		public readonly string s;
		public override byte Type { get { return 2; } }
		public override string GetStrValue()
		{
			return s;
		}
		public override int GetIntValue()
		{
			return int.Parse(s);
		}
		public override bool GetBooleanValue()
		{
			return int.Parse(s) != 0;
		}
		public override DateTime GetDateTimeValue()
		{
			//YYYY/MM/DD hh:mm型
			return DateTime.Parse(s);
		}
		public override string ToString()
		{
			return s.ToString();
		}
		public override void Write(StreamWriter writer, int tab, int max_tab)
		{
			writer.WriteLine("str " + s.ToString());
		}
	}
	class AmfObjectData : AmfData
	{
		public AmfObjectData()
		{
			Length = 0xFFFFFFFFu;
			Members = new List<AmfObjectMember>();
		}
		public AmfObjectData(UInt32 len)
		{
			Length = len;
			Members = new List<AmfObjectMember>();
		}
		public readonly List<AmfObjectMember> Members;
		public readonly UInt32 Length;
		
		public AmfData GetMember(string key)
		{
			foreach (AmfObjectMember mem in Members)
				if (mem.Key.Equals(key, StringComparison.OrdinalIgnoreCase))
					return mem.Value;
			return null;
		}
		
		public override byte Type { get { return 3; } }
		public override string ToString()
		{
			return Members.Count.ToString() + " members";
		}
		public override void Write(StreamWriter writer, int tab, int max_tab)
		{
			if(Length == 0xFFFFFFFFu)
				writer.WriteLine("obj " + Members.Count.ToString() + " members");
			else
				writer.WriteLine("objArray " + Members.Count.ToString() + " members/" + Length.ToString() + " Length");
			tab++;
			if (tab > max_tab)
				return;
			int i = 0;
			foreach (AmfObjectMember mem in Members)
			{

				writer.Write(new string(' ', tab * 2));
				if (i == 0 || i == Members.Count - 1)
				{
					writer.Write("+-");
				}
				else
					writer.Write("| ");
				writer.Write(mem.Key);
				writer.Write(":");
				mem.Value.Write(writer,tab,max_tab);
				i++;
			}
		}
	}

	class AmfObjectMember
	{
		public AmfObjectMember(string key, AmfData value)
		{
			this.Key = key;
			this.Value = value;
		}
		public readonly string Key;
		public readonly AmfData Value;
		public override string ToString()
		{
			return Key + ":" + Value.ToString();
		}
	}

	class AmfNullData : AmfData
	{
		public override byte Type { get { return 5; } }
		public override string ToString()
		{
			return "Null";
		}
		public override void Write(StreamWriter writer, int tab, int max_tab)
		{
			writer.WriteLine("Null");
		}
		public override string GetStrValue() { return null; }
		public override int GetIntValue() { return 0; }
		public override bool GetBooleanValue() { return false; }
	}
	class AmfUndefinedData : AmfData
	{
		public override byte Type { get { return 6; } }
		public override string ToString()
		{
			return "Undefined";
		}
		public override void Write(StreamWriter writer, int tab, int max_tab)
		{
			writer.WriteLine("Undefined");
		}
	}
	//試行錯誤の残骸
	//class AmfObjectArrayData : AmfData
	//{
	//    public AmfObjectArrayData(int length)
	//    {
	//        Array = new AmfObjectData[length];
	//    }
	//    public readonly AmfObjectData[] Array;
	//    public override byte Type { get { return 8; } }
	//    public override string ToString()
	//    {
	//        return Array.Length.ToString() + " objects";
	//    }
	//}

	//class AmfObjectArrayData2 : AmfData
	//{
	//    public AmfObjectArrayData2(int length)
	//    {
	//        Array = new AmfObjectMenber[length];
	//    }
	//    public readonly AmfObjectMenber[] Array;
	//    public override byte Type { get { return 8; } }
	//    public override string ToString()
	//    {
	//        return Array.Length.ToString() + " objects";
	//    }
	//}

	class AmfArrayData : AmfData
	{
		public AmfArrayData(int length)
		{
			Array = new AmfData[length];
		}
		public readonly AmfData[] Array;
		public override byte Type { get { return 0xA; } }
		public override string ToString()
		{
			return Array.Length.ToString() + " values";
		}
		public override void Write(StreamWriter writer, int tab, int max_tab)
		{
			writer.WriteLine("array " + Array.Length.ToString() + " values");
			tab++;
			if (tab > max_tab)
				return;
			int i = 0;
			foreach (AmfData amf in Array)
			{

				writer.Write(new string(' ', tab * 2));
				if (i == 0 || i == Array.Length - 1)
				{
					writer.Write("+-");
				}
				else
					writer.Write("| ");
				amf.Write(writer, tab,max_tab);
				i++;
			}
		}
	}





}
