﻿/*
*Copyright(c) Live2D Inc. All rights reserved.
*このソースコードの使用には、"http://www.live2d.com/eula/live2d-open-software-license-agreement_jp.html"ファイルにあるLive2D Open Software ライセンスが適用されます。
*Use of this source code is governed by a Live2D Open Software license that can be found in the "http://www.live2d.com/eula/live2d-open-software-license-agreement_en.html" file.
*/

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using UnityEngine;

namespace live2d.Mecanim.EditorExtensions
{
    /// <summary>
    /// Contains data and settings on mtn files.
    /// </summary>
    public class Mtn
    {
        /// <summary>
        /// Gets the duration of the mtn in seconds.
        /// </summary>
        public float Duration
        {
            get { return DurationInFrames / Fps; }
        }

        /// <summary>
        /// Gets the duration of the mtn in frames.
        /// </summary>
        public readonly int DurationInFrames;

        /// <summary>
        /// Gets the frames per second the mtn was sampled at.
        /// </summary>
        public readonly float Fps;

        /// <summary>
        /// Gets the parameter curves.
        /// </summary>
        public readonly MtnParameterCurve[] ParameterCurves;

        #region Constructors

        public Mtn(string mtnPath)
        {
            // Try to read in the mtn.
            try
            {
                var mtnData = File.ReadAllLines(Application.dataPath + "/" + mtnPath.Substring("Assets".Length));


                // Read in data.
                Fps = ReadFps(mtnData);
                ParameterCurves = ReadParameterCurves(mtnData);


                // Compute duration.
                DurationInFrames = ParameterCurves
                    .Select(curve => curve.DurationInFrames)
                    .Max();
            }
            catch (Exception)
            {
                
                throw new IOException("Failed to load mtn data at path " + mtnPath);
            }
        }

        #endregion

        #region Parse methods

        private static float ReadFps(IEnumerable<string> mtn)
        {
            foreach (var line in mtn)
            {
                // Skip non-fps lines.
                if (!line.StartsWith("$"))
                {
                    continue;
                }

                if (!line.Contains("fps"))
                {
                    continue;
                }


                return float.Parse(line
                    .Split('=')
                    .Last());
            }


            return 30.0f;
        }

        private static MtnParameterCurve[] ReadParameterCurves(IEnumerable<string> mtn)
        {
            var parameterCurves = new List<MtnParameterCurve>();

            foreach (var line in mtn)
            {
                // Skip non-parameter lines.
                if (string.IsNullOrEmpty(line) || line.StartsWith("#") || line.StartsWith("$"))
                {
                    continue;
                }

                if (line.StartsWith("VISIBLE:") || line.StartsWith("LAYOUT:"))
                {
                    continue;
                }


                var parameterName = line.Split(':', '=')[0];

                if (parameterName.Contains("."))
                {
                    parameterName = parameterName.Replace(".", "");
                }

                var values = line
                    .Split('=')[1]
                    .Split(',')

                    .Select(s => float.Parse(s))
                    .ToArray();


                parameterCurves.Add(new MtnParameterCurve(parameterName, values));
            }


            return parameterCurves.ToArray();
        }

        #endregion
    }
}