﻿/*
*Copyright(c) Live2D Inc. All rights reserved.
*このソースコードの使用には、"http://www.live2d.com/eula/live2d-open-software-license-agreement_jp.html"ファイルにあるLive2D Open Software ライセンスが適用されます。
*Use of this source code is governed by a Live2D Open Software license that can be found in the "http://www.live2d.com/eula/live2d-open-software-license-agreement_en.html" file.
*/

using System;
using System.Linq;
using UnityEditor;
using UnityEngine;

namespace live2d.Mecanim.EditorExtensions
{
    public class GenerateAnimationClipUi : EditorWindow
    {
        private void ShowDragAndDropAreaAndHandleMtnDrop()
        {
            var currentEvent = Event.current;
            var dropArea = EditorGUILayout.GetControlRect(GUILayout.Height(256.0f));


            EditorGUI.HelpBox(dropArea, "Drop mtn (*.mtn or *.mtn.bytes) file here", MessageType.Info);


            switch (currentEvent.type)
            {
                case EventType.DragUpdated:
                case EventType.DragPerform:
                    {
                        if (!dropArea.Contains(currentEvent.mousePosition))
                        {
                            return;
                        }


                        DragAndDrop.visualMode = DragAndDropVisualMode.Copy;


                        if (currentEvent.type == EventType.DragPerform)
                        {
                            foreach (var mtnPath in DragAndDrop.paths.Where(p => p.EndsWith(".mtn") || p.EndsWith(".mtn.bytes")))
                            {
                                var parametersScript = _parametersScript;
                                var animationClipPath = _animationClipOutputPath;
                                var animationClipAdditionalName = _animationClipAdditionalName;


                                // Generate an output path if necessary.
                                if (string.IsNullOrEmpty(animationClipPath))
                                {
                                    animationClipPath = mtnPath.Replace(mtnPath.EndsWith(".mtn") ? ".mtn" : animationClipAdditionalName + ".mtn.bytes", animationClipAdditionalName + ".anim");
                                }


                                GenerateAnimationClip(parametersScript, mtnPath, animationClipPath);


                                DragAndDrop.AcceptDrag();


                                return;
                            }
                        }


                        break;
                    }
                default:
                    {
                        break;
                    }
            }
        }

        private static void GenerateAnimationClip(MonoScript parametersClass, string mtnPath, string animationClipPath)
        {
            // Do checks.
            if (parametersClass == null || string.IsNullOrEmpty(mtnPath) || string.IsNullOrEmpty(animationClipPath))
            {
                throw new ArgumentNullException();
            }

            if (parametersClass.GetClass() == null)
            {
                throw new ArgumentException();
            }


            // Parse mtn.
            var mtn = new Mtn(mtnPath);


            // Generate animation clip.
            var animationClip = new AnimationClipGenerator()
                .SetMtn(mtn)
                .SetParametersType(parametersClass.GetClass())
                .GenerateAnimationClip();


            // Save animation clip.
            AssetDatabase.CreateAsset(animationClip, animationClipPath);
        }

        #region Unity editor integration

        [MenuItem("Assets/Generate/Animation Clip")]
        private static void ShowUi()
        {
            GetWindow<GenerateAnimationClipUi>();
        }

        #endregion

        #region Unity event handling

        private MonoScript _parametersScript;

        private string _animationClipOutputPath;
        private string _animationClipAdditionalName;

        private void OnGUI()
        {
            // Show parameter script drop box.
            var parametersScript = EditorGUILayout
                .ObjectField("Model Parameters Script", _parametersScript, typeof(MonoScript), true) as MonoScript;


            if (parametersScript != null && typeof(ModelParameters).IsAssignableFrom(parametersScript.GetClass()))
            {
                _parametersScript = parametersScript;
            }


            // Allow the user to set an output path.
            _animationClipOutputPath = EditorGUILayout
                .TextField("Animation Clip Output Path [optional]", _animationClipOutputPath);


            _animationClipAdditionalName = EditorGUILayout
                .TextField("Additional Name", _animationClipAdditionalName);

            // Show drag-and-drop area.
            ShowDragAndDropAreaAndHandleMtnDrop();
        }

        #endregion
    }
}
