﻿/*
*Copyright(c) Live2D Inc. All rights reserved.
*このソースコードの使用には、"http://www.live2d.com/eula/live2d-open-software-license-agreement_jp.html"ファイルにあるLive2D Open Software ライセンスが適用されます。
*Use of this source code is governed by a Live2D Open Software license that can be found in the "http://www.live2d.com/eula/live2d-open-software-license-agreement_en.html" file.
*/

using System.Collections.Generic;
using UnityEngine;
using UnityEngine.VR;
using live2d.Euclid;

[RequireComponent(typeof(Camera))]
public class EuclidVRCamera : MonoBehaviour
{
    [SerializeField]
    private List<EuclidModel> TargetEuclidModel = new List<EuclidModel>();
    [SerializeField]
    public float BothEyeOffset = 0.0f; // Adjust parallax

    private EuclidCameraBackend _euclidCameraBackendLeft = new EuclidCameraBackend(false);
    private EuclidCameraBackend _euclidCameraBackendRight = new EuclidCameraBackend(true);

    private float leftCameraOffset;
    private float rightCameraOffset;
    private float eyeOffset;

    private int _renderingEyeId;
    private int _currentEye = 0;

    public void AddEuclidModel(EuclidModel model)
    {
        if (TargetEuclidModel.Contains(model) || model == null) { return; }

        TargetEuclidModel.Add(model);
        var left = registerEuclidModel(_euclidCameraBackendLeft, model);
        var right = registerEuclidModel(_euclidCameraBackendRight, model);
        _euclidCameraBackendLeft.InitializeModel(transform, left);
        _euclidCameraBackendRight.InitializeModel(transform, right);
    }

    private EuclidModelBackend registerEuclidModel(EuclidCameraBackend backend, EuclidModel model)
    {
        if (model == null) { return null; }

        var modelBackend = model.AddBackend();
        backend.registerEuclidModel(modelBackend);

        return modelBackend;
    }

    public void ResetParallaxOffset()
    {
        var center = InputTracking.GetLocalPosition(VRNode.CenterEye);

        leftCameraOffset  = (center - InputTracking.GetLocalPosition(VRNode.LeftEye)).magnitude;
        rightCameraOffset = (center - InputTracking.GetLocalPosition(VRNode.RightEye)).magnitude;
        eyeOffset = (leftCameraOffset + rightCameraOffset) * BothEyeOffset * 0.5f;

    }

    #region Unity Functions
    private void Awake()
    {
        // Set ID for shader
        _renderingEyeId = Shader.PropertyToID("euclid_RenderingEye");

        foreach (var m in TargetEuclidModel)
        {
            registerEuclidModel(_euclidCameraBackendLeft, m);
            registerEuclidModel(_euclidCameraBackendRight, m);
        }
    }

    private void OnEnable()
    {
        _euclidCameraBackendLeft.InitializeModel(transform);
        _euclidCameraBackendRight.InitializeModel(transform);
    }

    private void Start()
    {
        ResetParallaxOffset();
    }

    private void Update()
    {
        var leftEyePos   = transform.position + -transform.right * leftCameraOffset;
        var rightEyePos  = transform.position +  transform.right * rightCameraOffset;

        // カメラや3Dモデルの情報からEuclidモデルの向きに関するパラメータを更新する
        _euclidCameraBackendLeft.UpdateEuclidModelParameter(leftEyePos);
        _euclidCameraBackendRight.UpdateEuclidModelParameter(rightEyePos);
    }


    private void LateUpdate()
    {
        var leftEyePos  = transform.position + -transform.right * (leftCameraOffset  + eyeOffset);
        var rightEyePos = transform.position +  transform.right * (rightCameraOffset + eyeOffset);

        // カメラや3Dモデルの情報からEuclidモデルの向きに関するパラメータを更新する
        _euclidCameraBackendLeft.LateUpdateEuclidModelParameter(leftEyePos);
        _euclidCameraBackendRight.LateUpdateEuclidModelParameter(rightEyePos);
    }

    private void OnPreRender()
    {
        Shader.SetGlobalInt(_renderingEyeId, _currentEye);

        // OnPreRenderは左右それぞれ1回呼ばれるので現在のターゲットアイを入れ替える
        _currentEye = 1 - _currentEye;
    }

    private void OnDisable()
    {
        _euclidCameraBackendLeft.Close();
        _euclidCameraBackendRight.Close();
    }

    private void OnDestroy()
    {
        _euclidCameraBackendLeft.Destroy();
        _euclidCameraBackendRight.Destroy();
    }
    #endregion
}
