#include "common.h"
#include "libWrapper.h"

#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _SELECT_FROM_H
#define _SELECT_FROM_H
#include "selectFrom.h"
#endif

#ifndef _EXTERN_H
#include "extern.h"
#define _EXTERN_H
#endif

#include "memsize.h"
#include "mmgr.h"
#include "port.h"
#include "host.h"
#include "cmp.h"
#include "chkptrlogdir.h"

/*******************************************************
 *
 * Declaration
 *
 *******************************************************/
extern void sendback(const int sockfd, const void *ptr, const int size, const int flag);
extern void recovery(void);

/*******************************************************
 *
 * Global variable
 *
 *******************************************************/
static MMGR Mmgr;
static unsigned int CidLBFM = 0;
static BOOLEAN HaltSystem = FALSE;

/*******************************************************
 *
 * Private Function
 *
 *******************************************************/
static void
allocBuffPool(unsigned const int cid)
{
  if ((Mmgr.cnxt[cid].buffPool = calloc(REMOTE_MEMORY_SIZE, sizeof(char))) == NULL)  ERR;
}

static void
initLockCnxt(unsigned const int cid)
{
  if (pthread_rwlock_init(&Mmgr.cnxt[cid].lock, NULL)) ERR;
}

static void
initHeadMctl(unsigned const int cid)
{
  Mmgr.cnxt[cid].head.end = (unsigned long)Mmgr.cnxt[cid].buffPool;
  Mmgr.cnxt[cid].head.begin = -1; /* Sentinel */
  Mmgr.cnxt[cid].tailP = &Mmgr.cnxt[cid].head;
}

static MCTL *
getMctl(const int addr, const int reqsiz)
{
  MCTL *mctlP;
  
  mctlP = (MCTL *)addr;
  bzero(mctlP, sizeof(MCTL) + reqsiz);
  mctlP->begin = addr + sizeof(MCTL);
  mctlP->end = mctlP->begin + reqsiz;
  mctlP->next = mctlP->prev = NULL;

  return mctlP;
}

static int
incrementCnxtid(unsigned int cid)
{
  if (++cid == NUM_CNXT) 
    cid = 0;
  return cid;
}

static void
cnxtSwitch(void)
{
  while ((Mmgr.cid == 0) && (CidLBFM != 0)) 
    usleep(1000);
  Mmgr.cid = incrementCnxtid(Mmgr.cid);
}

static void
writeBuffPool(char *p, unsigned long siz)
{
  int fd;
  char fname[BUFFSIZE];
  struct timeval now;

  gettimeofday(&now, NULL);
  bzero(fname, BUFFSIZE);
  sprintf(fname, "%s/%ld.%ld", CHKPTRLOGDIR, now.tv_sec, now.tv_usec);
  if ((fd = open(fname, O_WRONLY|O_CREAT|O_TRUNC|O_EXLOCK, 0644)) == -1) ERR;
  if (write(fd, p, siz) == -1) ERR;
  if (close(fd) == -1) ERR;
}

static void
writeInsertObj(char *p, unsigned long siz)
{
  int fd;
  char fname[BUFFSIZE];
  struct timeval now;
  MCTL mctl;
  
  bzero(&mctl, sizeof(MCTL));
  mctl.begin = (unsigned long)&mctl + (unsigned long)sizeof(MCTL);
  mctl.end = (unsigned long)&mctl + (unsigned long)sizeof(MCTL) + siz;
  gettimeofday(&now, NULL);
  bzero(fname, BUFFSIZE);
  sprintf(fname, "%s/%ld.%ld", CHKPTRLOGDIR, now.tv_sec, now.tv_usec);
  if ((fd = open(fname, O_WRONLY|O_CREAT|O_TRUNC|O_EXLOCK, 0644)) == -1) ERR;
  if (write(fd, &mctl, sizeof(MCTL)) == -1) ERR;
  if (write(fd, p, siz) == -1) ERR;
  if (close(fd) == -1) ERR;
}

static void
clearBuffPool(unsigned const int cid)
{
  bzero(Mmgr.cnxt[cid].buffPool, REMOTE_MEMORY_SIZE);
  initHeadMctl(cid);
}

static void
initCnxt(const int cid)
{
  initHeadMctl(cid);  
  initLockCnxt(cid);
}

static void *
logBufFileMaker(void)
{
  int err;

  for (; HaltSystem == FALSE; usleep(1000)) {
    while ((CidLBFM < Mmgr.cid) || (Mmgr.cid == 0 && CidLBFM != 0)) {
      if (pthread_rwlock_wrlock(&Mmgr.lock)) ERR;  
      if (pthread_rwlock_wrlock(&Mmgr.cnxt[CidLBFM].lock)) ERR;
      if (pthread_rwlock_unlock(&Mmgr.lock)) ERR;  
      writeBuffPool(Mmgr.cnxt[CidLBFM].buffPool, REMOTE_MEMORY_SIZE);
      clearBuffPool(CidLBFM);
      if ((err = pthread_rwlock_unlock(&Mmgr.cnxt[CidLBFM].lock))) ERR; 
      CidLBFM = incrementCnxtid(CidLBFM);
    }
  }

  /*
   * OK, ready to halt now
   */
  printf("Synchronizing memory resident data onto disk ...\n");
  writeBuffPool(Mmgr.cnxt[Mmgr.cid].buffPool, REMOTE_MEMORY_SIZE);
  exit(0);

  return NULL;
}

static void *
logBufFileReader(void)
{
  for (;; usleep(1000)) 
    recovery();

  return NULL;
}

/******************************************************************************
 *
 * Public Function
 *
 *****************************************************************************/
extern void
initMmgr(void)
{
  int i;

  /*
   * Just constructing data structure
   */
  bzero(&Mmgr, sizeof(MMGR));
  if (pthread_rwlock_init(&Mmgr.lock, NULL)) ERR;
  for (i = 0; i < NUM_CNXT; i++) {
    allocBuffPool(i);
    initCnxt(i);
  }
}

/* 
 * Cleanup data structure
 */
extern void
clearMmgr(void)
{
  int i;

  if (pthread_rwlock_destroy(&Mmgr.lock)) ERR;    
  if (close(Mmgr.chkPtrfd) == -1) ERR;
  for (i = 0; i < NUM_CNXT; i++) {
    if (pthread_rwlock_destroy(&Mmgr.cnxt[i].lock)) ERR;    
    free(Mmgr.cnxt[i].buffPool);
  }
}

extern void 
kmcpy(void *p, const unsigned long siz)
{
  if (pthread_rwlock_wrlock(&Mmgr.lock)) ERR;  
  if (pthread_rwlock_wrlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
  if ((unsigned long)(Mmgr.cnxt[Mmgr.cid].tailP->end + sizeof(MCTL) + siz) > 
      (unsigned long)(Mmgr.cnxt[Mmgr.cid].buffPool + REMOTE_MEMORY_SIZE)) {
    if (pthread_rwlock_unlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
    cnxtSwitch();
    if (pthread_rwlock_wrlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
  }
  if (pthread_rwlock_unlock(&Mmgr.lock)) ERR;    
  Mmgr.cnxt[Mmgr.cid].tailP->next = getMctl(Mmgr.cnxt[Mmgr.cid].tailP->end, siz);
  memcpy((void *)((unsigned long)Mmgr.cnxt[Mmgr.cid].tailP->next + sizeof(MCTL)), p, siz);
  Mmgr.cnxt[Mmgr.cid].tailP->next->prev = Mmgr.cnxt[Mmgr.cid].tailP;
  Mmgr.cnxt[Mmgr.cid].tailP = Mmgr.cnxt[Mmgr.cid].tailP->next;
  if (pthread_rwlock_unlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
}  

extern void 
kdcpy(void *p, const unsigned long siz)
{
  if (pthread_rwlock_wrlock(&Mmgr.lock)) ERR;  
  if (pthread_rwlock_wrlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
  if (Mmgr.cnxt[Mmgr.cid].head.end != (unsigned long)Mmgr.cnxt[Mmgr.cid].buffPool) {
    writeBuffPool(Mmgr.cnxt[Mmgr.cid].buffPool, REMOTE_MEMORY_SIZE);
    clearBuffPool(Mmgr.cid);
  }
  if (pthread_rwlock_unlock(&Mmgr.lock)) ERR;    
  writeInsertObj(p, siz);
  if (pthread_rwlock_unlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
}  

/*
extern void
createLogBufFileReader(void)
{
  pthread_t thread;

  if (pthread_create(&thread, NULL, (void *)logBufFileReader, NULL)) ERR;
  if (pthread_detach(thread)) ERR;
}

extern void
createLogBufFileMaker(void)
{
  pthread_t thread;

  if (pthread_create(&thread, NULL, (void *)logBufFileMaker, NULL)) ERR;
  if (pthread_detach(thread)) ERR;
}
*/

