#include "common.h"
#include "libWrapper.h"
#include <zlib.h>

#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _SELECT_FROM_H
#define _SELECT_FROM_H
#include "selectFrom.h"
#endif

#ifndef _EXTERN_H
#include "extern.h"
#define _EXTERN_H
#endif

#include "memsize.h"

#include "mmgr.h"
#include "port.h"
#include "host.h"
#include "cmp.h"

#define CHKPTRLOGDIR "/tmp/CHKPTRLOG"

/*******************************************************
 *
 * Declaration
 *
 *******************************************************/
extern void sendback(const int sockfd, const void *ptr, const int size, const int flag);

/*******************************************************
 *
 * Global variable
 *
 *******************************************************/
static MMGR Mmgr;
static BOOLEAN StopLogBufFileReader;
static BOOLEAN StopLogBufFileMaker;
static unsigned int CidLBFM = 0;

/*******************************************************
 *
 * Private Function
 *
 *******************************************************/
static void
allocBuffPool(int siz, unsigned const int cid)
{
  if ((Mmgr.cnxt[cid].buffPool = calloc(siz, sizeof(char))) == NULL)  ERR;
}

static void
initLockCnxt(unsigned const int cid)
{
  if (pthread_rwlock_init(&Mmgr.cnxt[cid].lock, NULL)) ERR;
}

static void
initHeadMctl(unsigned const int cid)
{
  Mmgr.cnxt[cid].head.end = (unsigned long)Mmgr.cnxt[cid].buffPool;
  Mmgr.cnxt[cid].head.begin = -1; /* Sentinel */
  Mmgr.cnxt[cid].tailP = &Mmgr.cnxt[cid].head;
}

static MCTL *
getMctl(const int addr, const int reqsiz)
{
  MCTL *mctlP;
  
  mctlP = (MCTL *)addr;
  bzero(mctlP, sizeof(MCTL) + reqsiz);
  mctlP->begin = addr + sizeof(MCTL);
  mctlP->end = mctlP->begin + reqsiz;
  mctlP->next = NULL;

  return mctlP;
}

static int
incrementCnxtid(unsigned int cid)
{
  if (++cid == NUM_CNXT) 
    cid = 0;
  return cid;
}

static void
cnxtSwitch(void)
{
  while ((Mmgr.cid == 0) && (CidLBFM != 0)) {
    usleep(1);
  }
  Mmgr.cid = incrementCnxtid(Mmgr.cid);
}

static void
logTransfer(int sockfd, unsigned int cid)
{
  int siz = REMOTE_MEMORY_SIZE;

  sendback(sockfd, &siz, sizeof(int), 0);
  sendback(sockfd, Mmgr.cnxt[cid].buffPool, siz, 0);
}

static void
endLogTransfer(int sockfd)
{
  int siz = -1;

  sendback(sockfd, &siz, sizeof(int), 0);
}

static void
writeBuffPool(char *p, unsigned long siz)
{
  int fd;
  char fname[BUFFSIZE];
  struct timeval now;
  
  gettimeofday(&now, NULL);
  bzero(fname, BUFFSIZE);
  sprintf(fname, "%s/%ld.%ld", CHKPTRLOGDIR, now.tv_sec, now.tv_usec);
  if ((fd = open(fname, O_WRONLY|O_CREAT|O_TRUNC|O_EXLOCK, 0644)) == -1) ERR;
  if (write(fd, p, siz) == -1) ERR;
  if (close(fd) == -1) ERR;
}

static void
writeInsertObj(char *p, unsigned long siz)
{
  int fd;
  char fname[BUFFSIZE];
  struct timeval now;
  MCTL mctl;
  
  bzero(&mctl, sizeof(MCTL));
  mctl.begin = (unsigned long)&mctl + (unsigned long)sizeof(MCTL);
  mctl.end = (unsigned long)&mctl + (unsigned long)sizeof(MCTL) + siz;
  gettimeofday(&now, NULL);
  bzero(fname, BUFFSIZE);
  sprintf(fname, "%s/%ld.%ld", CHKPTRLOGDIR, now.tv_sec, now.tv_usec);
  if ((fd = open(fname, O_WRONLY|O_CREAT|O_TRUNC|O_EXLOCK, 0644)) == -1) ERR;
  if (write(fd, &mctl, sizeof(MCTL)) == -1) ERR;
  if (write(fd, p, siz) == -1) ERR;
  if (close(fd) == -1) ERR;
}

static void
clearBuffPool(unsigned const int cid)
{
  bzero(Mmgr.cnxt[cid].buffPool, REMOTE_MEMORY_SIZE);
  initHeadMctl(cid);
}

static void
initCnxt(const int cid)
{
  initHeadMctl(cid);  
  initLockCnxt(cid);
}

static void
initMmgr(int siz)
{
  int i;
  
  if (Mmgr.thread != (pthread_t)NULL) {
    if (pthread_rwlock_destroy(&Mmgr.lock)) ERR;    
    if (close(Mmgr.chkPtrfd) == -1) ERR;
    for (i = 0; i < NUM_CNXT; i ++) {
      if (pthread_rwlock_destroy(&Mmgr.cnxt[i].lock)) ERR;    
      free(Mmgr.cnxt[i].buffPool);
    }
    pthread_cancel(Mmgr.thread); /* No need to be exit */
    Mmgr.thread = NULL;
  }

  bzero(&Mmgr, sizeof(MMGR));
  if (pthread_rwlock_init(&Mmgr.lock, NULL)) ERR;
  for (i = 0; i < NUM_CNXT; i ++) {
    allocBuffPool(siz, i);
    initCnxt(i);
  }
}

static void
initLogBufFileReader(void)
{
  initMmgr(REMOTE_MEMORY_SIZE);
  Mmgr.thread = pthread_self();
  Mmgr.chkPtrfd = connectCli(DBMS_CHKPTR_PORT, DBMS_HOST);
}

static void 
compressSend(char path[])
{
  FILE *fin;
  int flush, status;
  z_stream z;                     
  char *inbuf, *outbuf;
  int ack;

  /*
   * Prepare
   */
  if ((fin = fopen(path, "r")) == NULL) ERR;
  if ((inbuf = calloc(1, INBUFSIZ)) == NULL) ERR;
  if ((outbuf = calloc(1, OUTBUFSIZ)) == NULL) ERR;
  
  z.zalloc = Z_NULL;
  z.zfree = Z_NULL;
  z.opaque = Z_NULL;
  
  if (deflateInit(&z, Z_DEFAULT_COMPRESSION) != Z_OK) {
    fprintf(stderr, "deflateInit: %s\n", (z.msg) ? z.msg : "???");
    exit(1);
  }

  z.avail_in = 0;             
  z.next_out = outbuf;        
  z.avail_out = OUTBUFSIZ;    
  
  flush = Z_NO_FLUSH;

  /*
   * Compress
   */
  while (1) {
    if (z.avail_in == 0) {  
      z.next_in = inbuf;  
      z.avail_in = fread(inbuf, 1, INBUFSIZ, fin); 
      if (z.avail_in < INBUFSIZ) 
        flush = Z_FINISH;
    }
    status = deflate(&z, flush); 
    if (status == Z_STREAM_END) 
      break; 
    if (status != Z_OK) {   
      fprintf(stderr, "deflate: %s\n", (z.msg) ? z.msg : "???");
      exit(1);
    }
    if (z.avail_out == 0) { 
      z.avail_out = OUTBUFSIZ;
      outbuf = realloc(outbuf, z.total_out + OUTBUFSIZ);
      z.next_out = &outbuf[z.total_out]; 
    }
  }

  if (deflateEnd(&z) != Z_OK) {
    fprintf(stderr, "deflateEnd: %s\n", (z.msg) ? z.msg : "???");
    exit(1);
  }

  /*
   * Transfer
   */
  sendback(Mmgr.chkPtrfd, &z.total_out, sizeof(uLong), 0);
  if (send(Mmgr.chkPtrfd, outbuf, z.total_out, 0) == -1) {
    close(Mmgr.chkPtrfd);
    Mmgr.chkPtrfd = 0;
    pthread_exit(NULL);
  }
  if (recv(Mmgr.chkPtrfd, &ack, sizeof(ack), 0) == -1) {
    close(Mmgr.chkPtrfd);
    Mmgr.chkPtrfd = 0;
    pthread_exit(NULL);
  }
  
  /*
   * Cleanup
   */
  free(inbuf);
  free(outbuf);
  fclose(fin);
}

static void *
logBufFileMaker(void)
{
  while (1) {
    while ((CidLBFM < Mmgr.cid) || (Mmgr.cid == 0 && CidLBFM != 0)) {
      if (pthread_rwlock_wrlock(&Mmgr.lock)) ERR;  
      if (pthread_rwlock_wrlock(&Mmgr.cnxt[CidLBFM].lock)) ERR;
      if (pthread_rwlock_unlock(&Mmgr.lock)) ERR;  
      writeBuffPool(Mmgr.cnxt[CidLBFM].buffPool, REMOTE_MEMORY_SIZE);
      clearBuffPool(CidLBFM);
      if (pthread_rwlock_unlock(&Mmgr.cnxt[CidLBFM].lock)) ERR;
      CidLBFM = incrementCnxtid(CidLBFM);
    }
    usleep(1);
  }
}

static void *
logBufFileReader(void)
{
  DIR *dirP;
  struct dirent *direntP;
  char path[BUFFSIZE];

  /* This loop is broken by recovery msg from DBMS srv*/
  for (initLogBufFileReader(); StopLogBufFileReader == FALSE; usleep(1000)) {
    if ((dirP = opendir(CHKPTRLOGDIR)) == NULL) {
      P(dirP);
      ERR;
    }
    while (1) {
      direntP = readdir(dirP);
      if (direntP == NULL) {
        break;
      }
      else if ((strcmp(direntP->d_name, ".") != 0) && (strcmp(direntP->d_name, "..") != 0)) {
        bzero(path, BUFFSIZE);
        sprintf(path, "%s/%s", CHKPTRLOGDIR, direntP->d_name);
        //compressSend(path);
        if (unlink(path) == -1) ERR;
      }
    }
    if (closedir(dirP) == -1) ERR;
  }

  return NULL;
}

/**************************************************************************************
 *
 * Public Function
 *
 **************************************************************************************/
/*
 * First fit policy.
 * However, I consider only whether does space exist on the tail or not.
 */

/* 
 * KGCPY is for Group Commit 
 */
extern void 
kgcpy(void *p, const unsigned long siz)
{
  assert(siz == REMOTE_MEMORY_SIZE);
  if (pthread_rwlock_wrlock(&Mmgr.lock)) ERR;  
  memcpy((void *)(Mmgr.cnxt[Mmgr.cid].buffPool), p, siz);
  cnxtSwitch();
  if (pthread_rwlock_unlock(&Mmgr.lock)) ERR;  
}  

/* 
 * KMCPY is for P2P Commit
 */
extern void 
kmcpy(void *p, const unsigned long siz)
{
  if (pthread_rwlock_wrlock(&Mmgr.lock)) ERR;  
  if (pthread_rwlock_wrlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
  if ((unsigned long)(Mmgr.cnxt[Mmgr.cid].tailP->end + sizeof(MCTL) + siz) > 
      (unsigned long)(Mmgr.cnxt[Mmgr.cid].buffPool + REMOTE_MEMORY_SIZE)) {
    if (pthread_rwlock_unlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
    cnxtSwitch();
    if (pthread_rwlock_wrlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
  }
  if (pthread_rwlock_unlock(&Mmgr.lock)) ERR;    
  Mmgr.cnxt[Mmgr.cid].tailP->next = getMctl(Mmgr.cnxt[Mmgr.cid].tailP->end, siz);
  memcpy((void *)((unsigned long)Mmgr.cnxt[Mmgr.cid].tailP->next + sizeof(MCTL)), p, siz);
  Mmgr.cnxt[Mmgr.cid].tailP->next->prev = Mmgr.cnxt[Mmgr.cid].tailP;
  Mmgr.cnxt[Mmgr.cid].tailP = Mmgr.cnxt[Mmgr.cid].tailP->next;
  if (pthread_rwlock_unlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
}  

extern void 
kdcpy(void *p, const unsigned long siz)
{
  if (pthread_rwlock_wrlock(&Mmgr.lock)) ERR;  
  if (pthread_rwlock_wrlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
  if (Mmgr.cnxt[Mmgr.cid].head.end != (unsigned long)Mmgr.cnxt[Mmgr.cid].buffPool) {
    writeBuffPool(Mmgr.cnxt[Mmgr.cid].buffPool, REMOTE_MEMORY_SIZE);
    clearBuffPool(Mmgr.cid);
  }
  if (pthread_rwlock_unlock(&Mmgr.lock)) ERR;    
  writeInsertObj(p, siz);
  if (pthread_rwlock_unlock(&Mmgr.cnxt[Mmgr.cid].lock)) ERR;
}  

extern void
stopLogBufFileReader(void)
{
  StopLogBufFileReader = TRUE;
}

extern void
stopLogBufFileMaker(void)
{
  StopLogBufFileMaker = TRUE;
}

extern void
createLogBufFileReader(void)
{
  pthread_t thread;

  StopLogBufFileReader = FALSE;
  if (pthread_create(&thread, NULL, (void *)logBufFileReader, NULL)) ERR;
  if (pthread_detach(thread)) ERR;
}

extern void
createLogBufFileMaker(void)
{
  pthread_t thread;

  StopLogBufFileMaker = FALSE;
  if (pthread_create(&thread, NULL, (void *)logBufFileMaker, NULL)) ERR;
  if (pthread_detach(thread)) ERR;
}

extern void
allLogTransfer(int acptfd)
{
  int msg, ack, siz;
  int fd;
  char path[BUFFSIZE];
  char buffPool[REMOTE_MEMORY_SIZE];
  DIR *dirP;
  struct dirent *direntP;

  if (recv(acptfd, &msg, sizeof(int), 0) == -1) ERR;

  if ((dirP = opendir(CHKPTRLOGDIR)) == NULL) {
    P(dirP);
    ERR;
  }
  while (1) {
    direntP = readdir(dirP);
    if (direntP == NULL) {
      break;
    }
    else if ((strcmp(direntP->d_name, ".") != 0) && (strcmp(direntP->d_name, "..") != 0)) {
      bzero(path, BUFFSIZE);
      sprintf(path, "%s/%s", CHKPTRLOGDIR, direntP->d_name);
      if ((fd = open(path, O_RDONLY|O_SHLOCK)) == -1) {
        S(direntP->d_name);
        ERR;
      }
      if (read(fd, buffPool, REMOTE_MEMORY_SIZE) == -1) ERR;
      if (close(fd) == -1) ERR;
      
      siz = REMOTE_MEMORY_SIZE;
      sendback(acptfd, &siz, sizeof(int), 0);
      if (send(acptfd, buffPool, siz, 0) == -1) ERR;
      if (recv(acptfd, &ack, sizeof(ack), 0) == -1) ERR;
      
      if (unlink(path) == -1) ERR;
    }
  }
  if (closedir(dirP) == -1) ERR;

  if (Mmgr.cid == 0) {
    if ((Mmgr.cnxt[1].tailP != &Mmgr.cnxt[1].head) && (Mmgr.cnxt[1].tailP != NULL)) {
      logTransfer(acptfd, 1);
    }
    if ((Mmgr.cnxt[0].tailP != &Mmgr.cnxt[0].head) && (Mmgr.cnxt[0].tailP != NULL)) {
      logTransfer(acptfd, 0);
    }
  }
  else {
    if ((Mmgr.cnxt[0].tailP != &Mmgr.cnxt[0].head) && (Mmgr.cnxt[0].tailP != NULL)) {
      logTransfer(acptfd, 0);
    }
    if ((Mmgr.cnxt[1].tailP != &Mmgr.cnxt[1].head) && (Mmgr.cnxt[1].tailP != NULL)) {
      logTransfer(acptfd, 1);
    }
  }
  endLogTransfer(acptfd);
  if (close(acptfd) == -1) ERR;
}
