#include "common.h"
#include "libWrapper.h"

#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _SELECT_FROM_H
#define _SELECT_FROM_H
#include "selectFrom.h"
#endif

#ifndef _SIMSEQ_H
#define _SIMSEQ_H
#include "simseq.h"
#endif

/********************************************
 *
 * Defines
 *
 *******************************************/
#define DELTAPARAM 10
#define EPSILONPARAM 1.5
typedef struct _EPDELTA {
  double epsilon;
  int delta;
} EPDELTA;

/********************************************
 *
 * Function
 *
 *******************************************/
static EPDELTA *
initEpsilonDelta(const int n)
{
  EPDELTA *epsilonDeltaP;
  if ((epsilonDeltaP = calloc(n, sizeof(EPDELTA))) == NULL) ERR;
  return epsilonDeltaP;
}

static EPDELTA *
getEpsilon(const double *queryP, const int queryNum)
{
  int i;
  EPDELTA *epsilonDeltaP;
  double querySumX = 0.0;
  double queryMeanX = 0.0;
  double querySumSqX = 0.0;
  double querySumSqMeanX = 0.0;
  double varianceX = 0.0;
  double normalizeX;

  epsilonDeltaP = initEpsilonDelta(queryNum);
  
  for (i = 0; i < queryNum; i ++) {
    /* Sum and Second Power Sum Calculatation */
    querySumX = querySumX + queryP[i];
    querySumSqX = querySumSqX + queryP[i] * queryP[i];
  }
    /* Mean */
  queryMeanX = querySumX / queryNum;
  querySumSqMeanX = querySumSqX / queryNum;
  
  /* Variance */
  varianceX = queryNum * (querySumSqMeanX - queryMeanX * queryMeanX) / (queryNum - 1) ;
  
  /* Normalization */
  normalizeX = sqrt(varianceX);
  epsilonDeltaP->epsilon = normalizeX / EPSILONPARAM;

  return epsilonDeltaP;
}

inline 
int getDelta(const int queryNum)
{
  return queryNum / DELTAPARAM;
}

static EPDELTA *
createEpsilonDelta(const double *queryP, const int queryNum)
{
  EPDELTA *epsilonDeltaP;
  
  epsilonDeltaP = getEpsilon(queryP, queryNum);
  epsilonDeltaP->delta = getDelta(queryNum);
  
  return epsilonDeltaP;
}

inline double 
max(const int a, const int b)
{
  return (double)((a > b) ? a : b); 
}

inline double 
min(const int a, const int b)
{
  return (double)((a < b) ? a : b); 
}

static double **
initLcssTable(const int baseNum)
{
  int i;
  double **tablePtr = NULL;
  for (i = 0; i < baseNum; i ++) {
    if ((tablePtr = calloc(baseNum, sizeof(int *))) == NULL) ERR;
  }
  for (i = 0; i < baseNum; i ++) {
    if ((*(tablePtr + i) = calloc(baseNum, sizeof(int))) == NULL) ERR;
  }
  return tablePtr;
}

static void 
freeLcssTable(double **tablePtr, const int baseNum)
{
  int i;
  for (i = baseNum - 1; i >= 0;  i --) {
    free(*(tablePtr + i));
  }
  free(tablePtr);
}

static double 
getMaxFromLcssTable(double *sPtr, const int size)
{
  int i;
  double maxVal = sPtr[0];
  for (i = 1; i < size; i ++) {
    if (sPtr[i] > maxVal) {
      maxVal = sPtr[i];
    }
  }
  return maxVal;
}

static double 
calcLcss(double *baseP, double *queryP, int baseNum)
{
  int i;
  int j; 
  int delta;
  double maxVal;
  double simVal;
  double **lcssTable;
  EPDELTA *epsilonDeltaP;

  epsilonDeltaP = createEpsilonDelta(queryP, baseNum);
  lcssTable = initLcssTable(baseNum + 1);
  delta = epsilonDeltaP->delta;
  for (i = 0; i < baseNum; i ++) {
    for (j = i - delta; j <= i + delta; j ++) {
      if (j < 0 || j >= baseNum) {
        continue;
      }
      else if ((queryP[j] + epsilonDeltaP->epsilon) >= baseP[i]) {
        lcssTable[i + 1][j + 1] = lcssTable[i][j] + 1.0;
      }
      else if (lcssTable[i][j + 1] > lcssTable[i + 1][j]) {
        lcssTable[i + 1][j + 1] = lcssTable[i][j + 1];
      }
      else {
        lcssTable[i + 1][j + 1] = lcssTable[i + 1][j];
      }
    }
  }
  maxVal = getMaxFromLcssTable(lcssTable[baseNum], baseNum + 1);
  simVal = maxVal / baseNum;
  freeLcssTable(lcssTable, baseNum + 1);
  free(epsilonDeltaP);

  return simVal;
}


extern double
getLcssDistance(const OLDWINDOW window, const SENSOR *pSensor)
{
  double distance;
  double *sequence = NULL;
  
  //sequence = setSequenceInDatabase(pSensor, window.numOfElem);
  distance = calcLcss(window.sequence, sequence, window.numOfElem);
  free(sequence);

  return distance;
}
