#include "common.h"
#include "buffsize.h"
#include "libWrapper.h"
#include <sys/wait.h>
#include "schema.h"
#define RECORD_BUFFSIZE 128
#define RECORD_DIR "/tmp/record_dir"
#define MEGA (1000 * 1000)

typedef struct _RECORD {
  struct timeval begin;
  struct timeval end;
  long long duration;
} RECORD;

static void
calcDurationMicroSecond(RECORD recordP[], int maxrec)
{
  int recid;

  for (recid = 0; recid < maxrec; recid ++) {
    recordP[recid].duration = (recordP[recid].end.tv_sec - recordP[recid].begin.tv_sec) * MEGA + (recordP[recid].end.tv_usec - recordP[recid].begin.tv_usec);
  }
}

static int
findMinRecid(const RECORD recordP[], const int maxrec)
{
  int minRecid;
  int recid;
  long diff;

  minRecid = 0;
  for (recid = 1; recid < maxrec; recid ++) {
    diff = (recordP[minRecid].begin.tv_sec - recordP[recid].begin.tv_sec) * MEGA + (recordP[minRecid].begin.tv_usec - recordP[recid].begin.tv_usec);
    if (diff > 0) {
      minRecid = recid;
    }
  }

  return minRecid;
}

static int
findMaxRecid(const RECORD recordP[], const int maxrec)
{
  int maxRecid;
  int recid;
  long diff;

  maxRecid = 0;
  for (recid = 1; recid < maxrec; recid ++) {
    diff = (recordP[maxRecid].end.tv_sec - recordP[recid].end.tv_sec) * MEGA + (recordP[maxRecid].end.tv_usec - recordP[recid].end.tv_usec);
    if (diff < 0) {
      maxRecid = recid;
    }
  }

  return maxRecid;
}

static void
getTotalExecutionTime(RECORD recordP[], const int maxrec)
{
  int minRecid;
  int maxRecid;

  minRecid = findMinRecid(recordP, maxrec);
  maxRecid = findMaxRecid(recordP, maxrec);
  printf("total execution time:     %ld\n", (recordP[maxRecid].end.tv_sec - recordP[minRecid].begin.tv_sec) * MEGA + (recordP[maxRecid].end.tv_usec - recordP[minRecid].begin.tv_usec));
}

static void
getBestOfDuration(const RECORD recordP[], const int maxrec)
{
  int bestRecid;
  int recid;

  bestRecid = 0;
  for (recid = 1; recid < maxrec; recid ++) {
    if (recordP[recid].duration < recordP[bestRecid].duration) {
      bestRecid = recid;
    }
  }
  printf("Best duration:            %lld(id=%d)\n", recordP[bestRecid].duration, bestRecid);
}

static void
getWorstOfDuration(const RECORD recordP[], const int maxrec)
{
  int worstRecid;
  int recid;

  worstRecid = 0;
  for (recid = 1; recid < maxrec; recid ++) {
    if (recordP[recid].duration > recordP[worstRecid].duration) {
      worstRecid = recid;
    }
  }
  printf("Worst duration:           %lld(id=%d)\n", recordP[worstRecid].duration, worstRecid);
}

static void
getAverageAndDistributionOfDuration(const RECORD recordP[], const int maxrec)
{
  int recid;
  long long sum;
  long long average;
  double distribution;
  double total;
  double diff;
  
  for (sum = 0, recid = 0; recid < maxrec; recid ++) {
    sum += recordP[recid].duration;
  }
  average = sum / maxrec;

  printf("Average duration:         %lld USEC\n", average);
  printf("Average duration:         %lld MSEC\n", average / 1000);

  for (total = 0.0, recid = 0; recid < maxrec; recid ++) {
    diff = (double)(recordP[recid].duration - average);
    total += diff*diff;
  }
  distribution = sqrt((double)total);
  printf("Distribution of duration: %f\n", distribution);
}

static void
getResultAboutDuration(RECORD recordP[], const int maxrec)
{
  calcDurationMicroSecond(recordP, maxrec);
  getBestOfDuration(recordP, maxrec);
  getWorstOfDuration(recordP, maxrec);
  getAverageAndDistributionOfDuration(recordP, maxrec);
}

extern int 
main(int argc, char *argv[])
{
  int recid;
  int maxrec;
  int numappend;
  char filename[RECORD_BUFFSIZE];
  FILE *fp;
  RECORD *recordP;

  if (argc != 3) {
    printf("Usage: program maxrec numappend\n");fflush(stdout);
    exit(1);
  }
  maxrec = atoi(argv[1]);
  numappend = atoi(argv[2]);
  if ((recordP = calloc(maxrec, sizeof(RECORD))) == NULL) ERR;
  for (recid = 0; recid < maxrec; recid ++) {
    bzero(filename, RECORD_BUFFSIZE);
    sprintf(filename, "%s/%d/%d/%d", RECORD_DIR, maxrec, numappend, recid+1);    
    if ((fp = fopen(filename, "r")) == NULL) ERR;
    fscanf(fp, "%ld %ld", &recordP[recid].begin.tv_sec, &recordP[recid].begin.tv_usec);
    fscanf(fp, "%ld %ld", &recordP[recid].end.tv_sec, &recordP[recid].end.tv_usec);
    fclose(fp);
  }

  getTotalExecutionTime(recordP, maxrec);
  getResultAboutDuration(recordP, maxrec);
  free(recordP);

  return 0;
}
