#include "common.h"
#include <ctype.h>
#include "libWrapper.h"

#ifndef _BUFFSIZE_H
#include "buffsize.h"
#define _BUFFSIZE_H
#endif

#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _SELECT_FROM_H
#define _SELECT_FROM_H
#include "selectFrom.h"
#endif

#ifndef _LOCK_MODE_H
#define _LOCK_MODE_H
#include "lockMode.h"
#endif

#ifndef _SENSOR_IMAGE_SIZE_H
#define _SENSOR_IMAGE_SIZE_H
#include "sensorImageSize.h"
#endif

/********************
 *
 * Declaration
 *
 ********************/
extern void releaseCondition(CONDITION headCondition);
extern void unlockSchemaList(SCHEMA_LIST headSchemaList);
extern DBPAGE getStartRelPage(SCHEMA *schemaP);
extern DBPAGE getNextPage(SCHEMA *schemaP, DBPAGE page);
extern DBPAGE getStartSenPage(SCHEMA *schemaP, int attrid, int tplid);
extern ANSWER execSelection(const CONDITION *cndP, ANSWER answer);
extern TUPLE releaseTrashTupleForSelectOrDeleteOrUpdateOrAppend(ANSWER answer);
extern TUPLE sortAnswerTupleByOrder(ANSWER answer, const ORDER_INFO orderInfo);
extern double compEuclid(SELECT_ATTR *sap, SENSOR *sp, const int nt);
extern double compDTW(SELECT_ATTR *sap, SENSOR *sp, const int nt);
extern double compLCS(SELECT_ATTR *sap, SENSOR *sp, const int nt);
extern int fft(double RealData[], double ImageData[], int n, int flag, int normalize);
extern double compCorrel(SELECT_ATTR *sap, SENSOR *sp, const int nt);

/****************************************************************
 *
 * Global variables
 *
 ***************************************************************/
extern SCHEMA HeadSchema;

/****************************************************************
 *
 * Private Function
 *
 ***************************************************************/
static int
addAttrSize(ATTR_TYPE attrType)
{
  int size;

  switch (attrType) {

  case INT:      
    size = sizeof(int);    
    break;

  case DOUBLE:   
    size = sizeof(double); 
    break;

  case TEXT:     
    size = BUFFSIZE;       
    break;

  case SENSOR_SEGMENT_DOUBLE:
    size = sizeof(SENSOR_SEGMENT);      
    break;

  default:
    ERR;
    break;
  }

  return size;
}

static void
releaseSelectAttr(SELECT_ATTR headSelectAttr)
{
  SELECT_ATTR *pSelectAttr;

  while (headSelectAttr.next != NULL) {
    pSelectAttr = headSelectAttr.next;
    headSelectAttr.next = headSelectAttr.next->next;
    free(pSelectAttr);
  }
}

static void
releaseFromSchema(FROM_SCHEMA head)
{
  FROM_SCHEMA *p;

  while (head.next != NULL) {
    p = head.next;
    head.next = head.next->next;
    free(p);
  }
}

static int
getSensorArrayInfo(int o, int *saidp, const char query[], char an[])
{
	int l, r;
	char buff[BUFSIZ];

	while (isspace(query[o])) {
		o++;
		if (o == QUERY_BUFFSIZE) 
			return -1;
	}

	l = o;

	while (query[o] != ')') { /* max (...')' */

		if (query[o] == '[') { /* Sensor Type */

			/* said info */
			l = r = o;
			l--;
			while (isalpha(query[l]) || isdigit(query[l]))
				l--;
			
			l++; /* First character */
			strncpy(an, &query[l], r - l);
			
			/* said info */
			o++; /* skip '[' */
			while (isspace(query[o]))
				o++;
			l = o;
			while (!isspace(query[o]) && query[o] != ']')
				o++;
			r = o; /* ']' */
			bzero(buff, sizeof(buff));
			strncpy(buff, &query[l], r - l);
			*saidp = atoi(buff);
			o++; /* skip ']' */

			return o; 
		}
		o++;
		if (o == QUERY_BUFFSIZE) 
			return -1;
	}

	S(&query[o]);

	while (isspace(query[o]))
		o--;
	r = ++o;
	bzero(an, sizeof(an));
	strncpy(an, &query[l], r - l);

	return o;
}

static int
parsefunc(SELECT_ATTR *p, int o, char fn[], const char query[])
{
	if (strncmp(fn, "max", strlen(fn)) == 0) p->f = MAX;
	else if (strncmp(fn, "min", strlen(fn)) == 0) p->f = MIN;
	else if (strncmp(fn, "pow", strlen(fn)) == 0) p->f = POW;
	else if (strncmp(fn, "ftr", strlen(fn)) == 0) p->f = FTR;
	else if (strncmp(fn, "fti", strlen(fn)) == 0) p->f = FTI;
	else if (strncmp(fn, "cnt", strlen(fn)) == 0) p->f = CNT;
	else if (strncmp(fn, "avg", strlen(fn)) == 0) p->f = AVG;
	else if (strncmp(fn, "sum", strlen(fn)) == 0) p->f = SUM;
	else if (strncmp(fn, "euc", strlen(fn)) == 0) p->f = EUC;
	else if (strncmp(fn, "dtw", strlen(fn)) == 0) p->f = DTW;
	else if (strncmp(fn, "lcs", strlen(fn)) == 0) p->f = LCS;
	else if (strncmp(fn, "cor", strlen(fn)) == 0) p->f = COR;
	else ERR;

	/*    #
	 * max( 
	 */
	if ((o = getSensorArrayInfo(o, &p->said, query, p->attrName)) == -1)
		return -1;
	else 
		return o;
}

static int
getWindowNum(const char query[], int *op)
{
	int o = *op, l;
	char buff[BUFSIZ];

	for (; query[o] != ','; o++) {
		if (o == QUERY_BUFFSIZE)
			return -1;
	}
	
	for (++o; /* skip ',' */
			 !isdigit(query[o]); o++) {
		if (o == QUERY_BUFFSIZE)
			return -1;
	}
	
	for (l = o; /* skip ',' */
			 isdigit(query[o]); o++) {
		if (o == QUERY_BUFFSIZE)
			return -1;
	}
	bzero(buff, sizeof(buff));
	strncpy(buff, &query[l], o - l);

	*op = o;
	return atoi(buff);
}

static int
getFuncName(const char query[], const int right, char buff[])
{
	int o, l;

	/*              #
	 * select id, sw(cnt(0[0]), 2, 1) from r1
	 */
	for (o = right; query[o] != '('; o++) {
		if (o == QUERY_BUFFSIZE)
			return -1;
	}
	for (++o; /* skip '(' */
			 isspace(query[o]);	o++) {
		if (o == QUERY_BUFFSIZE)
			return -1;
	}
	for (l = o; query[o] != '('; o++) {
		if (o == QUERY_BUFFSIZE)
			return -1;
	}
	bzero(buff, BUFSIZ);
	strncpy(buff, &query[l], o - l);
	
	return o;
}

static SELECT_ATTR *
setNewAtr(const char query[], int left, int right, int *op)
{
	int o = *op;
  SELECT_ATTR *p;
  BOOLEAN isSchema;
  char buff[BUFSIZ];

	/* init */
  if ((p = calloc(1, sizeof(SELECT_ATTR))) == NULL) ERR;
	p->w = NUW;

  for (isSchema = FALSE, o = left; o < right; o++) {
    if (query[o] == '.') {
      isSchema = TRUE; 
      break;
    }
  }

  switch (isSchema) {

  case TRUE: 
    strncpy(p->schemaName, &query[left], o - left);
    strncpy(p->attrName, &query[o + 1], right - (o + 1));
    break;

  case FALSE: /* Attriubute or function or window */
    bzero(buff, sizeof(buff));
    strncpy(buff, &query[left], right - left); 

		if (strncmp(buff, "lw", strlen(buff)) == 0) {
			p->w = LST; /* Last Window    */
			if ((o = getFuncName(query, right, buff)) == -1)
				return NULL;				
			if ((o = parsefunc(p, o, buff, query)) == -1) 
				return NULL;

			switch (p->f) {
			case EUC:
			case DTW:
			case LCS:
			case COR:
				for (; query[o] != ','; o++) 
					if (o == QUERY_BUFFSIZE)
						return NULL;
				if ((o = getSensorArrayInfo(++o, &p->s_said, query, p->s_an)) == -1)
					return NULL;
				if ((p->nt = getWindowNum(query, &o)) == -1) {
					N;
					return NULL;
				}
				
				*op = o;
				return p;
				break;

			case POW:
			case MAX:
			case MIN:
			case CNT:
			case FTR:
			case FTI:
			case SUM:
			case AVG:
				if ((p->nt = getWindowNum(query, &o)) == -1) 
					return NULL;
				*op = o;
				return p;
				break;

			default:
				ERR; /* Never come here, just make compiler quiet */
				break;
			}
		}
    else if (strncmp(buff, "sw", strlen(buff)) == 0) {
			p->w = SLD; /* Sliding Window */

			if ((o = getFuncName(query, right, buff)) == -1)
				return NULL;				
			if ((o = parsefunc(p, o, buff, query)) == -1) 
				return NULL;

			switch (p->f) {
			case EUC:
			case DTW:
			case LCS:
			case COR:
				for (; query[o] != ','; o++) {
					if (o == QUERY_BUFFSIZE)
						return NULL;
				}
				if ((o = getSensorArrayInfo(++o, &p->s_said, query, p->s_an)) == -1)
					return NULL;

				if ((p->ns = getWindowNum(query, &o)) == -1) 
					return NULL;

				*op = o;
				return p;
				break;

			case POW:
			case MAX:
			case MIN:
			case FTR:
			case FTI:
			case CNT:
			case SUM:
			case AVG:
				if ((p->nt = getWindowNum(query, &o)) == -1) 
					return NULL;
				if ((p->ns = getWindowNum(query, &o)) == -1) 
					return NULL;
				*op = o;
				return p;
				break;

			default:
				ERR; /* Never come here, just make compiler quiet */
				break;
			}
		}
    else if ((strncmp(buff, "max", strlen(buff)) == 0) ||
						 (strncmp(buff, "min", strlen(buff)) == 0) ||
						 (strncmp(buff, "avg", strlen(buff)) == 0) ||
						 (strncmp(buff, "ftr", strlen(buff)) == 0) ||
						 (strncmp(buff, "fti", strlen(buff)) == 0) ||
						 (strncmp(buff, "sum", strlen(buff)) == 0) ||
						 (strncmp(buff, "cnt", strlen(buff)) == 0) ||
						 (strncmp(buff, "euc", strlen(buff)) == 0) ||
						 (strncmp(buff, "lcs", strlen(buff)) == 0) ||
						 (strncmp(buff, "dtw", strlen(buff)) == 0) ||
						 (strncmp(buff, "cor", strlen(buff)) == 0)) {
			o = parsefunc(p, ++o, buff, query);

			*op = o;
			return p;
    }
    else {
      p->f = NUF;
      strcpy(p->attrName, buff);
    }

    break;
  }

	*op = o;
  return p;
}

static SELECT_ATTR
getSelAtr(const char query[])
{
  int left = 0, right = 0, offset = 0;
  SELECT_ATTR head, *p;
  
  for (offset = 0; isspace(query[offset]); offset++) {
    if (offset == QUERY_BUFFSIZE) {
      head.err = TRUE; 
      return head;
    }
  }

  if (strncmp(&query[offset], "select", strlen("select")) == 0) 
    offset += strlen("select");
  else if (strncmp(&query[offset], "delete", strlen("delete")) == 0) 
    offset += strlen("delete");
  else {
    head.err = TRUE;
    return head;
  }
  
  for ( ;isspace(query[offset]); offset++) {
    if (offset == QUERY_BUFFSIZE) {
      head.err = TRUE; 
      return head;
    }
  }

  /*
   * Getting attributes & functions (max, min, count, ...)
   */
  bzero(&head, sizeof(SELECT_ATTR));
  for (left = offset, p = &head;
       offset <= QUERY_BUFFSIZE;
       left = offset) {

    /* Error check */
    if (offset == QUERY_BUFFSIZE) {
      head.err = TRUE; 
      return head;
    }

    for (; offset < QUERY_BUFFSIZE; offset++){
      if ((query[offset] == ' ') || 
          (query[offset] == ',') ||
          (query[offset] == '(')) { /* For functions such as "max" */
				right = offset;
        break;
      }
    } 

    if (offset == QUERY_BUFFSIZE) {
      head.err = TRUE; 
      return head;
    }
    else {
      if ((p->next = setNewAtr(query, left, right, &offset)) == NULL) {
				free(p);
        head.err = TRUE; 
        return head;
      }
      p = p->next;
    }

    while (1) {
      if (query[offset] == ',') {
        offset++;
        while (isspace(query[offset]))
          offset++;
        break;
      }
      else if (strncmp(&query[offset], "from", strlen("from")) == 0) 
        break;
      else 
        offset++;
    }

    if (strncmp(&query[offset], "from", strlen("from")) == 0) {
      break;
    }
    else if (query[offset] == '\n' || query[offset] == '\0') {
      head.err = TRUE; 
      return head;
    }
  }

  head.err = FALSE; /* No parse error */
  return head;
}

extern FROM_SCHEMA
getFromSchema(const char query[])
{
  size_t offset = 0, left = 0, right = 0;
  FROM_SCHEMA head, *p;

  /* Init */
  bzero(&head, sizeof(FROM_SCHEMA));

  for (offset = 0; offset < ((size_t)QUERY_BUFFSIZE - strlen("from")); offset++) {
    if (strncmp(&query[offset], "from", strlen("from")) == 0) 
      break;
  }
  if (offset == QUERY_BUFFSIZE) {
    head.err = TRUE;
    return head;
  }
  else {
    offset += strlen("from");
    for (; query[offset] == ' '; offset++) 
      ;
  }
  if ((query[offset] == '\n') || (query[offset] == '\0')) {
    head.err = TRUE;
    return head;
  }

  p = &head; while ((query[offset] != '\n') &&
         (query[offset] != '\0') &&
         (strncmp(&query[offset], "where", strlen("where")) != 0) &&
         (strncmp(&query[offset], "order", strlen("order")) != 0)) {

    for (;query[offset] == ' '; offset++) 
      ;

    for (left = offset; offset < QUERY_BUFFSIZE; offset++) {
      if ((query[offset] == ',')  ||
          (query[offset] == '\n') ||
          (query[offset] == '\0') ||
          (query[offset] == 'w' && strncmp(&query[offset], "where", strlen("where")) == 0) || 
          (query[offset] == ' ')) {
        right = offset;
        if (query[offset] == ',') 
          offset++;
        break;
      }
    }
    if (offset == QUERY_BUFFSIZE) {
      head.err = TRUE;
      return head;
    }
    else if (strncmp(&query[offset], "where", strlen("where")) != 0) {
      if ((p->next = calloc(1, sizeof(FROM_SCHEMA))) == NULL) ERR;
      p = p->next;
      strncpy(p->schemaName, &query[left], right-left);
    }
  }

  head.err = FALSE;  
  head.offset = offset;

  return head;
}

static int
parse4bin(CONDITION *cndP, const char query[], int buffid)
{
  int i;
  int left, right;
  char buff[BUFSIZ];

  while (query[buffid] == ' ') buffid++; left = buffid;
  while (query[buffid] != ' ') buffid++; right = buffid;
  strncpy(cndP->u.binary.leftText, &query[left], right-left);
  for (i = 0; i < (right - left); i++) {
    if (cndP->u.binary.leftText[i] == '[') {
      cndP->u.binary.leftText[i] = '\0';
      while (!isdigit(cndP->u.binary.leftText[i])) i++;
      for (left = i; isdigit(cndP->u.binary.leftText[i]); i++) ;
      bzero(buff, BUFSIZ);
      strncpy(buff, &cndP->u.binary.leftText[left], i - left);
      cndP->u.binary.said = atoi(buff);
      break;
    }
  }

  /* Skip spaces */
  while (query[buffid] == ' ') 
    buffid++;
  
  if (query[buffid] == '=') {
    cndP->u.binary.operand = EQUAL;
    buffid += strlen("=");
  }
  else if (query[buffid] == '<') {
    if (query[buffid+1] == '=') {
      cndP->u.binary.operand = LESS_EQUAL;
      buffid += strlen("<=");
    }
    else if (query[buffid+1] == '>') {
      cndP->u.binary.operand = NOT_EQUAL;
      buffid += strlen("<>");
    }
    else {
      cndP->u.binary.operand = LESS;
      buffid += strlen("<");
    }
  }
  else if (query[buffid] == '>') {
    if (query[buffid+1] == '=') {
      cndP->u.binary.operand = GREATER_EQUAL;
      buffid += strlen(">=");
    }
    else {
      cndP->u.binary.operand = GREATER;
      buffid += strlen(">");
    }
  }
  else {
    printf("No such operand\n");
    exit(1);
  }
  
  /* Right */
  while (query[buffid] == ' ') 
    buffid++; 
  left = buffid;

  while ((query[buffid] != ' ')  && 
         (query[buffid] != '\0') && 
         (query[buffid] != '\n') && 
         (query[buffid] != ';')) 
    buffid++; 
  right = buffid;

  strncpy(cndP->u.binary.rightText, &query[left], right - left);
  
  return buffid;
}

static int
parse4sim(CONDITION *cndP, const char query[], int o)
{
  int l, r;
  char buff[BUFFSIZE];

  while (query[o] != '(') o++;
  o++; /* skip '(' */

  /* 
   * Schema Name 
   */
  while (isspace(query[o])) {
    o++;
    if (o == QUERY_BUFFSIZE)
      return -1;
  }
  l = o;

  while (query[o] != '.') {
    o++;
    if (o == QUERY_BUFFSIZE)
      return -1;
  }
  r = o;
  strncpy(cndP->u.sim.sn, &query[l], r - l); /* Getting schema name */

  /* 
   * Attribute Name 
   */
  l = ++o; /* Skip '.' */
  while (!isspace(query[o]) && query[o] != '[') {
    o++;
    if (o == QUERY_BUFFSIZE)
      return -1;
  }
  r = o;
  strncpy(cndP->u.sim.an, &query[l], r - l);

  /* 
   * Getting Array ID: "sim(event.s[3] , W)" 
   *                                ^
   */
  while (query[o] != '[') {
    o++;
    if (o == QUERY_BUFFSIZE)
      return -1;
  }
  while (!isdigit(query[o])) {
    o++;
    if (o == QUERY_BUFFSIZE)
      return -1;
  }
  l = o;
  while (isdigit(query[o])) {
    o++;
    if (o == QUERY_BUFFSIZE)
      return -1;
  }
  r = o;
  bzero(buff, sizeof(buff));
  strncpy(buff, &query[l], r - l);
  cndP->u.sim.said = atoi(buff); /* ID of sensor */

  /* 
   * skip ',' in "sim(event.s , W)" 
   */
  while (query[o] != ',') o++;  
  o++; /* skip ',' */
  while (isspace(query[o])) o++;

  /* 
   * Window name
   */
  l = o; 
  while (!isspace(query[o]) && query[o] != ')') o++;
  r = o;
  strncpy(cndP->u.sim.wn, &query[l], r - l);

  /*
   * Metric
   */
  cndP->u.sim.mt = WARPING;

  /*
   * skip ')' in "sim(event.s , W)" 
   */
  while (query[o] != ')') o++;  
  o++; /* skip ')' */
  while (isspace(query[o])) o++;

  /*
   * Operand
   */
  if (query[o] == '=') {
    cndP->u.sim.op = EQUAL;
  }
  else if (query[o] == '<') {
    if (query[++o] == '>') 
      cndP->u.sim.op = NOT_EQUAL;      
    else if (query[o] == '=') 
      cndP->u.sim.op = LESS_EQUAL;      
    else 
      cndP->u.sim.op = LESS;      
  }
  else if (query[o++] == '>') {
    if (query[++o] == '=') 
      cndP->u.sim.op = GREATER_EQUAL;      
    else 
      cndP->u.sim.op = GREATER;      
  }
  else {
    ERR;
  }
  o++;

  /*
   * Threshold
   */
  while (isspace(query[o])) o++;
  l = o;
  while (!isspace(query[o]) && query[o] != '\0') o++;
  r = o;
  bzero(buff, BUFFSIZE);
  strncpy(buff, &query[l], r - l);
  cndP->u.sim.th = atof(buff);

  return o;
}

static int
getandor(const char query[], int buffid, ANDOR *andor)
{
  /* Andor(Connectivities) */
  if (query[buffid] == ' ') {
    while (query[buffid] == ' ') {
      buffid++;
    }
    if ((query[buffid] == 'a') && (strncmp(&query[buffid], "and", strlen("and")) == 0)) {
      *andor = AND;
      buffid += strlen("and");
    }
    else if ((query[buffid] == 'o') && (strncmp(&query[buffid], "or", strlen("or")) == 0)) {
      *andor = OR;
      buffid += strlen("or");
    }
  }

  return buffid;
}

static TYPE_OF_CONDITION
getTypeOfCondition(const char query[], const int o)
{
  if (!strncmp(&query[o], "simseq", strlen("simseq"))) {
    return SIMSEQ;
  }
  else {
    return BINARY;
  }
}

static CONDITION
parse4getcnd(const char query[], int o)
{
  CONDITION head, *p = NULL;
  ANDOR andor = OR;

  bzero(&head, sizeof(CONDITION));
  for (p = &head;
       query[o] != '\0' && query[o] != '\n' && query[o] != ';' && o < QUERY_BUFFSIZE; ) {
    if ((p->next = calloc(1, sizeof(CONDITION))) == NULL) ERR;
    p = p->next;
    p->andor = andor;
    p->typeOfCondition = getTypeOfCondition(query, o);
    switch (p->typeOfCondition) {
    case BINARY: 
      o = parse4bin(p, query, o); 
      break;
    case SIMSEQ: 
      if ((o = parse4sim(p, query, o)) == -1) { /* Error check */
        head.err = TRUE;
        return head;
      }
      break;
    default:
      break;
    }
    o = getandor(query, o, &andor);
  }

  return head;
}

static CONDITION
getcnd(const char query[])
{
  int buffid;
  CONDITION hcnd; /* Head Condition */

  /* Skip [WHERE] */
  bzero(&hcnd, sizeof(CONDITION));
  for (buffid = 0; buffid < QUERY_BUFFSIZE; buffid++) {
    if ((query[buffid] == 'w') && (strncmp(&query[buffid], "where", strlen("where")) == 0)) {
      buffid += strlen("where");
      while (query[buffid] == ' ') 
        buffid++;
      break;
    }
  }
  if (buffid == QUERY_BUFFSIZE) {
    hcnd.err = TRUE;
  }
  else {
    hcnd.err = FALSE;
    hcnd = parse4getcnd(query, buffid);
    if (hcnd.err == TRUE)
      return hcnd;
  }

  return hcnd;
}

static int
getPrjByteOfTupleInExecProjection(const int prjNumOfAttr, const ATTR *pPrjAttr)
{
  int i, prjByteOfTuple;

  for (prjByteOfTuple = 0, i = 0; i < prjNumOfAttr; i++) 
    prjByteOfTuple += pPrjAttr[i].sizOfAttr;

  return prjByteOfTuple;
}

extern ANSWER
execProjection(const SELECT_ATTR hsa, ANSWER answer)
{
  int i = 0;
  int prjsz = 0;  /* Size of Projection Tuple */
  int orgbo = 0, prjbo = 0; /* bo: byte offset */
  int orgaid = 0, prjaid = 0;
  int prjnba = 0; /* Projected, number of attrs */
  SELECT_ATTR *psa = NULL;
  TUPLE *pAnswerTuple = NULL;
  ATTR *prjatrP = NULL;
  void *tmp = NULL;

  /* Initialization */
  answer.parseMsg = PARSE_OK;

  if (strcmp(hsa.next->attrName, "*") == 0) 
    return answer;

  for (prjnba = 0, psa = hsa.next;
       psa != NULL; 
       psa = psa->next) {
    prjnba++;

    for (orgaid = 0; orgaid <= answer.numOfAttr; orgaid++) {
			if (orgaid == answer.numOfAttr) { /* Error check */
				answer.parseMsg = WRONG_ATTRIBUTE;
				return answer;
			}
      else if (strncmp(psa->attrName, answer.attr[orgaid].attrName, strlen(psa->attrName)) == 0) {
        answer.attr[orgaid].projected = TRUE;
        break;
      }
    }
	}

  if ((prjatrP = calloc(prjnba, sizeof(ATTR))) == NULL) ERR;
  for (prjaid = orgaid = 0; orgaid < answer.numOfAttr; orgaid++) {
    if (answer.attr[orgaid].projected == TRUE) {
      memcpy(&prjatrP[prjaid], &answer.attr[orgaid], sizeof(ATTR));
      prjaid++;
    }
  } 
  
  prjsz = getPrjByteOfTupleInExecProjection(prjnba, prjatrP);

  if ((tmp = calloc(1, prjsz)) == NULL) ERR;
  for (pAnswerTuple = answer.headTuple.next; 
       pAnswerTuple != NULL; 
       pAnswerTuple = pAnswerTuple->next) {
    for (orgaid = 0, prjaid = 0; 
         prjaid < prjnba; 
         orgaid++, prjaid++) {

			while (answer.attr[orgaid].projected == FALSE && orgaid <= answer.numOfAttr) {
				if (orgaid == answer.numOfAttr)
					ERR; /* This is fatal */
				else
					orgaid++;
			}

			for (orgbo = i = 0; i < orgaid; i++) 
        orgbo += answer.attr[i].sizOfAttr;
			
      for (prjbo = i = 0; i < prjaid; i++) 
        prjbo += prjatrP[i].sizOfAttr;

      switch (answer.attr[orgaid].attrType) {

      case INT:    
        memcpy(tmp + prjbo, pAnswerTuple->obj + orgbo, sizeof(INT));    
        break;

      case DOUBLE: 
        memcpy(tmp + prjbo, pAnswerTuple->obj + orgbo, sizeof(DOUBLE)); 
        break;

      case TEXT:   
        memcpy(tmp + prjbo, pAnswerTuple->obj + orgbo, BUFFSIZE);       
        break;

      case SENSOR_SEGMENT_DOUBLE: 
        memcpy(tmp + prjbo, pAnswerTuple->obj + orgbo, sizeof(SENSOR_SEGMENT));
        break;

      default:
        ERR;
        break;
      }
    } 
    free(pAnswerTuple->obj);

    if ((pAnswerTuple->obj = calloc(1, prjsz)) == NULL) ERR;
    memcpy(pAnswerTuple->obj, tmp, prjsz);
  }
  
  free(tmp);
  free(answer.attr);
  answer.numOfAttr = prjnba;
  answer.attr = prjatrP;

  return answer;
}

extern TUPLE 
relesaeTrashTupleForSelectOrDeleteOrUpdateOrAppend(ANSWER answer)
{
  TUPLE *p, *trash, *before;

  for (before = &answer.headTuple, p = before->next; p != NULL; p = before->next) {
    if (p->selected == FALSE) {
      trash = p;
      before->next = p->next;
      free(trash->obj);
      free(trash);
    }
    else {
      before = before->next;
    }
  }
  
  return answer.headTuple;
}

static ANSWER
exebin(CONDITION *cndP, ANSWER answer)
{
  int i;

  for (i = 0; i < answer.numOfAttr; i++) {  
    if (strcmp(cndP->u.binary.leftText, answer.attr[i].attrName) == 0) {
      cndP->u.binary.aid = i;
      break;
    }
  }

  return execSelection(cndP, answer);        
}

extern ANSWER
invokeSelection(const CONDITION head, ANSWER answer)
{
  CONDITION *p;

  for (p = head.next; p != NULL; p = p->next) {
    switch (p->typeOfCondition) {
    case BINARY: 
      answer = exebin(p, answer); 
      break;
    default:
			ERR;
      break;
    }
  }

  return answer;
}     

static void
page2tuple(TUPLE *tplP, const int sizTpl, SCHEMA *schemaP, const DBPAGE rpage, int oidInPage)
{
  int i, j, offset;
  SENSOR_SEGMENT *ssp;
  SENSOR *sp;
  DBPAGE page;
  char *p; /* for sensor data objects */

  /* For Relation */
  memcpy((tplP->obj + sizTpl), &rpage.data[schemaP->sizOfObj * oidInPage], schemaP->sizOfObj);
  
  /* For Sensor */
  for (offset = i = 0; i < schemaP->numOfAttr; offset += addAttrSize(schemaP->attr[i].attrType), i++) {
    if (schemaP->attr[i].attrType == SENSOR_SEGMENT_DOUBLE) {
      ssp = (SENSOR_SEGMENT *)(tplP->obj + sizTpl + offset);

      /* 
       * "NUF" and "NUW" shows default cases.
			 * NUF: Not Use Function
			 * NUW: Not Use Window 
       * If function is used, then this parameter will be changed 
       * before execProjection 
       */

      if ((ssp->next = calloc(1, sizeof(SENSOR_SEGMENT))) == NULL) ERR;
      ssp = ssp->next;
			ssp->f = NUF; 
			ssp->w = NUW; 
      ssp->attrid = i;
      for (sp = &ssp->headSensor, page = getStartSenPage(schemaP, i, rpage.pi.tupleid + oidInPage); 
           page.pi.offset != -1; 
           page = getNextPage(schemaP, page)) {
        for (p = page.data, j = 0; j < page.pi.numOfObj; j++) {
          if ((sp->next = calloc(1, sizeof(SENSOR))) == NULL) ERR;
					sp->next->prev = sp;
          sp = sp->next;
          memcpy(&sp->sdobj, p, sizeof(SDOBJ));

          p += sizeof(SDOBJ);
          if ((sp->sdobj.v = calloc(sp->sdobj.szsa, sizeof(double))) == NULL) ERR;
          memcpy(sp->sdobj.v, p, sizeof(double) * sp->sdobj.szsa);
          p += (sizeof(double) * sp->sdobj.szsa);
        }
      }
    }
  }
}

static TUPLE *
nestedLoopJoin(const int nbAttr, const int sizTpl, SCHEMA_LIST *slP)
{
  int i;
  int tplid;
  TUPLE head, *p;
  DBPAGE page;

  bzero(&head, sizeof(TUPLE));
  if (slP->next != NULL) { /* This is for join */
    for (tplid = 0, page = getStartRelPage(slP->pSchema); 
         page.pi.offset >= 0; 
         page = getNextPage(slP->pSchema, page)) {
      for (i = 0; i < page.pi.numOfObj; i++, tplid++) {
        if (slP->pSchema->garbageP == NULL || slP->pSchema->garbageP[i] == 0) { /* 1 means deleted */
					for (p = &head; p->next != NULL; p = p->next) {
						;
					}
					p->next = nestedLoopJoin((nbAttr + slP->pSchema->numOfAttr), (sizTpl + slP->pSchema->sizOfObj), slP->next);
					for (p = p->next; p != NULL; p = p->next) {
						page2tuple(p, sizTpl, slP->pSchema, page, i);
					}
				}
      }
    }
  }
  else {
    for (tplid = 0, p = &head, page = getStartRelPage(slP->pSchema); page.pi.offset >= 0; page = getNextPage(slP->pSchema, page)) {
      for (i = 0; i < page.pi.numOfObj; i++, tplid++) {
        if (slP->pSchema->garbageP == NULL || slP->pSchema->garbageP[tplid] == 0) { /* 1 means deleted */
					if ((p->next = calloc(1, sizeof(TUPLE))) == NULL) ERR;
					p->next->prev = p;
					p = p->next;
					if ((p->obj = calloc(1, (sizTpl + slP->pSchema->sizOfObj))) == NULL) ERR;
					page2tuple(p, sizTpl, slP->pSchema, page, i);
					p->selected = FALSE;
					p->certified = FALSE;
				}
			}
    }
  }

  return head.next;
}

extern ANSWER
execCartesianProduct(SCHEMA_LIST head)
{
  int i;
  ANSWER answer;
  SCHEMA_LIST *p;

  bzero(&answer, sizeof(ANSWER));
  for (p = head.next; p != NULL; p = p->next) 
    answer.numOfAttr += p->pSchema->numOfAttr;

  if ((answer.attr = calloc(answer.numOfAttr, sizeof(ATTR))) == NULL) ERR;
  for (i = 0, p = head.next; p != NULL; p = p->next) {
    memcpy(&answer.attr[i], p->pSchema->attr, p->pSchema->numOfAttr * sizeof(ATTR));
    i += p->pSchema->numOfAttr;
  }

  answer.headTuple.next = nestedLoopJoin(0, 0, head.next);
  for (i = 0; i < answer.numOfAttr; i++) {
    answer.attr[i].projected = FALSE;
    answer.attr[i].f = NUF; 
  }

  return answer;
}

extern TUPLE
selectAllOfTuple(ANSWER answer)
{
  TUPLE *p;

  for (p = answer.headTuple.next; p != NULL; p = p->next) 
    p->selected = TRUE;

  return answer.headTuple;
}

static SENSOR_SEGMENT *
compfunc(SELECT_ATTR *sap, SENSOR *sp, const int nt)
{
	int i, j;
	SENSOR_SEGMENT *p;
	double *ftr;
	double *fti;

	if ((p = calloc(1, sizeof(SENSOR_SEGMENT))) == NULL) ERR;
	p->f = sap->f;
	p->w = sap->w;

	switch (sap->f) {
	case MAX:
		p->v = sp->sdobj.v[sap->said];
		for (i = 0; sp && ((nt && i < nt) || !nt); sp = sp->next, i++) 
			if (p->v < sp->sdobj.v[sap->said]) 
				p->v = sp->sdobj.v[sap->said];
		break;
		
	case MIN:
		p->v = sp->sdobj.v[sap->said];
		for (i = 0; sp && ((nt && i < nt) || !nt); sp = sp->next, i++) 
			if (p->v > sp->sdobj.v[sap->said]) 
				p->v = sp->sdobj.v[sap->said];
		break;

	case POW:
		p->v = 0;
		for (i = 0; sp && ((nt && i < nt) || !nt); sp = sp->next, i++) 
			p->v += sp->sdobj.v[sap->said] * sp->sdobj.v[sap->said];
		break;
		
	case SUM:
		p->v = 0;
		for (i = 0; sp && ((nt && i < nt) || !nt); sp = sp->next, i++) 
			p->v += sp->sdobj.v[sap->said];
		break;
		
	case CNT:
		p->v = 0;
		for (i = 0; sp && ((nt && i < nt) || !nt); sp = sp->next, i++) 
			p->v++;
		break;
		
	case AVG:
		p->v = 0;
		for (i = 0; sp && ((nt && i < nt) || !nt); sp = sp->next, i++) 
			p->v += sp->sdobj.v[sap->said];
		p->v /= i;
		break;

		/*
		 *
		 * Signal Processing Function
		 *
		 */
	case FTR:
	case FTI:

		if ((ftr = calloc(nt, sizeof(double))) == NULL) ERR;
		if ((fti = calloc(nt, sizeof(double))) == NULL) ERR;
		for (i = 0; sp && ((nt && i < nt) || !nt); sp = sp->next, i++) 
			ftr[i] = sp->sdobj.v[sap->said];

		if (fft(ftr, fti, i, 1, 0) == -1) {
			free(ftr);
			free(fti);
			return NULL;
		}

		if (sap->f == FTR) {
			p->v = 0;
			for (j = 0; j < i; j++)
				if (ftr[j] > p->v)
					p->v = j;
		}
		else if (sap->f == FTI) {
			p->v = 0;
			for (j = 0; j < i; j++)
				if (fti[j] > p->v)
					p->v = j;
		}
		else {
			free(ftr);
			free(fti);
			return NULL;
		}

		free(ftr);
		free(fti);

		/* 
		 *
		 * Sequence Matching Functions
		 *
		 */
	case EUC:
		p->v = compEuclid(sap, sp, nt);
		break;

	case DTW:
		p->v = compDTW(sap, sp, nt);
		break;

	case LCS: 
		p->v = compLCS(sap, sp, nt);
		break;

	case COR:
		p->v = compCorrel(sap, sp, nt);
		break;

	case NUF:
		/* Just keep compiler quiet. Thus, if we are here, it must be fatal. */
		ERR;
		break;

	default:
		ERR;
		break;
	}

	return p;
}

static void
releaseSSP(SENSOR_SEGMENT *ssp)
{	
	SENSOR *sp;

	/* Release unused memories */
	while (ssp->headSensor.next != NULL) {
		sp = ssp->headSensor.next;
		ssp->headSensor.next = ssp->headSensor.next->next;
		free(sp);
	}
	free(ssp);
}

static ANSWER
executorCore(const SELECT_ATTR headSA, const CONDITION headC, const FROM_SCHEMA headFS)
{
  ANSWER answer;
  SCHEMA *baseP;
  FROM_SCHEMA *fromP;
  SCHEMA_LIST headSL, *listP;
  SELECT_ATTR *sap = headSA.next;

  bzero(&answer, sizeof(ANSWER));
  bzero(&headSL, sizeof(SCHEMA_LIST));

  /* 
   * Table checking
   */
  for (fromP = headFS.next, listP = &headSL; fromP != NULL; fromP  = fromP->next) {
    if ((listP->next = calloc(1, sizeof(SCHEMA_LIST))) == NULL) ERR;
    listP = listP->next;

    for (baseP = HeadSchema.next; 
         baseP != NULL && strcmp(baseP->schemaName, fromP->schemaName) != 0;
         baseP = baseP->next) {
      ;
    }
    
    if (baseP == NULL) {/* Error, table does not exist */
      SCHEMA_LIST *p;
      while (headSL.next != NULL) {
        p = headSL.next;
        headSL.next = headSL.next->next;
        free(p);
      }
      answer.parseMsg = TABLE_NOT_EXIST;
      return answer;
    }

    if ((pthread_rwlock_rdlock(&baseP->rwlock)) == -1) ERR;
    listP->pSchema = baseP;
  }

  /* Making a Big Table */
  answer = execCartesianProduct(headSL);
  if (headC.next == NULL) 
    answer.headTuple = selectAllOfTuple(answer);  
  else {
    answer = invokeSelection(headC, answer);
    answer.headTuple = releaseTrashTupleForSelectOrDeleteOrUpdateOrAppend(answer);  
  }


  /* 
   * Function Computation.
   *
   * Currently just only one funciton is allowed.
   * By default, ssp->f is set as "NUF" in page2tuple.
   */
  for (sap = headSA.next; sap != NULL; sap = sap->next) {
    int i = 0, offset = 0;

		if (sap->f != NUF) { /* Must be a function */

      for (offset = i = 0; i <= answer.numOfAttr; i++) {
        if (i == answer.numOfAttr) {
					answer.parseMsg = WRONG_ATTRIBUTE;
					return answer;
				}
        if (strcmp(sap->attrName, answer.attr[i].attrName) == 0) 
          break;
        offset += answer.attr[i].sizOfAttr;
      }

			/* Error Check */
			if (sap->said >= answer.attr[i].szsa || sap->said < 0) {
				answer.parseMsg = WRONG_SENSOR_ID;
				return answer;
			}

      if (answer.attr[i].attrType == SENSOR_SEGMENT_DOUBLE) {
				int j;
        SENSOR_SEGMENT *ssp;       /* General Use */
				SENSOR_SEGMENT *tssp, hss; /* Window & Function */
        SENSOR *sp;
        TUPLE *tp;

        for (tp = answer.headTuple.next; tp != NULL; tp = tp->next) {

					/* Cast */
					ssp = (SENSOR_SEGMENT *)(tp->obj + offset);

          if (ssp->next->headSensor.next == NULL)
            continue;

          /* 
           * These copies are for future use !
           *
           * ans: for printAttribute in "libKraft.c"
           * ssp: for printResultForRelation in "libKraft.c"
           */
          answer.attr[i].f = ssp->next->f = sap->f; 
          answer.attr[i].w = ssp->next->w = sap->w; 
					
					if (sap->w == SLD) { /* For subsequences using window */
						for (tssp = &hss, sp = ssp->next->headSensor.next; sp;) {
							tssp->next = compfunc(sap, sp, sap->nt);
							tssp = tssp->next;
							for (j = 0; j < sap->ns && sp; j++) 
								sp = sp->next;
						}
						tssp = hss.next;
					}
					else if (sap->w == LST) { /* For last subsequence */
						for (sp = ssp->next->headSensor.next; 
								 sp->next != NULL; 
								 sp = sp->next) ;
						for (j = 0; j < sap->nt - 1; j++)
							sp = sp->prev;
						if ((tssp = compfunc(sap, sp, sap->nt)) == NULL) {
							answer.parseMsg = FUNCTION_WRONG;
							return answer;
						}
					}
					else /* For a full sequence */ {
						if ((tssp = compfunc(sap, ssp->next->headSensor.next, 0)) == NULL) {
							answer.parseMsg = FUNCTION_WRONG;
							return answer;
						}
						tssp->f = sap->f;
						tssp->w = sap->w;
					}
					/* 
					 * Switch ssp,
					 *
					 * release it & attach the result of aggregation 
					 */
					releaseSSP(ssp->next);
					ssp->next = tssp;
				}
      } /* ENF IF (sensor_segment_double) */
      else {
        /* Computation for TUPLE is not implemented yet */
        answer.parseMsg = AGG4TUPLE_NOT_IMPLEMENTED;
        return answer;
      }
    }
  }

  /* Projection */
	
  answer = execProjection(headSA, answer);
  if (answer.parseMsg != PARSE_OK) 
	return answer;

  /* OK, unlock */
  unlockSchemaList(headSL);

  /* Everything seems fine */
  answer.parseMsg = PARSE_OK;


  return answer;
}

static ORDER_INFO
initGetOrderInfo(void)
{
  ORDER_INFO orderInfo;

  bzero(&orderInfo, sizeof(ORDER_INFO));
  orderInfo.orderType = NO_ORDER_TYPE;
  orderInfo.err = FALSE;

  return orderInfo;
}

static ORDER_INFO
getOrderInfo(const char query[], const char schemaName[])
{
  int queryid;
  int leftOfSchemaName;
  int rightOfSchemaName;
  int leftOfAttrName;
  int rightOfAttrName;
  ORDER_INFO orderInfo;

  orderInfo = initGetOrderInfo();
  for (queryid = 0;
       queryid < (int)(QUERY_BUFFSIZE - strlen("order"));
       queryid++) {
    if ((query[queryid] == 'o') && (strncmp(&query[queryid], "order", strlen("order")) == 0)) {
      queryid += strlen("order");
      break;
    }
  }
  if (queryid == (int)(QUERY_BUFFSIZE - strlen("order"))) {
    return orderInfo;
  }

  for (;queryid < (int)(QUERY_BUFFSIZE - strlen("by"));
       queryid++) {
    if ((query[queryid] == 'b') && (strncmp(&query[queryid], "by", strlen("by")) == 0)) {
      queryid += strlen("by");
      break;
    }
  }
  if (queryid == (int)(QUERY_BUFFSIZE - strlen("by"))) {
    return orderInfo;
  }

  while ((query[queryid] == ' ') || (query[queryid] == '\t')) {
    queryid++;
  }
  leftOfSchemaName = queryid;
  while ((query[queryid] != '.') && (query[queryid] != ' ') && (query[queryid] != '\t')) {
    queryid++;
  }
  if (query[queryid] == '.') { /* Both of schemaName and attrName are specified */
    rightOfSchemaName = queryid;
    strncpy(orderInfo.schemaName, &query[leftOfSchemaName], (rightOfSchemaName - leftOfSchemaName));

    queryid++; /* Next to '.' */
    leftOfAttrName = queryid;  
  }
  else { /* Only attrName is specified */
    strncpy(orderInfo.schemaName, schemaName, strlen(schemaName));
    leftOfAttrName = leftOfSchemaName;  
  }

  while ((query[queryid] != ' ') && (query[queryid] != '\t')) {
    queryid++;
  }
  rightOfAttrName = queryid;  
  strncpy(orderInfo.attrName, &query[leftOfAttrName], (rightOfAttrName - leftOfAttrName));

  orderInfo.orderType = ASC_ORDER_TYPE;
  for (;queryid < (int)(QUERY_BUFFSIZE - strlen("desc"));
       queryid++) {
    if ((query[queryid] == 'd') && (strncmp(&query[queryid], "desc", strlen("desc")) == 0)) {
      orderInfo.orderType = DESC_ORDER_TYPE;
      break;
    }
  }
  
  return orderInfo;
}

extern ANSWER
selectFrom(const char query[])
{
  SELECT_ATTR headSelectAttr;  
  CONDITION headCondition;  
  FROM_SCHEMA headFromSchema;
  ANSWER answer;
  ORDER_INFO orderInfo;

  /*
   * Attributes & function & windows
   */
  headSelectAttr = getSelAtr(query);
  if (headSelectAttr.err == TRUE) {
    answer.parseMsg = WRONG_ATTRIBUTE;    
    return answer;
  }

  /*
   * Tables
   */
  headFromSchema = getFromSchema(query);
  if (headFromSchema.err == TRUE) {
    answer.parseMsg = WRONG_SCHEMA;    
    return answer;
  }

  if ((query[headFromSchema.offset] != '\0') && (query[headFromSchema.offset] != '\n')) {
    headCondition = getcnd(query);
    if (headCondition.err == TRUE) {
      answer.parseMsg = WRONG_WHERE_CLAUSE;    
      return answer;
    }
  }
  else {
    bzero(&headCondition, sizeof(CONDITION));
    bzero(&orderInfo, sizeof(ORDER_INFO));
  }
  
  orderInfo = getOrderInfo(query, headFromSchema.next->schemaName);
  if (orderInfo.err == TRUE) {
    answer.parseMsg = WRONG_ORDER;
    return answer;
  }

  /*
   * Executor
   */
  answer = executorCore(headSelectAttr, headCondition, headFromSchema);
  if (answer.parseMsg != PARSE_OK)
    return answer;

  if (answer.attr != NULL) 
    answer.headTuple = sortAnswerTupleByOrder(answer, orderInfo);

  releaseSelectAttr(headSelectAttr);
  releaseFromSchema(headFromSchema);
  releaseCondition(headCondition);

  return answer;
} 

extern TUPLE
releaseTrashTupleForSelectOrDeleteOrUpdateOrAppend(ANSWER answer)
{
  TUPLE *p, *trash, *before;

  for (before = &answer.headTuple, p = before->next; p != NULL; p = before->next) {
    if (p->selected == FALSE) {
      trash = p;
      before->next = p->next;
      free(trash->obj);
      free(trash);
    }
    else {
      before = before->next;
    }
  }
  
  return answer.headTuple;
}
