#include "common.h"
#include "libWrapper.h"
#include <ctype.h>
#include "buffsize.h"

#ifndef _TRANS_INFO_H
#define _TRANS_INFO_H
#include "transInfo.h"
#endif

#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _DATADIR_H
#define _DATADIR_H
#include "dataDir.h"
#endif

/****************************************************************
 *
 * Declaration
 *
 ***************************************************************/
extern void unpin(DBPAGE *buffPageP);
extern void releaseThisSchema(SCHEMA *schemaP);
extern void initPagePtr(SCHEMA *schemaP);
extern void lockEntirePage(void);
extern void unlockEntirePage(void);
extern void logInsert(PERSISTENT_OBJ *persistP);
extern int getCurPageSize(DBPAGE *pageP);
extern void pin(DBPAGE *buffPageP, char mode);
extern SCHEMA *getSchema(const char schemaName[]);
extern DBPAGE *getEndPage(SCHEMA *schemaP, int attrid, int tupleid);
extern DBPAGE *getNewPage(SCHEMA *schemaP, int attrid, int tupleid);
extern PERSISTENT_OBJ *allocPrstInstObj(char *objP, const char schemaName[], const PAGEINFO pi);
extern DBPAGE getStartRelPage(SCHEMA *schemaP);
extern DBPAGE getNextPage(SCHEMA *schemaP, DBPAGE page);
extern char *getInsertPtr(SCHEMA *schemaP, const int tid);

/****************************************************************
 *
 * Gloval variable
 *
 ***************************************************************/
extern SCHEMA HeadSchema;

/****************************************************************
 *
 * Private Function
 *
 ***************************************************************/
static int 
getSchemaNameForInsert(const char query[], char schemaName[])
{
  int left, right, offset;

  for (offset = 0; offset < QUERY_BUFFSIZE; offset++) {
    if ((query[offset] == 'i') && (strncmp(&query[offset], "into", strlen("into")) == 0)) {
      offset += strlen("into");
      break;
    }
  }
  while (isspace(query[offset]) != 0) {offset++;} left = offset;
  while (isspace(query[offset]) == 0) {offset++;} right = offset;
  bzero(schemaName, BUFFSIZE);
  strncpy(schemaName, &query[left], right-left);

  return offset;
}

static int
getObjForInsert(char objForInsert[], const char query[], int offset)
{
  int left, right;

  while ((query[offset] == ' ')) {offset++;}  left = offset;
  while ((query[offset] != ' ') && (query[offset] != ',') && (query[offset] != ')')){offset++;}right = offset;
  bzero(objForInsert, BUFFSIZE);
  strncpy(objForInsert, &query[left], right-left);

  return offset;
}

static int
skipValues(int offset, const char query[])
{
  while (offset < QUERY_BUFFSIZE) {
    if ((query[offset] == 'v') && (strncmp(&query[offset], "values", strlen("values")) == 0)) {
      offset += strlen("values");
      break;
    }
    else 
      offset++;
  }
  if (offset == QUERY_BUFFSIZE) 
    return -1;

  while (isspace(query[offset]) != 0) 
    offset++;

  if (query[offset] != '(') {
    printf("Error: %s\n", &query[offset]);
    exit(1);
  }

  return ++offset; /* skip '(' */
}

static int
coverTuple(SCHEMA *schP, int qo, const char query[], char *tplobj)
{
  int i, tid, *iP;
  int bo; /* Byte Offset */
  char obj[BUFFSIZE];
  double *dP;

  for (bo = i = 0; 
       i < schP->numOfAttr; 
       bo += schP->attr[i].sizOfAttr, i++) {
    qo = getObjForInsert(obj, query, qo);
    switch (schP->attr[i].attrType) {
    case INT:    
      iP = (int *)(tplobj + bo);
      *iP = atoi(obj); 
      if (i == 0) { /* Must be a ID */
        tid = *iP;
      }
      break;
    case DOUBLE: 
      dP = (double *)(tplobj + bo);
      *dP = atof(obj); 
      break;
    case TEXT:   
      strcpy(tplobj + bo, obj);
      break;
    case SENSOR_SEGMENT_DOUBLE: 
      if ((strncmp("NULL", obj, strlen("NULL"))) && (strncmp("null", obj, strlen("null")))) 
        ERR;
      break;
    default: 
      ERR;
      break;
    }
    /* Search " " OR "," OR ")" */
    while ((query[qo] != ',') && (query[qo] != ')')) qo++;
    qo++; /* skip ',' or ')' */
  }

  return tid;
}

static PARSE_MSG
execInsert(SCHEMA *schemaP, const char query[], int queryOffset)
{
  int newMaxOffset;
  int tid;
  char *objP;

  /* Preparation */
  if ((objP = calloc(1, schemaP->sizOfObj)) == NULL) ERR;
  queryOffset = skipValues(queryOffset, query);
  if (queryOffset == -1) ERR;
  tid = coverTuple(schemaP, queryOffset, query, objP);
  if (schemaP->tid == 0) {
    if (tid != 0)
      return TUPLE_ID_MUST_BE_ZERO;
  }
  else if (tid != schemaP->tid) {
    if (schemaP->garbageP[tid] == 0) {/* This area is NOT garbage field */
      return TUPLE_ID_WRONG;
    }
    else { /* OK. Prepare for reusing this area */
      int fd;
      char val = 0; /* Changing value from 1 to 0 */
      char file[BUFSIZ], *p;
			char *hdp;

      /* 
       * Cleaning the disk area for 
       * reusing this tuple
       */
      bzero(file, BUFFSIZE);
			hdp = getenv("HOME");
      sprintf(file, "%s/%s/%s/garbage", hdp, DATA_DIR, schemaP->schemaName);
      if ((fd = open(file, O_WRONLY)) == -1) ERR;
      if (lseek(fd, tid * sizeof(char), SEEK_SET) == -1) ERR;
      if (write(fd, &val, sizeof(char)) == -1) ERR;
      if (close(fd) == -1) ERR;

      /* 
       * Get Buffer, here we get the lock of dbpage (i.e. pin)
       */
      lockEntirePage();
      if ((p = getInsertPtr(schemaP, tid)) == NULL) 
				return INSERT_FAILED;
      unlockEntirePage();
      memcpy(p, objP, schemaP->sizOfObj); /* Modify Buffer */
      unpin(schemaP->endP); /* Unlock schema */
      free(objP);

      /*
       * Reset Garbage Area
       */
      schemaP->garbageP[tid] = 0;

      return PARSE_OK;
    }
  }

  /*
   * Incresing the number of maximum tuples
   */
  schemaP->tid++; 

  /* 
   * Get Buffer, here we get the lock of dbpage (i.e. pin)
   */
  lockEntirePage();
  if (schemaP->endP == NULL) {
    /* 
     * 1: Page has a lock, 
     * 2: '0' in "getEndPage(schemaP, 0, tid)" means TUPLE. With sensor, 
     *    '0' must be changed to attribute id (see appendInto.c)
     */
    schemaP->endP = getEndPage(schemaP, 0, tid); 
  }
  else 
    pin(schemaP->endP, 'w');

  if ((schemaP->endP->pi.numOfObj + 1) * schemaP->endP->pi.sizOfObj > (int)(PAGESIZE - sizeof(PAGEINFO))) {
    newMaxOffset = schemaP->endP->pi.offset + PAGESIZE;
    unpin(schemaP->endP);
    schemaP->endP = getNewPage(schemaP, 0, tid); /* Page has a lock */
    schemaP->endP->pi.offset = newMaxOffset;
  }
  unlockEntirePage();

  /* Init buffer pointer for APPEND operations */
  schemaP->numOfObj++; 
  initPagePtr(schemaP);

  /* Modify Buffer */
  memcpy(&schemaP->endP->data[getCurPageSize(schemaP->endP)], objP, schemaP->sizOfObj);
  schemaP->endP->pi.numOfObj++; /* Required for page sync */
  unpin(schemaP->endP);
  free(objP);

  /* Garbage Area */
  if (schemaP->garbageP == NULL) {
    if ((schemaP->garbageP = calloc(tid + 1, sizeof(char))) == NULL) ERR;
  }
  else {
    if ((schemaP->garbageP = realloc(schemaP->garbageP, (tid + 1) * sizeof(char))) == NULL) ERR;    
  }

  return PARSE_OK;
}

/*********************
 *
 * Public Function
 *
 *********************/
extern PARSE_MSG
insert(const char query[])
{
  int queryOffset;
  char schemaName[BUFFSIZE];
  SCHEMA *schemaP;
  PARSE_MSG parseMsg = PARSE_OK;

  queryOffset = getSchemaNameForInsert(query, schemaName);
  schemaP = getSchema(schemaName);
  if (schemaP == NULL) {
    parseMsg = TABLE_NOT_EXIST;
    return parseMsg;
  }
  
  if (pthread_rwlock_rdlock(&schemaP->rwlock)) ERR;
  parseMsg = execInsert(schemaP, query, queryOffset);
  if (pthread_rwlock_unlock(&schemaP->rwlock)) ERR;

  return parseMsg;
}
