#include "common.h"
#include "libWrapper.h"

#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _SELECT_FROM_H
#define _SELECT_FROM_H
#include "selectFrom.h"
#endif

#ifndef _SIMSEQ_H
#define _SIMSEQ_H
#include "simseq.h"
#endif

#ifndef _WINDOW_H
#define _WINDOW_H
#include "window.h"
#endif

/*******************************************************
 *
 * Definitions
 *
 *******************************************************/
typedef struct _PATH {
  int s; /* Coordinates X in matrix */
  int t; /* Coordinates Y in matrix */
  double v;
} PATH;

/*******************************************************
 *
 * Global variables
 *
 *******************************************************/
static int PathLen;

/*******************************************************
 *
 * Private Functions
 *
 *******************************************************/
static void
setM(double **m, const double sa[], const double ta[], const int szw)
{
  int s, t;
  
  for (s = 0; s < szw; s++) {
    for (t = 0; t < szw; t++) {
      m[s][t] = abs(sa[s] - ta[t]);
    }
  }
}

static PATH *
allocPath(void)
{
  PATH *p;
  
  if ((p = (PATH *)calloc(PathLen, sizeof(PATH))) == NULL) ERR;

  return p;
}

static char 
getDirection(double **m, int i, PATH *p)
{
  int sv, tv, mv;
  char dir;

  sv = fabs((m[p[i - 1].t][p[i - 1].s + 1]) - (p[i - 1].v));
  tv = fabs((m[p[i - 1].t + 1][p[i - 1].s]) - (p[i - 1].v));
  mv = fabs((m[p[i - 1].t + 1][p[i - 1].s + 1]) - (p[i - 1].v));
  
  if (mv <= sv && mv <= tv) 
    dir = 'm';
  else if (sv <= tv) 
    dir = 's';
  else 
    dir = 't';

  return dir;
}

static PATH *
createWarpingPath(double **m, int *lnp, int szw)
{
	int i;
  char dir;
  PATH *p;

  p = allocPath();
  p[0].s = 0;
  p[0].t = 0;
  p[0].v = m[0][0];

  for (i = 0;;) {
    i++;
    dir = getDirection(m, i, p);
    switch (dir) {

    case 's':
      p[i].s = p[i - 1].s + 1;
      p[i].t = p[i - 1].t;
      p[i].v = m[p[i].t][p[i].s];      
      break;

    case 't':
      p[i].s = p[i - 1].s;
      p[i].t = p[i - 1].t + 1;
      p[i].v = m[p[i].t][p[i].s];
      break;

    case 'm':
      p[i].s = p[i - 1].s + 1;
      p[i].t = p[i - 1].t + 1;
      p[i].v = m[p[i].t][p[i].s];      
      break;

    default:
			ERR; /* Fatal */
      break;
    }

    /*
     * Check boundary condition
     */
    if (p[i].s == (szw - 1)) {
      while (p[i].t < szw - 1) {
        i++;
        p[i].t = p[i - 1].t + 1;
        p[i].s = p[i - 1].s;
      }
      break;
    }
    else if (p[i].t == (szw - 1)) {
      while (p[i].s < szw - 1) {
        i++;
        p[i].t = p[i - 1].t;
        p[i].s = p[i - 1].s + 1;
      }
      break;
    }
  }

  *lnp = ++i;

  return p;
}

static double
calcWarpingDistance(double **m, PATH *p, int lnp, int szw)
{
	/* lnp: Length of Path */
  int i, j, k;
  double d = 0.0; /* Distance */
  
  for (i = 0; i < szw; i++) {
    for (j = 0; j < szw; j++) {
      for (k = 0; k < lnp; k++) {
        if (p[k].t == i && p[k].s == j) {
					d += (m[i][j] * m[i][j]);
          break;
        }
      }
    }
  }

  return sqrt(d) / ((double)lnp);
}

static void
setPathLen(int szw)
{
  PathLen = szw * 2;
}

static double **
initM(int szw)
{
  int i;
  double **m;

  if ((m = (double **)calloc(szw, sizeof(double *))) == NULL) ERR;
  for (i = 0; i < szw; i++) 
    if ((m[i] = (double *)calloc(szw, sizeof(double))) == NULL) ERR;

  return m;
}

static void
release(double **m, int szw, PATH *p, double *s, double *t) 
{
  int i;

  for (i = 0; i < szw; i++) 
    free(m[i]);

  free(m);
  free(p);

	/* Sequence Array */
	free(s);
	free(t);
}

static double
doCompDTW(double s[], double t[], int szw)
{
  int lnp;
  double d;   /* Distance */
	double **m; /* Matrix */
  PATH *p;  

  setPathLen(szw);
  m = initM(szw);
  setM(m, s, t, szw);

  p = createWarpingPath(m, &lnp, szw);
  d = calcWarpingDistance(m, p, lnp, szw);

  release(m, szw, p, s, t);

  return d;
}

/****************************************************************
 *
 * Public Functions
 *
 ***************************************************************/
extern double 
compDTW(SELECT_ATTR *sap, SENSOR *sp, const int nt)
{
	int i;
	double *s, *t;
	SENSOR *tmp = sp;

	/* Getting length */
	for (i = 0; sp && ((nt && i < nt) || !nt); sp = sp->next, i++) ;

	/* Calculating DTW value */
	if ((s = calloc(i, sizeof(double))) == NULL) ERR;
	if ((t = calloc(i, sizeof(double))) == NULL) ERR;
	for (i = 0, sp = tmp; sp; sp = sp->next, i++) {
		s[i] = sp->sdobj.v[sap->said];
		t[i] = sp->sdobj.v[sap->s_said];
	}

	return doCompDTW(s, t, i);
}
