#include "common.h"
#include "libWrapper.h"
#include "schema.h"
#include "buffsize.h"
#include <ctype.h>

/****************************************************************
 *
 * Definition
 *
 ***************************************************************/
/****************************************************************
 *
 * Gloval variable
 *
 ***************************************************************/
static BTREE_PAGE Root;         /* Root node of B+-tree */
static KEYTYPE GlobalKey;       /* Key type */
static BOOLEAN Done;
static BOOLEAN Deleted;
static BOOLEAN UnderSize;     
static BTREE_PAGE NewPage;         /* New page generated by insert */
static BTREE_DATA *NewBtreeData;
static BTREE_DATA HeadData;
static BOOLEAN alreadyInitRoot = FALSE;
static int GlobalOffset = 0;
static char SchemaName[BUFFSIZE];
static char IndexFileName[BUFFSIZE];

/****************************************************************
 *
 * Prototype
 *
 ***************************************************************/

/****************************************************************
 *
 * Private Function
 *
 ***************************************************************/

static void 
writeNode(BTREE_PAGE p) 
{
  int fd;

  if ((fd = open(IndexFileName, O_WRONLY|O_APPEND)) == -1) ERR;
  if (lseek(fd, p.offset, SEEK_SET) == -1) ERR;
  if (write(fd, &p, sizeof(BTREE_PAGE)) == -1) ERR;
  if (fsync(fd) == -1) ERR;
  if (close(fd) == -1) ERR;
}

static BTREE_PAGE 
GetNewPage(void)  
{
  int i;
  BTREE_PAGE p;

  p.n = 0;
  for (i = 0; i < 2 * MIN_BTREE_FANOUT; i ++) {
    p.key[i] = -1;
  }
  
  p.offset = GlobalOffset * sizeof(BTREE_PAGE);
  GlobalOffset ++;
  writeNode(p);

  return p;
}

static BTREE_PAGE 
getNode(int offset) 
{
  int fd;
  BTREE_PAGE p;

  if ((fd = open(IndexFileName, O_RDONLY)) == -1) ERR;
  if (lseek(fd, offset, SEEK_SET) == -1) ERR;
  if (read(fd, &p, sizeof(BTREE_PAGE)) == -1) ERR;
  if (close(fd) == -1) ERR;

  return p;
}

static void 
search(void)  
{
  BTREE_PAGE p;
  int k;

  if (Root.offset == -1) {
    return;
  }
  else {
    p = getNode(Root.offset);
  }
              
  while (1) {
    k = 0;
    while (k < p.n && p.key[k] < GlobalKey) {
      k++;
    }
    if ((k < p.n) && 
        (p.key[k] == GlobalKey) &&
        (p.thisIsLeaf == TRUE)) {
#ifdef DEBUG
      printf("p->key[k]:  %d\n", p.key[k]);
      printf("key:        %d\n", p.u[k].data->key);
      printf("offset:     %d\n", p.u[k].data->offset);
#endif
      return;
    }
    if (p.thisIsLeaf == TRUE) {
      break;
    }
    else {
      p = getNode(p.u[k].branch);
    }
  }
}

void 
insertItem(BTREE_PAGE p, int k)
{
  int i;
  
  for (i = p.n; i > k; i--) {
    p.key[i] = p.key[i - 1];
    if (p.thisIsLeaf == FALSE) {
      p.u[i+1].branch = p.u[i].branch;
    }
    else {
      p.u[i].data = p.u[i-1].data;
    }
  }
  p.key[k] = GlobalKey;  

  if (p.thisIsLeaf == FALSE) {
    p.u[k+1].branch = NewPage.offset;  
  }
  else {
    p.u[k].data = NewBtreeData;
  }
  p.n++;

  writeNode(p);
}

static void 
split(BTREE_PAGE p, int k)  
{
  int j;
  int m;
  BTREE_PAGE q;

  if (k <= MIN_BTREE_FANOUT) {
    m = MIN_BTREE_FANOUT;  
  }
  else {
    m = MIN_BTREE_FANOUT + 1;
  }

  /*
   * Getting new page
   */
  q = GetNewPage();

  /*
   * Set thisIsLeaf or not
   */
  if (p.thisIsLeaf == TRUE) {
    q.thisIsLeaf = TRUE;
  }
  else {
    q.thisIsLeaf = FALSE;
  }

  /*
   * Copy
   */  
  for (j = m + 1; j <= 2 * MIN_BTREE_FANOUT; j++) {
    q.key[j - m - 1] = p.key[j - 1];
    if (p.thisIsLeaf == FALSE) {
      q.u[j-m].branch = p.u[j].branch;
    }
    else { /* thisIsLeaf */
      q.u[j-m-1].data = p.u[j-1].data;
    }
  }

  /*
   * Set the number of nodes
   */
  q.n = 2 * MIN_BTREE_FANOUT - m;  
  p.n = m;

  /*
   * Insert
   */
  if (k <= MIN_BTREE_FANOUT) {
    insertItem(p, k);
    p = getNode(p.offset);
  }
  else {
    insertItem(q, k - m);
    q = getNode(q.offset);
  }

  GlobalKey = p.key[p.n - 1];
  if (p.thisIsLeaf == FALSE) {
    q.u[0].branch = p.u[p.n].branch;
    p.n --;
  }

  /* 
   * A new page is inserted into NewPage and return 
   */
  NewPage = q; 

  /*
   * Sync
   */
  writeNode(p);
  writeNode(q);
}


/*
 * Function "insertsub"
 *
 * Tree is recursively traversed from p 
 */
static void 
insertSub(int offset)
{
  int k;
  BTREE_PAGE p;

  if (offset == -1) {
    Done = FALSE;  
    NewPage.offset = -1;  
    return;
  }
  else {
    p = getNode(offset);
  }

  k = 0;
  while ((k < p.n) && (p.key[k] < GlobalKey)) {
    k++;
  }
  if ((k < p.n) && (p.key[k] == GlobalKey)) {
    Done = TRUE;
    return;
  }

  if (p.thisIsLeaf == FALSE) {
    insertSub(p.u[k].branch);
    if (Done == TRUE) {
      return;
    }
  }
  if (p.n < 2 * MIN_BTREE_FANOUT) {/* Without split */
    insertItem(p, k);  
    Done = TRUE;
  } 
  else {/* With split */
    split(p, k);  
    Done = FALSE;
  }
}

static void 
insert(void)
{
  BTREE_PAGE p;

  insertSub(Root.offset);  
  if (Done == TRUE) {
    return;
  }

  p = GetNewPage();  
  p.key[0] = GlobalKey;
  p.n = 1;  
  if (alreadyInitRoot == TRUE) {
    p.thisIsLeaf = FALSE;
    p.u[0].branch = Root.offset;  
    p.u[1].branch = NewPage.offset;  
  }
  else {
    p.thisIsLeaf = TRUE;
    p.u[0].data = NewBtreeData;
    alreadyInitRoot = TRUE;
  }
  Root = p;
  writeNode(Root);
}

static void 
removeitem(BTREE_PAGE p, int k)
{
  while (k < p.n-1) {
    p.key[k] = p.key[k+1];
    if (p.thisIsLeaf == FALSE) {
      p.u[k].branch = p.u[k+1].branch;
    }
    else {
      p.u[k].data = p.u[k+1].data;
    }
    k ++;
  }
  if (p.thisIsLeaf == FALSE) {
    p.u[p.n-1].branch = p.u[p.n].branch;
  }

  p.n --;
  if (p.n < MIN_BTREE_FANOUT) {
    UnderSize = TRUE;
  }
  else {
    UnderSize = FALSE;
  }

  writeNode(p);
}

static void 
moveright(BTREE_PAGE p, int k)
{
  int j;
  BTREE_PAGE left;
  BTREE_PAGE right;
  BTREE_PAGE ptmp;

  left = getNode(p.u[k-1].branch);  
  right = getNode(p.u[k].branch);

  if (right.thisIsLeaf == FALSE) {
    right.u[right.n+1].branch = right.u[right.n].branch;
  }
  for (j = right.n; j >= 1; j--) {
    right.key[j] = right.key[j-1];
    if (right.thisIsLeaf == FALSE) {
      right.u[j].branch = right.u[j-1].branch;
    }
    else {
      right.u[j].data = right.u[j-1].data;      
    }
  }
  right.n ++;

  /*
   * Migrate rightest element at left node 
   * to the leftest element at right node
   */

  if (left.thisIsLeaf == FALSE) {
#ifdef DEBUG
    printf("NOT LEAF\n");
#endif
    p.key[k-1] = left.key[left.n-1];
    ptmp = getNode(left.u[left.n].branch);
    right.key[0] = ptmp.key[ptmp.n-1];
  }
  else {
#ifdef DEBUG
    printf("LEAF\n");
#endif
    p.key[k-1] = left.key[left.n-2];
    right.key[0] = left.key[left.n-1]; 
  }
#ifdef DEBUG
  printf("p.key[%d]: %d\n", k-1, p.key[k-1]);
  printf("right.key[0]: %d\n", right.key[0]);
#endif

  if (right.thisIsLeaf == FALSE) {  
    right.u[0].branch = left.u[left.n].branch;
  }
  else {
    right.u[0].data = left.u[left.n-1].data;
  }

  /*
   * Reducing number in left node
   */
  left.n --;

  /*
   * Sync
   */
  writeNode(p);
  writeNode(left);
  writeNode(right);
  
}

/*
 * Function "moveleft"
 *
 * Leftest element in p.branch[k] is 
 * moved to p.branch[k-1] via p.key[k-1]
 */
static void 
moveleft(BTREE_PAGE p, int k)
{
  int j;
  BTREE_PAGE left;
  BTREE_PAGE right;
  BTREE_PAGE ptmp;

#ifdef DEBUG
  printf("moveleft\n");
#endif

  /*
   * Set left and right
   */
  left = getNode(p.u[k-1].branch);  
  right = getNode(p.u[k].branch);

  /*
   * Set key
   */
  p.key[k-1] = right.key[0];
  if (left.thisIsLeaf == FALSE) {
    ptmp = getNode(left.u[left.n].branch);
    left.key[left.n] = ptmp.key[ptmp.n-1];
  }
  else {
    left.key[left.n] = right.key[0];
  }
#ifdef DEBUG
  printf("p.key[%d]: %d\n", k-1, p.key[k-1]);
  printf("left.key[%d]: %d\n", left.n, left.key[left.n]);
#endif
  if (left.thisIsLeaf == FALSE) {
    left.u[left.n+1].branch = right.u[0].branch;
  }
  else {
    left.u[left.n].data = right.u[0].data;
  }
  left.n++;

  /*
   * Shift in right node
   */
  if (left.thisIsLeaf == FALSE) {
    right.u[0].branch = right.u[1].branch;
  }
  for (j = 1; j < right.n; j++) {
    right.key[j-1] = right.key[j];
    if (left.thisIsLeaf == FALSE) {    
      right.u[j].branch = right.u[j+1].branch;
    }
    else {
      right.u[j-1].data = right.u[j].data;      
    }
  }
  right.n--;

  /*
   * Sync
   */
  writeNode(p);
  writeNode(left);
  writeNode(right);
}


/*
 * Function "coalesce"
 *
 * Coaleses p.branch[k-1] and p.branch[k]
 */
static void 
coalesce(BTREE_PAGE p, int k)  
{
  int j;
  BTREE_PAGE left;
  BTREE_PAGE right;
  BTREE_PAGE ptmp;

#ifdef DEBUG
  printf("coalesce\n");
  for (j = 0; j < p.n; j++) {
    printf("p.key[%d]: %d\n", j, p.key[j]);
  }
#endif
  if (p.thisIsLeaf == TRUE) {
    printf("thisIsLeaf == TRUE in coalesce\n");
    exit(1);
  }
  
  /*
   * Set left and right
   */
  right = getNode(p.u[k].branch);
  left = getNode(p.u[k-1].branch);

  /*
   * Migrate nodes from right to left
   */
  if (left.thisIsLeaf == FALSE) {
    ptmp = getNode(left.u[left.n].branch);
    left.key[left.n] = ptmp.key[ptmp.n-1];
    left.n++;
#ifdef DEBUG
    printf("left.key[%d]: %d\n", left.n, left.key[left.n]);
#endif
    for (j = 0; j < right.n; j++) {
      left.key[left.n] = right.key[j];
      left.u[left.n].branch = right.u[j].branch;
      left.n++;
    }
    left.u[left.n].branch = right.u[j].branch;
#ifdef DEBUG
    printf("left.n: %d\n", left.n);
#endif
  }
  else {
    for (j = 0; j < right.n; j++) {
      left.key[left.n] = right.key[j];
      left.u[left.n].data = right.u[j].data;
      left.n++;
    }
  }

  /* 
   * Reset the key at the upper node 
   * as the rightest in left node
   */
  if (left.thisIsLeaf == FALSE) {
    ptmp = getNode(left.u[left.n].branch);
    p.key[k-1] = ptmp.key[ptmp.n-1];
  }
  else {
    p.key[k-1] = left.key[left.n-1];
  }

  while (k < p.n) {
    p.key[k] = p.key[k+1];
    if (p.thisIsLeaf == FALSE) {
      p.u[k].branch = p.u[k+1].branch;
    }
    k ++;
  }

  /*
   * Check underflow
   */ 
  p.n --;
  if (p.n < MIN_BTREE_FANOUT) {
    UnderSize = TRUE;
  }
  else {
    UnderSize = FALSE;
  }

  /*
   * Sync
   */
  writeNode(left);
  writeNode(p);
}  


/*
 * Function "restore"
 */
static void 
restore(BTREE_PAGE p, int k)  
{
  BTREE_PAGE tmp;

  UnderSize = FALSE;
  if (k > 0) {
    tmp = getNode(p.u[k-1].branch);
    if (tmp.n > MIN_BTREE_FANOUT) {
      moveright(p, k);
    }
    else {
      coalesce(p, k);
    }
  } 
  else {
    tmp = getNode(p.u[1].branch);    
    if (tmp.n > MIN_BTREE_FANOUT) {
      moveleft(p, 1);
    }
    else {
      coalesce(p, 1);
    }
  }
}

/*
 * Function "deletesub"
 */
static void 
deletesub(BTREE_PAGE p)  
{
  int k;

  k = 0;
  while (k < p.n && p.key[k] < GlobalKey) {
    k++;
  }

  if (p.thisIsLeaf == TRUE) {
    if ((k < p.n) && (p.key[k] == GlobalKey)) {
      Deleted = TRUE;
      removeitem(p, k);
    }
    else {
      /* Not found */
    }
  }
  else {
    deletesub(getNode(p.u[k].branch));
    if (UnderSize) {
      restore(p, k);
    }
  }
}

static void 
delete(void)
{
  Deleted = UnderSize = FALSE;
  /* 
   * Traverse from root recursively 
   */
  if (Root.offset == -1) {
    return;
  }
  Root = getNode(Root.offset);
  deletesub(Root);  
  if (Deleted == TRUE) {
    if (Root.n == 0) { /* If root is empty */
      if (Root.thisIsLeaf == TRUE) {
#ifdef DEBUG
        printf("I am a leaf\n");
#endif
        Root.offset = -1;
        alreadyInitRoot = FALSE;
      }
      else {
        Root = getNode(Root.u[0].branch);  
      }
      writeNode(Root);
    }
  } 
}

static void 
makeListForBtree(const INDEX *pIndex, const SCHEMA *pSchema)
{
  int tupleid;
  BTREE_DATA *pBtreeData;
  TUPLE *pTuple;
  int *pInt;
  int attrid;
  int byteOffset;

  pBtreeData = &HeadData;
  for (pTuple = pSchema->headTuple.next, tupleid = 0; 
       tupleid < pSchema->numOfObj; 
       pTuple = pTuple->next, tupleid ++) {
    if ((pBtreeData->next = (BTREE_DATA *)calloc(1, sizeof(BTREE_DATA))) == NULL) ERR;
    pBtreeData = pBtreeData->next;
    pBtreeData->next = NULL;

    for (byteOffset = 0, attrid = 0; 
         attrid < pSchema->numOfAttr; 
         attrid ++) {
      if (strcmp(pIndex->attrName, pSchema->attr[attrid].attrName) == 0) {
        pInt = (int *)(pTuple->obj + byteOffset);
        break;
      }
      else {
        byteOffset += pSchema->attr[attrid].sizOfAttr;
      }
    }
    pBtreeData->key = *pInt;
    pBtreeData->offset = pTuple->offset;
  }
}

static void
initBtree(const INDEX *pIndex, const SCHEMA *pSchema)
{
  int fd;

  Root.offset = -1;
  alreadyInitRoot = FALSE;
  bzero(SchemaName, BUFFSIZE);
  strcpy(SchemaName, pSchema->schemaName);
  bzero(IndexFileName, BUFFSIZE);
  sprintf(IndexFileName, "BTREE_DATA/%s/btree-%s", SchemaName, pIndex->indexName);
  if ((fd = open(IndexFileName, O_CREAT|O_WRONLY, 0644)) == -1) ERR;
  if (close(fd) == -1) ERR;
}

/****************************************************************
 *
 * Public Function
 *
 ***************************************************************/
extern void
execCreateBtree(INDEX *pIndex, SCHEMA *pSchema)
{
  BTREE_DATA *pBtreeData;

  initBtree(pIndex, pSchema);
  makeListForBtree(pIndex, pSchema);

  for (pBtreeData = HeadData.next;
       pBtreeData != NULL;
       pBtreeData = pBtreeData->next) {
    GlobalKey = pBtreeData->key;
    NewBtreeData = pBtreeData;
    insert();
  }
}

extern void
prepareBtree(INDEX *pIndex, SCHEMA *pSchema)
{
  int fd;

  bzero(IndexFileName, BUFFSIZE);
  sprintf(IndexFileName, "DATA/%s/btree-%s", pSchema->schemaName, pIndex->indexName);
  if ((fd = open(IndexFileName, O_RDWR)) == -1) ERR;
  if (read(fd, &(pIndex->rootBtreePage), sizeof(BTREE_PAGE)) == -1) ERR;
  if (close(fd) == -1) ERR;
}
