/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2005-2008, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Software Foundation
 * All rights reserved.
 *
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU General Public License 2.0      (with    KONOHA_UNDER_GPL2)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/* ************************************************************************ */

#include"commons.h"

/* ************************************************************************ */

#ifdef __cplusplus
extern "C" {
#endif

/* ======================================================================== */

static Term *knh_Stmt_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt);
static int TERMs_typing(Ctx *ctx, Stmt *stmt, size_t n, Asm *abr, NameSpace *ns, knh_type_t reqt, int mode);

#define KNH_ASM_ASSERT(ctx, abr, c)   KNH_ASSERT(c)

/* ======================================================================== */
/* [getter] */

static
knh_fieldn_t knh_Token_getfnq(Ctx *ctx, Token *o)
{
	if(SP(o)->tt == TT_NAME) {
		return knh_tName_get_fnq(ctx, knh_Token_tobytes(o), FIELDN_NEWID);
	}
	else if(SP(o)->tt == TT_FN) {
		return DP(o)->fn;
	}
	DBG2_P("SP(o)->tt=%s", knh_token_tochar(SP(o)->tt));
	KNH_ASSERT(ctx == NULL);
	return FIELDN_NONAME;
}

/* ------------------------------------------------------------------------ */

static
knh_methodn_t knh_Token_getmn(Ctx *ctx, Token *o)
{
	DEBUG3_ASSERT(IS_Token(o));
	if(SP(o)->tt == TT_MN) {
		return DP(o)->mn;
	}
	else if(SP(o)->tt == TT_FN) {
		knh_methodn_t mn = FIELDN_UNMASK(DP(o)->fn);
		if(knh_Token_isGetter(o)) {
			return mn | KNH_FLAG_MN_GETTER;
		}
		else if(knh_Token_isSetter(o)) {
			return mn | KNH_FLAG_MN_SETTER;
		}
		return mn;
	}

	knh_bytes_t name = knh_Token_tobytes(o);
	if(SP(o)->tt == TT_NAME) {
		knh_methodn_t mn = knh_tName_getMethodn(ctx, name, METHODN_NEWID);
		if(knh_Token_isGetter(o)) {
			return mn | KNH_FLAG_MN_GETTER;
		}
		if(knh_Token_isSetter(o)) {
			return mn | KNH_FLAG_MN_SETTER;
		}
		return mn;
	}
	else if(SP(o)->tt == TT_CMETHODN) {
		knh_index_t idx = knh_bytes_rindex(name, '.');
		if(idx != -1) {
			name = knh_bytes_last(name, idx+1);
		}
		return knh_tName_getMethodn(ctx, name, METHODN_NEWID);

	}
	else if(SP(o)->tt == TT_MT) {
		return knh_tName_getMethodn(ctx, name, METHODN_NEWID) | KNH_FLAG_MN_MOVTEXT;
	}
	DBG2_P("SP(o)->tt=%s, '%s'", knh_token_tochar(SP(o)->tt), (char*)name.buf);
	return knh_tName_getMethodn(ctx, name, METHODN_NEWID);
}

/* ======================================================================== */
/* [type] */

static
knh_type_t knh_Token_gettype(Ctx *ctx, Token *o, NameSpace *ns, knh_class_t defc)
{
	KNH_ASSERT(IS_Token(o));
	if(SP(o)->tt == TT_ASIS) {
		return TYPE_Any;
	}
	else if(SP(o)->tt == TT_CID) {
		return NNTYPE_cid(DP(o)->cid);
	}
	else {
		knh_bytes_t name = knh_Token_tobytes(o);
		knh_type_t cid;
		if(ISB(name, "void")) {
			return TYPE_void;
		}
		if(knh_Token_isExceptionType(o)) {
			cid = CLASS_Exception;
		}
		else {
			cid = knh_NameSpace_getClass(ctx, ns, name);
			if(cid == CLASS_unknown) {
				cid = defc;
				knh_Token_perror(ctx, o, KMSG_UCLASSN);
				knh_Token_perrata(ctx, o, CLASSN(defc));
			}
		}
		if(knh_Token_isIteratorType(o)) {
			cid = knh_pmzclass(ctx, CLASS_Iterator, cid, CLASS_Any);
		}
		else if(knh_Token_isArrayType(o)) {
			cid = knh_pmzclass(ctx, CLASS_Array, cid, CLASS_Any);
		}
		if(knh_Token_isNotNullType(o)) {
			return NNTYPE_cid(cid);
		}
		else {
			return cid;
		}
	}
}

/* ------------------------------------------------------------------------ */

static
knh_class_t knh_Token_getcid(Ctx *ctx, Token *o, NameSpace *ns, knh_class_t defc)
{
	DEBUG3_ASSERT(IS_Token(o));
	if(SP(o)->tt == TT_CID) {
		return DP(o)->cid;
	}
	else {
		if(knh_Token_isExceptionType(o)) {
			return CLASS_Exception;
		}
		knh_bytes_t name = knh_Token_tobytes(o);
		knh_class_t cid = knh_NameSpace_getClass(ctx, ns, name);
		if(cid == CLASS_unknown) {

			if(defc == CLASS_unknown) {
				return cid;
			}
			cid = defc;
			if(knh_Token_isIteratorType(o)) {
				cid = knh_pmzclass(ctx, CLASS_Iterator, cid, CLASS_Nue);
			}
			else if(knh_Token_isArrayType(o)) {
				cid = knh_pmzclass(ctx, CLASS_Array, cid, CLASS_Nue);
			}
			knh_Token_perror(ctx, o, KMSG_UCLASSN);
			knh_Token_perrata(ctx, o, CLASSN(cid));
		}
		if(knh_Token_isIteratorType(o)) {
			cid = knh_pmzclass(ctx, CLASS_Iterator, cid, CLASS_Nue);
		}
		else if(knh_Token_isArrayType(o)) {
			cid = knh_pmzclass(ctx, CLASS_Array, cid, CLASS_Nue);
		}
		return cid;
	}
}

/* ------------------------------------------------------------------------ */

void knh_Asm_perrorMTD(Ctx *ctx, Asm *abr, int pe, knh_class_t cid, knh_methodn_t mn)
{
	char bufcm[CLASSNAME_BUFSIZ];
	knh_format_cmethodn(bufcm, sizeof(bufcm), cid, mn);
	knh_Asm_perror(ctx, abr, pe, bufcm);
}

/* ------------------------------------------------------------------------ */

void knh_Asm_perrorMPR(Ctx *ctx, Asm *abr, int pe, knh_class_t scid, knh_class_t tcid)
{
	char bufcm[CLASSNAME_BUFSIZ*2];
	knh_snprintf(bufcm, sizeof(bufcm), "%s --> %s", CTXCLASSN(scid), CTXCLASSN(tcid));
	knh_Asm_perror(ctx, abr, pe, bufcm);
}

/* ======================================================================== */
/* [CONST] */

#define _new_TermCONST(ctx, fln, d)   TM(new_TokenCONST(ctx, fln, d))

Token* new_TokenCONST(Ctx *ctx, Any *fln, Any *data)
{
	knh_Token_t *o = (Token*)new_Object_malloc(ctx, FLAG_Token, CLASS_Token, sizeof(knh_Token_struct));
	knh_Token_struct_init(ctx, DP(o), 0, NULL);
	knh_Token_setFL(o, fln);
	knh_Token_setCONST(ctx, o, data);
	return o;
}

/* ------------------------------------------------------------------------ */

void knh_Token_setCONST(Ctx *ctx, Token *o, Any *data)
{
	KNH_SETv(ctx, DP(o)->data, data);
	knh_Token_toCONST(o);
}

/* ------------------------------------------------------------------------ */

Token* knh_Token_toCONST(Token *o)
{
	SP(o)->tt = TT_CONST;
	DP(o)->type = knh_Object_cid(DP(o)->data);
	if(IS_NOTNULL(DP(o)->data)) {
		DP(o)->type = NNTYPE_cid(DP(o)->type);
	}
	return o;
}

/* ------------------------------------------------------------------------ */

Token* new_TokenNULL(Ctx *ctx, Any *fln, knh_type_t type)
{
	Token *tk = new_TokenCONST(ctx, fln, KNH_NULL);
	DP(tk)->type = CLASS_type(type);
	return tk;
}

/* ======================================================================== */
/* [DEFVAL] */

static
Token* knh_Token_toDEFVAL(Token *o, knh_class_t cid)
{
	SP(o)->tt = TT_DEFVAL;
	DP(o)->cid = cid;
	if(cid != CLASS_Nue) {
		DP(o)->type = NNTYPE_cid(cid);
	}
	else {
		DP(o)->type = cid;
	}
	return o;
}

/* ======================================================================== */
/* [SYSVAL] */

#define IS_SYSVAL(t,v)  (knh_bytes_strcasecmp(t, STEXT(v)) == 0)

static
int knh_Token_toSYSVAL(Ctx *ctx, Token *tk, Asm *abr)
{
	knh_bytes_t t = knh_Token_tobytes(tk);
	if(IS_SYSVAL(t, "CTX")) {
		SP(tk)->tt = TT_SYSVAL;
		DP(tk)->index = KNH_SYS_CTX; DP(tk)->type = NNTYPE_Context;
		return 1;
	}
	else if(IS_SYSVAL(t, "IN") || IS_SYSVAL(t, "STDIN")) {
		SP(tk)->tt = TT_SYSVAL;
		DP(tk)->index = KNH_SYS_STDIN; DP(tk)->type = NNTYPE_InputStream;
		return 1;
	}
	else if(IS_SYSVAL(t, "OUT") || IS_SYSVAL(t, "STDOUT")) {
		SP(tk)->tt = TT_SYSVAL;
		DP(tk)->index = KNH_SYS_STDOUT; DP(tk)->type = NNTYPE_OutputStream;
		return 1;
	}
	else if(IS_SYSVAL(t, "ERR") || IS_SYSVAL(t, "STDERR")) {
		SP(tk)->tt = TT_SYSVAL;
		DP(tk)->index = KNH_SYS_STDERR; DP(tk)->type = NNTYPE_OutputStream;
		return 1;
	}
	else if(IS_SYSVAL(t, "OS")) {
		SP(tk)->tt = TT_SYSVAL;
		DP(tk)->index = KNH_SYS_OS; DP(tk)->type = NNTYPE_System;
		return 1;
	}
	else if(ISB(t, "__")) {
		SP(tk)->tt = TT_SYSVAL;
		DP(tk)->index = KNH_SYS_SCRIPT; DP(tk)->type = NNTYPE_Script;
		return 1;
	}
	else if(IS_SYSVAL(t, "__line__")) {
		knh_Token_setCONST(ctx, tk, UP(new_Int(ctx, SP(tk)->line)));
		return 1;
	}
	else if(IS_SYSVAL(t, "__file__")) {
		knh_Token_setCONST(ctx, tk, UP(knh_tfileid_name(SP(tk)->fileid)));
		return 1;
	}
	else if(IS_SYSVAL(t, "__method__") || IS_SYSVAL(t, "__function__")) {
		knh_Token_setCONST(ctx, tk, UP(knh_Method_getURN(ctx, DP(abr)->mtd)));
		return 1;
	}
	else if(IS_SYSVAL(t, "__namespace__") || IS_SYSVAL(t, "__ns__")) {
		NameSpace *ns = knh_Context_getNameSpace(ctx);
		knh_Token_setCONST(ctx, tk, UP(DP(ns)->nsname));
		return 1;
	}
	else if(IS_SYSVAL(t, "EOL")) {
		knh_Token_setCONST(ctx, tk, UP(TS_EOL));
		return 1;
	}
	else if(IS_SYSVAL(t, "BEGIN")) {
		knh_Token_setCONST(ctx, tk, UP(TS_BEGIN));
		return 1;
	}
	else if(IS_SYSVAL(t, "END")) {
		knh_Token_setCONST(ctx, tk, UP(TS_END));
		return 1;
	}
	return 0;
}

/* ======================================================================== */
/* [STACK, FIELD, GLOBAL, MEMBER] */

static
void knh_Token_toSTACK(Ctx *ctx, Token *tk, int sfpidx, knh_type_t type)
{
	SP(tk)->tt = TT_STACK;
	DP(tk)->index = (knh_short_t)sfpidx;
	DP(tk)->type = type;
}

/* ------------------------------------------------------------------------ */

static
void knh_Token_toFIELD(Ctx *ctx, Token *tk, int idx, knh_type_t type)
{
	SP(tk)->tt = TT_FIELD;
	DP(tk)->index = (knh_short_t)idx;
	DP(tk)->type = type;
}

/* ------------------------------------------------------------------------ */

static
void knh_Token_toGLOBAL(Ctx *ctx, Token *tk, int idx, knh_type_t type)
{
	SP(tk)->tt = TT_GLOBAL;
	DP(tk)->index = (knh_short_t)idx;
	DP(tk)->type = type;
}

/* ------------------------------------------------------------------------ */

//static
//void knh_Token_toMEMBER(Ctx *ctx, Token *tk, int sfi, int idx, knh_type_t type)
//{
//	SP(tk)->tt = TT_MEMBER;
//	DP(tk)->index = (knh_short_t)sfi;
//	KNH_SETv(ctx, DP(tk)->data, new_Int(ctx, idx));
//	DP(tk)->type = type;
//}

/* ------------------------------------------------------------------------ */

static
knh_index_t knh_Asm_addVariableTable(Ctx *ctx, Asm *abr,
		knh_cfield_t *cf, size_t max, knh_flag_t flag, knh_type_t type, knh_fieldn_t fn, Object *value)
{
	size_t idx;
	for(idx = 0; idx < max; idx++) {
		if(cf[idx].fn == FIELDN_NONAME) {
			cf[idx].flag = flag;
			cf[idx].fn   = fn;
			cf[idx].type = type;
			KNH_SETv(ctx, cf[idx].value, value);
			return idx;
		}
		if(cf[idx].fn == fn) {
			if(cf[idx].type != type) {
				char buf[CLASSNAME_BUFSIZ];
				knh_snprintf(buf, sizeof(buf), "%s%s %s: %s%s", TYPEQN(cf->type), FIELDN(fn), TYPEQN(type));
				knh_Asm_perror(ctx, abr, KMSG_DIFFDECL, buf);
				return -1;
			}
			return idx;
		}
	}
	{
		char buf[CLASSNAME_BUFSIZ];
		knh_snprintf(buf, sizeof(buf), "%s%s %s", TYPEQN(type), FIELDN(fn));
		knh_Asm_perror(ctx, abr, KMSG_TOOMANYVARS, buf);
		return -1;
	}
}

/* ------------------------------------------------------------------------ */
/* [variable] */

static
void knh_Asm_declareScriptVariable(Ctx *ctx, Asm *abr, knh_flag_t flag, knh_type_t type, knh_fieldn_t fn, Object *value)
{
	knh_Script_t *scr = knh_Asm_getScript(ctx, abr);
	knh_class_t cid = knh_Object_cid(scr);
	knh_cfield_t *cf = DP(knh_tClass[cid].cstruct)->fields;
	knh_index_t idx = knh_Asm_addVariableTable(ctx, abr, cf, KNH_SCRIPT_FIELDSIZE, flag, type, fn, value);
	if(idx != -1) {
		KNH_SETv(ctx, scr->fields[idx], value);
	}
}

/* ------------------------------------------------------------------------ */

static
knh_index_t knh_Asm_declareVariable(Ctx *ctx, Asm *abr, knh_flag_t flag, knh_type_t type, knh_fieldn_t fn, Object *value)
{
	knh_index_t idx =
		knh_Asm_addVariableTable(ctx, abr, (knh_cfield_t*)DP(abr)->vars, KONOHA_LOCALSIZE, flag, type, fn, value);
	if(idx != -1 && (idx + 1) > DP(abr)->vars_size) {
		DP(abr)->vars_size = (knh_ushort_t)(idx + 1);
	}
	return idx;
}

/* ------------------------------------------------------------------------ */

knh_index_t knh_Asm_indexOfVariable(Asm *abr, knh_fieldn_t fnq)
{
	knh_fieldn_t fn = FIELDN_UNMASK(fnq);
	knh_index_t idx = 0;
	for(idx = 0; idx < DP(abr)->vars_size; idx++) {
		//DBG2_P("idx=%d, fn=%s, fnq=%s", idx, FIELDN(DP(abr)->vars[idx].fn), FIELDN(fn));
		if(DP(abr)->vars[idx].fn == fn) {
			return idx;
		}
		if(DP(abr)->vars[idx].fn == FIELDN_NONAME) {
			return -1;
		}
	}
	return -1;
}

#define knh_Asm_indexOfScriptVariable(abr, fnq)\
	knh_Class_queryField(knh_Object_cid(knh_Asm_getScript(ctx, abr)), fnq)

#define knh_Asm_indexOfFieldVariable(abr, fnq)\
	knh_Class_queryField(DP(abr)->this_cid, fnq)

/* ------------------------------------------------------------------------ */

static
knh_cfield_t *knh_Asm_cfieldOfVariable(Asm *abr, knh_index_t idx)
{
	KNH_ASSERT(idx != -1 && idx < KONOHA_LOCALSIZE);
	KNH_ASSERT(DP(abr)->vars[idx].fn != FIELDN_NONAME);
	return (knh_cfield_t*)&(DP(abr)->vars[idx]);
}

/* ------------------------------------------------------------------------ */

static
int knh_Asm_globalIndex(Ctx *ctx, Asm *abr)
{
	if(DP(abr)->globalidx == -1) {
		DP(abr)->globalidx =
			knh_Asm_declareVariable(ctx, abr, 0, NNTYPE_Script, FIELDN_GLOBAL, UP(knh_Asm_getScript(ctx, abr)));
		if(DP(abr)->globalidx == -1) return 0;
	}
	return 1;
}

/* ------------------------------------------------------------------------ */

static int knh_TokenNAME_typing(Ctx *ctx, Token *tk, Asm *abr)
{
	knh_fieldn_t fnq = knh_Token_getfnq(ctx, tk);
	if(fnq == FIELDN_NONAME) {
		goto L_ERROR;
	}

	if(FIELDN_IS_U1(fnq) || FIELDN_IS_SUPER(fnq)) goto L_FIELD;  /* _name */
	if(FIELDN_IS_U2(fnq)) goto L_GLOBAL; /* __name */

	knh_index_t idx = knh_Asm_indexOfVariable(abr, FIELDN_UNMASK(fnq));
	if(idx != -1) {
		knh_cfield_t *cf = knh_Asm_cfieldOfVariable(abr, idx);
		knh_type_t type = knh_pmztype_totype(ctx, cf->type, DP(abr)->this_cid);
		knh_Token_toSTACK(ctx, tk, idx, type);
		return 1;
	}

	L_FIELD:;
	idx = knh_Class_queryField(DP(abr)->this_cid, fnq);
	if(idx != -1) {
		knh_cfield_t *cf = knh_Class_fieldAt(DP(abr)->this_cid, idx);
		knh_type_t type = knh_pmztype_totype(ctx, cf->type, DP(abr)->this_cid);
		knh_Token_toFIELD(ctx, tk, idx, type);
		return 1;
	}
	if(FIELDN_IS_SUPER(fnq)) goto L_ERROR;

	L_GLOBAL:;
	{
		Script *scr = knh_Asm_getScript(ctx, abr);
		idx = knh_Class_queryField(knh_Object_cid(scr), fnq);
		if(idx != -1) {
			knh_cfield_t *cf = knh_Class_fieldAt(knh_Object_cid(scr), idx);
			knh_type_t type = knh_pmztype_totype(ctx, cf->type, knh_Object_cid(scr));
			knh_Token_toGLOBAL(ctx, tk, idx, type);
			return knh_Asm_globalIndex(ctx, abr);
		}
	}
	if(knh_Token_toSYSVAL(ctx, tk, abr)) {
		return 1;
	}

	L_ERROR:;
	knh_Token_perror(ctx, tk, KMSG_UVARN);
	return 0;
}

/* ------------------------------------------------------------------------ */

static
int knh_TokenCONSTN_typing(Ctx *ctx, Token *o, Asm *abr, NameSpace *ns)
{
	knh_bytes_t t = knh_Token_tobytes(o);
	knh_index_t loc = knh_bytes_rindex(t, '.');
	if(loc == -1) {
		if(knh_Token_toSYSVAL(ctx, o, abr)) {
			return 1;
		}
		if(knh_NameSpace_existsConst(ctx, ns, t)) {
			knh_Token_setCONST(ctx, o, knh_NameSpace_getConst(ctx, ns, t));
			return 1;
		}
	}
	else {
		knh_class_t cid = knh_NameSpace_getClass(ctx, ns, knh_bytes_first(t, loc));
		if(cid != CLASS_unknown) {
			char buf[CLASSNAME_BUFSIZ*2];
			knh_snprintf(buf, sizeof(buf), "%s.%s", CLASSN(cid), &t.buf[loc+1]);
			t = B(buf);
			if(knh_tConst_exists(ctx, t)) {
				knh_Token_setCONST(ctx, o, knh_tConst_value(ctx, t));
				return 1;
			}
		}
	}
	knh_Token_perror(ctx, o, KMSG_UCONSTN);
	return 0;
}

/* ======================================================================== */
/* [CLASSID] */

//CLASSID                @system @P99
//FIELDN                 @system @P99
//METHODN                @system @P99
//OPERATOR               @system @P99
//NOP                    @system @P99

/* ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------ */
/* [CLASSID] */

static
int knh_TokenTYEPN_typing(Ctx *ctx, Token *o, Asm *abr, NameSpace *ns)
{
	knh_class_t cid = knh_Token_getcid(ctx, o, ns, CLASS_unknown);
	if(cid == CLASS_unknown) {
		if(knh_Token_toSYSVAL(ctx, o, abr)) {
			return 1;
		}
		knh_bytes_t t = knh_Token_tobytes(o);
		if(knh_NameSpace_existsConst(ctx, ns, t)) {
			knh_Token_setCONST(ctx, o, knh_NameSpace_getConst(ctx, ns, t));
			return 1;
		}
		knh_Token_perror(ctx, o, KMSG_UCLASSN);
		return 0;
	}
	KNH_ASSERT_cid(cid);
	SP(o)->tt = TT_CLASSID;
	DP(o)->cid = cid;
	DP(o)->type = NNTYPE_Class;
	KNH_SETv(ctx, DP(o)->data, knh_tClass[cid].class_cid);
	return 1;
}

/* ------------------------------------------------------------------------ */

static
int knh_TokenCMETHODN_typing(Ctx *ctx, Token *o, NameSpace *ns)
{
	knh_bytes_t t = knh_Token_tobytes(o);
	knh_index_t idx = knh_bytes_index(t, '.');
	if(idx == -1) return 0;

	knh_bytes_t cname = knh_bytes_first(t, idx);
	knh_class_t cid = knh_NameSpace_getClass(ctx, ns, cname);
	if(cid == CLASS_unknown) {
		knh_Token_perror(ctx, o, KMSG_UCLASSN);
		return 0;
	}
	t = knh_bytes_last(t, idx+1);
	knh_methodn_t mn = knh_tName_getMethodn(ctx, t, METHODN_NONAME);
	Method *mtd = knh_Class_getMethod(ctx, cid, mn);
	if(IS_NULL(mtd)) {
		knh_Token_perror(ctx, o, KMSG_UMETHODN);
		return 0;
	}
	knh_Token_setCONST(ctx, o, UP(mtd));
	return 1;
}

/* ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------ */
/* [NUM] */

static
knh_class_t knh_Token_tagcNUM(Ctx *ctx, Token *o, knh_class_t reqc, NameSpace *ns)
{
	knh_bytes_t t = knh_Token_tobytes(o), tag = STEXT("");
	size_t i = 1;
	int ishex = 0;
	if(t.buf[0] == '0' && (t.buf[1] == 'x' || t.buf[1] == 'b')) {
		i = 2;
		ishex = 1;
	}
	for(; i < t.len; i++) {
		if(isdigit(t.buf[i]) || t.buf[i] == '_' || t.buf[i] == '.') continue;
		if((t.buf[i] == 'L' || t.buf[i] == 'l')) {
			if(t.buf[i+1] == 0) {
				return reqc;
			}
			else if(t.buf[i+1] == '[' || t.buf[i+1] == ':') {
				continue;
			}
		}
		if((t.buf[i] == 'E' || t.buf[i] == 'e')) {
			if(isdigit(t.buf[i+1]) || t.buf[i+1] == '-') {
				continue;
			}
		}
		if(t.buf[i] == '[') {
			int loc;
			tag.buf = t.buf + i + 1;
			tag.len = t.len - (i + 1);
			loc = knh_bytes_index(tag, ']');
			if(loc > 0) {
				tag = knh_bytes_first(tag, loc);
			}
			break;
		}
		else if(t.buf[i] == ':') {
			tag.buf = t.buf + i + 1;
			tag.len = t.len - (i + 1);
			break;
		}
		else {
			tag.buf = t.buf + i;
			tag.len = t.len - (i);
			break;
		}
	}
	if(tag.len == 0 || ishex) {
		return reqc;
	}
	else {
		knh_class_t tagc = knh_NameSpace_tagcid(ctx, ns, reqc, tag);
		if(tagc == CLASS_unknown) {
			knh_Token_perror(ctx, o, KMSG_UTAG);
			return reqc;
		}
		return tagc;
	}
}

/* ------------------------------------------------------------------------ */

static
knh_class_t knh_bytes_guessNUMcid(Ctx *ctx, knh_bytes_t t)
{
	size_t i;
	if(t.buf[0] == 0 && (t.buf[1] == 'x' || t.buf[1]=='b')) {
		return CLASS_Int;
	}
	for(i = 1; i < t.len; i++) {
		if(t.buf[i] == '_') {
#ifdef CLASS_Decimal
			return CLASS_Decimal;
#endif
		}
		else if(t.buf[i] == '.') {
			return CLASS_Float;
		}
		if(!isdigit(t.buf[i])) break;
	}
	return CLASS_Int;
}

/* ------------------------------------------------------------------------ */

static
int knh_TokenNUM_typing(Ctx *ctx, Token *o, NameSpace *ns, knh_class_t reqc)
{
	KNH_ASSERT_cid(reqc);
	knh_bytes_t t = knh_Token_tobytes(o);

	if(reqc == CLASS_Boolean) {
		knh_perror(ctx, SP(o)->fileid, SP(o)->line, KMSG_WTRUEFALSE, NULL);
		if(t.buf[0] == '0') {
			knh_perrata(ctx, SP(o)->fileid, SP(o)->line, "0", "false");
			knh_Token_setCONST(ctx, o, KNH_FALSE);
		}
		else {
			knh_perrata(ctx, SP(o)->fileid, SP(o)->line, (char*)t.buf, "true");
			knh_Token_setCONST(ctx, o, KNH_TRUE);
		}
		return 1;
	}

	knh_class_t req_bcid = knh_tClass[reqc].bcid;
	if(req_bcid != CLASS_Int && req_bcid != CLASS_Float) {
		req_bcid = knh_bytes_guessNUMcid(ctx, t);
		reqc = req_bcid;
	}

	if(req_bcid == CLASS_Float) {
		knh_float_t n = 0.0;
		if(!knh_bytes_parsefloat(t, &n)) {
			knh_Token_perror(ctx, o, KMSG_EABORT);  // NUMOVERFLOW
			knh_perrata(ctx, SP(o)->fileid, SP(o)->line, (char*)t.buf, "0.0");
		}
		knh_class_t tagc = knh_Token_tagcNUM(ctx, o, req_bcid, ns);
		if(tagc == reqc) {
			knh_Token_setCONST(ctx, o, UP(new_FloatX(ctx, reqc, n)));
		}else{
			knh_Token_setCONST(ctx, o, UP(new_FloatX(ctx, tagc, n)));
		}
	}
	else { /* if(req_bcid == CLASS_Int) */
		knh_integer_t n = 0;
		if(!knh_bytes_parseinteger(t, &n)) {
			knh_Token_perror(ctx, o, KMSG_EABORT /*NUMOVERFLOW*/);
			knh_perrata(ctx, SP(o)->fileid, SP(o)->line, (char*)t.buf, "0");
		}
		knh_class_t tagc = knh_Token_tagcNUM(ctx, o, req_bcid, ns);
		if(tagc == reqc) {
			knh_Token_setCONST(ctx, o, UP(new_IntX(ctx, reqc, n)));
		}else{
			knh_Token_setCONST(ctx, o, UP(new_IntX(ctx, tagc, n)));
		}
	}
	return 1;
}

/* ------------------------------------------------------------------------ */

static
void knh_TokenTSTR_typing(Ctx *ctx, Token *o, NameSpace *ns, knh_class_t reqt)
{
	knh_bytes_t t = knh_Token_tobytes(o);
	knh_index_t loc = knh_bytes_index(t, ':');
	if(loc <= -1) {
		KNH_ASSERT(IS_String(DP(o)->data));
		knh_Token_toCONST(o);
	}
	else {
		if(DP(o)->tt_next == TT_ADD || DP(o)->tt_next == TT_FMT) {
			TODO();
		}
		knh_Token_setCONST(ctx, o, new_Object_parseOf(ctx, (String*)DP(o)->data));
	}
}

/* ------------------------------------------------------------------------ */
/* [typing_Token] */

static
int knh_Token_typing(Ctx *ctx, Token *tk, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	if(SP(tk)->tt != TT_ASIS && knh_Token_isTyped(tk)) return 1;
	knh_class_t reqc = CLASS_type(reqt);

	switch(SP(tk)->tt) {
	case TT_TYPEN:
		return knh_TokenTYEPN_typing(ctx, tk, abr, ns);
	case TT_CONSTN:
		return knh_TokenCONSTN_typing(ctx, tk, abr, ns);

	case TT_FN :
	case TT_NAME: {
		int res = knh_TokenNAME_typing(ctx, tk, abr);
		//DBG2_P("NAME tt=%s, index=%d", knh_token_tochar(SP(tk)->tt), DP(tk)->index)
		return res;
	}
	case TT_NUM:
		return knh_TokenNUM_typing(ctx, tk, ns, reqc);
	case TT_CMETHODN:
		return knh_TokenCMETHODN_typing(ctx, tk, ns);
	case TT_URN:
		KNH_ASSERT(IS_String(DP(tk)->data));
		knh_Token_toCONST(tk);
		return 1;

	case TT_STR:
		KNH_ASSERT(IS_String(DP(tk)->data));
		knh_Token_toCONST(tk);
//		if(knh_tClass[reqc].bcid == CLASS_String) {
//			DP(o)->type = reqc;
//		}
		return 1;

	case TT_TSTR:
		knh_TokenTSTR_typing(ctx, tk, ns, reqt);
		return 1;

	case TT_ASIS:
		/* This is used in DECL for default value */
		DP(tk)->cid  = reqc;
		DP(tk)->type = reqt;
		return 1;

	default:
		DBG2_P("unknown tt=%s", knh_token_tochar(SP(tk)->tt));
	}

	knh_Token_perror(ctx, tk, KMSG_ESYNTAX);
	return 0;
}

/* ======================================================================== */
/* [TERMs] */

static
int TERMs_isCONST(Stmt *stmt, size_t n)
{
	Token *tk = DP(stmt)->tokens[n];
	return (IS_Token(tk) && SP(tk)->tt == TT_CONST);
}

/* ------------------------------------------------------------------------ */

static Object *TERMs_const(Stmt *stmt, size_t n)
{
	KNH_ASSERT(TERMs_isCONST(stmt, n));
	Token *tk = DP(stmt)->tokens[n];
	return DP(tk)->data;
}

/* ------------------------------------------------------------------------ */

static int TERMs_isNULL(Stmt *stmt, size_t n)
{
	if(n < DP(stmt)->size) {
		return (IS_Token(DP(stmt)->tokens[n]) && IS_NULL(DP(DP(stmt)->tokens[n])->data));
	}
	return 0;
}

/* ------------------------------------------------------------------------ */

int TERMs_isTRUE(Stmt *stmt, size_t n)
{
	if(n < DP(stmt)->size) {
		return (IS_Token(DP(stmt)->tokens[n]) && IS_TRUE(DP(DP(stmt)->tokens[n])->data));
	}
	return 0;
}

/* ------------------------------------------------------------------------ */

int TERMs_isFALSE(Stmt *stmt, size_t n)
{
	if(n < DP(stmt)->size) {
		return (IS_Token(DP(stmt)->tokens[n]) && IS_FALSE(DP(DP(stmt)->tokens[n])->data));
	}
	return 0;
}

/* ------------------------------------------------------------------------ */

static Object *TERMs_value(Ctx *ctx, Stmt *stmt, size_t n, knh_type_t type)
{
	Token *tk = DP(stmt)->tokens[n];
	if(IS_Token(tk)) {
		if(SP(tk)->tt == TT_CONST) return DP(tk)->data;
		if(SP(tk)->tt == TT_ASIS) {
			return IS_NNTYPE(type) ?
					knh_tClass_defaultValueNN(ctx, CLASS_type(type)) : KNH_NULL;
		}
	}
	return KNH_NULL;
}

/* ------------------------------------------------------------------------ */

static int TERMs_isASIS(Stmt *stmt, size_t n)
{
	Token *tk = DP(stmt)->tokens[n];
	return (IS_Token(tk) && SP(tk)->tt == TT_ASIS);
}

/* ------------------------------------------------------------------------ */

static int TERMs_isCLASSID(Stmt *stmt, size_t n)
{
	Token *tk = DP(stmt)->tokens[n];
	return (IS_Token(tk) && SP(tk)->tt == TT_CLASSID);
}

/* ------------------------------------------------------------------------ */

#define _TERMs_getcid(stmt, n)    CLASS_type(TERMs_gettype(stmt, n))
#define _TERMs_getbcid(stmt, n)   knh_tClass[TERMs_getcid(stmt,n)].bcid

knh_type_t TERMs_gettype(Stmt *stmt, size_t n)
{
	if(IS_Token(DP(stmt)->tokens[n])) {
		Token *tk = DP(stmt)->tokens[n];
		KNH_ASSERT(knh_Token_isTyped(tk));
		return DP(tk)->type;
	}
	else {
		Stmt *stmt2 = DP(stmt)->stmts[n];
		KNH_ASSERT(knh_Stmt_isTyped(stmt2));
		return DP(stmt2)->type;
	}
}

/* ------------------------------------------------------------------------ */

static
void TERMs_perrorTYPE(Ctx *ctx, Stmt *stmt, size_t n, int pe, knh_type_t reqt)
{
//	switch(SP(stmt)->stt) {
//	case STT_CALL:
//	case STT_NEW:
//		{
//			char bufnp[CLASSNAME_BUFSIZ];
//			Method *mtd = DP(DP(stmt)->tokens[0])->mtd;
//			DEBUG3_ASSERT(IS_Method(mtd));
//			knh_format_methodparam(bufnp, sizeof(bufnp), DP(mtd)->mn, n - 1);
//			knh_perror(ctx, SP(stmt)->fileid, SP(stmt)->line, pe, bufnp);
//		}
//		break;
//	case STT_RETURN:
//		if(n == 0) {
//			knh_perror(ctx, SP(stmt)->fileid, SP(stmt)->line, pe, NULL);
//		}
//		else {
//			char bufn[40];
//			knh_snprintf(bufn, sizeof(bufn), "..., #%d", (int)(n + 1));
//			knh_perror(ctx, SP(stmt)->fileid, SP(stmt)->line, pe, bufn);
//		}
//		break;
//
//	case STT_LET:
//		knh_perror(ctx, SP(stmt)->fileid, SP(stmt)->line, pe, NULL);
//		break;
//
//	default :
		knh_perror(ctx, SP(stmt)->fileid, SP(stmt)->line, pe, NULL);
//	}
}


/* ------------------------------------------------------------------------ */

static
knh_type_t knh_Asm_typeinfer(Ctx *ctx, Asm *abr, knh_type_t type, char *varname)
{
	if(type == CLASS_Nue) type = CLASS_Any;

	if(type == CLASS_Any || type == NNTYPE_Any) {
		knh_Asm_perror(ctx, abr, KMSG_WTYPEINF, varname);
	}
	return type;
}

/* ======================================================================== */
/* [STMT] */

void knh_Stmt_setType(Ctx *ctx, Stmt *stmt, knh_type_t type)
{
	//DBG2_P("stt=%s, type=%s%s", knh_stmt_tochar(SP(stmt)->stt), TYPEQN(type));
	DP(stmt)->type = type;
	knh_Stmt_setTyped(stmt, 1);
}

/* ------------------------------------------------------------------------ */
/* [DECL] */

static
knh_bool_t knh_Asm_existsName(Ctx *ctx, Asm *abr, knh_fieldn_t fnq)
{
	knh_index_t idx = -1;
	knh_fieldn_t fn = FIELDN_UNMASK(fnq);
	if(FIELDN_IS_U2(fnq)) goto L_GLOBAL;
	if(FIELDN_IS_U1(fnq)) goto L_FIELD;

	idx = knh_Asm_indexOfVariable(abr, fn);
	if(idx != -1) return 1;

	L_FIELD:;
	idx = knh_Class_queryField(DP(abr)->this_cid, fnq);
	if(idx != -1) return 1;

	L_GLOBAL:;
	idx = knh_Class_queryField(knh_Object_cid(knh_Asm_getScript(ctx, abr)), fnq);
	if(idx != -1) return 1;
	return 0;
}

/* ------------------------------------------------------------------------ */

static
Object *knh_StmtDECL_value(Ctx *ctx, knh_type_t type)
{
	return IS_NNTYPE(type) ? knh_tClass_defaultValueNN(ctx, CLASS_type(type)) : KNH_NULL;
}

/* ------------------------------------------------------------------------ */

static void knh_StmtDECL_toLET(Ctx *ctx, Stmt *stmt)
{
	DEBUG3_ASSERT(SP(stmt)->stt == STT_DECL);
	SP(stmt)->stt = STT_LET;
	KNH_SETv(ctx, DP(stmt)->terms[0], DP(stmt)->terms[1]);
	KNH_SETv(ctx, DP(stmt)->terms[1], DP(stmt)->terms[2]);
	KNH_SETv(ctx, DP(stmt)->terms[2], KNH_NULL);
	DP(stmt)->size = 2;
}

/* ------------------------------------------------------------------------ */

Term * knh_StmtDECL_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	int level = DP(abr)->level;
	knh_fieldn_t fnq = knh_Token_getfnq(ctx, StmtDECL_name(stmt));
	knh_fieldn_t fn  = FIELDN_UNMASK(fnq);
	knh_flag_t flag  = knh_Stmt_metaflag__field(ctx, stmt);

	knh_type_t pmztype  = knh_Token_gettype(ctx, StmtDECL_type(stmt), ns, CLASS_Any);
	knh_type_t type = knh_pmztype_totype(ctx, pmztype, DP(abr)->this_cid);
	//knh_class_t var_cid = CLASS_type(type);

	/* Type name value */
	if(!TERMs_typing(ctx, stmt, 2, abr, ns, type, TCHECK_)) {
		return NULL;
	}

	if(level == 0) {  /* SCRIPT VARIABLE */
		flag = flag | KNH_FLAG_CFF_GETTER | KNH_FLAG_CFF_SETTER;
		Object *value = knh_StmtDECL_value(ctx, type);
		knh_Asm_declareScriptVariable(ctx, abr, flag, type, fn, value);
		if(TERMs_isASIS(stmt, 2)) {
			knh_Stmt_done(ctx, stmt);
		}
		else {
			knh_StmtDECL_toLET(ctx, stmt);
		}
	}
	else if(level == 1) { /* FIELD VARIABLE */
		if(!FIELDN_IS_U1(fnq)) {
			flag |= KNH_FLAG_CFF_GETTER | KNH_FLAG_CFF_SETTER;
		}
		Object *value = TERMs_value(ctx, stmt, 2, type);
		if(!TERMs_isCONST(stmt, 2) && !TERMs_isASIS(stmt, 2)) {
			knh_Asm_perror(ctx, abr, KMSG_IGFIELDVALUE, FIELDN(fn));
		}
		knh_Asm_declareVariable(ctx, abr, flag, type, fn, value);
		knh_Stmt_done(ctx, stmt);
	}
	else if(level == -1 || level == -2) {  /* -1 OUTER_PARAM, -2: INNER_PARAM */
		Object *value = KNH_NULL;
		if(TERMs_isCONST(stmt, 2)) {
			value = TERMs_const(stmt, 2);
			if(IS_NOTNULL(value)) {
				if(TERMs_isASIS(stmt, 0)) {  /* (a = 1) ==> Int a = 1 */
					type = knh_Object_cid(value);
				}
				else {  /* Int! a = 1 ==> Int a = 1 */
					type = CLASS_type(type);
				}
				if(level == -2) {
					type = NNTYPE_cid(type);  // a = 1 ==> Int! if inner
				}
			}
		}
		knh_Asm_declareVariable(ctx, abr, flag, type, fn, value);
	}
	else {
		if(FIELDN_IS_U2(fnq) || FIELDN_IS_U1(fnq)) {
			knh_Asm_perror(ctx, abr, KMSG_EABORT/*KMSG_IGQNAME*/, knh_Token_tochar(StmtDECL_name(stmt)));
		}
		Object *value = knh_StmtDECL_value(ctx, type);
		if(TERMs_isASIS(stmt, 2)) {
			knh_Asm_declareVariable(ctx, abr, flag, type, fn, value);
			knh_Stmt_done(ctx, stmt);
		}
		else {
			knh_Asm_declareVariable(ctx, abr, flag, type, fn, value);
			knh_StmtDECL_toLET(ctx, stmt);
		}
	}
	return TM(stmt);
}

/* ======================================================================== */
/* [LET] */

static
int TERMs_isCONSTNAME(Ctx *ctx, Stmt *stmt, size_t n, Asm *abr, NameSpace *ns)
{
	Token *tk = DP(stmt)->tokens[n];
	if(!IS_Token(tk)) return 0;
	if(SP(tk)->tt == TT_CONSTN) {
		if(knh_Token_toSYSVAL(ctx, tk, abr)) {
			return 0;
		}
		return 1;
	}
	if(SP(tk)->tt == TT_TYPEN) {
		knh_bytes_t cname = knh_Token_tobytes(tk);
		size_t i;
		for(i = 0; i < cname.len; i++) {
			if(islower(cname.buf[i])) return 0;
		}
		knh_class_t cid = knh_NameSpace_getClass(ctx, ns, cname);
		return (cid == CLASS_unknown);
	}
	return 0;
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtLET_declCONST(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, int level)
{
	Token *tk = DP(stmt)->tokens[0];
	int pe = KMSG_ESYNTAX;
	if(level > 1) {
		pe = KMSG_EABORT /*KMSG_NOTHERE*/; goto L_PERROR;
	}

	if(!TERMs_typing(ctx, stmt, 1, abr, ns, TYPE_Any, TWARN_)) {
		return NULL;
	}

	if(!TERMs_isCONST(stmt, 1)) {
		pe = KMSG_NNCONST; goto L_PERROR;
	}

	Object *value = TERMs_const(stmt, 1);
	knh_bytes_t cn = knh_Token_tobytes(tk);
	knh_index_t dotidx = knh_bytes_index(cn, '.');

	if(level == 0) {
		if(dotidx == -1) {
			if(knh_NameSpace_existsConst(ctx, ns, cn)) {
				pe = KMSG_DUPCONST; goto L_PERROR;
			}
			KNH_ASSERT(IS_String(DP(tk)->text));
			knh_NameSpace_addConst(ctx, ns, DP(tk)->text, value);
		}
		else {
			knh_bytes_t fn = knh_bytes_first(cn, dotidx);
			knh_class_t cid = knh_NameSpace_getClass(ctx, ns, fn);
			if(cid == CLASS_unknown) {
				pe = KMSG_UCLASSN; goto L_PERROR;
			}
			if(!knh_tConst_addClassConst(ctx, cid, knh_bytes_last(cn, dotidx+1), value)) {
				pe = KMSG_DUPCONST; goto L_PERROR;
			}
		}
	}
	else if(level == 1) {
		knh_class_t cid = DP(abr)->this_cid;
		if(dotidx != -1) {
			pe = KMSG_EABORT/*KMSG_NOTHERE*/; goto L_PERROR;
		}
		if(!knh_tConst_addClassConst(ctx, cid, knh_bytes_last(cn, dotidx+1), value)) {
			pe = KMSG_DUPCONST; goto L_PERROR;
		}
	}
	knh_Stmt_done(ctx, stmt);
	return TM(stmt);

	L_PERROR:
	knh_Token_perror(ctx, tk, pe);
	return NULL;
}

/* ------------------------------------------------------------------------ */

Term *knh_StmtLET_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	int level = DP(abr)->level;
	if(TERMs_isCONSTNAME(ctx, stmt, 0, abr, ns)) {
		return knh_StmtLET_declCONST(ctx, stmt, abr, ns, level);
	}
	if(knh_Token_isTyped(DP(stmt)->tokens[0])) {
		return TM(stmt);
	}

	knh_fieldn_t fnq = knh_Token_getfnq(ctx, DP(stmt)->tokens[0]);
	knh_fieldn_t fn = FIELDN_UNMASK(fnq);
	if(fn == FIELDN_NONAME) {
		knh_Asm_perror(ctx, abr, KMSG_ELVALUE, FIELDN(fn));
		return NULL;
	}

	knh_flag_t  flag = KNH_FLAG_CFF_AUTONAME;
	int existsName = 0;
	if(level == 0) {  /* SCRIPT LEVEL */
		if(knh_Asm_indexOfScriptVariable(abr, fnq) != -1) {
			existsName = 1;
		}
	}
	else if(level > 1) {  /* LOCAL LEVEL */
		if(knh_Asm_existsName(ctx, abr, fnq)) {
			existsName = 1;
		}
		else if(FIELDN_IS_U1(fnq) || FIELDN_IS_U2(fnq)) {
			knh_Token_perror(ctx, DP(stmt)->tokens[0], KMSG_UVARN);
			return NULL;
		}
	}

	if(existsName == 0){   /* TYPE INFERENCING : a = 1 */
		if(!TERMs_typing(ctx, stmt, 1, abr, ns, CLASS_Any, TWARN_)) {
			return NULL;
		}
		knh_type_t type = TERMs_gettype(stmt, 1);
		type = knh_Asm_typeinfer(ctx, abr, type, FIELDN(fn));
		Object *value = TERMs_value(ctx, stmt, 1, type);

		if(level == 0) { /* SCRIPT LEVEL */
			Script *scr = knh_Asm_getScript(ctx, abr);
			DP(abr)->this_cid = knh_Object_cid(scr);
			//KNH_ASSERT(DP(abr)->vars_size == 1);  /* That means prepared */
			flag |= KNH_FLAG_CFF_GETTER | KNH_FLAG_CFF_SETTER;
			knh_Asm_declareScriptVariable(ctx, abr, flag, type, fn, value);
		}
		else if(level == 1) {  /* CLASS FIELD LEVEL */
			if(!TERMs_isCONST(stmt, 1)) {
				knh_Asm_perror(ctx, abr, KMSG_IGFIELDVALUE, FIELDN(fn));
			}
			// flag |= KNH_FLAG_CFF_GETTER | KNH_FLAG_CFF_SETTER;
			knh_Asm_declareVariable(ctx, abr, flag, type, fn, value);
			knh_Stmt_done(ctx, stmt);
			return TM(stmt);
		}
		else if(level > 1) { /* LOCAL_LEVEL */
			knh_Asm_declareVariable(ctx, abr, flag, type, fn, value);
		}
		if(!TERMs_typing(ctx, stmt, 0, abr, ns, TYPE_Any, TCHECK_)) {
			return NULL;
		}
		knh_Stmt_setType(ctx, stmt, type);
	}
	else {
		Token *tk = DP(stmt)->tokens[0];
		if(!TERMs_typing(ctx, stmt, 0, abr, ns, TYPE_Any, TCHECK_)) {
			return NULL;
		}
		tk = DP(stmt)->tokens[0];
		if(SP(tk)->tt != TT_STACK && SP(tk)->tt == TT_GLOBAL && SP(tk)->tt == TT_FIELD) {
			knh_Asm_perror(ctx, abr, KMSG_ELVALUE, FIELDN(fn));
			return NULL;
		}
		knh_type_t type = TERMs_gettype(stmt, 0);
		if(!TERMs_typing(ctx, stmt, 1, abr, ns, type, TCHECK_)) {
			return NULL;
		}
		knh_Stmt_setType(ctx, stmt, type);
	}
	if(reqt != TYPE_void) {
		TODO();
	}
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */
/* [CALL1] */

static
Term *knh_StmtCALL1_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	DEBUG3_ASSERT(DP(stmt)->size == 1);
	if(reqt == TYPE_void) {
		TERMs_typing(ctx, stmt, 0, abr, ns, TYPE_Any, TWARN_);
		knh_Stmt_setType(ctx, stmt, TERMs_gettype(stmt, 0));
		return TM(stmt);
	}
	else {
		TERMs_typing(ctx, stmt, 0, abr, ns, reqt, TCHECK_);
		return DP(stmt)->terms[0];
	}
}

/* ------------------------------------------------------------------------ */
/* [CALL] */

static
Term *knh_StmtCALLBASE_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_methodn_t mn)
{
	Token *tk = DP(stmt)->tokens[1];
	if(IS_Stmt(tk) || SP(tk)->tt != TT_ASIS) {
		if(!TERMs_typing(ctx, stmt, 1, abr, ns, CLASS_Any, TWARN_)) {
			return NULL;
		}
		if(TERMs_isCLASSID(stmt, 1)) {
			Token *tk = DP(stmt)->tokens[1];
			knh_Token_toDEFVAL(tk, DP(tk)->cid);
		}
		return TM(stmt);
	}

	/* (func _ ...) */
	/* built-in function */
	int pe = KMSG_EBUILTINFUNC;
	if(mn == METHODN_typeof) {
		if(DP(stmt)->size != 3) goto L_ERROR;
		if(!TERMs_typing(ctx, stmt, 2, abr, ns, CLASS_Any, TWARN_)) {
			knh_Token_setCONST(ctx, tk, UP(new_Class__type(ctx, TERMs_gettype(stmt, 2))));
			return TM(tk);
		}
		return NULL;
	}
	else if(mn == METHODN_default) {
		if(DP(stmt)->size != 3) goto L_ERROR;
		if(!TERMs_typing(ctx, stmt, 2, abr, ns, CLASS_Any, TWARN_)) {
			if(TERMs_isCLASSID(stmt, 2)) {
				tk = DP(stmt)->tokens[2];
				knh_Token_toDEFVAL(tk, DP(tk)->cid);
			}
			else {
				knh_Token_toDEFVAL(tk, TERMs_getcid(stmt, 2));
			}
			return TM(tk);
		}
		return NULL;
	}
	else if(mn == METHODN_defined) {
		if(DP(stmt)->size != 3) goto L_ERROR;
		if(TERMs_typing(ctx, stmt, 2, abr, ns, CLASS_Any, TWARN_)) {
			knh_Token_setCONST(ctx, tk, KNH_TRUE);
		}
		else {
			knh_Token_setCONST(ctx, tk, KNH_FALSE);
		}
		return TM(tk);
	}
	else {
		char bufmn[CLASSNAME_BUFSIZ];
		knh_format_methodn(bufmn, sizeof(bufmn), mn);

		//DEBUG3("1. lookup function.. %s()", bufmn);
		knh_class_t mtd_cid = knh_NameSpace_getFuncClass(ctx, ns, B(bufmn));
		if(mtd_cid != CLASS_unknown) {
			KNH_ASSERT_cid(mtd_cid);
			knh_Token_toDEFVAL(tk, mtd_cid);
			return TM(stmt);
		}

		//DEBUG3("2. lookup this.%s()", bufmn);
		if(DP(abr)->vars[0].fn == FIELDN_this) {
			mtd_cid = CLASS_type(DP(abr)->vars[0].type);
			Method *mtd = knh_Class_getMethod(ctx, mtd_cid, mn);
			if(IS_NOTNULL(mtd)) {
				knh_Token_toSTACK(ctx, tk, 0, DP(abr)->vars[0].type);
				return TM(stmt);
			}
		}

		//DEBUG3("3. lookup script function %s()", bufmn);
		Script *scr = knh_Asm_getScript(ctx, abr);
		mtd_cid = knh_Object_cid(scr);
		Method *mtd = knh_Class_getMethod(ctx, mtd_cid, mn);
		//KNH_ASSERT(IS_NOTNULL(mtd));
		if(IS_NOTNULL(mtd)) {
			knh_Token_setCONST(ctx, tk, UP(scr));
			return TM(stmt);
		}
		pe = KMSG_UMETHODN;
	}

	L_ERROR:;
	knh_Asm_perror(ctx, abr, pe, knh_Token_tochar(DP(stmt)->tokens[0]));
	return NULL;
}

/* ------------------------------------------------------------------------ */

static
void KNH_BOX(Ctx *ctx, knh_sfp_t *sfp, knh_type_t type)
{
	knh_class_t cid = CLASS_type(type);
	KNH_ASSERT_cid(cid);
	knh_class_t bcid = knh_tClass[cid].bcid;
	if(bcid == CLASS_Int || bcid == CLASS_Float || bcid == CLASS_Boolean) {
		if(IS_NNTYPE(type) || IS_NOTNULL(sfp[0].o)) {
			KNH_SETv(ctx, sfp[0].o, knh_boxing(ctx, sfp, cid));
		}
	}
}

/* ------------------------------------------------------------------------ */

static void KNH_UNBOX(Ctx *ctx, knh_sfp_t *sfp)
{
	sfp[0].data = (sfp[0].i)->n.ivalue;
}

/* ------------------------------------------------------------------------ */

static Term *knh_StmtCALL_toCONST(Ctx *ctx, Stmt *stmt, Method *mtd)
{
	if(!knh_Method_isConst(mtd)) return TM(stmt);
	if(DP(mtd)->mn == METHODN_isNull && IS_NNTYPE(TERMs_gettype(stmt, 1))) {
		return TM(new_TokenCONST(ctx, DP(stmt)->terms[0], KNH_FALSE));
	}
	if(DP(mtd)->mn == METHODN_isNotNull && IS_NNTYPE(TERMs_gettype(stmt, 1))) {
		return TM(new_TokenCONST(ctx, DP(stmt)->terms[0], KNH_TRUE));
	}

	knh_sfp_t *lsfp = KNH_LOCAL(ctx);
	int i;
	for(i = 1; i < DP(stmt)->size; i++) {
		Token *tk = DP(stmt)->tokens[i];
		if(!IS_Token(tk) || SP(tk)->tt != TT_CONST) {
			return TM(stmt);
		}
		KNH_MOV(ctx, lsfp[i].o, DP(tk)->data);
		KNH_UNBOX(ctx, &lsfp[i]);
	}
	DEBUG3("STMT TO CONST ..");
	KNH_SCALL(ctx, lsfp, 0, mtd, DP(stmt)->size);
	KNH_BOX(ctx, &lsfp[0], knh_Method_rtype(mtd));
	knh_Token_setCONST(ctx, DP(stmt)->tokens[0], lsfp[0].o);
	return DP(stmt)->terms[0];
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtPARAMS_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_class_t mtd_cid, Method *mtd)
{
	KNH_ASSERT(IS_Method(mtd));
	size_t i, size = DP(stmt)->size;
	knh_type_t rtype = knh_pmztype_totype(ctx, knh_Method_rtype(mtd), mtd_cid);
	knh_Stmt_setType(ctx, stmt, rtype);

	for(i = 0; i < knh_Method_psize(mtd); i++) {
		knh_type_t reqt = knh_pmztype_totype(ctx, knh_Method_ptype(mtd, i), mtd_cid);
		size_t n = i + 2; //DEBUG3("reqt[%d]=%d,%s%s", n, reqt, TYPEQN(reqt));
		if(n < size) {
			if(!TERMs_typing(ctx, stmt, n, abr, ns, reqt, TCHECK_)) {
				return NULL;
			}
		}
		else {
			if(IS_NNTYPE(reqt)) {
				knh_Asm_perrorMTD(ctx, abr, KMSG_ETOOFEWPARAMS, mtd_cid, DP(mtd)->mn);
				return NULL;
			}
			else if(!knh_Method_isVarArgs(mtd)) {
				Token *tk = new_TokenCONST(ctx, UP(stmt), KNH_NULL);
				DP(tk)->type = CLASS_type(reqt);
				knh_Stmt_add(ctx, stmt, UP(tk));
			}
		}
	}
	if(knh_Method_isVarArgs(mtd)) {
		knh_type_t reqt = knh_pmztype_totype(ctx, knh_Method_ptype(mtd, knh_Method_psize(mtd) - 1), mtd_cid);
		for(/*i = knh_Method_psize(mtd)*/; i < size; i++) {
			if(!TERMs_typing(ctx, stmt, i, abr, ns, reqt, TCHECK_)) {
				return NULL;
			}
		}
	}
	else if(i + 2 != size) {
		//DBG2_P("i+2=%d, size=%d", i+2, size);
		knh_Asm_perrorMTD(ctx, abr, KMSG_WTOOMANYPARAMS, mtd_cid, DP(mtd)->mn);
		DP(stmt)->size = i + 2;
	}
	return knh_StmtCALL_toCONST(ctx, stmt, mtd);
}

/* ------------------------------------------------------------------------ */

static
void knh_Token_toMTD(Ctx *ctx, Token *tk, knh_methodn_t mn, Method *mtd)
{
	SP(tk)->tt = TT_MN;
	DP(tk)->mn = mn;
	KNH_SETv(ctx, DP(tk)->data, mtd);
	DP(tk)->type = TYPE_Method;
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtCALL_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_class_t reqt)
{
	DEBUG3_ASSERT(DP(stmt)->size > 1);
	knh_methodn_t mn = knh_Token_getmn(ctx, DP(stmt)->tokens[0]);
	Term *tm = knh_StmtCALLBASE_typing(ctx, stmt, abr, ns, mn);
	if(tm == NULL || IS_Token(tm)) {
		return tm;
	}

	knh_type_t    btype   = TERMs_gettype(stmt, 1);
	knh_class_t   mtd_cid = CLASS_type(btype);
	Method *mtd = knh_Class_getMethod(ctx, mtd_cid, mn);
	if(mtd_cid != CLASS_Any && IS_NULL(mtd)) {
		knh_Asm_perrorMTD(ctx, abr, KMSG_UMETHODN, mtd_cid, mn);
		return NULL;
	}

	knh_Token_toMTD(ctx, DP(stmt)->tokens[0], mn, mtd);

	if(IS_NOTNULL(mtd)) {
		return knh_StmtPARAMS_typing(ctx, stmt, abr, ns, mtd_cid, mtd);
	}
	else {
		int i;
		for(i = 2; i < DP(stmt)->size; i++) {
			if(!TERMs_typing(ctx, stmt, i, abr, ns, TYPE_Any, TWARN_)) {
				return NULL;
			}
		}
		knh_Stmt_setType(ctx, stmt, TYPE_Any);
		return UP(stmt);
	}
}

/* ------------------------------------------------------------------------ */
/* [NEW] */

static
Term *knh_StmtNEW_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_class_t reqt)
{
	DEBUG3_ASSERT(DP(stmt)->size > 1);
	//knh_flag_t flag = knh_Stmt_metaflag__new(ctx, stmt);
	knh_methodn_t mn = knh_Token_getmn(ctx, DP(stmt)->tokens[0]);

	knh_class_t mtd_cid = knh_Token_getcid(ctx, DP(stmt)->tokens[1], ns, CLASS_unknown);
	if(mtd_cid == CLASS_Exception) {
		mn = METHODN_new__init;
	}

	if(mtd_cid == CLASS_unknown) {
		knh_Token_perror(ctx, DP(stmt)->tokens[1], KMSG_UCLASSN);
		knh_Token_perrata(ctx, DP(stmt)->tokens[0], "null");
		return TM(new_TokenNULL(ctx, FL(DP(stmt)->tokens[1]), reqt));
	}


	if(mtd_cid == CLASS_Array) {
		knh_class_t reqc = CLASS_type(reqt); KNH_ASSERT_cid(reqc);
		if(knh_tClass[reqc].bcid == CLASS_Array) {
			mtd_cid = reqc;
		}
		else if(reqc == CLASS_Any && mn == METHODN_new__init && DP(stmt)->size > 2) {
			if(!TERMs_typing(ctx, stmt, 2, abr, ns, CLASS_Any, TWARN_)) {
				return NULL;
			}
			knh_type_t icid = TERMs_getcid(stmt, 2);
			if(icid != CLASS_Any) {
				int i;
				for(i = 3; i < DP(stmt)->size; i++) {
					if(!TERMs_typing(ctx, stmt, i, abr, ns, CLASS_Any, TWARN_)) {
						return NULL;
					}
					if(icid != TERMs_getcid(stmt, i)) {
						icid = CLASS_Any;
						break;
					}
				}
				if(icid != CLASS_Any) {
					mtd_cid = knh_class_Array(ctx, icid);
				}
			}
		}
	}

	KNH_ASSERT_cid(mtd_cid);

	Method *mtd = knh_Class_getMethod(ctx, mtd_cid, mn);
	if(IS_NULL(mtd)) {
		char bufnw[CLASSNAME_BUFSIZ];
		knh_snprintf(bufnw, sizeof(bufnw), "%s %s(...)", knh_Token_tochar(DP(stmt)->tokens[0]), CLASSN(mtd_cid));
		knh_Asm_perror(ctx, abr, KMSG_UNEW, bufnw);
		return NULL;
	}

//	if(!knh_Token_isExceptionType(DP(stmt)->tokens[1])) {
//		KNH_SETv(ctx, DP(stmt)->terms[1], new_TokenCONST(ctx, FL(DP(stmt)->terms[0]), UP(knh_tClass[mtd_cid].class_cid)));
//	}
	knh_Token_toMTD(ctx, DP(stmt)->tokens[0], mn, mtd);
	knh_StmtPARAMS_typing(ctx, stmt, abr, ns, mtd_cid, mtd);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */
/* [OP] */

/* ======================================================================== */
/* [OP] */

/* ------------------------------------------------------------------------ */
//1 2 3 4  => (((1 + 2) + 3) + 4)

static
Term *knh_StmtTOBINARY_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	size_t i;
	while(DP(stmt)->size > 3) {
		Stmt *newstmt = new_Stmt(ctx, 0, STT_OP);
		knh_Stmt_add(ctx, newstmt, DP(stmt)->terms[0]);
		knh_Stmt_add(ctx, newstmt, DP(stmt)->terms[1]);
		knh_Stmt_add(ctx, newstmt, DP(stmt)->terms[2]);
		KNH_SETv(ctx, DP(stmt)->terms[1], newstmt);
		DP(stmt)->size -= 1;
		for(i = 2; i < DP(stmt)->size; i++) {
			KNH_SETv(ctx, DP(stmt)->terms[i], DP(stmt)->terms[i+1]);
		}
	}
	return knh_StmtEXPR_typing(ctx, stmt, abr, ns, reqt);
}

/* ------------------------------------------------------------------------ */

static
knh_class_t knh_StmtOPADD_basecid(Ctx *ctx, Stmt *stmt)
{
	KNH_ASSERT(DP(stmt)->size == 3);

	knh_class_t cid1 = TERMs_getcid(stmt, 1);
	knh_class_t cid2 = TERMs_getcid(stmt, 2);

	if(cid1 == CLASS_Any || cid2 == CLASS_Any) return CLASS_Any;
	if(cid1 == cid2) return cid1;

	knh_class_t bcid1 = knh_tClass[cid1].bcid;
	knh_class_t bcid2 = knh_tClass[cid2].bcid;

	if(bcid1 == bcid2) return bcid1;

	if(bcid1 == CLASS_Float && bcid2 == CLASS_Int) {
		return cid1;
	}
	if(bcid2 == CLASS_Float && bcid1 == CLASS_Int) {
		return cid2;
	}
	return cid1;
}

/* ------------------------------------------------------------------------ */

static
knh_class_t knh_StmtOPEQ_basecid(Ctx *ctx, Stmt *stmt)
{
	knh_class_t cid1 = TERMs_getcid(stmt, 1);
	knh_class_t cid2 = TERMs_getcid(stmt, 2);

	if(cid1 == cid2) return cid1;
	if(cid1 == CLASS_Any || cid2 == CLASS_Any) return CLASS_Any;

	knh_class_t bcid1 = knh_tClass[cid1].bcid;
	knh_class_t bcid2 = knh_tClass[cid2].bcid;

	if(bcid1 == cid2) return bcid1;
	if(bcid2 == cid1) return bcid2;

	if(bcid1 == CLASS_Float && bcid2 == CLASS_Int) {
		return cid1;
	}
	if(bcid2 == CLASS_Float && bcid1 == CLASS_Int) {
		return cid2;
	}
	return CLASS_unknown;
}

/* ------------------------------------------------------------------------ */

static
int knh_Stmt_checkOPSIZE(Ctx *ctx,Stmt *stmt, size_t n)
{
	if(n < 2) {
		knh_perror(ctx, SP(stmt)->fileid, SP(stmt)->line, KMSG_EABORT, NULL);
		return 0;
	}
	return 1;
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtOP_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	DEBUG3_ASSERT(DP(stmt)->size > 1);
	size_t opsize = DP(stmt)->size - 1;
	Token *optk = DP(stmt)->tokens[0];
	knh_methodn_t mn = knh_token_tomethodn(SP(optk)->tt);
	KNH_ASM_ASSERT(ctx, abr, mn != METHODN_NONAME);
	knh_class_t mtd_cid = CLASS_unknown;

	size_t i;
	for(i = 1; i < opsize + 1; i++) {
		if(!TERMs_typing(ctx, stmt, i, abr, ns, CLASS_Any, TWARN_)) {
			return NULL;
		}
		if(TERMs_getbcid(stmt, i) == CLASS_String) {
			mtd_cid = CLASS_String;
		}
	}

	switch(mn) {
	case METHODN_opAdd:
	{
		if(opsize == 1) { /* +1 */
			return DP(stmt)->terms[1];
		}
		if(mtd_cid == CLASS_String) {
			if(opsize > 2) {
				mn = METHODN_concat;
			}
			break;
		}
	}
	case METHODN_opSub:
	case METHODN_opMul:
	case METHODN_opDiv:
	{
		if(opsize > 2) {
			return knh_StmtTOBINARY_typing(ctx, stmt, abr, ns, reqt);
		}
		if(!knh_Stmt_checkOPSIZE(ctx, stmt, 2)) {
			return NULL;
		}
		mtd_cid = knh_StmtOPADD_basecid(ctx, stmt);
		break;
	}

	case METHODN_opEq:
	{
		if(!knh_Stmt_checkOPSIZE(ctx, stmt, 2)) {
			return NULL;
		}
		if(TERMs_isNULL(stmt, 1)) {
			mn = METHODN_isNull;
			mtd_cid = CLASS_Object;
			DP(stmt)->size = 2;
			break;
		}
		mtd_cid = knh_StmtOPEQ_basecid(ctx, stmt);
		if(mtd_cid == CLASS_unknown) {
			knh_Asm_perror(ctx, abr, KMSG_WOPCMP, NULL);
			knh_Token_perrata(ctx, optk, "false");
			return new_TermCONST(ctx, FL(optk), KNH_FALSE); /* CONST */
		}
		break;
	}
	case METHODN_opNeq:
	{
		if(!knh_Stmt_checkOPSIZE(ctx, stmt, 2)) {
			return NULL;
		}
		if(TERMs_isNULL(stmt, 1)) {
			mn = METHODN_isNotNull;
			mtd_cid = CLASS_Object;
			DP(stmt)->size = 2;
			break;
		}
		mtd_cid = knh_StmtOPEQ_basecid(ctx, stmt);
		if(mtd_cid == CLASS_unknown) {
			knh_Asm_perror(ctx, abr, KMSG_WOPCMP, NULL);
			knh_Token_perrata(ctx, optk, "true");
			return new_TermCONST(ctx, FL(optk), KNH_TRUE); /* CONST */
		}
		break;
	}

	case METHODN_opGt: case METHODN_opGte:
	case METHODN_opLt: case METHODN_opLte:
	{
		if(!knh_Stmt_checkOPSIZE(ctx, stmt, 2)) {
			return NULL;
		}
		mtd_cid = knh_StmtOPEQ_basecid(ctx, stmt);
		if(mtd_cid == CLASS_unknown) {
			knh_Asm_perror(ctx, abr, KMSG_WOPCMP, NULL);
			return NULL;
		}
		break;
	}

	case METHODN_opHas:
	{
		if(!knh_Stmt_checkOPSIZE(ctx, stmt, 2)) {
			return NULL;
		}
		Term *temp = DP(stmt)->terms[1];
		DP(stmt)->terms[1] = DP(stmt)->terms[2];
		DP(stmt)->terms[2] = temp;
		mtd_cid = TERMs_getcid(stmt, 1);
		break;
	}

	case METHODN_opLor:
	{
		if(opsize == 1) {
			mn = METHODN_opSize;
			mtd_cid = TERMs_getcid(stmt, 1);
		}
		else {
			mtd_cid = CLASS_Int;
		}
	}
	break;

	default:
		mtd_cid = TERMs_getcid(stmt, 1);
		break;
	}

	KNH_ASM_ASSERT(ctx, abr, mtd_cid != CLASS_unknown);
	Method *mtd = knh_Class_getMethod(ctx, mtd_cid, mn);
	if(IS_NULL(mtd)) {
		knh_Asm_perror(ctx, abr, KMSG_UOP, knh_token_tochar(SP(optk)->tt));
		return NULL;
	}
	knh_Token_toMTD(ctx, DP(stmt)->tokens[0], mn, mtd);
	if(!TERMs_typing(ctx, stmt, 1, abr, ns, mtd_cid, TCHECK_)) {
		return NULL;
	}
	return knh_StmtPARAMS_typing(ctx, stmt, abr, ns, mtd_cid, mtd);
}

/* ------------------------------------------------------------------------ */
/* [MAPCAST] */

static
void knh_Token_toMPR(Ctx *ctx, Token *tk, knh_class_t cid, Mapper *mpr)
{
	SP(tk)->tt = TT_MPR;
	DP(tk)->cid = cid;
	KNH_SETv(ctx, DP(tk)->data, mpr);
	DP(tk)->type = knh_Mapper_isTotal(mpr) ? NNTYPE_cid(DP(mpr)->tcid) : DP(mpr)->tcid;
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtMAPCAST_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	DEBUG3_ASSERT(DP(stmt)->size > 1);
	if(!TERMs_typing(ctx, stmt, 0, abr, ns, TYPE_Any, TWARN_)) {
		return NULL;
	}
	if(!TERMs_isCLASSID(stmt, 0)) {
		knh_Token_perror(ctx, DP(stmt)->tokens[0], KMSG_UCLASSN);
		return NULL;
	}

	Token *tk = DP(stmt)->tokens[0];
	knh_class_t mprcid = DP(tk)->cid;
	if(mprcid == CLASS_Any) mprcid = CLASS_type(reqt);

	if(!TERMs_typing(ctx, stmt, 1, abr, ns, mprcid, TWARN_)) {
		return NULL;
	}
	if(mprcid == CLASS_Any) return DP(stmt)->terms[1];

	if(TERMs_isNULL(stmt, 1)) {
		if(knh_Token_isNotNullType(tk)) {
			knh_Token_setCONST(ctx, DP(stmt)->tokens[1], knh_tClass_defaultValueNN(ctx, mprcid));
		}
		return DP(stmt)->terms[1];
	}

	knh_type_t exprt = TERMs_gettype(stmt, 1);
	knh_type_t exprc = CLASS_type(exprt);
	knh_Stmt_setType(ctx, stmt, knh_Token_isNotNullType(tk) ? NNTYPE_cid(mprcid) : mprcid);

	if(exprc == CLASS_Any && mprcid != CLASS_Any) {
		knh_Token_toMPR(ctx, DP(stmt)->tokens[0], mprcid, (Mapper*)KNH_NULL);
		return TM(stmt);
	}

	if(mprcid == exprc || knh_class_instanceof(mprcid, exprc)) {
		if(knh_Token_isNotNullType(tk)) {
			DEBUG3("Nonnull cast");
			if(!IS_NNTYPE(exprt)) {
				knh_Stmt_setNNCAST(stmt, 1);
				knh_Token_toMPR(ctx, tk, mprcid, (Mapper*)KNH_NULL);
				return TM(stmt);
			}
		}
		DEBUG3("UPCAST (%s)%s", CLASSN(mprcid), CLASSN(exprc));
		return DP(stmt)->terms[1];
	}

	Mapper *mpr = knh_Class_getMapper(ctx, exprc, mprcid);
	if(IS_NULL(mpr)) {
		knh_Asm_perrorMPR(ctx, abr, KMSG_UMAP, exprc, mprcid);
		return NULL;
	}

	if(knh_Mapper_isConst(mpr) && TERMs_isCONST(stmt, 1)) {
		DEBUG3("MAPCAST TO CONST ..");
		Token *tk2 = DP(stmt)->tokens[1];
		knh_sfp_t *lsfp = KNH_LOCAL(ctx);
		KNH_MOV(ctx, lsfp[0].o, DP(tk2)->data);
		KNH_UNBOX(ctx, &lsfp[0]);
		KNH_SMAP(ctx, lsfp, 0, mpr);
		KNH_BOX(ctx, &lsfp[0], DP(mpr)->tcid);
		knh_Token_setCONST(ctx, tk2, lsfp[0].o);
		return TM(tk2);
	}
	knh_Token_toMPR(ctx, tk, mprcid, mpr);
	return TM(stmt);
}

/* ======================================================================== */
/* [MT,AND,OR,] */

static
Term *knh_StmtMT_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	DEBUG3_ASSERT(DP(stmt)->size > 1);
	Token *tk = DP(stmt)->tokens[0];
	knh_methodn_t mn = knh_Token_getmn(ctx, tk);
	if(!TERMs_typing(ctx, stmt, 1, abr, ns, CLASS_Any, TCHECK_)) {
		return NULL;
	}
	if(knh_bytes_isOptionalMT(knh_Token_tobytes(tk))) {
		KNH_ASSERT(DP(stmt)->size == 2);
		KNH_ASSERT(IS_String(DP(tk)->data));
		knh_Stmt_add(ctx, stmt, TM(new_TokenCONST(ctx, FL(tk), DP(tk)->data)));
	}
	knh_Token_toMTD(ctx, tk, mn, (Method*)KNH_NULL);
	knh_Stmt_setType(ctx, stmt, NNTYPE_String);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtAND_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	size_t i;
	for(i = 0; i < DP(stmt)->size; i++) {
		if(!TERMs_typing(ctx, stmt, i, abr, ns, NNTYPE_Boolean, TCHECK_)) {
			return NULL;
		}
		if(TERMs_isTRUE(stmt, i)) {
			return new_TermCONST(ctx, UP(stmt), KNH_TRUE);
		}
	}
	knh_Stmt_setType(ctx, stmt, NNTYPE_Boolean);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtOR_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	size_t i;
	for(i = 0; i < DP(stmt)->size; i++) {
		if(!TERMs_typing(ctx, stmt, i, abr, ns, NNTYPE_Boolean, TCHECK_)) {
			return NULL;
		}
		if(TERMs_isTRUE(stmt, i)) {
			return new_TermCONST(ctx, UP(stmt), KNH_TRUE);
		}
	}
	knh_Stmt_setType(ctx, stmt, NNTYPE_Boolean);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtALT_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	DEBUG3_ASSERT(DP(stmt)->size > 1);
	size_t i;
	knh_class_t reqc = CLASS_type(reqt);
	for(i = 0; i < DP(stmt)->size; i++) {
		if(!TERMs_typing(ctx, stmt, i, abr, ns, reqc, TCHECK_)) {
			return NULL;
		}
		knh_type_t type = TERMs_gettype(stmt, i);
		if(IS_NNTYPE(type)) {
			DP(stmt)->size = i + 1;
			knh_Stmt_setType(ctx, stmt, NNTYPE_cid(reqc));
			return TM(stmt);
		}
	}
	knh_Stmt_setType(ctx, stmt, reqt);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtTRI_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	DEBUG3_ASSERT(DP(stmt)->size == 3);
	int res = 1;
	if(!TERMs_typing(ctx, stmt, 0, abr, ns, NNTYPE_Boolean, TCHECK_)) res = 0;
	if(!TERMs_typing(ctx, stmt, 1, abr, ns, reqt, TCHECK_)) res = 0;
	if(!TERMs_typing(ctx, stmt, 2, abr, ns, reqt, TCHECK_)) res = 0;
	if(res == 0) return NULL;

	if(TERMs_isCONST(stmt, 0)) {
		if(IS_TRUE(TERMs_const(stmt, 0))) {
			return DP(stmt)->terms[1];
		}
		else {
			return DP(stmt)->terms[2];
		}
	}
	knh_type_t type = TERMs_gettype(stmt, 1);
	knh_type_t type2 = TERMs_gettype(stmt, 2);
	if(IS_NNTYPE(type) && IS_NNTYPE(type2)) {
		reqt = NNTYPE_cid(reqt);
	}
	knh_Stmt_setType(ctx, stmt, reqt);
	return TM(stmt);
}

/* ======================================================================== */

Term *knh_StmtEXPR_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_class_t reqt)
{
	knh_stmt_t stt = SP(stmt)->stt;
	switch(stt) {
	case STT_CALL1:
		return knh_StmtCALL1_typing(ctx, stmt, abr, ns, reqt);
	case STT_LET:
		return knh_StmtLET_typing(ctx, stmt, abr, ns, reqt);
	case STT_CALL:
		return knh_StmtCALL_typing(ctx, stmt, abr, ns, reqt);
	case STT_NEW:
		return knh_StmtNEW_typing(ctx, stmt, abr, ns, reqt);
	case STT_OP:
		return knh_StmtOP_typing(ctx, stmt, abr, ns, reqt);
	case STT_MAPCAST:
		return knh_StmtMAPCAST_typing(ctx, stmt, abr, ns, reqt);
	case STT_MT:
		return knh_StmtMT_typing(ctx, stmt, abr, ns, reqt);
	case STT_AND:
		return knh_StmtAND_typing(ctx, stmt, abr, ns, reqt);
	case STT_OR:
		return knh_StmtOR_typing(ctx, stmt, abr, ns, reqt);
	case STT_ALT:
		return knh_StmtALT_typing(ctx, stmt, abr, ns, reqt);
	case STT_TRI:
		return knh_StmtTRI_typing(ctx, stmt, abr, ns, reqt);
	default:
		DEBUG3("undefined stmt=%s", knh_stmt_tochar(stt));
	}
	return NULL;
}

/* ------------------------------------------------------------------------ */

static
Term *new_TermINCAST(Ctx *ctx, knh_class_t reqc, Stmt *stmt, size_t n)
{
	knh_class_t varc = TERMs_getcid(stmt, n);
	Mapper *mpr = knh_Class_getMapper(ctx, varc, reqc);
	if(IS_NULL(mpr) && varc != TYPE_Any) {
		DBG2_P("cannot convert");
		return NULL;
	}
	if(TERMs_isCONST(stmt, n)) {
		Token *tk2 = DP(stmt)->tokens[n];
		knh_sfp_t *lsfp = KNH_LOCAL(ctx);
		KNH_MOV(ctx, lsfp[0].o, DP(tk2)->data);
		KNH_UNBOX(ctx, &lsfp[0]);
		KNH_SMAP(ctx, lsfp, 0, mpr);
		KNH_BOX(ctx, &lsfp[0], DP(mpr)->tcid);
		knh_Token_setCONST(ctx, tk2, lsfp[0].o);
		return TM(tk2);
	}
	else {
		Token *tk = new_TokenNULL(ctx, FL(stmt), CLASS_Any);
		knh_Token_toMPR(ctx, tk, reqc, mpr);
		Stmt *cstmt = new_Stmt(ctx, 0, STT_MAPCAST);
		knh_Stmt_add(ctx, cstmt, TM(tk));
		knh_Stmt_add(ctx, cstmt, DP(stmt)->terms[n]);
		knh_Stmt_setType(ctx, cstmt, DP(tk)->type);
		return TM(cstmt);
	}
}

/* ------------------------------------------------------------------------ */

static
int TERMs_typecheck(Ctx *ctx, Stmt *stmt, size_t n, Asm *abr, knh_type_t reqt, int mode)
{
	knh_type_t vart = TERMs_gettype(stmt, n);
	knh_class_t varc = CLASS_type(vart);
	knh_class_t reqc = CLASS_type(reqt);

	if(TERMs_isNULL(stmt, n)) {
		//DBG2_P("reqc=%s varc=%s", CTXCLASSN(reqc), CTXCLASSN(varc));
		if(IS_NNTYPE(reqt) && varc != reqc) {
			TERMs_perrorTYPE(ctx, stmt, n, KMSG_EABORT /*KMSG_ENULL*/, reqt);
			return 0;
		}
		return 1;
	}

	if(vart == TYPE_void) {
		TERMs_perrorTYPE(ctx, stmt, n, KMSG_EABORT/*KMSG_EVOID*/, reqt);
		return 0;
	}

	if(reqt == NNTYPE_Boolean && vart != NNTYPE_Boolean) {
		//DBG2_P("stt=%s, vart=%s%s", knh_stmt_tochar(SP(stmt)->stt), TYPEQN(vart));
		TERMs_perrorTYPE(ctx, stmt, n, KMSG_TPEXPR/*KMSG_EVOID*/, reqt);
		return 0;
	}

	if(vart == reqt || reqc == CLASS_Any || knh_class_instanceof(varc, reqc)) return 1;

	if(varc == CLASS_Any ||
		(varc == CLASS_Float && reqc == CLASS_Int) ||
		(varc == CLASS_Int && CLASS_Float)) {
		mode = TCONV_;
	}

	if(mode == TCONV_) {
		Term *mcast = new_TermINCAST(ctx, reqc, stmt, n);
		if(mcast == NULL) {
			return 0;
		}
		KNH_SETv(ctx, DP(stmt)->stmts[n], mcast);
		return 1;
	}
	return 0;
}

/* ------------------------------------------------------------------------ */

static
int TERMs_typing(Ctx *ctx, Stmt *stmt, size_t n, Asm *abr, NameSpace *ns, knh_type_t reqt, int mode)
{
	if(SP(stmt)->stt == STT_ERR) return 0;

	if(IS_Token(DP(stmt)->tokens[n])) {
		Token *tk = DP(stmt)->tokens[n];
		if(SP(tk)->tt == TT_ASIS || !knh_Token_isTyped(tk) ) {
			if(!knh_Token_typing(ctx, tk, abr, ns, reqt)) {
				return 0;
			}
		}
	}
	else {
		Stmt *stmt_n = DP(stmt)->stmts[n];
		if(!knh_Stmt_isTyped(stmt_n)) {
			Term *tm = knh_Stmt_typing(ctx, stmt_n, abr, ns, reqt);
			if(tm == NULL) {
				return 0;
			}
			if(tm != DP(stmt)->terms[n]) {
				KNH_SETv(ctx, DP(stmt)->stmts[n], tm);
			}
		}
	}
	if(reqt == TYPE_Any) return 1;
	if(mode == TWARN_) return 1;
	return TERMs_typecheck(ctx, stmt, n, abr, reqt, mode);
}

/* ======================================================================== */

static
int TERMs_typingBLOCK(Ctx *ctx, Stmt *stmt, size_t n, Asm *abr, NameSpace *ns)
{
	KNH_ASSERT(IS_Stmt(DP(stmt)->terms[n]));
	return knh_Stmt_typingBLOCK(ctx, DP(stmt)->stmts[n], abr, ns, 1);
}

/* ======================================================================== */
/* [IF] */

void knh_Stmt_toBLOCK(Ctx *ctx, Stmt *stmt, size_t n)
{
	KNH_ASSERT(DP(stmt)->size > 0);
	SP(stmt)->stt = STT_BLOCK;
	KNH_SETv(ctx, DP(stmt)->terms[0], DP(stmt)->terms[n]);
	DP(stmt)->size = 1;
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtIF_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	if(!TERMs_typing(ctx, stmt, 0, abr, ns, NNTYPE_Boolean, TCHECK_)) {
		return NULL;
	}

	if(TERMs_isCONST(stmt, 0)) {
		if(IS_TRUE(TERMs_const(stmt, 0))) {
			TERMs_typingBLOCK(ctx, stmt, 1, abr, ns);
			knh_Stmt_done(ctx, DP(stmt)->stmts[2]);
			knh_Stmt_toBLOCK(ctx, stmt, 1);
		}
		if(IS_FALSE(TERMs_const(stmt, 0))) {
			knh_Stmt_done(ctx, DP(stmt)->stmts[1]);
			TERMs_typingBLOCK(ctx, stmt, 2, abr, ns);
			knh_Stmt_toBLOCK(ctx, stmt, 2);
		}
	}
	else {
		TERMs_typingBLOCK(ctx, stmt, 1, abr, ns);
		TERMs_typingBLOCK(ctx, stmt, 2, abr, ns);
	}
	return TM(stmt);
}


/* ------------------------------------------------------------------------ */

static
Term *knh_StmtSWITCH_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	knh_Asm_perror(ctx, abr, KMSG_EFUTURE, "switch");
	knh_Stmt_done(ctx, stmt);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtWHILE_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	if(!TERMs_typing(ctx, stmt, 0, abr, ns, NNTYPE_Boolean, TCHECK_)) {
		return NULL;
	}

	if(TERMs_isCONST(stmt, 0)) {
		if(IS_TRUE(TERMs_const(stmt, 0))) { /* while(true) */
			TERMs_typingBLOCK(ctx, stmt, 1, abr, ns);
			knh_Stmt_toBLOCK(ctx, stmt, 1);
		}
		if(IS_FALSE(TERMs_const(stmt, 0))) { /* while(false) */
			knh_Stmt_done(ctx, stmt);
		}
	}
	else {
		TERMs_typingBLOCK(ctx, stmt, 1, abr, ns);
	}
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtDO_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	TERMs_typingBLOCK(ctx, stmt, 0, abr, ns);
	if(!TERMs_typing(ctx, stmt, 1, abr, ns, NNTYPE_Boolean, TCHECK_)) {
		return NULL;
	}
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtFOR_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	/* i = 0*/
	if(!TERMs_typingBLOCK(ctx, stmt, 0, abr, ns)) return NULL;
	/* i < N */
	if(!TERMs_typing(ctx, stmt, 1, abr, ns, NNTYPE_Boolean, TCHECK_)) return NULL;

	/* i++ */
	if(!TERMs_typingBLOCK(ctx, stmt, 2, abr, ns)) return NULL;

	if(TERMs_isCONST(stmt, 1)) {
		if(IS_TRUE(TERMs_const(stmt, 1))) {
			TERMs_typingBLOCK(ctx, stmt, 3, abr, ns);
			knh_Stmt_toBLOCK(ctx, stmt, 3);
		}
		if(IS_FALSE(TERMs_const(stmt, 0))) {
			return DP(stmt)->terms[0];
		}
	}
	else {
		TERMs_typingBLOCK(ctx, stmt, 3, abr, ns);
	}
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */
/* [FOREACH] */

static
knh_fieldn_t knh_tName_local(Ctx *ctx, const char *fmt, int level)
{
	char bufn[40];
	knh_snprintf(bufn, sizeof(bufn), fmt, level);
	knh_bytes_t tname = B(bufn);
	knh_index_t idx = knh_DictIdx_index(ctx, DP(knh_rootSystem)->tfieldnDictIdx, tname);
	if(idx == -1) {
		String *s = new_String(ctx, tname, NULL);
		idx = knh_DictIdx_add__fast(ctx, DP(knh_rootSystem)->tfieldnDictIdx, s);
	}
	return (knh_fieldn_t)idx;
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtFOREACH_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	knh_type_t type = TYPE_Any;
	knh_fieldn_t fn = knh_Token_getfnq(ctx, StmtFOREACH_name(stmt));
	if(TERMs_isASIS(stmt, FOREACH_type)) {  /* foreach(n in itr) */
		knh_index_t idx = knh_Asm_indexOfVariable(abr, fn);
		if(idx == -1) { DEBUG3("type inferencing..");
			if(!TERMs_typing(ctx, stmt, FOREACH_iter, abr, ns, TYPE_Any, TWARN_)) {
				return NULL;
			}
			if(TERMs_getbcid(stmt, FOREACH_iter) != CLASS_Iterator) {
				Stmt *stmt_call = new_Stmt(ctx, 0, STT_CALL);
				Token *tk = new_TokenMN(ctx, FL(stmt), METHODN_opItr);
				knh_Stmt_add(ctx, stmt_call, UP(tk));
				knh_Stmt_add(ctx, stmt_call, StmtFOREACH_iter(stmt));
				KNH_SETv(ctx, DP(stmt)->stmts[FOREACH_iter], stmt_call);
				if(!TERMs_typing(ctx, stmt, FOREACH_iter, abr, ns, NNTYPE_Iterator, TCHECK_)) {
					return NULL;
				}
			}
			knh_class_t itrcid = TERMs_getcid(stmt, FOREACH_iter);
			type = NNTYPE_cid(knh_tClass[itrcid].p1);
			type = knh_Asm_typeinfer(ctx, abr, type, FIELDN(fn));
			knh_Asm_declareVariable(ctx, abr, 0, type, fn, knh_tClass_defaultValueNN(ctx, CLASS_type(type)));
			if(!TERMs_typing(ctx, stmt, FOREACH_name, abr, ns, type, TCHECK_)) {
				return NULL;
			}
			goto L_BLOCK;
		}
	}
	else { DEBUG3("defined type .."); /* foreach(String line in itr) */
		type = knh_Token_gettype(ctx, StmtFOREACH_type(stmt), ns, CLASS_Any);
		type = knh_pmztype_totype(ctx, type, DP(abr)->this_cid);
		knh_index_t idx = knh_Asm_indexOfVariable(abr, fn);
		if(idx == -1) { DEBUG3("defining new type ..");
			knh_Asm_declareVariable(ctx, abr, /*flag*/0, type, fn, KNH_NULL);
		}
	}
	DBG2_P("type=%s%s", TYPEQN(type));
	if(!TERMs_typing(ctx, stmt, FOREACH_name, abr, ns, type, TWARN_)) {
		return NULL;
	}
	knh_class_t itrcid = knh_class_Iterator(ctx, TERMs_getcid(stmt, FOREACH_name));
	if(!TERMs_typing(ctx, stmt, FOREACH_iter, abr, ns, NNTYPE_cid(itrcid), TWARN_)) {
		return NULL;
	}

	L_BLOCK:;
	int sfpidx = knh_Asm_declareVariable(ctx, abr, 0, TYPE_Any,
			knh_tName_local(ctx, "__S%d__", DP(abr)->level), KNH_NULL);
	DBG2_P("sfpidx=%d", sfpidx);
	if(sfpidx == -1) return NULL;
	Token *tkitr = new_TokenNULL(ctx, FL(stmt), itrcid);
	knh_Token_toSTACK(ctx, tkitr, sfpidx, itrcid);
	knh_Stmt_add(ctx, stmt, TM(tkitr));

	if(!TERMs_typing(ctx, stmt, FOREACH_where, abr, ns, NNTYPE_Boolean, TCHECK_)) {
		return NULL;
	}
	if(TERMs_isFALSE(stmt, FOREACH_where)) {
		knh_Stmt_done(ctx, stmt);
		return TM(stmt);
	}
	TERMs_typingBLOCK(ctx, stmt, FOREACH_loop, abr, ns);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtTRY_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	int sfpidx = knh_Asm_declareVariable(ctx, abr,
			0, TYPE_Any, knh_tName_local(ctx, "__E%d__", DP(abr)->level), KNH_NULL);
	DBG2_P("sfpidx=%d", sfpidx);
	if(sfpidx == -1) return NULL;
	Token *tkhdr = new_TokenNULL(ctx, FL(stmt), CLASS_ExceptionHandler);
	knh_Token_toSTACK(ctx, tkhdr, sfpidx, CLASS_ExceptionHandler);
	knh_Stmt_add(ctx, stmt, TM(tkhdr));

	TERMs_typingBLOCK(ctx, stmt, TRY_try, abr, ns);
	{
		Stmt *instmt = DP(stmt)->stmts[TRY_catch];
		while(SP(instmt)->stt == STT_CATCH) {
			knh_fieldn_t fn = knh_Token_getfnq(ctx, DP(instmt)->tokens[1]);
			if(fn == FIELDN_NONAME) {
				knh_Asm_perror(ctx, abr, KMSG_ESYNTAX, knh_Token_tochar(DP(instmt)->tokens[1]));
				return NULL;
			}
			knh_Asm_declareVariable(ctx, abr, 0, TYPE_Exception, fn, KNH_NULL);
			if(TERMs_typing(ctx, instmt, 1, abr, ns, TYPE_Exception, TCHECK_)) {
				knh_Stmt_done(ctx, instmt);
			}
			else {
				TERMs_typingBLOCK(ctx, instmt, 2, abr, ns);
			}
			instmt = DP(instmt)->next;
		}
	}
	TERMs_typingBLOCK(ctx, stmt, TRY_finally, abr, ns);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtTHROW_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	if(!TERMs_typing(ctx, stmt, 0, abr, ns, NNTYPE_Exception, TCHECK_)) {
		return NULL;
	}
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
void knh_StmtRETURN_addValue(Ctx *ctx, Stmt *stmt, knh_type_t rtype)
{
	KNH_ASSERT(SP(stmt)->stt == STT_RETURN && DP(stmt)->size == 0);
	if(rtype != TYPE_void) {
		Token *tk = new_TokenNULL(ctx, FL(stmt), rtype);
		if(IS_NNTYPE(rtype)) {
			knh_Token_toDEFVAL(tk, CLASS_type(rtype));
		}
		knh_Stmt_add(ctx, stmt, TM(tk));
	}
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtRETURN_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	knh_type_t rtype = DP(abr)->rtype;
	if(rtype == TYPE_void) {
		if(DP(stmt)->size > 0) {
			knh_Asm_perror(ctx, abr, KMSG_EABORT/*KMSG_IGGRETURN*/, NULL);
			DP(stmt)->size = 0;
		}
	}
	else if(knh_Method_isNew(DP(abr)->mtd)) {
		if(DP(stmt)->size > 0) {
			knh_Asm_perror(ctx, abr, KMSG_IGRETURN, NULL);
			DP(stmt)->size = 1;
		}
		else {
			Token *tk = new_TokenNULL(ctx, FL(stmt), rtype);
			knh_Stmt_add(ctx, stmt, TM(tk));
		}
		knh_Token_toSTACK(ctx, DP(stmt)->tokens[0], 0, rtype);
	}
	else if(DP(stmt)->size == 0) {
		knh_Asm_perror(ctx, abr, KMSG_NORETURNVALUE, NULL);
		knh_StmtRETURN_addValue(ctx, stmt, rtype);
	}
	else if(!TERMs_typing(ctx, stmt, 0, abr, ns, rtype, TCHECK_)){
		return NULL;
	}
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtPRINT_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	size_t i;
	for(i = 0; i < DP(stmt)->size; i++) {
		if(!TERMs_typing(ctx, stmt, i, abr, ns, TYPE_Any, TWARN_)) {
			return NULL;
		}
	}
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
Term *knh_StmtASSERT_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	if(!TERMs_typing(ctx, stmt, 0, abr, ns, NNTYPE_Boolean, TWARN_)) {
		return NULL;
	}
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------ */
/* [METHOD] */

static
knh_type_t knh_StmtMETHOD_rtype(Ctx *ctx, Stmt *stmt, NameSpace *ns)
{
	return knh_Token_gettype(ctx, DP(stmt)->tokens[0], ns, CLASS_Any);
}

/* ------------------------------------------------------------------------ */

static
knh_class_t knh_StmtMETHOD_class(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, int level)
{
	Token *tk = DP(stmt)->tokens[1];
	KNH_ASSERT(IS_Token(tk));
	if(SP(tk)->tt == TT_ASIS) {
		if(level == 0) {
			Script *scr = knh_Asm_getScript(ctx, abr);
			return knh_Object_cid(scr);
		}
		else {
			return DP(abr)->this_cid;
		}
	}
	{
		knh_bytes_t name = knh_Token_tobytes(tk);
		if(SP(tk)->tt == TT_CMETHODN) {
			knh_index_t idx = knh_bytes_rindex(name, '.');
			KNH_ASSERT(idx != 1);
			name = knh_bytes_first(name, idx);
		}

		knh_class_t cid = knh_NameSpace_getClass(ctx, ns, name);
		if(cid == CLASS_unknown) {
			knh_Asm_perror(ctx, abr, KMSG_UCLASSN, (char*)name.buf);
			cid = DP(abr)->this_cid;
		}
		else if(level > 0 && cid != DP(abr)->this_cid) {
			knh_Asm_perror(ctx, abr, KMSG_DIFFCLASSN, (char*)name.buf);
			knh_Token_perrata(ctx, tk, CLASSN(cid));
			cid = DP(abr)->this_cid;
		}
		return cid;
	}
}

/* ------------------------------------------------------------------------ */

static
knh_methodn_t knh_StmtMETHOD_name(Ctx *ctx, Stmt *stmt, Asm *abr, int level)
{
	Token *tk = DP(stmt)->tokens[2];
	KNH_ASSERT(IS_Token(tk));
	if(SP(tk)->tt == TT_ASIS) {
		if(level == 1) {
			return METHODN_new;
		}
		else {
			TODO();
			return METHODN_lambda;
		}
	}
	else {
		knh_bytes_t name = knh_Token_tobytes(tk);
		knh_index_t idx = knh_bytes_rindex(name, '.');
		if(idx != -1) {
			name = knh_bytes_last(name, idx);
		}
		return knh_Token_getmn(ctx, tk);
	}
}

/* ------------------------------------------------------------------------ */

#define _KNH_OUTERPARAMS -1
#define _KNH_INNERPARAMS -2

int knh_Stmt_initParams(Ctx *ctx, Stmt *pstmt, Asm *abr, NameSpace *ns, int level)
{
	DP(abr)->vars[0].flag = 0;
	DP(abr)->vars[0].type = NNTYPE_cid(DP(abr)->this_cid);
	DP(abr)->vars[0].fn   = FIELDN_this;
	KNH_SETv(ctx, DP(abr)->vars[0].value, KNH_NULL);
	DP(abr)->vars_size = 1;
	while(IS_Stmt(pstmt)) {
		if(SP(pstmt)->stt == STT_DECL) {
			int templevel = DP(abr)->level;
			DP(abr)->level = level;
			Term *tm = knh_StmtDECL_typing(ctx, pstmt, abr, ns);
			DP(abr)->level = templevel;
			if(tm == NULL) return 0;
			KNH_ASSERT(pstmt == (Stmt*)tm);
		}
		pstmt = DP(pstmt)->next;
	}
	return 1;
}

/* ------------------------------------------------------------------------ */

static
MethodField *knh_tMethodField_find(Ctx *ctx, Asm *abr, knh_type_t rztype, int psize)
{
	int i, n = 0, size = knh_tMethodField_size;
	L_CONTINUE:;
	while(n < size) {
		MethodField *mf = knh_tMethodField(n);
		if(psize != DP(mf)->size) {
			n++;
			continue;
		}
		if(DP(mf)->params[0].type != rztype) {
			n++;
			continue;
		}
		for(i = 1; i < psize; i++) {
			knh_type_t vtype = DP(abr)->vars[i].type;
			if(IS_NOTNULL(DP(abr)->vars[i].value)) {
				vtype = CLASS_type(vtype);
			}
			if(DP(mf)->params[i].type != vtype) {
				n++;
				goto L_CONTINUE;
			}
		}
//		DEBUG3("Found defined method field");
		return mf;
	}
	return NULL;
}

/* ------------------------------------------------------------------------ */

static
int knh_Stmt_checkLastReturn(Ctx *ctx, Stmt *stmt, Method *mtd)
{
	Stmt *last_stmt = knh_Stmt_tail(stmt);
	knh_stmt_t stt = SP(last_stmt)->stt;
	if(stt == STT_RETURN || stt == STT_THROW || stt == STT_ERR) {
		return 1;
	}
	else if(stt == STT_IF) {
		if(knh_Stmt_checkLastReturn(ctx, StmtIF_truecase(last_stmt), NULL) &&
				knh_Stmt_checkLastReturn(ctx, StmtIF_falsecase(last_stmt), NULL)) {
			return 1;
		}
	}
	if(mtd == NULL) return 0;

	/* Generate default return statement */
	Stmt *rstmt = new_Stmt(ctx, 0, STT_RETURN);
	if(knh_Method_isNew(mtd) || knh_Method_rtype(mtd) == TYPE_void) {
		//last_stmt = knh_StmtNULL_tail_append(ctx, last_stmt, rstmt);
	}
	else { /* return default value */
		knh_perror(ctx, SP(last_stmt)->fileid, SP(last_stmt)->line, KMSG_NORETURN, NULL);
		knh_StmtRETURN_addValue(ctx, rstmt,
				knh_pmztype_totype(ctx, knh_Method_rtype(mtd), DP(mtd)->cid));
	}
	last_stmt = knh_StmtNULL_tail_append(ctx, last_stmt, rstmt);
	return 1;
}

/* ------------------------------------------------------------------------ */

Term * knh_StmtMETHOD_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	int level = DP(abr)->level;
	knh_flag_t flag   = knh_StmtMETHOD_flag(ctx, stmt);
	knh_type_t rztype = knh_StmtMETHOD_rtype(ctx, stmt, ns);
	knh_class_t mtd_cid = knh_StmtMETHOD_class(ctx, stmt, abr, ns, level);
	knh_methodn_t mn = knh_StmtMETHOD_name(ctx, stmt, abr, level);
	int pe = KMSG_ESYNTAX;

	if(mtd_cid == CLASS_unknown ||
		!knh_Stmt_initParams(ctx, StmtMETHOD_params(stmt), abr, ns, KNH_OUTERPARAMS)) {
		return NULL;
	}
	Method *mtd = knh_Class_getMethod(ctx, mtd_cid, mn);

	/* Check the type of Constructor */
	if(knh_methodn_isNew(mn) &&
			(mtd_cid != CLASS_type(rztype) && mtd_cid != CLASS_This)) {
		knh_Asm_perror(ctx, abr, KMSG_DIFFNEWTYPE, NULL);
		knh_Token_perrata(ctx, StmtMETHOD_rtype(stmt), CLASSN(mtd_cid));
		rztype = mtd_cid;
	}

	/* New method, and constructors are always new */
	if(IS_NULL(mtd) || knh_methodn_isNew(mn)) {
		int i, psize = DP(abr)->vars_size;
		MethodField *mf = knh_tMethodField_find(ctx, abr, rztype, psize);
		if(mf == NULL) {
			mf = new_MethodField(ctx, flag, psize);
			DP(mf)->params[0].type = rztype;
			DP(mf)->params[0].fn   = FIELDN_return;
			for(i = 1; i < psize; i++) {
				if(IS_NOTNULL(DP(abr)->vars[i].value)) {
					DP(mf)->params[i].type = CLASS_type(DP(abr)->vars[i].type);
				}
				else {
					DP(mf)->params[i].type = DP(abr)->vars[i].type;
				}
				DP(mf)->params[i].fn = DP(abr)->vars[i].fn;
			}
			knh_tMethodField_add(ctx, mf);
		}
		mtd = new_Method(ctx, flag, mtd_cid, mn, NULL);
		KNH_SETv(ctx, DP(mtd)->mf, mf);
		knh_Class_addMethod(ctx, mtd_cid, mtd);
	}
	else {
		int i, psize = DP(abr)->vars_size;
		MethodField *mf = DP(mtd)->mf;
		//DEBUG3("(mtd)->cid=%s, mtd_cid=%s", CLASSN(DP(mtd)->cid), CLASSN(mtd_cid));
		if(DP(mtd)->cid != mtd_cid) { /* Overriding */
			if(knh_Method_isFinal(mtd)) { /* CHECK @Final */
				knh_Asm_perrorMTD(ctx, abr, KMSG_EOVERRIDE, DP(mtd)->cid, mn);
				return NULL;
			}
			mtd = new_Method(ctx, flag, mtd_cid, mn, NULL);
			KNH_SETv(ctx, DP(mtd)->mf, mf);
			knh_Class_addMethod(ctx, mtd_cid, mtd);
		}

		if(!knh_StmtMETA_isOverride(stmt) &&
				!knh_Method_isAbstract(mtd) && DP(mtd)->mn != METHODN_main) {
			/* Konoha: To override, you need @Override */
			knh_Asm_perrorMTD(ctx, abr, KMSG_AOVERRIDE, mtd_cid, mn);
			return NULL;
		}

		if(psize != DP(mf)->size) {
			pe = KMSG_DIFFPSIZE; goto L_PERROR;
		}

		if(DP(mf)->params[0].type != rztype) {
			pe = KMSG_DIFFRTYPE; goto L_PERROR;
		}

		for(i = 1; i < psize; i++) {
			knh_type_t ptype = DP(mf)->params[i].type;
			knh_type_t vtype = DP(abr)->vars[i].type;
			if(IS_NOTNULL(DP(abr)->vars[i].value)) {
				vtype = CLASS_type(vtype);
			}
			if(ptype != vtype) {
				pe = KMSG_DIFFPTYPE; goto L_PERROR;
			}
		}
	}
	{
		if(DP(stmt)->size == 4) {
			DEBUG3("@Abstract");
			knh_Stmt_done(ctx, stmt);
		}
		else {
			knh_Stmt_checkLastReturn(ctx, StmtMETHOD_instmt(stmt), mtd);
			knh_Token_setCONST(ctx, StmtMETHOD_method(stmt), UP(mtd));
		}
		return TM(stmt);
	}

	L_PERROR:;
	knh_wbuf_t wb = knh_Context_wbuf(ctx);
	knh_Method__k(ctx, mtd, wb.w, (String*)KNH_NULL);
	knh_Asm_perror(ctx, abr, pe, knh_wbuf_top(wb));
	knh_wbuf_clear(wb);
	return NULL;
}


/* ------------------------------------------------------------------------ */

Term *knh_StmtFORMAT_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	knh_Stmt_done(ctx, stmt);
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */

static
void knh_Asm_initClassField(Ctx *ctx, Asm *abr, knh_class_t cid)
{
	DP(abr)->flag = 0;
	DP(abr)->this_cid = cid;
	DP(abr)->level = 0;
	knh_int_t i;
	DP(abr)->vars_size = 0;
	for(i = 0; i < KONOHA_LOCALSIZE; i++) {
		DP(abr)->vars[i].flag  = 0;
		DP(abr)->vars[i].type  = TYPE_Any;
		DP(abr)->vars[i].fn    = FIELDN_NONAME;
//		DP(abr)->vars[i].count = 0;
		KNH_SETv(ctx, DP(abr)->vars[i].value, KNH_NULL);
	}
}

/* ------------------------------------------------------------------------ */

static
void knh_Asm_declareClassField(Ctx *ctx, Asm *abr, knh_class_t cid)
{
	knh_tClass_t *TC = (knh_tClass_t*)(&knh_tClass[cid]);
	DEBUG3_ASSERT(IS_ClassStruct(TC->cstruct));
	DEBUG3_ASSERT(DP(TC->cstruct)->fields == NULL);
	{
		int i, fsize = DP(abr)->vars_size;
		knh_cfield_t *cf = (knh_cfield_t*)KNH_MALLOC(ctx, sizeof(knh_cfield_t) * fsize);
		DEBUG3("class %s fsize=%d", CLASSN(cid), fsize);
		for(i = 0; i < fsize; i++) {
			cf[i].flag = DP(abr)->vars[i].flag;
			cf[i].type = DP(abr)->vars[i].type;
			cf[i].fn = DP(abr)->vars[i].fn;
			KNH_INITv(cf[i].value, DP(abr)->vars[i].value);
		}
		DP(TC->cstruct)->fields =cf;
		DP(TC->cstruct)->fsize = fsize;
		DP(TC->cstruct)->sid = BSIZE_TOSID(fsize);
		TC->sid = BSIZE_TOSID(fsize);
		TC->bsize = fsize + TC->offset;
		TC->size = sizeof(Object*) * TC->bsize;
		knh_NameSpace_setClass(ctx, knh_rootNameSpace, TC->lname, cid);
		DBG2_({
			DBG2_P("HERE IS DEFINED STRUCT");
			knh_cfield_dump(ctx, cf, 0, fsize, KNH_STDOUT);
		})
	}
}

/* ------------------------------------------------------------------------ */

Term *knh_StmtCLASS_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns)
{
	knh_class_t prev_cid = DP(abr)->this_cid;
	knh_class_t this_cid = DP(StmtCLASS_class(stmt))->cid;

	knh_Asm_initClassField(ctx, abr, this_cid);

	Stmt *instmt = StmtCLASS_instmt(stmt);
	while(IS_Stmt(instmt)) {
		DP(abr)->line = SP(instmt)->line;
		DP(abr)->level = 1;
		if(SP(instmt)->stt == STT_DECL) {
			knh_StmtDECL_typing(ctx, instmt, abr, ns);
			knh_Stmt_done(ctx, instmt);
		}
		else if(SP(instmt)->stt == STT_LET) {
			knh_StmtLET_typing(ctx, instmt, abr, ns, NNTYPE_void);
			knh_Stmt_done(ctx, instmt);
		}
		instmt = DP(instmt)->next;
	}
	knh_Asm_declareClassField(ctx, abr, this_cid);

	instmt = StmtCLASS_instmt(stmt);
	while(IS_Stmt(instmt)) {
		DP(abr)->line = SP(instmt)->line;
		DP(abr)->level = 1;
		if(SP(instmt)->stt == STT_METHOD) {
			knh_StmtMETHOD_typing(ctx, instmt, abr, ns);
		}
		else if(SP(instmt)->stt == STT_FORMAT) {
			DP(abr)->line = SP(instmt)->line;
			knh_StmtFORMAT_typing(ctx, instmt, abr, ns);
		}
		else {
			knh_Asm_perror(ctx, abr, KMSG_EABORT/*NOTHERE*/, knh_stmt_tochar(SP(instmt)->stt));
		}
		instmt = DP(instmt)->next;
	}
	DP(abr)->this_cid = prev_cid;
	return TM(stmt);
}

/* ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------ */

static
Term *knh_Stmt_typing(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, knh_type_t reqt)
{
	/* check statement or expr */
	if(reqt == TYPE_void) {  /* Statement */
		switch(SP(stmt)->stt) {
		case STT_BLOCK:
			if(!TERMs_typingBLOCK(ctx, stmt, 0, abr, ns)) {
				return NULL;
			}
		case STT_DECL:
			return knh_StmtDECL_typing(ctx, stmt, abr, ns);
		case STT_IF:
			return knh_StmtIF_typing(ctx, stmt, abr, ns);
		case STT_SWITCH:
			return knh_StmtSWITCH_typing(ctx, stmt, abr, ns);
		case STT_WHILE:
			return knh_StmtWHILE_typing(ctx, stmt, abr, ns);
		case STT_DO:
			return knh_StmtDO_typing(ctx, stmt, abr, ns);
		case STT_FOR:
			return knh_StmtFOR_typing(ctx, stmt, abr, ns);
		case STT_FOREACH:
			return knh_StmtFOREACH_typing(ctx, stmt, abr, ns);
		case STT_BREAK:
			return TM(stmt);
		case STT_CONTINUE:
			return TM(stmt);
		case STT_TRY:
			return knh_StmtTRY_typing(ctx, stmt, abr, ns);
		case STT_THROW:
			return knh_StmtTHROW_typing(ctx, stmt, abr, ns);
		case STT_CALL1:
			return knh_StmtCALL1_typing(ctx, stmt, abr, ns, TYPE_void);
		case STT_RETURN:
			return knh_StmtRETURN_typing(ctx, stmt, abr, ns);
		case STT_PRINT:
			return knh_StmtPRINT_typing(ctx, stmt, abr, ns);
		case STT_ASSERT:
			return knh_StmtASSERT_typing(ctx, stmt, abr, ns);
		case STT_DONE:
			return TM(stmt);
		}
	}
	if(!knh_stmt_isExpr(SP(stmt)->stt)) {
		DBG2_P("stt=%s", knh_stmt_tochar(SP(stmt)->stt));
		knh_Asm_perror(ctx, abr, KMSG_EABORT /* NOT_HERE*/, NULL);
		return NULL;
	}
	return knh_StmtEXPR_typing(ctx, stmt, abr, ns, reqt);
}

/* ------------------------------------------------------------------------ */

static
void knh_Stmt_toRuntimeError(Ctx *ctx, Stmt *stmt, Stmt *errstmt)
{
	//knh_Stmt_done(ctx, stmt);
	SP(stmt)->stt = STT_ERR;
	//KNH_ASSERT(ctx == NULL);
	{
		char buf[CLASSNAME_BUFSIZ*2];
		knh_snprintf(buf, sizeof(buf), "Compiler!!: running an error code at %s %d",
				knh_stmt_tochar(SP(errstmt)->stt), (int)SP(errstmt)->line);
		DBG2_P("ERR: %s", buf);
		KNH_SETv(ctx, DP(stmt)->errMsg, new_String(ctx, B(buf), NULL));
	}
	KNH_SETv(ctx, DP(stmt)->next, KNH_NULL);
}

/* ------------------------------------------------------------------------ */

int
knh_Stmt_typingBLOCK(Ctx *ctx, Stmt *stmt, Asm *abr, NameSpace *ns, int isIteration)
{
	int level = DP(abr)->level ;
	Stmt *cur = stmt;
	while(IS_Stmt(cur)) {
		DP(abr)->level = level + 1;
		DP(abr)->line = SP(cur)->line;
		Term *tm = knh_Stmt_typing(ctx, cur, abr, ns, TYPE_void);
		DP(abr)->level = level;
		if(tm == NULL) {
			knh_Stmt_toRuntimeError(ctx, stmt, cur);
			return 0;
		}
		if(!isIteration) break;
		cur = DP(cur)->next;
	}
	return 1;
}

/* ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------ */


#ifdef __cplusplus
}
#endif
