/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2005-2008, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Software Foundation
 * All rights reserved.
 *
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU General Public License 2.0      (with    KONOHA_UNDER_GPL2)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/* ************************************************************************ */

#include"commons.h"

/* ************************************************************************ */

#ifdef __cplusplus
extern "C" {
#endif


/* ======================================================================== */
/* [constructor] */

#ifndef KNH_ARRAY_INITSIZE
#define KNH_ARRAY_INITSIZE  16
#endif/*KNH_ARRAY_INITSIZE*/

/* ------------------------------------------------------------------------ */
/* @method This! Array.new(Int init) */

METHOD knh__Array_new(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t init = IS_NULL(sfp[1].o) ? KNH_ARRAY_INITSIZE: ARG_int(sfp[1]);
	if(init > 0) {
		o->list = knh_oarray_malloc(ctx, init, KNH_NULL);
		o->size = 0;
	}
	KNH_RETURN(ctx, sfp, o);
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.new:array(Int n) */

METHOD knh__Array_new__array(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t init = IS_NULL(sfp[1].o) ? KNH_ARRAY_INITSIZE : ARG_int(sfp[1]);
	knh_class_t p1 = knh_tClass[knh_Object_cid(o)].p1;
	Object *v = (p1 == CLASS_Any) ? KNH_NULL : knh_tClass_defaultValue(ctx, p1);
	if(init > 0) {
		o->list = knh_oarray_malloc(ctx, init, v);
		o->size = init;
	}
	KNH_RETURN(ctx, sfp, o);
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.new:array2D(Int! x, Int! y) */

METHOD knh__Array_new__array2D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t init = ARG_int(sfp[1]) * ARG_int(sfp[2]);
	knh_class_t p1 = knh_tClass[knh_Object_cid(o)].p1;
	Object *v = (p1 == CLASS_Any) ? KNH_NULL : knh_tClass_defaultValue(ctx, p1);
	if(init > 0) {
		o->list = knh_oarray_malloc(ctx, init, v);
		o->size = init;
		knh_oarray_set_opsize(o->list, ARG_int(sfp[1]));
	}
	KNH_RETURN(ctx, sfp, o);
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.new:array3D(Int! x, Int! y, Int! z) */

METHOD knh__Array_new__array3D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t init = ARG_int(sfp[1]) * ARG_int(sfp[2]) * ARG_int(sfp[3]);
	knh_class_t p1 = knh_tClass[knh_Object_cid(o)].p1;
	Object *v = (p1 == CLASS_Any) ? KNH_NULL : knh_tClass_defaultValue(ctx, p1);
	if(init > 0) {
		o->list = knh_oarray_malloc(ctx, init, v);
		o->size = init;
		knh_oarray_setop1(o->list, ARG_int(sfp[1]));
		knh_oarray_setop2(o->list, ARG_int(sfp[2]));
	}
	KNH_RETURN(ctx, sfp, o);
}

/* ------------------------------------------------------------------------ */
/* @method[VARARGS] This! Array.new:init(Any1 value) @VARARGS */

METHOD knh__Array_new__init(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	knh_sfp_t *v = sfp + 1;
	knh_vargc_t ac = knh_sfp_argc(ctx, v);
	size_t i;
	for(i = 0; i < ac; i++) {
		knh_Array_add(ctx, o, v[i].o);
	}
	KNH_RETURN(ctx, sfp, o);
}

/* ======================================================================== */
/* [method] */

/* ------------------------------------------------------------------------ */
/* @method void Array.add(Any1 value) */

METHOD knh__Array_add(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		if(o->size == knh_oarray_capacity(o->list)) {
			o->list = knh_oarray_grow(ctx, o->list);
		}
		KNH_ASSERT(o->size < knh_oarray_capacity(o->list));
		KNH_SETv(ctx, o->list[o->size], sfp[1].o);
		o->size++;
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method[VARARGS] void Array.opLshift(Any1 v) */

METHOD knh__Array_opLshift(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		knh_sfp_t *v = sfp + 1;
		knh_vargc_t ac = knh_sfp_argc(ctx, v);
		size_t i;
		for(i = 0; i < ac; i++) {
			knh_Array_add(ctx, o, v[i].o);
		}
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ======================================================================== */
/* ------------------------------------------------------------------------ */
/* @method Int! Array.getSize() */

METHOD knh__Array_getSize(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	KNH_RETURN_Int(ctx, sfp, o->size);
}

/* ------------------------------------------------------------------------ */
/* @method Any1 Array.get(Int! n) */

METHOD knh__Array_get(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t n2 = knh_array_index(ctx, ARG_int(sfp[1]), o->size);
	KNH_RETURN(ctx, sfp, o->list[n2]);
}

/* ------------------------------------------------------------------------ */
/* @method Any1 Array.get2D(Int! x, Int! y) */

METHOD knh__Array_get2D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t x = knh_oarray_opsize(o->list), y = o->size / x;
	size_t n2 = knh_array_index(ctx, ARG_int(sfp[1]), x) + (knh_array_index(ctx, ARG_int(sfp[2]), y) * x);
	KNH_RETURN(ctx, sfp, o->list[n2]);
}

/* ------------------------------------------------------------------------ */
/* @method Any1 Array.get3D(Int! x, Int! y, Int! z) */

METHOD knh__Array_get3D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t n2 = knh_array_index(ctx, ARG_int(sfp[1]) +
			(ARG_int(sfp[2]) * knh_oarray_op1(o->list)) + (ARG_int(sfp[3]) * knh_oarray_op1(o->list) * knh_oarray_op2(o->list)), o->size);
	KNH_RETURN(ctx, sfp, o->list[n2]);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.set(Int! n, Any1 v) */

METHOD knh__Array_set(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t n2 = knh_array_index(ctx, ARG_int(sfp[1]), o->size);
		KNH_SETv(ctx, o->list[n2], sfp[2].o);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.set2D(Int! x, Int! y, Any1 v) */

METHOD knh__Array_set2D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t x = knh_oarray_opsize(o->list), y = o->size / x;
		size_t n2 = knh_array_index(ctx, ARG_int(sfp[1]), x) + (knh_array_index(ctx, ARG_int(sfp[2]), y) * x);
		KNH_SETv(ctx, o->list[n2], sfp[3].o);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.set3D(Int! x, Int! y, Int! z, Any1 v) */

METHOD knh__Array_set3D(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t n2 = knh_array_index(ctx, ARG_int(sfp[1]) +
				(ARG_int(sfp[2]) * knh_oarray_op1(o->list)) + (ARG_int(sfp[3]) * knh_oarray_op1(o->list) * knh_oarray_op2(o->list)), o->size);
		KNH_SETv(ctx, o->list[n2], sfp[4].o);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.setAll(Any1 v) */

METHOD knh__Array_setAll(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t i;
		for(i = 0; i < o->size; i++) {
			KNH_SETv(ctx, o->list[i], sfp[1].o);
		}
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.opSubset(Int s, Int e) @Debug */

METHOD knh__Array_opSubset(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(o->size > 0) {
		size_t s = IS_NULL(sfp[1].o) ? 0 : knh_array_index(ctx, ARG_int(sfp[1]), o->size);
		size_t e = IS_NULL(sfp[2].o) ? (o->size) : knh_array_index(ctx, ARG_int(sfp[2]), o->size);
		Array *newo = (Array*)new_Object__init(ctx, o->h.flag, o->h.cid);
		newo->list = knh_oarray_copy(ctx, o->list, s, e - s);
		newo->size = e - s;
		KNH_RETURN(ctx, sfp, newo);
	}
	else {
		KNH_RETURN(ctx, sfp, new_Object__init(ctx, o->h.flag, o->h.cid));
	}
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.opSubsete(Int s, Int e) @Debug */

METHOD knh__Array_opSubsete(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(o->size > 0) {
		size_t s = IS_NULL(sfp[1].o) ? 0 : knh_array_index(ctx, ARG_int(sfp[1]), o->size);
		size_t e = IS_NULL(sfp[2].o) ? (o->size) : knh_array_index(ctx, ARG_int(sfp[2]), o->size) + 1;
		Array *newo = (Array*)new_Object__init(ctx, o->h.flag, o->h.cid);
		newo->list = knh_oarray_copy(ctx, o->list, s, e - s);
		newo->size = e - s;
		KNH_RETURN(ctx, sfp, newo);
	}
	else {
		KNH_RETURN(ctx, sfp, new_Object__init(ctx, o->h.flag, o->h.cid));
	}
}

/* ------------------------------------------------------------------------ */
/* @method This! Array.opOffset(Int s, Int offset) @Debug */

METHOD knh__Array_opOffset(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(o->size > 0) {
		size_t s = IS_NULL(sfp[1].o) ? 0 : knh_array_index(ctx, ARG_int(sfp[1]), o->size);
		knh_int_t offset = IS_NULL(sfp[2].o) ? 0 : ARG_int(sfp[2]);
		Array *newo = (Array*)new_Object__init(ctx, o->h.flag, o->h.cid);
		if(offset == 0) {

		}
		else if(offset > 0){
			newo->list = knh_oarray_copy(ctx, o->list, s, offset);
			newo->size = offset;
		}
		else {
			knh_int_t ss = s + offset;
			if(ss < 0) ss = 0;
			newo->list = knh_oarray_copy(ctx, o->list, ss, s - ss);
			newo->size = s - ss;
		}
		KNH_RETURN(ctx, sfp, newo);
	}
	else {
		KNH_RETURN(ctx, sfp, new_Object__init(ctx, o->h.flag, o->h.cid));
	}
}

/* ======================================================================== */

/* ------------------------------------------------------------------------ */
/* @method void Array.insert(Int! n, Any1 v) */

METHOD knh__Array_insert(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t i, n = knh_array_index(ctx, ARG_int(sfp[1]), o->size);
		if(o->size == knh_oarray_capacity(o->list)) {
			o->list = knh_oarray_grow(ctx, o->list);
		}
		KNH_ASSERT(o->size < knh_oarray_capacity(o->list));
		Object *temp = o->list[o->size];
		o->size++;
		for(i = o->size - 1; i > n ; i++) {
			o->list[i] = o->list[i-1];
		}
		o->list[n] = temp;
		KNH_SETv(ctx, o->list[n], sfp[2].o);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.remove(Int! n) */

METHOD knh__Array_remove(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t n = knh_array_index(ctx, ARG_int(sfp[1]), o->size);
		knh_Array_remove(ctx, o, n);
	}
	KNH_RETURN_void(ctx,sfp);
}

/* ------------------------------------------------------------------------ */
/* @method Any1 Array.pop() */

METHOD knh__Array_pop(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o) && o->size > 0) {
		o->size--;
		KNH_RETURN(ctx, sfp, o->list[o->size]);
	}else {
		KNH_RETURN(ctx,sfp, KNH_NULL);
	}
}

/* ------------------------------------------------------------------------ */
/* @method void Array.clear() */

void knh_Array_clear(Ctx *ctx, Array *o)
{
	if(!knh_Object_isImmutable(o)) {
		size_t i;
		for(i = 0; i < o->size; i++) {
			KNH_SETv(ctx, o->list[i], KNH_NULL);
		}
		o->size = 0;
	}
}

/* ------------------------------------------------------------------------ */
/* @method Int! Array.indexOf(Any1 v) */

METHOD knh__Array_indexOf(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t i;
	for(i = 0; i < o->size; i++) {
		if(knh_Object_compareTo(o->list[i], sfp[1].o) == 0) {
			KNH_RETURN_Int(ctx, sfp, i);
		}
	}
	KNH_RETURN_Int(ctx, sfp, -1); // Not Found
}

/* ------------------------------------------------------------------------ */
/* @method Int! Array.lastIndexOf(Any1 v) */

METHOD knh__Array_lastIndexOf(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t i;
	for(i = o->size - 1; i >= 0; i--) {
		if(knh_Object_compareTo(o->list[i], sfp[1].o) == 0) {
			KNH_RETURN_Int(ctx, sfp, i);
		}
	}
	KNH_RETURN_Int(ctx, sfp, -1); // Not Found
}

/* ------------------------------------------------------------------------ */
/* @method Boolean! Array.opHas(Any1 v) */

METHOD knh__Array_opHas(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	size_t i;
	for(i = 0; i < o->size; i++) {
		if(knh_Object_compareTo(o->list[i], sfp[1].o) == 0) {
			KNH_RETURN_Boolean(ctx, sfp, 1);
		}
	}
	KNH_RETURN_Boolean(ctx, sfp, 0); // Not Found
}

/* ======================================================================== */
/* [Collections] */

/* ------------------------------------------------------------------------ */
/* @method void Array.sort() */

METHOD knh__Array_sort(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		knh_qsort(o->list, o->size, sizeof(Object*),
					(int (*)(const void*, const void*))knh_Object_compareTo);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.reverse() */

METHOD knh__Array_reverse(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t i;
		for(i = 0; i < o->size / 2; i++) {
			size_t n = o->size - i;
			Object *temp = o->list[i];
			o->list[i] = o->list[n];
			o->list[n] = temp;
		}
		TODO_THROW(ctx);
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.swap(Int! m, Int! n) */

METHOD knh__Array_swap(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t m = knh_array_index(ctx, ARG_int(sfp[1]), o->size);
		size_t n = knh_array_index(ctx, ARG_int(sfp[1]), o->size);
		Object *temp = o->list[n];
		o->list[n] = o->list[m];
		o->list[m] = temp;
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ------------------------------------------------------------------------ */
/* @method void Array.shuffle() */

METHOD knh__Array_shuffle(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	if(!knh_Object_isImmutable(o)) {
		size_t i;
		for(i = 0; i < o->size; i++) {
			size_t m = knh_rand() % o->size;
			size_t n = knh_rand() % o->size;
			Object *temp = o->list[n];
			o->list[n] = o->list[m];
			o->list[m] = temp;
		}
	}
	KNH_RETURN_void(ctx, sfp);
}

/* ======================================================================== */
/* [movabletext] */

/* ------------------------------------------------------------------------ */
/* @method void Array.%k(OutputStream w, String m) */

void knh_Array__k(Ctx *ctx, Array *o, OutputStream *w, String *m)
{
	knh_putc(ctx, w, '[');
	size_t c;
	for(c = 0; c < o->size; c++) {
		if(c > 0) {
			knh_write_delim(ctx,w);
		}
		knh_format(ctx, w, METHODN__k, o->list[c], KNH_NULL);
	}
	knh_putc(ctx, w, ']');
}

/* ======================================================================== */
/* [mapping] */

static
Object* knh_Array_var_next(Ctx *ctx, Iterator *it)
{
	Array *o = (Array*)knh_Iterator_source(it);
	KNH_ASSERT(IS_bArray(o));
	size_t pos = knh_Iterator_pos(it);
	while(pos < o->size) {
		if(IS_NOTNULL(o->list[pos])) {
			knh_Iterator_setpos(it, pos+1);
			return o->list[pos];
		}
		pos++;
	}
	return KNH_VOID;
}

/* ------------------------------------------------------------------------ */
/* @map Array Iterator! */

Iterator* knh_Array_Iterator(Ctx *ctx, Array *o, Mapper *mpr)
{
	return new_Iterator(ctx, knh_tClass[o->h.cid].p1, UP(o), knh_Array_var_next);
}

/* ------------------------------------------------------------------------ */
/* @method Any1.. Array.opItr() */

METHOD knh__Array_opItr(Ctx *ctx, knh_sfp_t *sfp)
{
	Array *o = (Array*)sfp[0].o;
	KNH_RETURN(ctx, sfp, new_Iterator(ctx, knh_tClass[o->h.cid].p1, UP(o), knh_Array_var_next));
}

/* ------------------------------------------------------------------------ */
/* @map Iterator Array! */

Array* knh_Iterator_Array(Ctx *ctx, Iterator *o, Mapper *mpr)
{
	Array *a = new_Array(ctx, knh_tClass[o->h.cid].p1, 0);
	Object *v = o->fnext_1(ctx, o);
	while(IS_NOTNULL(v)) {
		knh_Array_add(ctx, a, v);
		v = o->fnext_1(ctx, o);
	}
	return a;
}

/* ------------------------------------------------------------------------ */

Mapper *knh_tMapper_newIteratorArray(Ctx *ctx, knh_class_t icid, knh_class_t acid)
{
	Mapper *mpr = new_Mapper(ctx, KNH_FLAG_MMF_TOTAL, acid, icid, (knh_fmapper)knh_Array_Iterator, KNH_NULL);
	knh_ClassMap_add(ctx, knh_tClass[acid].cmap, mpr);
	mpr = new_Mapper(ctx, KNH_FLAG_MMF_TOTAL, icid, acid, (knh_fmapper)knh_Iterator_Array, KNH_NULL);
	knh_ClassMap_add(ctx, knh_tClass[icid].cmap, mpr);
	return mpr;
}

/* ------------------------------------------------------------------------ */

Mapper *knh_tMapper_newArrayIterator(Ctx *ctx, knh_class_t acid, knh_class_t icid)
{
	Mapper *mpr = new_Mapper(ctx, KNH_FLAG_MMF_TOTAL, icid, acid, (knh_fmapper)knh_Iterator_Array, KNH_NULL);
	knh_ClassMap_add(ctx, knh_tClass[icid].cmap, mpr);
	mpr = new_Mapper(ctx, KNH_FLAG_MMF_TOTAL, acid, icid, (knh_fmapper)knh_Array_Iterator, KNH_NULL);
	knh_ClassMap_add(ctx, knh_tClass[acid].cmap, mpr);
	return mpr;
}

/* ------------------------------------------------------------------------ */

#ifdef __cplusplus
}
#endif
