from qt import *
from Queue import Queue

from kittywidget import *
from utils import *
from ui import ICONPATH
from bbs import Board


class TopicListWidget(KittyTabWidget):

    """Widget to display topic list

    _pages is used to store board-page association.  Each key of the
    dictionary is a Board instance and value is a widget (currently
    QListView) showing the topic list of the board.
    """

    class _ListViewItem(QListViewItem):
        def __init__(self, topic, *args):
            QListViewItem.__init__(self, *args)
            self._topic = topic

        def topic(self):
            return self._topic

    _newtab = True

    def __init__(self, parent):
        KittyTabWidget.__init__(self, parent)
        self._context_menu = self._ContextMenu(self)
        self._pages = {}
        self._queue = Queue()
        self.connect(self._tab_menu, PYSIGNAL("requestClose"), self._close)
        self.connect(self._tab_menu, PYSIGNAL("requestReload"),
                     self._reload_pointed_page)
        self.connect(self,
                     PYSIGNAL("tabMidPressed"),
                     self._close_pointed_page)
        self.connect(self,
                     PYSIGNAL("tabRightPressed"),
                     self._tab_menu.popup)
        self.connect(self,
                     PYSIGNAL("tabLeftDoubleClicked"),
                     self._reload_current_page)

        self.connect(self._context_menu,
                     PYSIGNAL("requestOpenMessages"),
                     self._request_open)
        # Start thread
        self._timer = QTimer()
        self.connect(self._timer, SIGNAL("timeout()"), self._reduce_queue)
        self._timer.start(100, False)

    def display(self, board, topics, newtab=None):
        """Interface to display the topic list of board
        
        This method actually submit board into the local queue.  If 
        newtab is True, the new tab will be created for this board.
        """
        assert isinstance(board, Board)
        if newtab==None:
            newtab = self._newtab
        self._queue.put((board, topics, newtab))

    def close_current_page(self):
        """Close currently displayed page.

        If no page is open, thid method does nothing.
        """
        w = self.currentPage()
        if w!=None:
            self._close(w)

    def current_board(self):
        widget = self.currentPage()
        d = dict([(w,b) for (b,w) in self._pages.items()])
        return d.get(widget, None)

    def _request_open(self, item, newtab=None):
        """Request to open given item

        This method emits requestOpen(topic) pysignal.
        """
        self.emit(PYSIGNAL("requestOpenMessages"), (item.topic(), newtab))

    def _reduce_queue(self):
        if not self._queue.empty():
            self._display(*self._queue.get())

    def _display(self, board, topics, newtab):
        """Display the topic list of board, used from sub thread.
        
        If this window is already displaying the board or newtab is 
        False, the tab is  shown.  Otherwise, create a new tab and 
        display the list in the tab.
        """
        if board in self._pages.keys():
            listview = self._pages[board]
        elif newtab==False and len(self._pages)>0:
            listview = self.currentPage()
            self.changeTab(listview, board.title())
            self._forget_page(self.currentPage())

        else:
            listview = QListView(self)
            listview.setAllColumnsShowFocus(True)
            listview.setSorting(0, True)
            listview.addColumn("No")
            listview.addColumn(u("タイトル"))
            listview.addColumn(u("レス"))
            listview.addColumn(u("既得"))
            self.connect(listview,
                         SIGNAL("doubleClicked(QListViewItem *)"),
                         self._request_open)
            self.connect(listview,
                         SIGNAL("returnPressed(QListViewItem *)"),
                         self._request_open)
            self.connect(
              listview,
              SIGNAL("rightButtonClicked(QListViewItem *,const QPoint &, int)"),
              self._popup_context_menu)
            self.addTab(listview, board.title())

        listview.clear()
        i = 1
        for topic in topics:
            self._ListViewItem(topic,
                               listview,
                               "%03d"%i,
                               topic.title(),
                               "%d" % topic.remote_size(),
                               "%d" % (topic.local_size() or 0))
            i += 1
        self._pages[board] = listview
        self.showPage(self._pages[board])

    def _popup_context_menu(self, item, position, column):
        self._context_menu.popup(item, position)

    def _close_pointed_page(self, page, event):
        """Close page pointed by mouse cursor"""
        self._close(page)
    
    def _close(self, page_widget):
        """Close tab that contains page_widget

        In addition to call removePage() method, this method removes the 
        widget from self._pages dictionary.
        """
        self.removePage(page_widget)
        self._forget_page(page_widget)
        page_widget.close()

    def _reload_current_page(self):
        self._reload(self.current_board())

    def _reload_pointed_page(self, page):
        board = dict([(w,b) for b,w in self._pages.items()]).get(page, None)
        if not board is None:
            self._reload(board)

    def _reload(self, board):
        if board!=None:
            self.emit(PYSIGNAL("requestSyncTopicList"), (board,))

    def _forget_page(self, widget):
        """Remove widget from self._pages"""
        boards = dict([(w,b) for b,w in self._pages.items()])
        del self._pages[boards.get(widget, None)]

    class _ContextMenu(QPopupMenu):
        def __init__(self, parent):
            QPopupMenu.__init__(self, parent)
            self._view = parent
            self._target_item = None
            self.insertItem(u("新しいタブで開く"), self._open_in_new_tab)
            self.insertItem(u("今のタブで開く"), self._open_in_current_tab)

        def popup(self, item, position):
            self._target_item = item
            QPopupMenu.popup(self, position)

        def _open_in_new_tab(self):
            self.emit(PYSIGNAL("requestOpenMessages"),(self._target_item,True))

        def _open_in_current_tab(self):
            self.emit(PYSIGNAL("requestOpenMessages"),(self._target_item,False))

