/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "thread.h"

#include <config.h>

#include <stdlib.h>
#include <unistd.h>

#include <kurl.h>
#include <kdebug.h>
#include <kprotocolmanager.h>
#include <kglobal.h>
#include <kstandarddirs.h>

#include <kio/slaveconfig.h>
#include <kio/jobclasses.h>
#include <kio/scheduler.h>
#include <kio/netaccess.h>

#include <qwidget.h>
#include <qapplication.h>
#include <qregexp.h>

#include "board.h"
#include "kita_misc.h"
#include "qcp932codec.h"
#include "threadinfo.h"

using namespace Kita;

QDict<Thread>* Thread::m_threadDict = 0;

Thread::Thread()
{}

Thread::Thread( const QString& datURL, const QString& threadName )
        : m_datURL( datURL ), m_threadName( threadName )
{}

Thread::~Thread()
{}

const QString& Thread::datURL() const
{
    return m_datURL;
}

const QString Thread::url() const
{
    return datToThread( m_datURL );
}

const QString Thread::datID() const
{
    return KURL( m_datURL ).filename().section( ".", 0, 0 );
}

int Thread::resNum() const
{
    return KitaThreadInfo::resNum( m_datURL );
}

const QString& Thread::name() const
{
    if ( ! Kita::Thread::getByURL( m_datURL ) ->m_threadName.isEmpty() ) {
        return Kita::Thread::getByURL( m_datURL ) ->m_threadName;
    }
    return m_threadName;
}

void Thread::setResNum( int resNum )
{
    KitaThreadInfo::setResNum( m_datURL, resNum );
}

const QString& Thread::boardName() const
{
    Board * board = Board::getByURL( boardURL() );
    return board->name();
}

const QString Thread::boardURL() const
{
    return datToBoard( m_datURL );
}

const QString Thread::boardID() const
{
    return KURL( datToBoard( m_datURL ) ).fileName();
}

const QString Thread::toXmlFragment() const
{
    QString ret;

    ret += "<thread xmlns=\"http://kita.sourceforge.jp/ns/thread\">\n";
    ret += QString( "<daturl>%1</daturl>\n" ).arg( m_datURL );
    ret += QString( "<name>%1</name>\n" ).arg( m_threadName );
    ret += QString( "<resnum>%1</resnum>\n" ).arg( resNum() );

    // FIXME: Board饹饳ԡ
    ret += "<board xmlns=\"http://kita.sourceforge.jp/ns/board\">\n";
    ret += QString( "<url>%1</url>\n" ).arg( boardURL() );
    ret += QString( "<name>%1</name>\n" ).arg( boardName() );
    ret += "</board>\n";

    ret += "</thread>\n";

    return ret;
}

Thread* Thread::fromXml( const QString& xml )
{
    Thread * thread;

    ThreadXmlParser parser;
    QXmlSimpleReader reader;
    QXmlInputSource source;
    source.setData( xml );
    reader.setContentHandler( &parser );
    reader.parse( &source );

    if ( parser.isValid() ) {
        thread = parser.getThread();
    } else {
        kdError() << "inValid" << endl;
    }

    return thread;
}

Thread* Thread::getByURL( const QString& datURL )
{
    if ( m_threadDict == 0 ) {
        m_threadDict = new QDict<Thread>();
    }

    if ( m_threadDict->find( datURL ) ) {
        return m_threadDict->find( datURL );
    }

    Thread* newThread = new Thread( datURL );
    m_threadDict->insert( datURL, newThread );

    return newThread;
}

void Thread::setName( const QString& datURL, const QString& threadName )
{
    if ( datURL.isNull() ) {
        return ;
    }

    if ( m_threadDict == 0 ) {
        m_threadDict = new QDict<Thread>();
    }

    if ( m_threadDict->find( datURL ) ) {
        Thread * thread = getByURL( datURL );
        thread->m_threadName = threadName;
    } else {
        Thread* newThread = new Thread( datURL, threadName );
        m_threadDict->insert( datURL, newThread );
    }

    return ;
}

NullThread::NullThread()
{}

NullThread::~NullThread()
{}

ThreadXmlParser::ThreadXmlParser()
        : m_boardParser( 0 ), m_inBoard( false ), m_isValid( false ), m_inThread( false )
{};

bool ThreadXmlParser::startElement( const QString& namespaceURI, const QString& localName, const QString& qName, const QXmlAttributes& atts )
{
    if ( m_inBoard ) {
        m_boardParser->startElement( namespaceURI, localName, qName, atts );
    } else if ( m_inThread ) {
        if ( localName == "daturl" ) {} else if ( localName == "name" ) {} else if ( localName == "resnum" ) {} else if ( localName == "board" ) {
            m_inBoard = true;
            m_boardParser = new BoardXmlParser();
            m_boardParser->startElement( namespaceURI, localName, qName, atts );
        } else {
            // error
            return false;
        }
    } else {
        if ( localName == "thread" ) {
            m_inThread = true;
        } else {
            // error
            return false;
        }
    }

    return true;
}

// TODO: <resnum>FavoritesǤϻȤʤȤˤʤäΤǺɬפꡣ
bool ThreadXmlParser::endElement( const QString& namespaceURI, const QString& localName, const QString& qName )
{
    if ( localName == "board" ) {
        m_inBoard = false;
        m_boardParser->endElement( namespaceURI, localName, qName );

        delete m_boardParser;
        m_boardParser = 0;

        // create board;
    } else if ( m_inBoard ) {
        m_boardParser->endElement( namespaceURI, localName, qName );
    } else if ( localName == "thread" ) {
        m_inThread = false;
        Kita::Thread::setName( m_datUrlStr, m_threadName );
        m_thread = Kita::Thread::getByURL( m_datUrlStr );
        KitaThreadInfo::setResNum( m_datUrlStr, m_resNum );
        m_isValid = true;
        // create thread;
    } else if ( localName == "daturl" ) {
        m_datUrlStr = m_characters;
    } else if ( localName == "name" ) {
        m_threadName = m_characters;
    } else if ( localName == "resnum" ) {
        m_resNum = m_characters.toInt();
    } else {
        // error
        return false;
    }
    return true;
}

bool ThreadXmlParser::characters( const QString& ch )
{
    if ( m_inBoard ) {
        m_boardParser->characters( ch );
    } else {
        m_characters = ch;
    }
    return true;
}

// copied from http.cc
QString ThreadAccess::getCachePath( const KURL& url )
{
    const QChar seperator = '_';

    QString CEF = url.path();

    int p = CEF.find( '/' );

    while ( p != -1 ) {
        CEF[ p ] = seperator;
        p = CEF.find( '/', p );
    }

    QString host = url.host().lower();
    CEF = host + CEF + '_';

    QString dir;
    if ( url.protocol() == "k2ch" ) {
        dir = KGlobal::dirs() ->saveLocation( "cache", "k2ch" );
    } else {
        dir = KProtocolManager::cacheDir();
    }
    kdDebug( 7103 ) << "dir = " << dir << endl;
    if ( dir[ dir.length() - 1 ] != '/' )
        dir += "/";

    int l = host.length();
    for ( int i = 0; i < l; i++ ) {
        if ( host[ i ].isLetter() && ( host[ i ] != 'w' ) ) {
            dir += host[ i ];
            break;
        }
    }
    if ( dir[ dir.length() - 1 ] == '/' )
        dir += "0";

    unsigned long hash = 0x00000000;
    QCString u = url.url().latin1();
    for ( int i = u.length(); i--; ) {
        hash = ( hash * 12211 + u[ i ] ) % 2147483563;
    }

    QString hashString;
    hashString.sprintf( "%08lx", hash );

    CEF = CEF + hashString;

    CEF = dir + "/" + CEF;

    return CEF;
}

// copied from http.cc
// å夬ȤϤƤmallocΰ˥ԡ֤
// å夬ʤȤ0֤
char* ThreadAccess::getCacheData( const KURL& url )
{
    QString cachePath = getCachePath( url );

    FILE *fs = fopen( QFile::encodeName( cachePath ), "r" );
    if ( !fs ) {
        return 0;
    }

    char buffer[ 401 ];
    bool ok = true;

    // CacheRevision
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;
    if ( ok && ( strcmp( buffer, "7\n" ) != 0 ) )
        ok = false;

    // URL
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;
    if ( ok ) {
        int l = strlen( buffer );
        if ( l > 0 )
            buffer[ l - 1 ] = 0; // Strip newline
        if ( url.url() != buffer ) {
            ok = false; // Hash collision
        }
    }

    // Creation Date
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // Expiration Date
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // ETag
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // Last-Modified
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // Mime-Type
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // Charset
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    if ( ok ) {
        struct stat buf;
        ::stat( QFile::encodeName( cachePath ), &buf );
        int pos = ftell( fs );
        int datasize = buf.st_size - pos;

        char* ret = static_cast<char *>( malloc( datasize + 1 ) );
        fread( ret, datasize, 1, fs );
        ret[ datasize ] = '\0';
        fclose( fs );
        return ret;
    }

    fclose( fs );
    unlink( QFile::encodeName( cachePath ) );
    return 0;
}

// å˽񤭹ߡ
// partial dataäƤΤǤ񤭴롣
void ThreadAccess::writeCacheData( const KURL& url )
{
    QString cachePath = getCachePath( url );

    FILE *fs = fopen( QFile::encodeName( cachePath ), "r+" );
    if ( !fs ) {
        return ;
    }

    char buffer[ 401 ];
    bool ok = true;

    // CacheRevision
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;
    if ( ok && ( strcmp( buffer, "7\n" ) != 0 ) )
        ok = false;

    // URL
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;
    if ( ok ) {
        int l = strlen( buffer );
        if ( l > 0 )
            buffer[ l - 1 ] = 0; // Strip newline
        if ( url.url() != buffer ) {
            ok = false; // Hash collision
        }
    }

    // Creation Date
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // Expiration Date
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // ETag
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // Last-Modified
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // Mime-Type
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    // Charset
    if ( ok && ( !fgets( buffer, 400, fs ) ) )
        ok = false;

    if ( ok ) {
        if ( m_orgData && responseCode() == 304 ) {
            // dat餷
            m_threadData = m_orgData;
        } else if ( m_orgData && responseCode() == 206 ) {
            QCString orgData = QCString( m_orgData, strlen( m_orgData ) );
            m_threadData = orgData + m_threadData;
        }
        fwrite( m_threadData, m_threadData.size(), 1, fs );
        fclose( fs );

        if ( m_orgData ) {
            free( m_orgData );
            m_orgData = 0;
        }
        return ;
    }

    fclose( fs );
    unlink( QFile::encodeName( cachePath ) );
    return ;
}

QString ThreadAccess::get()
{
    m_orgData = getCacheData( m_thread->datURL() );

    // copy original cache file
    QString cachePath = getCachePath( m_thread->datURL() );
    QString backupPath = cachePath + ".kita_backup";
    KIO::NetAccess::copy( cachePath, backupPath );

    if ( KURL( m_thread->datURL() ).protocol() != "k2ch" ) {
        KIO::SlaveConfig::self() ->setConfigData( "http",
                KURL( m_thread->datURL() ).host(),
                "UserAgent",
                QString( "Monazilla/1.00 (Kita/%1)" ).arg( VERSION ) );
    }

    KIO::TransferJob* job = KIO::get( m_thread->datURL(), true, true );
    m_currentJob = job;

    connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             SLOT( slotReceiveThreadData( KIO::Job*, const QByteArray& ) ) );
    connect( job, SIGNAL( result( KIO::Job* ) ), SLOT( slotThreadResult( KIO::Job* ) ) );
    connect( job, SIGNAL( redirection( KIO::Job *, const KURL& ) ), SLOT( slotRedirection( KIO::Job *, const KURL& ) ) );

    // use 'HTTP-Headers' metadata.
    job->addMetaData( "PropagateHttpHeader", "true" );
    if ( m_orgData ) {
        job->addMetaData( "resume", QString::number( strlen( m_orgData ) - 1 ) );
        job->addMetaData( "AllowCompressedPage", "false" );
    }

    enter_loop();

    writeCacheData( m_thread->datURL() );
    KIO::NetAccess::del( backupPath );

    QCp932Codec codec;
    return codec.toUnicode( m_threadData );
}

// from netaccess.cpp
void qt_enter_modal( QWidget* widget );
void qt_leave_modal( QWidget* widget );

void ThreadAccess::enter_loop()
{
    QWidget dummy( 0, 0, WType_Dialog | WShowModal );
    dummy.setFocusPolicy( QWidget::NoFocus );
    qt_enter_modal( &dummy );
    qApp->enter_loop();
    qt_leave_modal( &dummy );
}

void ThreadAccess::slotRedirection( KIO::Job *, const KURL & newURL )
{
    // datƤȻפΤǥå򸵤᤹
    QString cachePath = getCachePath( m_thread->datURL() );
    QString backupPath = cachePath + ".kita_backup";
    KIO::NetAccess::copy( backupPath, cachePath );
    KIO::NetAccess::del( backupPath );
}

void ThreadAccess::slotReceiveThreadData( KIO::Job*, const QByteArray& data )
{
    QCString cstr( data.data(), data.size() + 1 );
    m_threadData.append( cstr );
}

void ThreadAccess::slotThreadResult( KIO::Job* job )
{
    m_currentJob = 0;
    if ( job->error() ) {
        job->showErrorDialog();
    } else {
        m_header = job->queryMetaData( "HTTP-Headers" );
    }
    qApp->exit_loop();
}

void ThreadAccess::killJob()
{
    if ( m_currentJob ) m_currentJob->kill();
}

int ThreadAccess::serverTime()
{
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_header );
    QRegExp regexp( "Date: (...), (..) (...) (....) (..:..:..) .*" );
    QString dateStr = headerList.grep( regexp ) [ 0 ];
    if ( regexp.search( dateStr ) == -1 ) {
        // invalid date format
        return QDateTime::currentDateTime().toTime_t();
    } else {
        // I hate this format ;p
        QString usLocalDateStr = regexp.cap( 1 ) + " " + regexp.cap( 3 ) + " " +
                                 regexp.cap( 2 ) + " " + regexp.cap( 5 ) + " " +
                                 regexp.cap( 4 );

        // 1970/01/01 00:00:00 GMT
        QDateTime zeroTime( QDate( 1970, 1, 1 ), QTime( 0, 0 ) );
        return zeroTime.secsTo( QDateTime::fromString( usLocalDateStr ) );
    }
}

int ThreadAccess::responseCode()
{
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_header );
    QRegExp regexp( "HTTP/1\\.[01] ([0-9]+) .*" );
    QString dateStr = headerList.grep( regexp ) [ 0 ];
    if ( regexp.search( dateStr ) == -1 ) {
        // invalid response
        return 0;
    } else {
        return regexp.cap( 1 ).toInt();
    }
}

#include "thread.moc"
