/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *   linux 󥫥פ餰ο                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "kitathreadview.h"

#include <kurl.h>
#include <khtmlview.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kaction.h>
#include <kmessagebox.h>
#include <kdebug.h>

#include <dom/html_inline.h>
#include <dom/html_base.h>
#include <dom/html_list.h>

#include <qlayout.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qprogressdialog.h>
#include <qcursor.h>
#include <qtextbrowser.h>
#include <qmessagebox.h>
#include <qdatetime.h>

#include "kitahtmlpart.h"
#include "kitawritedialog.h"
#include "../kitacacheinfo.h"
#include "kita2ch.h"

#include "libkita/comment.h"
#include "libkita/qcp932codec.h"
#include "libkita/favoritethreads.h"

#define MAX_LABEL_LENGTH 60

static const char* cookie_title = "񤭹߳ǧ";
static const char* cookie_message =
"Ƴǧ\n"
"Ƥ줿Ƥϥԡ¸ѡž礬ޤ\n"
"Ƥ˴ؤȯǤƼԤ˵ޤ\n"
"\n"
"Ǥ餦Ȥƽ񤭹ߤޤ\n";

static const char* kokomade_yonda = "<p style=\"background-color: #CCCCCC; text-align: center\">"
"ޤɤ"
"</p>";

KitaThreadView::KitaThreadView(QWidget *parent, const char *name)
    : KitaThreadViewBase(parent, name)
    , m_access( 0 )
    , m_parent( parent )
    , m_popup( 0 )
{
  m_threadPart = new KitaHTMLPart(threadFrame);
  QHBoxLayout* aLayout = new QHBoxLayout(threadFrame);
  aLayout->addWidget(m_threadPart->view());

  m_threadPart->setZoomFactor(120); // XXX
  {
    SearchButton->setPixmap( SmallIcon("find") );
    HighLightButton->setPixmap( SmallIcon("idea") );
    ReloadButton->setPixmap( SmallIcon("reload") );
    GobackAnchorButton->setPixmap( SmallIcon("2leftarrow") );
    BookmarkButton->setPixmap( SmallIcon("bookmark_add") );
  }

  setAcceptDrops( true ); // DND Drop eneble: 2nd stage. - enable on "KitaThreadView" widget and disable on the others(child widgets of "KitaThreadView").
  threadFrame->setAcceptDrops( false ); // don't treat Drop event on child.
  m_threadPart->view()->setAcceptDrops( false ); // don't treat Drop event on child.

  m_threadPart->enableMetaRefresh( false ); //disable <meta refresh="...">


  connect( writeButton, SIGNAL(clicked()), SLOT(slotWriteButtonClicked()));
  connect( m_threadPart, SIGNAL( nodeActivated(const DOM::Node &) ), SLOT( slotDOMNodeActivated(const DOM::Node &) ) );
  connect( m_threadPart, SIGNAL( onURL(const QString&) ), SLOT( slotOnURL(const QString&) ) );
  connect( m_threadPart, SIGNAL( setLocationBarURL(const QString &) ),
  			 SIGNAL( setLocationBarURL(const QString &) ) );
  connect( BookmarkButton, SIGNAL( toggled(bool) ), SLOT( slotBookmarkButtonClicked(bool) ) );
  connect( SearchButton, SIGNAL( clicked() ), SLOT( slotSearchButton() ) );
  connect( SearchCombo, SIGNAL( activated(int) ), SLOT( slotSearchButton() ) );
  connect( HighLightButton, SIGNAL( toggled(bool) ), SLOT( slotHighLightenButton(bool) ) );
  connect( GobackAnchorButton, SIGNAL( clicked() ), m_threadPart, SLOT( gobackAnchor() ) );
  connect( ReloadButton, SIGNAL( clicked() ), SLOT( slotReloadButton() ) );

  KParts::BrowserExtension * ext = m_threadPart->browserExtension();
  connect( ext, SIGNAL( openURLRequest(const KURL&, const KParts::URLArgs&) ),
                 SLOT( slotOpenURLRequest(const KURL&, const KParts::URLArgs&) ) );
  connect( ext, SIGNAL( createNewWindow (const KURL&, const KParts::URLArgs&) ),
      		SIGNAL( createNewWindow (const KURL&, const KParts::URLArgs&) ) );
  connect( ext, SIGNAL( setLocationBarURL(const QString &) ),
      		SIGNAL( setLocationBarURL(const QString &) ) );
  connect( ext, SIGNAL( enableAction(const char*, bool) ),
      		SIGNAL( enableAction(const char*, bool) ) );

  connect( ext, SIGNAL( popupMenu(KXMLGUIClient *, const QPoint&, const KURL&, const QString&, mode_t) ),
      this, SLOT( slotPopupMenu(KXMLGUIClient *, const QPoint&, const KURL&, const QString&, mode_t) ) );

}

KitaThreadView::~KitaThreadView() {}

const QString KitaThreadView::threadName() const
{
  return m_thread.name();
}

const KURL KitaThreadView::threadURL() const
{
  return m_thread.url();
}

void KitaThreadView::slotDOMNodeActivated(const DOM::Node &node)
{
  { //process Anchor tags. Anchor tags not proccessed here cause 'emit KParts::BrowserExtention::openURLRequest()'
    DOM::HTMLAnchorElement anchor = node;

    if ( ! anchor.href().isEmpty() )
    {
      kdDebug() << "AnchorNodeActivated::" << endl;
    } // end: anchor.href().isEmpty()
  } // end of Anchor tags.
}

void KitaThreadView::showThread( const Kita::Thread& thread )
{
  { //reset member variables associated with a thread.
    m_threadPart->reset();
  }

  m_thread = thread;

  m_access = new Kita::ThreadAccess( thread );
  connect( m_access, SIGNAL( redirection( const QString& ) ), SIGNAL( setLocationBarURL( const QString& ) ) );
  QString result = m_access->get();
  m_serverTime = getServerTimeFromHttpHeaders( m_access->getHeader() );
  update( result );
}

int KitaThreadView::getServerTimeFromHttpHeaders( const QString& headers )
{
  // parse HTTP headers
  QStringList headerList = QStringList::split("\n", headers);
  QRegExp regexp("Date: (...), (..) (...) (....) (..:..:..) .*");
  QString dateStr = headerList.grep(regexp)[0];
  if(regexp.search(dateStr) == -1) {
    // invalid date format
    return QDateTime::currentDateTime().toTime_t();
  } else {
    // I hate this format ;p
    QString usLocalDateStr = regexp.cap(1) + " " + regexp.cap(3) + " " +
                             regexp.cap(2) + " " + regexp.cap(5) + " " + regexp.cap(4);

    // 1970/01/01 00:00:00 GMT
    QDateTime zeroTime(QDate(1970, 1, 1), QTime(0, 0));
    return zeroTime.secsTo(QDateTime::fromString(usLocalDateStr));
  }
}

void KitaThreadView::setSubjectLabel(const QString& boardName, const QString& threadName)
{
  QString disp;
  if ( boardName.isEmpty() ) {
    disp = threadName;
  } else {
    disp = QString("[%1] %2").arg(boardName).arg(threadName);
  }

  disp.truncate( MAX_LABEL_LENGTH );
  subjectLabel->setText( disp );
}

void KitaThreadView::update( const QString& result )
{
  QStringList lines = QStringList::split( "\n", result );
  QString text;
  int num = 0;

  int total = result.length();
  int step = 0;
  int divide = total / ( 100 / 5 );
  int next = divide;
  int prevResNum = m_thread.resNum();
  QProgressDialog * progress = new QProgressDialog( m_parent );
  progress->setTotalSteps( static_cast<int>(total * 0.7) );
  progress->setLabelText( "Parse DAT file.....");
  progress->show();

  text += "<html><head>";
  text += "</head><body>";

  for( QStringList::iterator it = lines.begin(); it != lines.end(); ++it ) {
    QString line = (*it);

    if( line.isEmpty() ) {
      continue;
    }

    Kita::Comment comment(line);

    num++;
    if(num == 1) {
      m_thread.setName(comment.getSubject());
    }
    text += comment.toHtml(num);

    if( prevResNum == num ) {
      text += QString::fromLocal8Bit( kokomade_yonda );
    }

    if( (step+=line.length()) > next ) {
      progress->setProgress( step );
      next += divide;
    }
  }
  delete progress;

  m_thread.setResNum(num);
  emit thread( m_thread );

  setSubjectLabel( m_thread.boardName(), m_thread.name() );

  text += footer( prevResNum );
  text += "</body></html>";

  m_threadPart->displayContentsAndGotoURL( text, m_thread );
  updateButton();

  emit showThreadCompleted( m_thread.url() );
}

void KitaThreadView::updateButton()
{
  writeButton->setEnabled( true );
  BookmarkButton->setEnabled( true );
  ReloadButton->setEnabled( true );

  if ( HighLightButton->isOn() ) {
    HighLightButton->toggle();
  }
  if( FavoriteThreads::getInstance()->contains( m_thread.datURL().url() ) ) {
    BookmarkButton->setOn( true );
  } else {
    BookmarkButton->setOn( false );
  }
}

const QString KitaThreadView::footer( int prevResNum ) const
{
  QString text;
  int target;
  for ( target = 1; target < m_thread.resNum(); target += 100 ) {
    text += QString(" <a href=\"#%1\">%3</a> ").arg(target).arg(target);
  }
  text += QString(" <a href=\"#%1\">New</a> ").arg( prevResNum + 1 );
  text += "<br/>";

  return text;
}

void KitaThreadView::slotShowErrorDialog( const QString& input, const KURL& )
{
  kdDebug() << "'" << input << "'" << endl;
  Kita::WriteResult writeResult(input);

  kdDebug() << "code = " << writeResult.code() << endl;
  switch( writeResult.code() ) {
  case Kita::K2ch_Unknown:
    // probably OK.
    emit writeSucceeded();
    slotReloadButton();
    break;
  case Kita::K2ch_True:
  case Kita::K2ch_False:
  case Kita::K2ch_Error:
  case Kita::K2ch_Check:
    KMessageBox::error(0, writeResult.message(), writeResult.title());
    break;
  case Kita::K2ch_Cookie:
    if( KMessageBox::questionYesNo(0, QString::fromLocal8Bit(cookie_message),
                                      QString::fromLocal8Bit(cookie_title)) == KMessageBox::Yes) {
      KitaWriteDialog* dialog = openDialog( m_postInfo );
      dialog->postMessage();
    } else {
      KitaWriteDialog* dialog = openDialog( m_postInfo );
      dialog->show();
    }
    break;
  default:
    break;
  }
}

KitaWriteDialog* KitaThreadView::openDialog( const Kita::PostInfo& info )
{
  KitaWriteDialog* new_dialog = KitaWriteDialog::open( info, m_thread.boardName(), m_thread.name() );
  connect( new_dialog, SIGNAL( postStarted( KIO::Job *, const Kita::PostInfo& ) ),
           this, SLOT( slotPostStarted( KIO::Job*, const Kita::PostInfo&) ) );
  connect( new_dialog, SIGNAL( postResponse(const QString&, const KURL&) ),
      	   this, SLOT( slotShowErrorDialog(const QString&, const KURL&) ) );
  connect( new_dialog, SIGNAL( postResponse(const QString&, const KURL&) ),
      	   this, SIGNAL( postResponse(const QString&, const KURL&) ) );
  return new_dialog;
}

void KitaThreadView::slotWriteButtonClicked()
{
//  QSjisCodec cp932Codec;
  KURL bbscgiURL = KURL( m_thread.boardUrl(), "../test/bbs.cgi");
  bbscgiURL.setProtocol( "http" );

  Kita::PostInfo info;
  info.host = bbscgiURL.host();
  info.bbs = m_thread.boardId();
  info.key = m_thread.datID();
  info.time = QString("%1").arg(m_serverTime);

  KitaWriteDialog* dialog = openDialog( info );
  dialog->show(); // work asynchronus.
}

void KitaThreadView::slotSearchButton()
{
  insertSearchCombo();
  QStringList list = parseSearchQuery( SearchCombo->currentText() );
  searchNext( list );
}

void KitaThreadView::slotHighLightenButton(bool yes)
{
  insertSearchCombo();
  QStringList list = parseSearchQuery( SearchCombo->currentText() );
  m_threadPart->highLighten( yes, list );
}

void KitaThreadView::insertSearchCombo()
{
  for( int count = 0; count < SearchCombo->count(); ++count ) {
    if ( SearchCombo->text( count ) == SearchCombo->currentText() ) {
      // found
      return;
    }
  }
  SearchCombo->insertItem( SearchCombo->currentText() );
}

QStringList KitaThreadView::parseSearchQuery(const QString &input) const
{
  QStringList tmp = QStringList::split( ' ', input );
  QStringList ret_list;
  QRegExp truncSpace("\\s*$");
  QStringList::iterator it = tmp.begin();
  for( ; it != tmp.end(); ++it )
    ret_list += (*it).replace( truncSpace, "" );
  return ret_list;
}

void KitaThreadView::searchNext(const QStringList &query)
{
  if ( query.isEmpty() ) return;
  if ( ! HighLightButton->isOn() ) {
    HighLightButton->toggle();
    m_threadPart->resetHit(); //A next jump-search target reset to '0'.
    // Process works asynchronusly. So Firstly, we don't do jump-search as a simple solution.
    return;
  }
  if ( query != m_threadPart->prevQuery() ) {
    m_threadPart->highLighten( true, query );
    m_threadPart->resetHit(); //A next jump-search target reset to '0'.
    return;
  }

  m_threadPart->gotoAnchor( QString("highlighten%1").arg( m_threadPart->nextHit() )  );
}

void KitaThreadView::slotOnURL(const QString& url)
{
  emit signalChangeStatusbar(url);

  // TODO: enterEvent()ơäƤȤtipɽ뤳ȡ
  if ( ! url.isEmpty() ) {
    KHTMLView* view = m_threadPart->view();

    if( m_popup ) {
      delete m_popup;
    }

    m_popup = new QFrame( view, "res_popup", WStyle_Customize | WStyle_NoBorder | WStyle_Tool | WType_TopLevel );
    m_browser = new QTextBrowser( m_popup );
    m_browser->zoomOut( static_cast<int>( m_browser->pointSize() * 0.3 ) );
    m_browser->setPaletteBackgroundColor( "yellow" );
    m_browser->setWordWrap( QTextEdit::NoWrap );
    m_browser->setResizePolicy( QScrollView::AutoOne );
    connect( m_threadPart->view(), SIGNAL( leave() ), SLOT( hidePopup() ) );

    int refNum = 0;
    if ( url.at(0) == '#' ) {
      refNum = url.mid(1).toInt();
    } else {
      KURL filteredURL = filterReadCGI( KURL( m_thread.datURL(), url ) );
      refNum = filteredURL.ref().toInt();
    }

    if( refNum != 0 ) {
      DOM::NodeList comments = m_threadPart->htmlDocument().body().getElementsByTagName( "dl" );
      DOM::HTMLDListElement comment_dl = static_cast<DOM::HTMLDListElement>(comments.item( refNum - 1 ));
      QString innerHTML = comment_dl.innerHTML().string();

      m_browser->setText( innerHTML );
      m_browser->resize( m_browser->contentsWidth() + 10, m_browser->contentsHeight() );
      m_popup->adjustSize();

      QPoint pos = QCursor::pos();
      pos -= QPoint( 0, m_popup->height() ) + QPoint( -10, 10 );
      m_popup->move( pos );

      m_popup->show();
    }
  } else {
    m_popup->hide();
  }
}

void KitaThreadView::hidePopup()
{
  if( m_popup ) {
    m_popup->hide();
  }
}

void KitaThreadView::slotPopupMenu( KXMLGUIClient *client, const QPoint &global, const KURL &url, const QString &mimeType, mode_t mode)
{
  KActionCollection * collection = client->actionCollection();
  KAction * action;
  action = new KAction( i18n("goback anchor"), SmallIcon("idea"), KShortcut(), m_threadPart, SLOT( gobackAnchor() ), collection, "goback_anchor" );
  emit popupMenu(client, global, url, mimeType, mode);
}

void KitaThreadView::slotOpenURLRequest(const KURL& url, const KParts::URLArgs& args)
{
  if ( url.url().at(0) == '#' ) {
    m_threadPart->gotoAnchor( url.ref() );
    return;
  }
  KURL datURL = filterReadCGI( url );
  if ( datURL.host() == m_thread.datURL().host()
    && datURL.path() == m_thread.datURL().path() )
  {
    if ( datURL.hasRef() ) m_threadPart->gotoAnchor( datURL.ref() );
    return;
  }
  emit openURLRequest(datURL, args);
}

void KitaThreadView::slotReloadButton()
{
  showThread( m_thread );
}

KURL KitaThreadView::filterReadCGI(const KURL& url)
{
  KURL  newURL = url;
  if ( url.path().contains("/test/read.cgi") ) {
    newURL.setProtocol( m_thread.datURL().protocol() );
    QString tmp = url.path().section("/test/read.cgi", 1);

    QString newPath = QString( "/%1/dat/%2.dat" )
               .arg( tmp.section('/', 1, 1) )
	       .arg( tmp.section('/', 2, 2) );
    newURL.setPath( newPath );

    QString refBase = tmp.section('/', 3);
    if ( ! refBase.isEmpty() ) {
      QString newRef =  refBase.section('-', 0, 0);
      if( ! newRef.isEmpty() ) newURL.setRef( newRef );
      else if ( refBase.at(0) == '-' ) newURL.setRef("1");
      else newURL.setRef( refBase );
    }
  }
  kdDebug() << "newURL: " << newURL.url() << endl;
  return newURL;
}

void KitaThreadView::setFont( const QFont& font )
{
  m_threadPart->setStandardFont(font.family());
  subjectLabel->setFont(font);
}

void KitaThreadView::slotPostStarted( KIO::Job*, const Kita::PostInfo& info)
{
  m_postInfo = info;
}

void KitaThreadView::slotBookmarkButtonClicked( bool on )
{
  emit bookmarked( m_thread, on );
}

void KitaThreadView::killJob()
{
  if( m_access ) {
    m_access->killJob();
  }
}
// vim:sw=2:
