/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "kita.h"
#include "pref.h"
#include "kitathreadview.h"
#include "kitasubjectview.h"
#include "kitaboardview.h"
#include "kitacacheinfo.h"
#include "kitathreadtabwidget.h"
#include "kitasubjecttabwidget.h"

#include "libkita/thread.h"
#include "libkita/favoritethreads.h"

#include <qdragobject.h>
#include <kprinter.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qtextcodec.h>
#include <qevent.h>

#include <klineedit.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <kurlrequesterdlg.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kdebug.h>

#include <kedittoolbar.h>

#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>

KitaMainWindow::KitaMainWindow()
    : KDockMainWindow( 0, "Kita" ),
    ///      m_view(new KitaView(this)),
    m_printer(0)
{
  // accept dnd
  setAcceptDrops(true);

  // setup view, dock
  setupView();

  // then, setup our actions
  setupActions();

  // load cache
  loadCache();

  // load favorites
  loadFavorites();

  // and a status bar
  statusBar()->show();

  // apply the saved mainwindow settings, if any, and ask the mainwindow
  // to automatically save settings if changed: window size, toolbar
  // position, icon size, etc.
  setAutoSaveSettings();

  readConfig(KGlobal::config());

  // allow the view to change the statusbar and caption
  connect( m_threadView, SIGNAL( signalChangeStatusbar( const QString& ) ),
                         SLOT( changeStatusbar( const QString& ) ) );

  connect( m_boardView, SIGNAL( clicked( const Kita::Board& ) ),
           m_subjectView, SLOT( loadBoard( const Kita::Board& ) ) );

  connect( m_subjectView, SIGNAL( signalShowThread( const Kita::Thread& ) ),
           m_threadView, SLOT( showThread( const Kita::Thread& ) ) );

  // ad-hoc
  connect( m_subjectView, SIGNAL( signalShowThreadWithNewTab( const Kita::Thread& ) ),
           m_threadView, SLOT( showThreadWithNewTab( const Kita::Thread& ) ) );

  connect( m_threadView, SIGNAL( thread(const Kita::Thread& ) ),
                         SLOT( updateThreadCache( const Kita::Thread& ) ) );

  connect( m_threadView, SIGNAL( thread( const Kita::Thread& ) ),
           m_subjectView, SLOT( updateThread( const Kita::Thread& ) ) );

  connect( m_threadView, SIGNAL( thread( const Kita::Thread& ) ),
                         SLOT( slotThread( const Kita::Thread& ) ) );

  connect( m_threadView, SIGNAL( writeSucceeded() ),
                         SLOT( slotWriteSucceeded() ) );

  connect( m_threadView, SIGNAL( showThreadCompleted( const KURL& ) ),
                         SLOT( setUrl( const KURL& ) ) );

  connect( m_subjectView, SIGNAL( loadBoardCompleted( const KURL& ) ),
                         SLOT( setUrl( const KURL& ) ) );

  connect( m_threadView, SIGNAL( bookmarked( const Kita::Thread&, bool ) ),
                         SLOT( bookmark( const Kita::Thread&, bool ) ) );

  QPopupMenu* popup = static_cast<QPopupMenu*>( factory()->container( "window", this ) );
  connect( popup, SIGNAL( aboutToShow() ), SLOT( windowMenuAboutToShow() ) );

  m_boardView->loadBoardList();
}

KitaMainWindow::~KitaMainWindow()
{
  saveFavorites();

  saveCache();
  writeConfig(KGlobal::config());
}

void KitaMainWindow::load(const KURL& url)
{
  QString target;
  // the below code is what you should normally do.  in this
  // example case, we want the url to our own.  you probably
  // want to use this code instead for your app

#if 0
  // download the contents
  if (KIO::NetAccess::download(url, target)) {
    // set our caption
    setCaption(url);

    // load in the file (target is always local)
    loadFile(target);

    // and remove the temp file
    KIO::NetAccess::removeTempFile(target);
  }
#endif

  setCaption(url.url());
  //    m_view->openURL(url);
}

void KitaMainWindow::setupActions()
{
  KStdAction::openNew(this, SLOT(fileNew()), actionCollection());
  KStdAction::quit(kapp, SLOT(quit()), actionCollection());
  KStdAction::copy( this, SLOT( slotEditCopy() ), actionCollection() );

  m_toolbarAction = KStdAction::showToolbar(this,
                                            SLOT(optionsShowToolbar()),
                                            actionCollection());

  m_statusbarAction = KStdAction::showStatusbar(this,
                                                SLOT(optionsShowStatusbar()),
                                                actionCollection());

  m_boardListAction = new KToggleAction( i18n("Show board list"),
                                         0,
                                         this,
                                         SLOT( windowShowBoardlist() ),
                                         actionCollection(),
                                         "window_show_board_list" );

  m_subjectListAction = new KToggleAction( i18n("Show subject list"),
                                           0,
                                           this,
                                           SLOT( windowShowSubjectlist() ),
                                           actionCollection(),
                                           "window_show_subject_list" );

  m_urlLine = new KLineEdit("", 0);
  new KWidgetAction(m_urlLine, i18n("URL Line"), 0,
                    this, SLOT(slotUrlLine()) /* dummy */,
                    actionCollection(), "url_line_action");

  KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection());
  KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection());
  KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());
  // this doesn't do anything useful.  it's just here to illustrate
  // how to insert a custom menu and menu item
  /*    KAction *custom = new KAction(i18n("Cus&tom Menuitem"), 0,
                                    this, SLOT(optionsPreferences()),
                                    actionCollection(), "custom_action");*/

  new KAction(i18n("Load BoardList"), 0,
              m_boardView, SLOT(loadBoardList()), actionCollection(), "load_board_list");

  createGUI();
}

void KitaMainWindow::slotUrlLine()
{
  kdDebug() << __func__;
}

void KitaMainWindow::writeConfig(KConfig *config)
{
  config->setGroup("Global");
  config->writeEntry("Font", KitaConfig::font());
}

void KitaMainWindow::saveProperties(KConfig *config)
{
  // the 'config' object points to the session managed
  // config file.  anything you write here will be available
  // later when this app is restored

  //    if (m_view->currentURL() != QString::null)
  //        config->writeEntry("lastURL", m_view->currentURL());
  writeConfig(config);
}

void KitaMainWindow::readConfig(KConfig* config)
{
  config->setGroup("Global");
  KitaConfig::setFont(config->readFontEntry("Font"));

  setFont(KitaConfig::font());
}

void KitaMainWindow::readProperties(KConfig *config)
{
  // the 'config' object points to the session managed
  // config file.  this function is automatically called whenever
  // the app is being restored.  read in here whatever you wrote
  // in 'saveProperties'

  //    QString url = config->readEntry("lastURL");

  //    if (url != QString::null)
  //        m_view->openURL(KURL(url));
  readConfig(config);
}

void KitaMainWindow::dragEnterEvent(QDragEnterEvent *event)
{
  // accept uri drops only
  event->accept(QUriDrag::canDecode(event));
}

void KitaMainWindow::dropEvent(QDropEvent *event)
{
  // this is a very simplistic implementation of a drop event.  we
  // will only accept a dropped URL.  the Qt dnd code can do *much*
  // much more, so please read the docs there
  QStrList uri;

  // see if we can decode a URI.. if not, just ignore it
  if (QUriDrag::decode(event, uri)) {
    // okay, we have a URI.. process it
    QString url, target;
    url = uri.first();

    // load in the file
    load(KURL(url));
  }
}

void KitaMainWindow::fileNew()
{
  // this slot is called whenever the File->New menu is selected,
  // the New shortcut is pressed (usually CTRL+N) or the New toolbar
  // button is clicked

  // create a new window
  (new KitaMainWindow)->show();
}

void KitaMainWindow::fileOpen()
{
  // this slot is called whenever the File->Open menu is selected,
  // the Open shortcut is pressed (usually CTRL+O) or the Open toolbar
  // button is clicked
  //    KURL url = KURLRequesterDlg::getURL(QString::null, this, i18n("Open Location") );
  //    if (!url.isEmpty())
  //        m_view->openURL(url);
}

void KitaMainWindow::fileSave()
{
  // this slot is called whenever the File->Save menu is selected,
  // the Save shortcut is pressed (usually CTRL+S) or the Save toolbar
  // button is clicked

  // save the current file
}

void KitaMainWindow::fileSaveAs()
{
  // this slot is called whenever the File->Save As menu is selected,
  KURL file_url = KFileDialog::getSaveURL();
  if (!file_url.isEmpty() && !file_url.isMalformed()) {
    // save your info, here
  }
}

void KitaMainWindow::filePrint()
{
  // this slot is called whenever the File->Print menu is selected,
  // the Print shortcut is pressed (usually CTRL+P) or the Print toolbar
  // button is clicked
  /*    if (!m_printer) m_printer = new KPrinter;
      if (m_printer->setup(this))
      {
          // setup the printer.  with Qt, you always "print" to a
          // QPainter.. whether the output medium is a pixmap, a screen,
          // or paper
          QPainter p;
          p.begin(m_printer);

          // we let our view do the actual printing
          QPaintDeviceMetrics metrics(m_printer);
          m_view->print(&p, metrics.height(), metrics.width());

          // and send the result to the printer
          p.end();
      }*/
}

void KitaMainWindow::optionsShowToolbar()
{
  // this is all very cut and paste code for showing/hiding the
  // toolbar
  if (m_toolbarAction->isChecked())
    toolBar()->show();
  else
    toolBar()->hide();
}

void KitaMainWindow::optionsShowStatusbar()
{
  // this is all very cut and paste code for showing/hiding the
  // statusbar
  if ( m_statusbarAction->isChecked() )
    statusBar()->show();
  else
    statusBar()->hide();
}

void KitaMainWindow::optionsConfigureKeys()
{
  KKeyDialog::configureKeys(actionCollection(), "kitaui.rc");
}

void KitaMainWindow::optionsConfigureToolbars()
{
  // use the standard toolbar editor
  saveMainWindowSettings( KGlobal::config(), settingsGroup() );
  KEditToolbar dlg(actionCollection());
  connect( &dlg, SIGNAL( newToolbarConfig() ),
                 SLOT( newToolbarConfig() ) );
  dlg.exec();
}

void KitaMainWindow::newToolbarConfig()
{
  // this slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
  // recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
  createGUI();
  applyMainWindowSettings( KGlobal::config(), settingsGroup() );
}

void KitaMainWindow::windowShowBoardlist()
{
  if ( m_boardListAction->isChecked() )
    m_boardDock->show();
  else
    m_boardDock->hide();
}

void KitaMainWindow::windowShowSubjectlist()
{
  if ( m_subjectListAction->isChecked() )
    m_subjectDock->show();
  else
    m_subjectDock->hide();
}

void KitaMainWindow::windowMenuAboutToShow()
{
  m_boardListAction->setChecked( m_boardDock->isVisible() );
  m_subjectListAction->setChecked( m_subjectDock->isVisible() );
}

void KitaMainWindow::optionsPreferences()
{
  // popup some sort of preference dialog, here
  KitaPreferences dlg;

  connect(&dlg, SIGNAL( fontChanged( const QFont& ) ),
                SLOT( setFont( const QFont& ) ) );
  if (dlg.exec()) {
    // redo your settings
  }
}

void KitaMainWindow::changeStatusbar(const QString& text)
{
  // display the text on the statusbar
  statusBar()->message(text);
}

void KitaMainWindow::changeCaption(const QString& text)
{
  // display the text on the caption
  setCaption(text);
}

void KitaMainWindow::setupView()
{
  KDockWidget* mainDock;
  mainDock = createDockWidget( "main", 0L, 0L, i18n("main") );
  m_threadView = new KitaThreadTabWidget( mainDock );
  mainDock->setWidget( m_threadView );
  mainDock->setDockSite( KDockWidget::DockFullSite );
  mainDock->setEnableDocking( KDockWidget::DockNone );
  connect( m_threadView, SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ),
                         SLOT( slotOpenURLRequest( const KURL&, const KParts::URLArgs& ) ) );

  m_subjectDock = createDockWidget( "subject", 0L, 0L, i18n("subject") );
//  m_subjectView = new KitaSubjectView( m_subjectDock );
  m_subjectView = new KitaSubjectTabWidget( m_subjectDock );
  m_subjectDock->setWidget( m_subjectView );
  m_subjectDock->setDockSite( KDockWidget::DockFullSite );

  m_boardDock = createDockWidget( "board", 0L, 0L, i18n("board") );
  m_boardView = new KitaBoardView( m_boardDock );
  m_boardDock->setWidget( m_boardView );
  m_boardDock->setDockSite( KDockWidget::DockFullSite );

  setView( mainDock );
  setMainDockWidget( mainDock );

  m_boardDock->manualDock( mainDock, KDockWidget::DockLeft, 20 );
  m_subjectDock->manualDock( mainDock, KDockWidget::DockTop );
}

void KitaMainWindow::slotOpenURLRequest(const KURL& url, const KParts::URLArgs&)
{
  new KRun(url);
}

void KitaMainWindow::updateThreadCache( const Kita::Thread& thread )
{
  KitaCacheInfo::getInstance()->setReadNum( thread.datURL(), thread.resNum() );
  KitaCacheInfo::getInstance()->setResNum( thread.datURL(), thread.resNum() );
}

void KitaMainWindow::loadCache()
{
  KitaCacheInfo* cache = KitaCacheInfo::getInstance();
  QString cacheConfigPath = locateLocal("appdata", "cache");
  QFile file(cacheConfigPath);
  if( file.open(IO_ReadOnly) ) {
    QDataStream stream(&file);
    stream >> *cache;
  }
}

void KitaMainWindow::saveCache()
{
  KitaCacheInfo* cache = KitaCacheInfo::getInstance();
  QString cacheConfigPath = locateLocal("appdata", "cache");
  QFile file(cacheConfigPath);
  if( file.open(IO_WriteOnly) ) {
    QDataStream stream(&file);
    stream << *cache;
  }
}

/*!
    \fn KitaMainWindow::setFont(QFont& font)
 */
void KitaMainWindow::setFont( const QFont& font )
{
  KitaConfig::setFont(font);
  m_threadView->setFont(font);
  m_subjectView->setFont(font);
  m_boardView->setFont(font);
}

void KitaMainWindow::slotWriteSucceeded()
{
  statusBar()->message(i18n("posting succeeded."));
}

void KitaMainWindow::setUrl(const KURL& url)
{
  m_urlLine->setText(url.url());
}

void KitaMainWindow::bookmark( const Kita::Thread& thread, bool on )
{
  Kita::Thread* t = new Kita::Thread( thread.getBoard(), thread.datURL(), thread.name(), thread.resNum() );
  FavoriteThreads* favorite = FavoriteThreads::getInstance();

  if( on ) {
    favorite->insert( t );
  } else {
    favorite->remove( t->datURL().url() );
  }
}

void KitaMainWindow::loadFavorites()
{
  QString favoritesConfigPath = locateLocal("appdata", "favorites.xml");
  QFile file( favoritesConfigPath );
  if( file.open(IO_ReadOnly) ) {
    FavoriteThreads::fromXml( QString::fromUtf8( file.readAll().data() ) );
  }
}

void KitaMainWindow::saveFavorites()
{
  QString favoritesConfigPath = locateLocal("appdata", "favorites.xml");
  QFile file( favoritesConfigPath );
  if( file.open(IO_WriteOnly) ) {
    QTextStream stream(&file);
    stream.setEncoding( QTextStream::UnicodeUTF8 );
    stream << FavoriteThreads::getInstance()->toXml();
  }
}

void KitaMainWindow::slotThread( const Kita::Thread& thread )
{
  setCaption( thread.name() );
}

void KitaMainWindow::slotEditCopy()
{
  QWidget* widget = kapp->focusWidget();

  if( widget ) {
    QKeyEvent e( QEvent::KeyPress, Key_C, 'C', ControlButton );
    QApplication::sendEvent( widget, &e );
  }
}

#include "kita.moc"
