/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "board.h"

#include <kio/slaveconfig.h>
#include <kio/netaccess.h>

#include <kfilterdev.h>

#include <qregexp.h>

#include "qcp932codec.h"

using namespace Kita;

Board::Board()
{
}

Board::Board(const KURL& boardURL, const QString boardName) : m_boardURL(boardURL), m_boardName(boardName)
{
}

Board::~Board()
{
}

const KURL Board::subjectTxtURL() const
{
  if( m_boardURL.isEmpty() ) {
    return KURL();
  } else {
    return KURL(m_boardURL, "./subject.txt");
  }
}

const KURL& Board::url() const
{
  return m_boardURL;
}

const QString& Board::name() const
{
  return m_boardName;
}

const QString Board::id() const
{
  return m_boardURL.fileName();
}

const QPtrList<Thread> Board::getThreadList() const
{
  if( m_boardURL.isEmpty() ) {
    qDebug("Waring: board url is empty!");
    return QPtrList<Thread>();
  }
  QPtrList<Thread> threadList;

  KIO::SlaveConfig::self()->setConfigData("http", m_boardURL.host() , "UserAgent", "Monazilla/1.00 (Kita/0.6)");
  QString tmpFile;
  if(KIO::NetAccess::download(subjectTxtURL(), tmpFile)) {
    QIODevice* tmpDevice = KFilterDev::deviceForFile(tmpFile, "application/x-gzip");
    tmpDevice->open(IO_ReadOnly);

    QCp932Codec cp932Codec;
    QTextStream stream(tmpDevice);
    stream.setCodec(&cp932Codec);

    // parse subject.txt(only one format...)
    QRegExp regexp("(\\d+\\.dat)<>(.*)\\((\\d+)\\)");
    QString line;

    while((line = stream.readLine()) != QString::null) {
      int pos = regexp.search(line);
      if(pos != -1) {
        QString fname = regexp.cap(1);
        QString subject = regexp.cap(2);
        QString num = regexp.cap(3);

        KURL datURL = m_boardURL;
        datURL.addPath("/dat/" + fname);
        Kita::Thread* thread = new Kita::Thread(*this, datURL, subject, num.toInt());
        threadList.append(thread);
      }
    }

    KIO::NetAccess::removeTempFile(tmpFile);
  }
  return threadList;
}

const QString Board::toXmlFragment() const
{
  QString ret;

  ret += "<board xmlns=\"http://kita.sourceforge.jp/ns/board\">\n";
  ret += QString("<url>%1</url>\n").arg( m_boardURL.prettyURL() );
  ret += QString("<name>%1</name>\n").arg( m_boardName );
  ret += "</board>\n";

  return ret;
}

Board Board::fromXml( const QString& xml )
{
  Board board;

  BoardXmlParser parser;
  QXmlSimpleReader reader;
  QXmlInputSource source;
  source.setData( xml );
  reader.setContentHandler( &parser );
  reader.parse( &source );

  if( parser.isValid() ) {
    board = parser.getBoard();
  } else {
    qDebug("inValid");
  }
  return board;
}

bool Board::test()
{
  qDebug("Board::test()");
  Board board = Board::fromXml( "<board>\n<url>http://pc.2ch.net/linux/</url>\n<name>Linux</name>\n</board>\n" );
  if( board.url().prettyURL() != "http://pc.2ch.net/linux/" ) qDebug("inValid: line=%d", __LINE__);
  if( board.name() != "Linux" ) qDebug("inValid: line=%d", __LINE__);

  board = Board::fromXml( "  <board>\n    <url>http://pc.2ch.net/linux/</url>\n    <name>Linux</name>\n  </board>\n" );
  if( board.url().prettyURL() != "http://pc.2ch.net/linux/" ) qDebug("inValid: line=%d", __LINE__);
  if( board.name() != "Linux" ) qDebug("inValid: line=%d", __LINE__);

  return true;
}

bool BoardXmlParser::startElement( const QString&, const QString& localName, const QString&, const QXmlAttributes& )
{
  if( m_inBoard == true ) {
    if( localName == "url" ) {
    } else if( localName == "name" ) {
    } else {
      // error
      return false;
    }
  } else {
    if( localName == "board" ) {
      m_inBoard = true;
    } else {
      // error
      return false;
    }
  }
  return true;
}

bool BoardXmlParser::endElement( const QString&, const QString& localName, const QString& )
{
  if( localName == "board" ) {
    m_inBoard = false;
    m_board = Kita::Board( KURL( m_urlStr ), m_nameStr );
    m_isValid = true;
    // create board;
  } else if( localName == "url" ) {
    m_urlStr = m_characters;
  } else if( localName == "name" ) {
    m_nameStr = m_characters;
  } else {
    // error
    return false;
  }
  return true;
}

bool BoardXmlParser::characters( const QString& ch )
{
  m_characters = ch;
  return true;
}
