/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "favoritelistview.h"

#include "libkita/favoritethreads.h"
#include "libkita/board.h"

#include <qmap.h>
#include <qapplication.h>
#include <qclipboard.h>

#include <klocale.h>
#include <kpopupmenu.h>
#include <krun.h>

enum FavoriteListViewRows {
  Row_Board,
  Row_Icon,
  Row_Subject,
  Row_ResNum,
  Row_Read,
  Row_Unread,
  Row_DatName,
  Row_DatURL
};

FavoriteListView::FavoriteListView(QWidget* parent, const char *name)
 : KListView(parent, name)
{
  addColumn(i18n("Board"));
  addColumn("");
  addColumn(i18n("Title"));
  addColumn(i18n("ResNum"));
  addColumn(i18n("ReadNum"));
  addColumn(i18n("Unread"));
  addColumn(i18n("Dat"));

  setColumnWidth(Row_Subject, 400);
  setColumnWidthMode(Row_Subject, QListView::Manual);

  connect( this, SIGNAL( clicked(QListViewItem*) ),
                 SLOT( loadThread(QListViewItem*) ) );
  connect( this, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
                 SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
}

FavoriteListView::~FavoriteListView()
{
}

void FavoriteListView::update()
{
  const QDict<Kita::Thread>& threads = FavoriteThreads::getInstance()->threads();
//  FavoriteThreads::const_iterator it;

  clear();

  QDictIterator<Kita::Thread> it( threads );
  for(; it.current(); ++it) {
    const Kita::Thread* thread = it.current();
    const Kita::Board* board = thread->getBoard();

    new KListViewItem( this,
                       board->name(),
                       "",
                       thread->name(),
                       QString("%1").arg(thread->resNum(), 4),
                       "",
                       "",
                       thread->datURL().fileName(),
                       thread->datURL().prettyURL() );
  }
}

void FavoriteListView::loadThread( QListViewItem* item )
{
  if( ! item ) return;

  const QDict<Kita::Thread>& threads = FavoriteThreads::getInstance()->threads();
//  FavoriteThreads::const_iterator it;

  QDictIterator<Kita::Thread> it( threads );
  for(; it.current(); ++it) {
    const Kita::Thread* thread = it.current();

    if( thread->datURL().prettyURL() == item->text( Row_DatURL ) ) {
      emit signalShowThread( *thread );
    }
  }
}

void FavoriteListView::updateThread( const Kita::Thread& updated_thread )
{
  for( QListViewItem* item = firstChild(); item; item = item->nextSibling() ) {
    if( item->text( Row_DatURL ) == updated_thread.datURL().prettyURL() ) {
      item->setText( Row_ResNum, QString("%1").arg( updated_thread.resNum(), 4 ) );
    }
  }
}

// TODO: KitaSubjectView饹Ʊ᥽åɤȤۤȤƱ
void FavoriteListView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
  if( item == 0 ) {
    return;
  }

  KPopupMenu popup( 0 );
  popup.insertItem( i18n("Open with Web Browser"), 0 );
  popup.insertItem( i18n("Open with new tab"), 1 );
  popup.insertItem( i18n("Copy title and URL"), 2 );

  QString datName = item->text(Row_DatName);
  KURL datURL = KURL( item->text(Row_DatURL) );

  qDebug("datURL = %s", datURL.url().latin1());
  Kita::Board board = Kita::Board( KURL( datURL, ".." ) );
  qDebug("board.url = %s", board.url().url().latin1());
  Kita::Thread thread( board, datURL );
  qDebug("thread.url = %s", thread.url().url().latin1());
  thread.setName( item->text( Row_Subject ) );
  thread.setResNum( item->text( Row_Read ).toInt() );

  QClipboard* clipboard = QApplication::clipboard();

  switch( popup.exec( point ) ) {
  case 0:
    KRun::runURL( thread.url(), "text/html" );
    break;
  case 1:
    emit signalShowThreadWithNewTab( thread );
    break;
  case 2:
    clipboard->setText( thread.name() + "\n" + thread.url().url() );
    break;
  default:
    break;
  }
}
