/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "thread.h"

#include <kurl.h>
#include <kio/slaveconfig.h>
#include <kio/jobclasses.h>
#include <kio/scheduler.h>

#include <qwidget.h>
#include <qapplication.h>

#include "board.h"
#include "qcp932codec.h"

using namespace Kita;

Thread::Thread() : m_board( 0 )
{
}

Thread::Thread(const Board& board, const KURL& datURL, const QString threadName, int resNum)
  : m_datURL(datURL), m_threadName(threadName), m_resNum(resNum)
{
  m_datID = m_datURL.filename().section(".", 0, 0);
  m_board = new Board( board.url(), board.name() );
  m_boardID = m_board->id();
}

Thread::Thread(const Board* board, const KURL& datURL, const QString threadName, int resNum)
  : m_datURL(datURL), m_threadName(threadName), m_resNum(resNum)
{
  m_datID = m_datURL.filename().section(".", 0, 0);
  m_board = new Board( board->url(), board->name() );
  m_boardID = m_board->id();
}

Thread::Thread( const Thread& obj )
{
  m_datURL = obj.m_datURL;
  m_datID = obj.m_datID;
  m_boardID = obj.m_boardID;
  m_threadName = obj.m_threadName;
  m_resNum = obj.m_resNum;
  m_board = new Board( obj.m_board->url(), obj.m_board->name() );
}

Thread::~Thread()
{
  delete m_board;
}

Thread& Thread::operator=( const Thread& obj )
{
  m_datURL = obj.m_datURL;
  m_datID = obj.m_datID;
  m_boardID = obj.m_boardID;
  m_threadName = obj.m_threadName;
  m_resNum = obj.m_resNum;
  m_board = new Board( obj.m_board->url(), obj.m_board->name() );

  return *this;
}

const KURL& Thread::datURL() const
{
  return m_datURL;
}

const KURL Thread::url() const
{
  KURL url = KURL(m_datURL, "/test/read.cgi/");
  url.addPath(m_boardID + "/");
  url.addPath(m_datID + "/");

  return url;
}

const QString& Thread::datID() const
{
  return m_datID;
}

int Thread::resNum() const
{
  return m_resNum;
}

const QString& Thread::name() const
{
  return m_threadName;
}

const QString& Thread::boardName() const
{
  return m_board->name();
}

const KURL& Thread::boardUrl() const
{
  return m_board->url();
}

const QString Thread::boardId() const
{
  return m_board->id();
}

const QString Thread::toXmlFragment() const
{
  QString ret;

  ret += "<thread xmlns=\"http://kita.sourceforge.jp/ns/thread\">\n";
  ret += QString("<daturl>%1</daturl>\n").arg( m_datURL.prettyURL() );
  ret += QString("<name>%1</name>\n").arg( m_threadName );
  ret += QString("<resnum>%1</resnum>\n").arg( m_resNum );
  ret += m_board->toXmlFragment();
  ret += "</thread>\n";

  return ret;
}

Thread Thread::fromXml( const QString& xml )
{
  Thread thread;

  ThreadXmlParser parser;
  QXmlSimpleReader reader;
  QXmlInputSource source;
  source.setData( xml );
  reader.setContentHandler( &parser );
  reader.parse( &source );

  if( parser.isValid() ) {
    thread = parser.getThread();
  } else {
    qDebug("inValid");
  }

  return thread;
}

bool Thread::test()
{
  qDebug("Thread::test()");
  Thread thread = Thread::fromXml( "<thread xmlns=\"http://kita.sourceforge.jp/ns/thread\">\n"
                                   "<daturl>http://pc.2ch.net/linux/dat/1022744633.dat</daturl>\n"
                                   "<name>2ch browser thread</name>\n"
                                   "<resnum>700</resnum>\n"
                                   "<board xmlns=\"http://kita.sourceforge.jp/ns/board\">\n"
                                   "<url>http://pc.2ch.net/linux/</url>\n"
                                   "<name>Linux</name>\n"
                                   "</board>\n"
                                   "</thread>\n" );
  if( thread.datURL().prettyURL() != "http://pc.2ch.net/linux/dat/1022744633.dat" ) qDebug("inValid: line=%d", __LINE__);
  if( thread.url().prettyURL() != "http://pc.2ch.net/test/read.cgi/linux/1022744633/" ) qDebug("inValid: line=%d", __LINE__);
  if( thread.name() != "2ch browser thread" ) qDebug("inValid: line=%d", __LINE__);
  if( thread.resNum() != 700 ) qDebug("inValid: line=%d", __LINE__);
  if( thread.getBoard()->url().prettyURL() != "http://pc.2ch.net/linux/" ) qDebug("inValid: line=%d", __LINE__);
  if( thread.getBoard()->name() != "Linux" ) qDebug("inValid: line=%d", __LINE__);

  Thread thread2 = Thread::fromXml( thread.toXmlFragment() );

  if( thread2.datURL().prettyURL() != "http://pc.2ch.net/linux/dat/1022744633.dat" ) qDebug("inValid: line=%d", __LINE__);
  if( thread2.url().prettyURL() != "http://pc.2ch.net/test/read.cgi/linux/1022744633/" ) qDebug("inValid: line=%d", __LINE__);
  if( thread2.name() != "2ch browser thread" ) qDebug("inValid: line=%d", __LINE__);
  if( thread2.resNum() != 700 ) qDebug("inValid: line=%d", __LINE__);
  if( thread2.getBoard()->url().prettyURL() != "http://pc.2ch.net/linux/" ) qDebug("inValid: line=%d", __LINE__);
  if( thread2.getBoard()->name() != "Linux" ) qDebug("inValid: line=%d", __LINE__);

  qDebug( "%s", thread.toXmlFragment().latin1() );

  thread = Thread::fromXml( "<thread xmlns=\"http://kita.sourceforge.jp/ns/thread\">\n"
                            "  <daturl>http://pc.2ch.net/linux/dat/1022744633.dat</daturl>\n"
                            "  <name>2ch browser thread</name>\n"
                            "  <resnum>700</resnum>\n"
                            "  <board xmlns=\"http://kita.sourceforge.jp/ns/board\">\n"
                            "    <url>http://pc.2ch.net/linux/</url>\n"
                            "    <name>Linux</name>\n"
                            "  </board>\n"
                            "</thread>\n" );
  if( thread.datURL().prettyURL() != "http://pc.2ch.net/linux/dat/1022744633.dat" ) qDebug("inValid: line=%d", __LINE__);
  if( thread.url().prettyURL() != "http://pc.2ch.net/test/read.cgi/linux/1022744633/" ) qDebug("inValid: line=%d", __LINE__);
  if( thread.name() != "2ch browser thread" ) qDebug("inValid: line=%d", __LINE__);
  if( thread.resNum() != 700 ) qDebug("inValid: line=%d", __LINE__);
  if( thread.getBoard()->url().prettyURL() != "http://pc.2ch.net/linux/" ) qDebug("inValid: line=%d", __LINE__);
  if( thread.getBoard()->name() != "Linux" ) qDebug("inValid: line=%d", __LINE__);

  return true;
}

bool ThreadXmlParser::startElement( const QString& namespaceURI, const QString& localName, const QString& qName, const QXmlAttributes& atts )
{
  if( m_inBoard ) {
    m_boardParser->startElement( namespaceURI, localName, qName, atts );
  } else if( m_inThread ) {
    if( localName == "daturl" ) {
    } else if( localName == "name" ) {
    } else if( localName == "resnum" ) {
    } else if( localName == "board" ) {
      m_inBoard = true;
      m_boardParser = new BoardXmlParser();
      m_boardParser->startElement( namespaceURI, localName, qName, atts );
    } else {
      // error
      return false;
    }
  } else {
    if( localName == "thread" ) {
      m_inThread = true;
    } else {
      // error
      return false;
    }
  }

  return true;
}

bool ThreadXmlParser::endElement( const QString& namespaceURI, const QString& localName, const QString& qName )
{
  if( localName == "board" ) {
    m_inBoard = false;
    m_boardParser->endElement( namespaceURI, localName, qName );
    Board board = m_boardParser->getBoard();
    m_board = new Board( board.url(), board.name() );

    delete m_boardParser;
    m_boardParser = 0;

    // create board;
  } else if( m_inBoard ) {
    m_boardParser->endElement( namespaceURI, localName, qName );
  } else if( localName == "thread" ) {
    m_inThread = false;
    m_thread = Kita::Thread( *m_board, KURL( m_datUrlStr ), m_threadName, m_resNum );
    m_isValid = true;
    // create thread;
  } else if( localName == "daturl" ) {
    m_datUrlStr = m_characters;
  } else if( localName == "name" ) {
    m_threadName = m_characters;
  } else if( localName == "resnum" ) {
    m_resNum = m_characters.toInt();
  } else {
    // error
    return false;
  }
  return true;
}

bool ThreadXmlParser::characters( const QString& ch )
{
  if( m_inBoard ) {
    m_boardParser->characters( ch );
  } else {
    m_characters = ch;
  }
  return true;
}

QString ThreadAccess::get()
{
  if ( m_thread.datURL().protocol() != "k2ch" ) {
    KIO::SlaveConfig::self()->setConfigData("http", m_thread.datURL().host() ,
                                          "UserAgent", "Monazilla/1.00 (Kita/0.51)");
  }

  KIO::TransferJob* job = KIO::get(m_thread.datURL(), true, true);
  m_currentJob = job;

  connect(job, SIGNAL(data(KIO::Job*, const QByteArray&)),
          SLOT(slotReceiveThreadData(KIO::Job*, const QByteArray&)));
  connect(job, SIGNAL(result(KIO::Job*)), SLOT(slotThreadResult(KIO::Job*)));
  connect(job, SIGNAL(redirection(KIO::Job *, const KURL&) ), SLOT(slotRedirection(KIO::Job *, const KURL&) ) );

  // use 'HTTP-Headers' metadata.
  job->addMetaData("PropagateHttpHeader", "true");

  enter_loop();

  QCp932Codec codec;
  QTextStream stream(m_threadData, IO_ReadOnly);
  stream.setCodec(&codec);

  return stream.read();
}

// from netaccess.cpp
void qt_enter_modal( QWidget* widget );
void qt_leave_modal( QWidget* widget );

void ThreadAccess::enter_loop()
{
  QWidget dummy( 0, 0, WType_Dialog | WShowModal );
  dummy.setFocusPolicy( QWidget::NoFocus );
  qt_enter_modal(&dummy);
  qApp->enter_loop();
  qt_leave_modal(&dummy);
}

void ThreadAccess::slotRedirection(KIO::Job *, const KURL & newURL)
{
  qDebug("  Redirected to newURL: %s\n", newURL.url().latin1() );
  m_thread = Kita::Thread( m_thread.getBoard(), newURL);
  emit redirection( newURL.url() );
}

void ThreadAccess::slotReceiveThreadData( KIO::Job*, const QByteArray& data )
{
  QCString cstr(data.data(), data.size()+1);
  m_threadData.append(cstr);
}

void ThreadAccess::slotThreadResult(KIO::Job* job)
{
  m_currentJob = 0;
  if(job->error()) {
    job->showErrorDialog();
  } else {
    m_header = job->queryMetaData("HTTP-Headers");
  }
  qApp->exit_loop();
}

void ThreadAccess::killJob()
{
  if( m_currentJob ) m_currentJob->kill();
}

#include "thread.moc"
