/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <kio/netaccess.h>

#include <qfile.h>
#include <qtextcodec.h>
#include <qstringlist.h>
#include <qregexp.h>
#include <klistview.h>
#include <qvaluelist.h>

#include "libkita/qcp932codec.h"
#include "part/kita2ch.h"
#include "kitaboardview.h"
#include "kita.h"
#include "libkita/board.h"
#include "libkita/category.h"


KitaBoardView::KitaBoardView(QWidget *parent, const char *name)
    : KitaBoardViewBase(parent, name)
{
  m_boardList->setSorting(-1);
  m_boardList->addColumn("name");
  connect(m_boardList, SIGNAL(executed(QListViewItem*)), SLOT( loadBoard(QListViewItem*) ));
}

KitaBoardView::~KitaBoardView()
{}

QString KitaBoardView::getCategory(const QString& line) const
{
  QRegExp regexp("<BR><BR><B>(.*)</B><BR>", false);
  if(regexp.search(line) != -1) {
    return regexp.cap(1);
  } else {
    return QString::null;
  }
}

Kita::Board KitaBoardView::getBoard(const QString& line) const
{
  QRegExp regexp("<A HREF=(.*)>(.*)</A>", false);
  if( regexp.search(line) != -1 ) {
    QString board_url = regexp.cap(1);
    QString board_title = regexp.cap(2);
    return Kita::Board(board_url, board_title);
  } else {
    return Kita::Board(KURL());
  }
}

bool KitaBoardView::isBoardUrl(const KURL& url) const
{
  QRegExp url_2ch("http://.*\\.2ch\\.net/.*");
  QRegExp url_bbspink("http://.*\\.bbspink\\.com/.*");
  QRegExp url_www_2ch("http://www\\.2ch\\.net/.*");

  if ( url.isEmpty() ) return false;

  QString str = url.prettyURL();

  if ( url_2ch.search(str) == -1 && url_bbspink.search(str) == -1 ) return false;
  if ( url_www_2ch.search(str) != -1 ) return false;

  return true;
}

void KitaBoardView::loadBoardList()
{
  QString tmpFile;
  QString url = "http://www.ff.iij4u.or.jp/~ch2/bbsmenu.html";
  if( ! KIO::NetAccess::download(url, tmpFile) )
  {
    return;
  }

  QFile file(tmpFile);
  if( ! file.open(IO_ReadOnly) )
  {
    return;
  }

  QTextStream stream(&file);
  QCp932Codec cp932Codec;
  stream.setCodec(&cp932Codec);
  QPtrList<Kita::Category> categoryList = getCategoryList(stream.read());

  // clear list
  m_boardList->clear();

  KListViewItem* previous = 0;
  for(Kita::Category* category = categoryList.first(); category; category = categoryList.next())
  {
    QValueList<Kita::Board> board_list = category->getBoardList();

    if( board_list.isEmpty() ) {
      continue;
    }

    KListViewItem* categoryItem = new KListViewItem(m_boardList, previous, category->name());

    KListViewItem* previousBoard = 0;
    for(QValueListIterator<Kita::Board> it = board_list.begin(); it != board_list.end(); ++it)
    {
      Kita::Board board = (*it);
      previousBoard = new KListViewItem(categoryItem, previousBoard, board.name(), board.url().prettyURL());
    }
    previous = categoryItem;
  }
}

QPtrList<Kita::Category> KitaBoardView::getCategoryList(const QString& html) const
{
  QPtrList<Kita::Category> result;
  result.setAutoDelete( true );

  QStringList lines = QStringList::split("\n", html);
  QStringList::iterator it;

  Kita::Category* current_category = 0;
  for(it = lines.begin(); it != lines.end(); ++it) {
    QString category_name = getCategory(*it);
    if( category_name != QString::null ) {
      current_category = new Kita::Category(category_name);
      result.append(current_category);
    } else {
      Kita::Board board = getBoard(*it);

      if( isBoardUrl(board.url()) ) {
        current_category->append(board);
      }
    }
  }
//  printCateogryList(result);
  return result;
}

void KitaBoardView::printCateogryList(QPtrList<Kita::Category>& list)
{
  Kita::Category* category;
  for(category = list.first(); category; category = list.next()) {
    if(category->name() == QString::null) continue;

    QValueList<Kita::Board> board_list = category->getBoardList();
    QValueListIterator<Kita::Board> it2;
    for(it2 = board_list.begin(); it2 != board_list.end(); ++it2) {
      qDebug("category:%s name:%s url:%s", (const char *)category->name().local8Bit(),
      (const char*)(*it2).name().local8Bit(),
      (*it2).url().prettyURL().latin1());
    }
  }
}

void KitaBoardView::loadBoard( QListViewItem* item )
{
  QString boardName = item->text(0);
  KURL url = item->text(1);

  if(item->text(1).isEmpty()) {
    return;
  }

  Kita::Board board(url, boardName);
  emit clicked( board );
}

void KitaBoardView::setFont( const QFont& font )
{
  m_boardList->setFont(font);
}
