/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@wakaba.jp                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

// include files for Qt
#include <qtoolbutton.h>
#include <qtextbrowser.h>
#include <qtextcodec.h>
#include <qregexp.h>
#include <qfile.h>

// kdelibs/kio
#include <kio/slaveconfig.h>
#include <kio/netaccess.h>

// include files for KDE
#include <kfilterdev.h>
#include <klistview.h>
#include <kiconloader.h>
#include <klocale.h>
 
#include "kitasubjectview.h"
#include "kita.h"
#include "kitacacheinfo.h"
#include "part/kita2ch.h"

enum SubjectViewRows {
  Row_ID,
  Row_Icon,
  Row_Subject,
  Row_ResNum,
  Row_Vested,
  Row_Unread,
  Row_DatName,
};

KitaSubjectView::KitaSubjectView(QWidget *parent, const char *name)
  : KitaSubjectViewBase(parent, name)
{
  {
    SearchButton->setPixmap( SmallIcon("find") );
    HideButton->setPixmap( SmallIcon("filter") );
  }
  
  subjectList->addColumn(i18n("No."));
  subjectList->addColumn("");
  subjectList->addColumn(i18n("Title"));
  subjectList->addColumn(i18n("ResNum"));
  subjectList->addColumn(i18n("Vested"));
  subjectList->addColumn(i18n("Unread"));
  subjectList->addColumn(i18n("Dat"));

  subjectList->setColumnWidth(Row_Subject, 400);
  subjectList->setColumnWidthMode(Row_Subject, QListView::Manual);

  connect(subjectList, SIGNAL(clicked(QListViewItem*)), SLOT(loadThread(QListViewItem*)));
  connect( SearchButton, SIGNAL( clicked() ), SLOT( slotSearchButton() ) );
  connect( SearchCombo, SIGNAL( activated(int) ), SLOT( slotSearchButton() ) );
  connect(this, SIGNAL(signalSubjectListClicked(QListViewItem*)), subjectList, SIGNAL(clicked(QListViewItem*)));
  connect( HideButton, SIGNAL( toggled(bool) ), SLOT( slotHideButton(bool) ) );
}

KitaSubjectView::~KitaSubjectView()
{
}

void KitaSubjectView::reloadSubject()
{
}

void KitaSubjectView::loadThread(QListViewItem* item)
{
  QString datName = item->text(Row_DatName);
  KURL datURL = m_board.url();
  datURL.addPath("/dat/" + datName);
  emit signalShowThread(m_board, Kita2ch::Thread(datURL));
}

void KitaSubjectView::slotLoadBoard(const KURL& url, const QString& boardName)
{
  // reset member variables.
  {
    m_hitList.clear();
    m_nextHitIndex = 0;
    m_prevquery = "";
  }

  m_board = Kita2ch::Board(url, boardName);
  m_board.refreshChildList();

  KitaCacheInfo* cache = KitaCacheInfo::getInstance();  
  QPtrList<Kita2ch::Thread> threadList = m_board.getThreadList();
  
  // clear list
  subjectList->clear();
  
  for(int i=0; i<threadList.count(); i++) {
    Kita2ch::Thread* thread = threadList.at(i);
    
    int vested = cache->getVestedNum(thread->datURL());
    new KListViewItem(subjectList,
                      QString("%1").arg(i+1, 4),
                      "",
                      thread->name(),
                      QString("%1").arg(thread->resNum(), 4),
                      vested > 0 ? QString::number(vested) : "",
                      vested > 0 ? QString::number(thread->resNum() - vested) : "",
                      thread->datURL().fileName());
  }

  if( HideButton->isOn() ) {
    HideButton->toggle();
  }
}

void KitaSubjectView::slotSearchButton()
{
  insertSearchCombo();
  QStringList list = parseSearchQuery( SearchCombo->currentText() );
  searchNext( list );
}

void KitaSubjectView::insertSearchCombo()
{
  int count;
  bool found = false;

  for( count = 0; count < SearchCombo->count(); ++count ) {
    if ( SearchCombo->text( count ) ==  SearchCombo->currentText() ) {
      found = true;
      break;
    }
  }
  if ( ! found ) SearchCombo->insertItem( SearchCombo->currentText() );
}

QStringList KitaSubjectView::parseSearchQuery(const QString &input)
{
  QStringList tmp = QStringList::split( ' ', input );
  QStringList ret_list;
  QRegExp truncSpace("\\s*$");
  QStringList::iterator it = tmp.begin();
  for( ; it != tmp.end(); ++it )
    ret_list += (*it).replace( truncSpace, "" );
  return ret_list;
}

void KitaSubjectView::searchNext(const QStringList &query)
{
  if ( query.isEmpty() ) return;

  if ( query != m_prevquery ) {
    searchAll( query );
    slotHideButton( HideButton->isOn() );
    m_nextHitIndex = 0; //A next jump-search target reset to '0'.
    return;
  }

  if(m_nextHitIndex >= m_hitList.size()) return;

  KListViewItem* item = m_hitList[m_nextHitIndex];
  subjectList->ensureItemVisible(item);
  subjectList->setSelected(item, true);
  m_nextHitIndex++;
  if(m_nextHitIndex >= m_hitList.size()) m_nextHitIndex = 0;
}

void KitaSubjectView::searchAll(const QStringList &query)
{
  m_hitList.clear();
  m_prevquery = query;

  QListViewItemIterator listIt( subjectList );
  while( listIt.current() != 0 ) {
    KListViewItem* item = static_cast<KListViewItem*>(listIt.current());
    item->setPixmap(Row_Icon, 0);

    QStringList::const_iterator queryIt = query.begin();
    for( ; queryIt != query.end(); ++queryIt ) {
      if( item->text(Row_Subject).contains(*queryIt) ) {
        item->setPixmap(Row_Icon, SmallIcon("find") );
        m_hitList.append(item);
        break;
      }
    }
    ++listIt;
  }
}

void KitaSubjectView::slotHideButton(bool on)
{
  if(m_hitList.empty()) return;

  QListViewItemIterator listIt( subjectList );
  while( listIt.current() != 0 ) {
    KListViewItem* item = static_cast<KListViewItem*>(listIt.current());
    if( on && ! item->pixmap(Row_Icon)) {
      item->setVisible(false);
    } else {
      item->setVisible(true);
    }
    ++listIt;
  }
}

void KitaSubjectView::slotFontChanged(QFont& font)
{
}
