/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "boardmanager.h"
#include "qcp932codec.h"
#include "downloadmanager.h"
#include "cache.h"
#include "favoritethreads.h"
#include "thread.h"
#include "threadinfo.h"
#include "favoriteboards.h"
#include "kita_misc.h"
#include "threadindex.h"

#include <config.h>  /* VERSION */

#include <kdebug.h>
#include <kio/netaccess.h>
#include <kfilterdev.h>
#include <kio/slaveconfig.h>
#include <kdeversion.h>

#include <qfile.h>
#include <qdir.h>
#include <qregexp.h>
#include <qdatetime.h>
#include <qeucjpcodec.h>

using namespace Kita;


/*---------------------------------------------------------*/

/* BoardData */

BoardData::BoardData( const QString& boardName,
                      const QString& hostname,
                      const QString& rootPath,
                      const QString& delimiter,
                      const QString& bbsPath,
                      const QString& ext,
                      int boardtype )
{
    m_readIdx = FALSE;
    m_boardName = boardName;
    m_rootPath = rootPath;
    m_delimiter = delimiter;
    m_bbsPath = bbsPath;
    m_ext = ext;
    m_type = boardtype;

    /* set hostname and create URL of board */
    setHostName( hostname );

    /* create default key */
    QStringList keyHosts = m_hostname;
    createKeys( keyHosts );

    /* reset SETTING.TXT */
    setSettingLoaded( FALSE );
}

BoardData::~BoardData()
{}


/* public */
void BoardData::setHostName( const QString& hostName )
{
    m_hostname = hostName;

    /* m_basePath = (hostname)/(rootPath)/(bbsPath)/ */
    m_basePath = m_hostname + m_rootPath + m_bbsPath + "/";

    switch ( m_type ) {

    case Board_MachiBBS:  /* m_cgiBasePath = (hostname)/(rootPath)/(delimiter)?BBS=(bbsPath) */
        m_cgiBasePath = m_hostname + m_rootPath + m_delimiter + "?BBS=" + m_bbsPath.mid( 1 );
        break;

        /* m_cgiBasePath = (hostname)/(rootPath)/(delimiter)/(bbsPath)/ */
    default:
        m_cgiBasePath = m_hostname + m_rootPath + m_delimiter + m_bbsPath + "/";
        break;
    }
}


/*---------------------------*/
/* information */

/* public */
const bool BoardData::readIdx() const
{
    return m_readIdx;
}

/* public */
void BoardData::setReadIdx( bool idx )
{
    m_readIdx = idx;
}

/* public */
const QString& BoardData::boardName() const
{
    return m_boardName;
}

/* public */
const QString& BoardData::hostName() const
{
    return m_hostname;
}

/* public */
const QString& BoardData::rootPath() const
{
    return m_rootPath;
}

/* public */
const QString& BoardData::delimiter() const
{
    return m_delimiter;
}

/* public */
const QString& BoardData::bbsPath() const
{
    return m_bbsPath;
}

/* public */
const QString& BoardData::ext() const
{
    return m_ext;
}

/* public */
const int BoardData::type() const
{
    return m_type;
}

/* public */
const QString& BoardData::basePath() const
{
    return m_basePath;
}

/* public */
const QString& BoardData::cgiBasePath() const
{
    return m_cgiBasePath;
}


/*---------------------------*/
/* SETTING.TXT */

/* public */
const QString BoardData::settingURL() const
{
    return m_basePath + "SETTING.TXT";
}

/* public */
const bool BoardData::settingLoaded() const
{
    return m_settingLoaded;
}

/* public */
const QString& BoardData::defaultName() const
{
    return m_defaultName;
}

/* public */
const int BoardData::lineNum() const
{
    return m_linenum;
}

/* public */
const int BoardData::msgCount() const
{
    return m_msgCount;
}

/* public */
const KURL& BoardData::titleImgURL() const
{
    return m_titleImgURL;
}

/* public */
void BoardData::setSettingLoaded( bool set )
{
    m_settingLoaded = set;
    if ( ! set ) {
            m_defaultName = QString::null;
            m_linenum = 0;
            m_msgCount = 0;
            m_titleImgURL = QString::null;
        }
}

/* public */
void BoardData::setDefaultName( const QString& newName )
{
    m_defaultName = newName;
}

/* public */
void BoardData::setLineNum( int newLine )
{
    m_linenum = newLine;
}

/* public */
void BoardData::setMsgCount( int msgCount )
{
    m_msgCount = msgCount;
}

/* public */
void BoardData::setTitleImgURL( const KURL& url )
{
    m_titleImgURL = url;
}


/*---------------------------*/
/* keys */

/* create keys of DB */ /* public */
void BoardData::createKeys( const QStringList& keyHostList )
{
    /* reset keys */
    m_keyBasePathList.clear();
    m_keyCgiBasePathList.clear();
    m_keyHostList.clear();

    m_keyHostList = keyHostList;

    /* m_basePath = (hostname)/(rootPath)/(bbsPath)/ */
    for ( unsigned int i = 0; i < m_keyHostList.count(); ++i ) {
        if ( m_keyHostList[ i ].length() > 0 )
            m_keyBasePathList += m_keyHostList[ i ] + m_rootPath + m_bbsPath + "/";
    }

    switch ( m_type ) {

    case Board_MachiBBS:  /* m_cgiBasePath = (hostname)/(rootPath)/(delimiter)?BBS=(bbsPath) */
        for ( unsigned int i = 0; i < m_keyHostList.count(); ++i )
            m_keyCgiBasePathList += m_keyHostList[ i ] + m_rootPath + m_delimiter
                                    + "?BBS=" + m_bbsPath.mid( 1 );
        break;

        /* m_cgiBasePath = (hostname)/(rootPath)/(delimiter)/(bbsPath)/ */
    default:
        for ( unsigned int i = 0; i < m_keyHostList.count(); ++i )
            m_keyCgiBasePathList += m_keyHostList[ i ] + m_rootPath + m_delimiter + m_bbsPath + "/";
        break;
    }
}

/* public */
const QStringList& BoardData::keyHostList() const
{
    return m_keyHostList;
}

/* public */
const QStringList& BoardData::keyBasePathList() const
{
    return m_keyBasePathList;
}

/* public */
const QStringList& BoardData::keyCgiBasePathList() const
{
    return m_keyCgiBasePathList;
}




/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/

/* BoardManager */

QCp932Codec* Kita::BoardManager::m_cp932Codec = NULL;
QEucJpCodec* Kita::BoardManager::m_eucJpCodec = NULL;
BoardDataList Kita::BoardManager::m_boardDataList;
BoardData* Kita::BoardManager::m_previousBoardData = NULL;
QString Kita::BoardManager::m_previousBoardURL;


BoardManager::BoardManager()
{
    clearBoardData();
}


BoardManager::~BoardManager()
{
    clearBoardData();
}

/* (hostname)/(rootPath)/(bbsPath)/ */ /* public */ /* static */
const QString BoardManager::boardURL( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return QString::null;

    return bdata->basePath();
}

/* public */ /* static */
const QStringList BoardManager::allBoardURLList()
{
    QStringList urlList;
    urlList.clear();

    for ( BoardDataList::Iterator it = m_boardDataList.begin(); it != m_boardDataList.end(); ++it )
        urlList += ( *it ) ->basePath();

    return urlList;
}

/* (hostname)/(rootPath) */ /* public */ /* static */
const QString BoardManager::boardRoot( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return QString::null;

    return bdata->hostName() + bdata->rootPath();
}

/* (bbspath) */ /* public */ /* static */
const QString BoardManager::boardPath( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return QString::null;

    return bdata->bbsPath();
}

/* (ext) */ /* public */ /* static */
const QString BoardManager::ext( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return QString::null;

    return bdata->ext();
}

/* ID of board for writing */ /* public */ /* static */
const QString BoardManager::boardID( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return QString::null;

    return bdata->bbsPath().mid( 1 ); /* remove "/" */
}


/* (hostname)/(rootPath)/(bbsPath)/subject.txt */ /* public */ /* static */
const QString BoardManager::subjectURL( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return QString::null;

    return bdata->basePath() + "subject.txt";
}


/* public */ /* static */
const QString BoardManager::boardName( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return QString::null;

    return bdata->boardName();
}


/* public */ /* static */
const int BoardManager::type( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return Board_Unknown;

    return bdata->type();
}


/*---------------------------*/
/* ThreadList */


/*  get list of pointers of Thread classes.     */
/*
  Input:
 
  url:  URL of board.
  oldLogs: If TRUE, search cache and get list of pointer of old threads.
  online: online or offline mode.
 
  Output:
 
  threadList: list of pointers of Thread classes.
  oldLogList: list of pointers of old threads.
 
                                                 */ /* public */ /* static */
void BoardManager::getThreadList(

    /* input */
    const KURL& url,
    bool oldLogs,
    bool online,

    /* output */
    QPtrList< Thread >& threadList,
    QPtrList< Thread >& oldLogList )
{
    threadList.clear();
    oldLogList.clear();

    /* get all obtained threads list from cache */
    if ( url.prettyURL() == "http://virtual/obtained/" ) {

        QStringList bbslist = allBoardURLList();

        /* search all cache dirs */
        for ( QStringList::iterator it = bbslist.begin() ; it != bbslist.end(); ++it ) {

            getCachedThreadList( ( *it ), threadList );
        }

        return ;
    }

    /*-------------------------*/

    BoardData* bdata = getBoardData( url );
    if ( bdata == NULL ) return ;

    /* download subject.txt */
    if ( online ) {

        /* make directory */
        QString cacheDir = Cache::baseDir() + Cache::serverDir( url ) + Cache::boardDir( url );
        if ( !Kita::mkdir( cacheDir ) ) return ;

        KIO::SlaveConfig::self() ->setConfigData( "http",
                url.host() ,
                "UserAgent",
                QString( "Monazilla/1.00 (Kita/%1)" ).arg( VERSION ) );
        QString subjectPath = Cache::getSubjectPath( url );
#if KDE_IS_VERSION( 3, 2, 0 )
        KIO::NetAccess::download( subjectURL( url ), subjectPath, NULL );
#else
        KIO::NetAccess::download( subjectURL( url ), subjectPath );
#endif

        /* download SETTING.TXT */
        loadBBSSetting( url );
    }

    /* open and read subject.txt */
    readSubjectTxt( bdata, url, threadList );

    /* get old logs */
    if ( oldLogs ) {

        QPtrList< Thread > tmpList;
        tmpList.clear();
        getCachedThreadList( url, tmpList );

        for ( unsigned i = 0; i < tmpList.count(); i++ ) {

            if ( threadList.contains( tmpList.at( i ) ) == 0 ) oldLogList.append( tmpList.at( i ) );
        }
    }
}


/* read the cache dir & get the list of all threads. */ /* private */ /* static */
void BoardManager::getCachedThreadList( const KURL& url, QPtrList< Thread >& threadList )
{
    QString cacheDir = Cache::baseDir() + Cache::serverDir( url ) + Cache::boardDir( url );
    QDir d( cacheDir );
    if ( d.exists() ) {

        /* get all file names */
        QString ext = BoardManager::getBoardData( url ) ->ext();
        QString boardURL = BoardManager::getBoardData( url ) ->basePath();
        QStringList flist = d.entryList( "*" + ext );

        for ( QStringList::iterator it = flist.begin(); it != flist.end(); ++it ) {
            if ( ( *it ) == QString::null ) continue;

            QString datURL = boardURL + "dat/" + ( *it );

            /* read idx file */
            Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
            if ( thread == NULL ) {

                thread = Kita::Thread::getByURL( datURL );
                if ( thread == NULL ) continue;
                ThreadIndex::loadIndex( thread, datURL, FALSE );
            }

            if ( thread != NULL ) threadList.append( thread );
        }
    }
}



/* open subject.txt and get list of Thread classes */ /* private */ /* static */
bool BoardManager::readSubjectTxt( BoardData* bdata, const KURL& url, QPtrList< Thread >& threadList )
{
    /* get all names of cached files to read idx.  */
    QStringList cacheList;
    if ( !bdata->readIdx() ) {

        QString cacheDir = Cache::baseDir() + Cache::serverDir( url ) + Cache::boardDir( url );
        QDir d( cacheDir );
        if ( d.exists() ) {
            QString ext = BoardManager::getBoardData( url ) ->ext();
            cacheList = d.entryList( "*" + ext );
        }
    }

    /* open subject.txt */
    QString subjectPath = Cache::getSubjectPath( url );
    QIODevice * device = KFilterDev::deviceForFile( subjectPath, "application/x-gzip" );
    if ( !device->open( IO_ReadOnly ) ) return FALSE;

    QTextStream stream( device );

    if ( BoardManager::type( url ) == Board_JBBS ) {
        if ( !m_eucJpCodec ) m_eucJpCodec = new QEucJpCodec();
        stream.setCodec( m_eucJpCodec );
    } else {
        if ( !m_cp932Codec ) m_cp932Codec = new QCp932Codec();
        stream.setCodec( m_cp932Codec );
    }

    // parse subject.txt(only one format...)
    // FIXME: need to refactoring
    QRegExp regexp;
    switch ( BoardManager::type( url ) ) {

    case Board_MachiBBS:
    case Board_JBBS:
        regexp.setPattern( "(\\d+\\.cgi),(.*)\\((\\d+)\\)" );
        break;

    default:
        regexp.setPattern( "(\\d+\\.dat)<>(.*)\\((\\d+)\\)" );
        break;
    }
    QString line;

    while ( ( line = stream.readLine() ) != QString::null ) {
        int pos = regexp.search( line );
        if ( pos != -1 ) {
            QString fname = regexp.cap( 1 );
            QString subject = regexp.cap( 2 );
            QString num = regexp.cap( 3 );

            /* get pointer of Thread class */
            QString datURL = boardURL( url ) + "dat/" + fname;
            Kita::Thread* thread = Kita::Thread::getByURL( datURL );
            if ( threadList.find( thread ) == -1 ) {
                threadList.append( thread );
            }

            /* set thread name */
            thread->setThreadName( subject );

            /* load index file */
            if ( !bdata->readIdx() ) {

                if ( cacheList.contains( fname ) ) ThreadIndex::loadIndex( thread, datURL, FALSE );
            }

            /* update res num */
            int newNum = num.toInt();
            if ( thread->readNum() ) { /* cache exists */
                int oldNum = thread->resNum();

                if ( newNum > oldNum ) {
                    Kita::ThreadIndex::setResNum( datURL, newNum );
                }
            }
            thread->setResNum( newNum );
        }
    }

    device->close();
    bdata->setReadIdx( TRUE ); /* never read idx files again */

    return TRUE;
}

/*---------------------------*/
/* BoardData */

/* reset all BoardData */ /* public */ /* static */
void BoardManager::clearBoardData()
{
    for ( BoardDataList::Iterator it = m_boardDataList.begin(); it != m_boardDataList.end(); ++it )
        delete( *it );

    m_boardDataList.clear();
    m_previousBoardData = NULL;
    m_previousBoardURL = QString::null;
}

/* enroll board        */

/* Input: board, boardName, type, test
   Output: oldURL
 
   return: 
   If board is already enrolled, return Board_enrollEnrolled and oldURL is QString::null.
   If board is new board, return Board_enrollNew and oldURL is QString::null.
   If board is moved, return Board_enrollMoved, and oldURL is old URL.
 
   Note that board is NOT enrolled when board is moved.
   To enroll new URL, call BoardManager::moveBoard(). 
 
   "int type" is type of board. It could be "Kita::Board_Unknown". See also parseBoardURL().
   
   If "bool test" is TRUE, this function just checks if the board is enrolled (never enroll board).
 
*/ 
/* public */ /* static */
int BoardManager::enrollBoard( const KURL& url, const QString& boardName, QString& oldURL, int type, bool test )
{
    QString hostname;
    QString rootPath;
    QString delimiter;
    QString bbsPath;
    QString ext;
    type = parseBoardURL( url, type, hostname, rootPath, delimiter, bbsPath, ext );
    oldURL = QString::null;

    if ( type == Board_Unknown ) return Board_enrollFailed;

    /* check if the board is enrolled or moved. */
    for ( BoardDataList::Iterator it = m_boardDataList.begin(); it != m_boardDataList.end(); ++it ) {

        if ( ( *it ) ->boardName() == boardName
                && ( *it ) ->type() == type
                && ( *it ) ->bbsPath() == bbsPath ) {

            if ( ( *it ) ->hostName() == hostname
                    && ( *it ) ->rootPath() == rootPath ) { /* enrolled */
                return Board_enrollEnrolled;
            } else { /* moved */
                oldURL = ( *it ) ->basePath();
                return Board_enrollMoved;
            }
        }
    }

    /* test only */
    if ( test ) return Board_enrollNew;

    /* enroll new board */
    BoardData* bdata = new BoardData( boardName, hostname, rootPath, delimiter, bbsPath, ext, type );
    m_boardDataList.append( bdata );

    return Board_enrollNew;
}


/* parse board URL      */
/* return board type.   */ /* private */ /* static */
int BoardManager::parseBoardURL(

    /* input */
    const KURL& url,
    int type,   /* If type = Board_Unknown, type will be decided according to url. */

    /* output */
    QString& hostname,
    QString& rootPath,
    QString& delimiter,
    QString& bbsPath,
    QString& ext )
{
    hostname = url.protocol() + "://" + url.host();
    rootPath = QString::null;
    delimiter = QString::null;
    bbsPath = QString::null;
    ext = QString::null;

    /* decide type */
    if ( type == Board_Unknown ) {

        if ( url.host().contains( "machi.to" ) ) type = Board_MachiBBS;
        else if ( url.host().contains( "jbbs.livedoor.jp" ) ) type = Board_JBBS;
        else type = Board_2ch;
    }

    /* parse */
    switch ( type ) {

    case Board_MachiBBS:     /* MACHI : http:// *.machi.to/(bbsPath)/ */

        delimiter = "/bbs/read.pl";
        bbsPath = url.filename();
        ext = ".cgi";
        break;

    case Board_JBBS:   /* JBBS : http://jbbs.livedoor.jp/(bbsPath)/ */

        delimiter = "/bbs/read.cgi";
        bbsPath = url.prettyURL().remove( hostname );
        type = Board_JBBS;
        ext = ".cgi";
        break;

    case Board_FlashCGI:  /* test for Flash CGI/Mini Thread  */

        delimiter = "/test/read.cgi";
        bbsPath = url.filename();
        rootPath = url.prettyURL().remove( hostname + "/" ).remove( bbsPath + "/" );
        if ( rootPath.length() == 0 ) rootPath = QString::null;
        ext = ".dat";
        break;

    default:   /* 2ch : http://(hostname)/(rootPath)/(bbsPath)/ */

        delimiter = "/test/read.cgi";
        bbsPath = url.filename();
        rootPath = url.prettyURL().remove( hostname + "/" ).remove( bbsPath + "/" );
        if ( rootPath.length() == 0 ) rootPath = QString::null;
        ext = ".dat";
        type = Board_2ch;
        break;
    }

    /* For example, if bbsPath = "linux/", then m_bbsPath = "/linux" */
    const QRegExp exp( "/$" );
    rootPath.remove( exp );
    bbsPath.remove( exp );
    if ( rootPath != QString::null && rootPath.at( 0 ) != '/' ) rootPath = "/" + rootPath;
    if ( bbsPath != QString::null && bbsPath.at( 0 ) != '/' ) bbsPath = "/" + bbsPath;

    return type;
}


/* public */ /* static */
bool BoardManager::isEnrolled( const KURL& url )
{
    if ( getBoardData( url ) == NULL ) return FALSE;
    return TRUE;
}


/* public */ /* static */
BoardData* BoardManager::getBoardData( const KURL& url )
{
    if ( url.isEmpty() ) return NULL;
    QString urlstr = url.prettyURL();

    /* cache */
    if ( m_previousBoardData != NULL && m_previousBoardURL == urlstr ) return m_previousBoardData;

    for ( BoardDataList::Iterator it = m_boardDataList.begin(); it != m_boardDataList.end(); ++it ) {

        int count = ( *it ) ->keyBasePathList().count();
        for ( int i = 0; i < count ; ++i ) {
            if ( urlstr.contains( ( *it ) ->keyBasePathList() [ i ] )
                    || urlstr.contains( ( *it ) ->keyCgiBasePathList() [ i ] ) ) {

                /* cache */
                m_previousBoardData = ( *it );
                m_previousBoardURL = urlstr;

                return ( *it );
            }
        }
    }

    return NULL;
}



/*--------------------------------*/
/* BBSHISTORY */


/* load the bbs history file ( BBSHISTORY ), and create keys of Data Base.  */
/* Before calling this, enroll the board by enrollBoard().                  */
/*
    ex) If the host of board moved like :
 
    http:://aaa.com -> http://bbb.com -> http://ccc.com -> http://ddd.com
    
    then, BBSHISTORY is
 
    http://ccc.com
    http://bbb.com
    http://aaa.com
 
*/ /* public */ /* static */
bool BoardManager::loadBBSHistory( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return FALSE;

    QStringList keyHosts = bdata->hostName();

    QFile file( Cache::getBBSHistoryPath( url ) );
    if ( file.open( IO_ReadOnly ) ) {

        QTextStream ts( &file );

        QString line;
        while ( !ts.eof() ) {

            line = ts.readLine();
            keyHosts += line;
        }

        bdata->createKeys( keyHosts );
        file.close();

        return TRUE;
    }

    return FALSE;
}


/* public */ /* static */
bool BoardManager::moveBoard( const KURL& fromURL, const KURL& toURL )
{
    QString oldhost = fromURL.protocol() + "://" + fromURL.host();
    QString newhost = toURL.protocol() + "://" + toURL.host();

    const QRegExp exp( "/$" );
    QString oldURL = fromURL.prettyURL();
    QString newURL = toURL.prettyURL();
    oldURL.remove( exp );
    newURL.remove( exp );
    oldURL += "/";
    newURL += "/";

    if ( oldURL == newURL ) return FALSE;

    /* Is oldURL enrolled? */
    BoardData* bdata = getBoardData( oldURL );
    if ( bdata == NULL ) {

        /* Is newURL enrolled? */
        bdata = getBoardData( newURL );
        if ( bdata == NULL ) return FALSE;
    }


    /*---------------------------*/
    /* update BoardData */

    /* get the path of old cache */
    bdata->setHostName( oldhost );
    QStringList keyHosts = bdata->keyHostList();
    keyHosts.remove( oldhost );
    keyHosts.prepend( oldhost );
    bdata->createKeys( keyHosts );
    QString oldCachePath = Cache::baseDir() + Cache::serverDir( bdata->basePath() )
                           + Cache::boardDir( bdata->basePath() );

    /* update URL */
    bdata->setHostName( newhost );

    /* update keys */
    /* The order of keyHosts will be like this:
       
      newhost      
      oldhost      
      foohost1
      foohost2
      
    */
    keyHosts = bdata->keyHostList();
    keyHosts.remove( oldhost );
    keyHosts.prepend( oldhost );
    keyHosts.remove( newhost );
    keyHosts.prepend( newhost );
    bdata->createKeys( keyHosts );

    /* reset BoardData */
    bdata->setReadIdx( FALSE );
    bdata->setSettingLoaded( FALSE );


    /*---------------------------*/
    /* move cache dir */

    QDir qdir;
    if ( ! qdir.exists( oldCachePath ) ) return TRUE;

    /* mkdir new server dir */
    QString newCachePath = Cache::baseDir() + Cache::serverDir( bdata->basePath() );
    Kita::mkdir( newCachePath );

    /* backup old dir */
    newCachePath += Cache::boardDir( bdata->basePath() );
    if ( qdir.exists ( newCachePath ) ) {
        QString bkupPath = newCachePath;
        bkupPath.truncate( bkupPath.length() - 1 ); /* remove '/' */
        bkupPath += "." + QString().setNum( QDateTime::currentDateTime().toTime_t() );
        qdir.rename( newCachePath, bkupPath );
    }

    /* move cache dir */
    if ( qdir.exists( oldCachePath ) ) {
        qdir.rename( oldCachePath, newCachePath );
    } else Kita::mkdir( newCachePath );

    /* make old dir */
    if ( ! qdir.exists( oldCachePath ) ) {
        Kita::mkdir( oldCachePath );
        /* create BBS_MOVED */
        QString movedPath = oldCachePath + "/BBS_MOVED";
        QFile file( movedPath );
        if ( file.open( IO_WriteOnly ) ) {
            QTextStream stream( &file );
            stream << newURL << endl;
        }
        file.close();
    }

    /*---------------------------*/
    /* update BBSHISTRY */

    QFile file( Cache::getBBSHistoryPath( bdata->basePath() ) );
    if ( file.open( IO_WriteOnly ) ) {

        QTextStream ts( &file );

        keyHosts.remove( newhost );
        for ( QStringList::iterator it = keyHosts.begin() ; it != keyHosts.end(); ++it ) {
            ts << ( *it ) << endl;
        }

        file.close();
    }


    /*---------------------------*/
    /* update other information */
    FavoriteThreads::replace( oldURL, newURL );
    Kita::Thread::replace( oldURL, newURL );
    KitaThreadInfo::replace( oldURL, newURL );
    Kita::FavoriteBoards::replace( oldURL, newURL );

    return TRUE;
}


/*--------------------------------*/
/* SETTING.TXT  */


/* public */ /* static */
bool BoardManager::loadBBSSetting( const KURL& url, bool reload )
{
    /* Is board enrolled ? */
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return FALSE;
    if ( bdata->type() != Board_2ch ) return FALSE;

    KURL settingURL = bdata->settingURL();
    QString path = Cache::getSettingPath( url );

    /* now loading */
    if ( DownloadManager::isLoadingNow( settingURL ) ) return FALSE;

    /* already loaded */
    if ( bdata->settingLoaded() && !reload ) return TRUE;

    /* reset names, linenum, etc. */
    bdata->setSettingLoaded( FALSE );

    /* download SETTING.TXT.*/
    DownloadManager::download( settingURL, path );
    bdata->setSettingLoaded( TRUE );

    return TRUE;
}


/* public */ /* static */
QString BoardManager::getBBSDefaultName( const KURL& url )
{
    BoardData * bdata = openSettingTxt( url );
    if ( bdata == NULL ) bdata = getBoardData( url );
    if ( bdata == NULL ) return "(default name)";
    if ( bdata->defaultName() == QString::null ) return "(default name)";
    return bdata->defaultName();
}


/* public */ /* static */
int BoardManager::getBBSMaxLine( const KURL& url )
{
    BoardData * bdata = openSettingTxt( url );
    if ( bdata == NULL ) bdata = getBoardData( url );
    if ( bdata == NULL ) return 0;
    return bdata->lineNum();
}

/* public */ /* static */
int BoardManager::getBBSMsgCount( const KURL& url )
{
    BoardData * bdata = openSettingTxt( url );
    if ( bdata == NULL ) bdata = getBoardData( url );
    if ( bdata == NULL ) return 0;
    return bdata->msgCount();
}


/* public */ /* static */
const KURL BoardManager::titleImgURL( const KURL& url )
{
    BoardData * bdata = openSettingTxt( url );
    if ( bdata == NULL ) bdata = getBoardData( url );
    if ( bdata == NULL ) return QString::null;
    return bdata->titleImgURL();
}

/* open local SETTING.TXT, then get names, linenum, etc. */ /* private */ /* static */
BoardData* BoardManager::openSettingTxt( const KURL& url )
{
    BoardData * bdata = getBoardData( url );
    if ( bdata == NULL ) return NULL;
    if ( !bdata->settingLoaded() ) return NULL;
    if ( DownloadManager::isLoadingNow( bdata->settingURL() ) ) return NULL;
    if ( bdata->defaultName() != QString::null ) return bdata;

    QFile file( Cache::getSettingPath( url ) );
    if ( file.open( IO_ReadOnly ) ) {

        QTextStream ts( &file );
        if ( m_cp932Codec == NULL ) m_cp932Codec = new QCp932Codec();
        ts.setCodec( m_cp932Codec );

        QString line;
        while ( !ts.eof() ) {

            line = ts.readLine();

            /* default name */
            QString key = "BBS_NONAME_NAME=";
            if ( line.find( key ) != -1 ) bdata->setDefaultName( line.remove( key ) );

            /* 0ch type */
            key = "NANASI_NAME=";
            if ( line.find( key ) != -1 ) bdata->setDefaultName( line.remove( key ) );

            /* line number */
            key = "BBS_LINE_NUMBER=";
            if ( line.find( key ) != -1 ) bdata->setLineNum( line.remove( key ).toInt() * 2 );

            /* msg count */
            key = "BBS_MESSAGE_COUNT=";
            if ( line.find( key ) != -1 ) bdata->setMsgCount( line.remove( key ).toInt() );

            /* title image */
            key = "BBS_TITLE_PICTURE=";
            if ( line.find( key ) != -1 ) {
                QString path = line.remove( key );
                KURL titleImgURL = KURL( bdata->basePath(), path );
                bdata->setTitleImgURL( titleImgURL );
            }
        }

        file.close();
    }

    if ( bdata->defaultName() == QString::null ) bdata->setDefaultName( "(default name)" );

    return bdata;
}
