/**************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto , (c)2004 by 421                 *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
**************************************************************************/

#include <qlayout.h>
#include <qregexp.h>
#include <kaction.h>
#include <kstdaccel.h>
#include <kpopupmenu.h>
#include <qmessagebox.h>
#include <klocale.h>

#include "libkita/datmanager.h"
#include "libkita/kita_misc.h"
#include "libkita/kita-utf8.h"
#include "libkita/parsemisc.h"
#include "libkita/signalcollection.h"

#include "kitanavi.h"
#include "kitathreadview.h"

#define MAX_TABLABEL_LEN 8

/*----------------------------------------*/
/* class KitaNavi                         */
/*----------------------------------------*/

KitaNavi* KitaNavi::m_kitanavi = NULL;

/* Don't call this directly. Call getKitaNavi instead. */
KitaNavi::KitaNavi(
    QWidget* parent,
    int navimode)
        : QFrame( parent, "kita_navif" , WType_TopLevel)
{
    /* default size of kita navi */
    const int kitanavi_wd = 600;
    const int kitanavi_ht = 400;

    resize(kitanavi_wd,kitanavi_ht);

    /* mode is defined in kitanavi.h */
    m_navimode = navimode;

    m_qtw = new KitaNaviTab(m_navimode,this, "kita_navitab", 0 );
    m_qtw->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
    QVBoxLayout *vl = new QVBoxLayout(this);
    vl->addWidget(m_qtw);

    /* emit when this widget is deactivated */
    connect( this, SIGNAL( windowDeactivated() ),
             Kita::SignalCollection::getInstance(), SIGNAL( windowDeactivated() ));

    /* If this widget is active and receives
       signal isKitaActive, then emit signal kitaIsActive. */
    connect( Kita::SignalCollection::getInstance(), SIGNAL( isKitaActive() ),
             this, SLOT( slotIsKitaActive() ) );

    connect( this, SIGNAL( kitaIsActive() ),
             Kita::SignalCollection::getInstance(), SIGNAL( kitaIsActive() ) );

    /* set keyboard accelerator */
    KActionCollection *kac = new KActionCollection(this);
    if(m_navimode == KITANAVI_NAVI) KStdAction::find( this, SLOT( slotEditFind() ), kac);
    new KAction(i18n( "Close Current Thread Tab" ),
                KStdAccel::close(),this,SLOT( slotCloseCurrentTab() ),kac,"close_tab" );
}


KitaNavi::~KitaNavi()
{
    if(m_qtw) delete m_qtw;
}



/*-------------------------*/
/* show thread on the tab  */
/*-------------------------*/  /* private */
void KitaNavi::addTab(int mode,
                      const KURL& datURL,
                      const QString& query,
                      int startnum,
                      int endnum)
{

    if(m_navimode != KITANAVI_NAVI) return;

    if(isMinimized()) showNormal();
    raise();
    setActiveWindow();

    const int margin = 5;
    int count = 0,totalNum = 0;
    QString headstr = QString::null;
    QString footstr = QString::null;
    QString anchor = QString::null;
    QString cmbstr = QString::null;
    QString str, urlstr;
    QStringList queries;
    QRegExp qrx(" +");
    bool ext_OR;
    QString labelstr;

    /* set title and label */
    switch(mode)
    {

    case KITANAVI_SHOWRES:

        if(startnum == endnum) labelstr = QString().setNum(startnum);
        else labelstr = QString( "%1-%2" ).arg(startnum).arg(endnum);

        break;

    case KITANAVI_EXTBYID:
    case KITANAVI_EXTBYWORD:
    case KITANAVI_EXTBYNAME:
        labelstr = query
                   + " :"+ Kita::ParseMisc::utf8ToUnicode(KITAUTF8_EXTRACT);
        break;

    case KITANAVI_SHOWRESTREE:
    case KITANAVI_SHOWREVRESTREE:
        labelstr = QString().setNum(startnum);
        break;

    }
    labelstr += " :"+QString( "[%1] %2" )
                .arg( Kita::DatManager::thread_boardName( datURL ) )
                .arg( Kita::DatManager::thread_name( datURL ) );

    /* create KitaThreadView and setup it */
    KitaThreadView *view = new KitaThreadView(m_qtw);
    str = Kita::unescape( labelstr.left( MAX_TABLABEL_LEN ) );
    m_qtw->addTab(view,str);
    m_qtw->setTabToolTip(view,labelstr);

    view->setup( datURL, VIEWMODE_KITANAVI );
    totalNum = Kita::DatManager::getReadNum( datURL );

    /*---------------------------------------------*/
    /* copy data from DatManager to KitaThreadView */

    switch(mode)
    {

    case KITANAVI_SHOWRES:

        for(int i = QMAX(1,startnum-margin); i <= QMIN(totalNum,endnum+margin); i++)
        {
            if(i == startnum) view->domAppendBelt("belt1");
            str = Kita::DatManager::getDat( datURL ,i);
            view->domAppendData(str,i,i);
            if(i == endnum) view->domAppendBelt("belt2");
        }
        if(startnum == endnum) headstr = QString("No. %1").arg(startnum);
        else headstr = QString("No. %1 - %2").arg(startnum).arg(endnum);
        footstr = headstr;

        anchor = QString().setNum(startnum-1);

        break;

        /*--------------------------------*/

    case KITANAVI_EXTBYID:

        for(int i = 1; i<= totalNum; i++)
        {
            if(Kita::DatManager::checkID( datURL, query, i ))
            {
                str = Kita::DatManager::getDat( datURL, i );
                view->domAppendData(str,i,i);
                count++;
            }
        }
        headstr = QString("ID:%1 : [%2]").arg(query).arg(count);
        footstr = headstr;
        anchor = "header";
        break;

        /*--------------------------------*/

    case KITANAVI_EXTBYNAME:

        for(int i = 1; i<= totalNum; i++)
        {
            if( Kita::DatManager::getPlainName( datURL, i ) == query )
            {
                str = Kita::DatManager::getDat( datURL, i );
                view->domAppendData( str, i, i );
                count++;
            }
        }
        headstr = Kita::ParseMisc::utf8ToUnicode(KITAUTF8_EXTRACT)
                  +QString(": %1 : [%2]").arg( query ).arg( count );
        footstr = headstr;
        anchor = "header";
        break;

        /*--------------------------------*/

    case KITANAVI_EXTBYWORD:

        if(query.at(0) == '|')
        { /* OR */
            ext_OR = TRUE;
            str = query.mid(1);
        }
        else
        { /* AND */
            ext_OR = FALSE;
            str = query;
        }
        str.replace(Kita::ParseMisc::utf8ToUnicode(KITAUTF8_ZENSPACE)," ");
        str.replace(qrx," "); /* pack space */
        queries  = QStringList::split(" ", str);

        for(int i = 1; i<= totalNum; i++)
        {
            if(Kita::DatManager::checkWord( datURL, queries, i, ext_OR ))
            {
                str = Kita::DatManager::getDat( datURL, i );
                view->domAppendData(str,i,i);
                count++;
            }
        }
        headstr = Kita::ParseMisc::utf8ToUnicode(KITAUTF8_EXTRACT)
                  +QString(": %1 : [%2]").arg(query).arg(count);
        footstr = headstr;
        anchor = "header";
        cmbstr = "?"+query;
        break;

        /*--------------------------------*/

    case KITANAVI_SHOWRESTREE:
    case KITANAVI_SHOWREVRESTREE:

        str = QString("[%1] %2 : No.%3 [%4]<br>")
              .arg( Kita::DatManager::thread_boardName( datURL ) )
              .arg( Kita::DatManager::thread_name( datURL ) )
              .arg(startnum ).arg( endnum );
        urlstr = Kita::DatManager::thread_url( datURL ) + QString().setNum( startnum );
        str += "<a href=\""+urlstr+"\">"+urlstr+"</a><br>";
        str += Kita::DatManager::getHtml( datURL, startnum, startnum );
        if( endnum )
        {
            str += "<div style=\"background-color: #CCCCCC; text-align: center\"><br></div>";
            str += query;
        }
        view->domSetInnerHTML(str);

        break;

    default:
        break;
    }

    m_qtw->showPage(view);
    view->domApplyChange(labelstr,headstr,footstr,anchor,cmbstr);

    raise();
    setActiveWindow();
}


void KitaNavi::slotChangeCaption( const QString& text )
{
    setCaption("KitaNavi: "+text);
}


/* for keyboard acceleration */
void KitaNavi::slotCloseCurrentTab()
{
    QWidget* view = m_qtw->currentPage();
    m_qtw->myRemovePage( view ,TRUE);
}

/* for keyboard acceleration */
void KitaNavi::slotEditFind()
{

    KitaThreadView* view = static_cast<KitaThreadView*>(m_qtw->currentPage());
    view->focusSearchCombo();
}

/* when window is deactivated, emit signal to popup */ /* protected */
void KitaNavi::windowActivationChange ( bool  )
{
    if(!isActiveWindow()) emit windowDeactivated();
}

/* private slot */
void KitaNavi::slotIsKitaActive()
{
    if(isActiveWindow()) emit kitaIsActive();
}


/*----------------------------------------------------------------------*/
/* interfaces */

/* static & public */

/* Don't forget to call this in KitaMainWindow::setupView */ /* static & public */
void KitaNavi::createKitaNavi(QWidget* parent, KitaThreadTabWidget* )
{
    if(m_kitanavi == NULL)
    {
        m_kitanavi = new KitaNavi(parent,KITANAVI_NAVI);
    }
}

KitaNavi* KitaNavi::getKitaNavi()
{
    if(m_kitanavi) m_kitanavi->show();
    return m_kitanavi;
}


/* show res */
void KitaNavi::showKitaNavi( const KURL& datURL, int startnum, int endnum)
{

    if( ! m_kitanavi ) return;
    m_kitanavi->show();

    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    m_kitanavi->addTab( KITANAVI_SHOWRES, parsedURL, QString::null, startnum, endnum );
}



/* extract by ID */
void KitaNavi::showKitaNaviByID(const KURL& datURL, QString strid)
{

    if( ! m_kitanavi ) return;
    m_kitanavi->show();

    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    m_kitanavi->addTab( KITANAVI_EXTBYID, parsedURL, strid, 0, 0 );
}



/* extract by keywords */
void KitaNavi::showKitaNaviByWord( const KURL& datURL, QString query)
{

    if( ! m_kitanavi ) return;
    m_kitanavi->show();

    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    m_kitanavi->addTab( KITANAVI_EXTBYWORD, parsedURL, query, 0, 0 );
}


/* extract by name */
void KitaNavi::showKitaNaviByName( const KURL& datURL, QString query)
{

    if( ! m_kitanavi ) return;
    m_kitanavi->show();

    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    m_kitanavi->addTab( KITANAVI_EXTBYNAME, parsedURL, query, 0, 0 );
}



/* res tree */
void KitaNavi::showKitaNaviResTree( const KURL& datURL, int resNum)
{

    if( ! m_kitanavi ) return;
    m_kitanavi->show();

    int num;
    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    QString htmlstr = Kita::DatManager::getTreeByRes( parsedURL, resNum, num);
    m_kitanavi->addTab( KITANAVI_SHOWRESTREE, parsedURL, htmlstr, resNum, num);
}


/* reverse res tree */
void KitaNavi::showKitaNaviRevResTree( const KURL& datURL, int resNum)
{

    if( ! m_kitanavi ) return;
    m_kitanavi->show();

    int num;
    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    QString htmlstr = Kita::DatManager::getTreeByResReverse( parsedURL, resNum, num );
    m_kitanavi->addTab( KITANAVI_SHOWREVRESTREE, parsedURL, htmlstr, resNum, num );
}


/*----------------------------------------------------------------------*/


/*----------------------------------------*/
/* class KitaNaviTabBar                   */
/*----------------------------------------*/

/* tabbar for kitanavi */

/* tab menu */
void KitaNaviTabBar::contextMenuEvent( QContextMenuEvent* e )
{
    KitaNaviTab* tw = static_cast<KitaNaviTab*>(parentWidget());
    QWidget* view = tw->currentPage();
    QWidget* curview = view;
    QString str;
    int max = tw->count();
    int curnum = tw->currentPageIndex();

    KPopupMenu popup( 0 );
    popup.insertItem(  i18n( "Close this tab" ) , 1 );
    popup.insertSeparator();

    popup.insertItem( i18n( "Close other tabs" ),  3 );
    popup.insertItem( i18n( "Close right tabs" ),  4 );
    popup.insertItem( i18n( "Close left tabs" ),  5 );
    popup.insertItem( i18n( "Close all tabs" ),  2 );

    switch ( popup.exec( e->globalPos() ) )
    {

    case 1: /* close */
        if(view != NULL) tw->myRemovePage( view ,TRUE);
        break;

    case 2: /* all */
        while(tw->count() > 0 && view != NULL) view = tw->myRemovePage( view ,TRUE);
        break;

    case 3: /* other */
        for(int i=0,i2 = 0;i < max;i++)
        {
            view = tw->page(i2);
            if(view != NULL)
            {
                if(view != curview) tw->myRemovePage( view ,TRUE);
                else i2++;
            }
        }
        break;

    case 4: /* right */
        for(int i=curnum+1 ; i<max;i++)
        {
            view = tw->page(curnum+1);
            if(view != NULL) tw->myRemovePage( view ,TRUE);
        }
        break;

    case 5: /* left */
        for(int i=0 ;i<curnum ;i++)
        {
            view = tw->page(0);
            if(view != NULL) tw->myRemovePage( view ,TRUE);
        }
        break;

    default:
        break;
    }
}

/*---------------------------------------------------------------------------*/

/*-------------------------------------*/
/* class KitaNaviTab                   */
/*-------------------------------------*/

/* tab widget for kita navi */

KitaNaviTab::KitaNaviTab(int mode,  QWidget* parent, const char* name, WFlags f)
        : QTabWidget( parent, name, f )
{
    setTabBar( new KitaNaviTabBar(mode, this ) );

    connect( this, SIGNAL( currentChanged ( QWidget * ) ),
             SLOT( slotCurrentChanged ( QWidget * ) ) );

    m_navimode = mode;
    m_prevview = NULL;
}


KitaNaviTab::~KitaNaviTab()
{

    if(m_prevview)
    {
        delete m_prevview;
        m_prevview = NULL;
    }

    QWidget* view = currentPage();
    while(count() > 0 && view != NULL) view = myRemovePage( view ,TRUE);
}


void KitaNaviTab::slotCurrentChanged( QWidget * w)
{
    static_cast<KitaThreadView *>( w ) ->showStatusBar( "" );
}


/*----------------------------------------*/
/* remove page                            */
/* remove w, then return new current page */
/*----------------------------------------*/
QWidget* KitaNaviTab::myRemovePage(QWidget* w, bool delCurView)
{

    QWidget* retw;
    if(m_prevview)
    {
        delete m_prevview;
        m_prevview = NULL;
    }

    switch(m_navimode)
    {

    case KITANAVI_NAVI:
        KitaThreadView* view;
        view = static_cast<KitaThreadView*>(w);
        removePage( view );

        /* Calling this function from KitaThreadView itself will
           causes the crush. So, backup the instance and delete it later. */
        if(delCurView) delete view;
        else m_prevview = view;

        break;
    }

    retw = currentPage();
    if(retw == NULL) parentWidget()->hide();

    return retw;
}

