/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitasubjecttabwidget.h"

#include <kpopupmenu.h>
#include <kdebug.h>
#include <klocale.h>
#include <krun.h>
#include <kstdaccel.h>
#include <kaction.h>

#include <qapplication.h>
#include <qclipboard.h>

#include "libkita/signalcollection.h"

KitaSubjectTabWidget::KitaSubjectTabWidget( QWidget* parent, const char* name, WFlags f )
        : KitaTabWidgetBase( parent, name, f )
{
    setXMLFile( "kitasubjecttabwidgetui.rc" );

    setTabBar( new SubjectTabBar( this ) );
    KitaSubjectView * view = new KitaSubjectView( this );
    addTab( view, "        " );
    m_latestView = view;
    m_viewList.append( view );

    m_favoriteList = new FavoriteListView( this, "favoriteTab" );
    addTab( m_favoriteList, i18n( "Favorite" ) );

    setupActions();
    connectSignals( m_favoriteList );
    connectSignals( view );
}

KitaSubjectTabWidget::~KitaSubjectTabWidget()
{}

void KitaSubjectTabWidget::loadBoard( const QString& boardURL, bool withNewTab )
{
    KitaSubjectView * view = findView( boardURL );
    QString boardName = Kita::Board::getName( boardURL );
    if ( view ) {
        m_latestView = view;
        setTabLabel( m_latestView, boardName );
    } else if ( withNewTab ) {
        KitaSubjectView * newView = new KitaSubjectView( this );
        connectSignals( newView );
        insertTab( newView, boardName, count() - 1 );
        m_latestView = newView;
        m_viewList.append( newView );
    } else {
        setTabLabel( m_latestView, boardName );
    }

    m_latestView->loadBoard( boardURL );
    setTabToolTip( m_latestView, boardName );

    showPage( m_latestView );
}

void KitaSubjectTabWidget::connectSignals( Kita::ThreadListView* view )
{
    connect( view, SIGNAL( showThreadRequested( const QString&, bool ) ),
             this, SIGNAL( showThreadRequested( const QString&, bool ) ) );
}

KitaSubjectView* KitaSubjectTabWidget::findView( const QString& boardURL )
{
    KitaSubjectView * view;
    for ( view = m_viewList.first(); view; view = m_viewList.next() ) {
        if ( view->boardURL() == boardURL ) {
            return view;
        }
        if ( view->boardURL().isEmpty() ) {
            return view;  // default view
        }
    }
    return 0;
}

void KitaSubjectTabWidget::slotCurrentChanged( QWidget* widget )
{
    KitaTabWidgetBase::slotCurrentChanged( widget );
}

/* protected */ /* virtual */
void KitaSubjectTabWidget::deleteWidget( QWidget* w )
{
    KitaSubjectView * view = static_cast<KitaSubjectView *>( w );
    if ( count() <= 2 ) {
        return ;
        // favoriteȤΤȤϲ⤷ʤ
    }
    m_viewList.remove( view );
    removePage( view );
    delete view;

    // FIXME: favoriteϰֺˤʤȤ
    setCurrentPage( 0 );
    m_latestView = static_cast<KitaSubjectView *>( currentPage() );
}


/* private */
KitaSubjectView* KitaSubjectTabWidget::isSubjectView( QWidget* w )
{
    KitaSubjectView * view = NULL;
    if ( w ) {
        if ( w->isA( "KitaSubjectView" ) ) view = static_cast< KitaSubjectView* >( w );
    }

    return view;
}

/*--------------------------------*/
/* KitaSubjectView actions */

/* private */
void KitaSubjectTabWidget::setupActions()
{
    new KAction( i18n( "Find" ),
                 "find",
                 KStdAccel::find(),
                 this,
                 SLOT( slotFocusSearchCombo() ),
                 actionCollection(),
                 "threadview_find" );

    new KAction( i18n( "Reload" ),
                 "reload",
                 KStdAccel::reload(),
                 this,
                 SLOT( slotReloadButton() ),
                 actionCollection(),
                 "subjectview_reload" );
}

/* public slot */
void KitaSubjectTabWidget::slotReloadButton()
{
    KitaSubjectView * view = isSubjectView( currentPage() );
    if ( view ) view->reloadSubject();
}

/* public slot */
void KitaSubjectTabWidget::slotFocusSearchCombo()
{
    KitaSubjectView * view = isSubjectView( currentPage() );
    if ( view ) view->slotFocusSearchCombo();
}





/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/



SubjectTabBar::SubjectTabBar( QWidget* parent, const char* name )
        : KitaTabBarBase( parent, name )
{}

SubjectTabBar::~SubjectTabBar()
{}


/* private */ /* virtual */
void SubjectTabBar::showPopupMenu( int idx, QPoint global )
{
    enum{
        MENU_CLOSE,
        MENU_CLOSEOTHER,
        MENU_CLOSELEFT,
        MENU_CLOSERIGHT,
        MENU_OPENBROWSER,
        MENU_COPYTITLE
    };

    KitaSubjectTabWidget* tabwidget = static_cast<KitaSubjectTabWidget*>( parentWidget() );
    KActionCollection * collection = tabwidget->actionCollection();

    if ( QString::compare( tabwidget->page( idx ) ->name(), "favoriteTab" ) == 0 ) return ;

    KPopupMenu* popup = new KPopupMenu( this );
    popup->clear();

    popup->insertItem( i18n( "Close this tab" ) , MENU_CLOSE );
    collection->action( "tab_prevtab" ) ->plug( popup );
    collection->action( "tab_nexttab" ) ->plug( popup );
    popup->insertSeparator();

    popup->insertItem( i18n( "Close Other Tabs" ) , MENU_CLOSEOTHER );
    popup->insertItem( i18n( "Close right tabs" ) , MENU_CLOSERIGHT );
    popup->insertItem( i18n( "Close left tabs" ) , MENU_CLOSELEFT );
    collection->action( "tab_closealltab" ) ->plug( popup );

    popup->insertSeparator();
    popup->insertItem( i18n( "Open with Web Browser" ), MENU_OPENBROWSER );
    popup->insertItem( i18n( "Copy title and URL" ), MENU_COPYTITLE );

    popup->insertSeparator();
    collection->action( "tab_configkeys" ) ->plug( popup );

    KitaSubjectView* subjectView = static_cast<KitaSubjectView *>( tabwidget->page( idx ) );
    QClipboard* clipboard = QApplication::clipboard();

    int ret = popup->exec( global );
    delete popup;

    switch ( ret ) {
    case MENU_CLOSE: tabwidget->slotCloseTab( idx ); break;
    case MENU_CLOSEOTHER: tabwidget->slotCloseOtherTab( idx ); break;
    case MENU_CLOSERIGHT: tabwidget->slotCloseRightTab( idx ); break;
    case MENU_CLOSELEFT: tabwidget->slotCloseLeftTab( idx ); break;
    case MENU_OPENBROWSER:
        KRun::runURL( subjectView->boardURL(), "text/html" );
        break;
    case MENU_COPYTITLE:
        QString cliptxt = subjectView->boardName() + "\n" + subjectView->boardURL();
        clipboard->setText( cliptxt , QClipboard::Clipboard );
        clipboard->setText( cliptxt , QClipboard::Selection );
        break;
    }
}



/*--------------------------------------------------------------*/
/*--------------------------------------------------------------*/
/*--------------------------------------------------------------*/


KitaSubjectDock::KitaSubjectDock( KDockManager* dockManager,
                                  const char* name,
                                  const QPixmap &pixmap,
                                  QWidget* parent,
                                  const QString& strCaption,
                                  const QString& strTabPageLabel,
                                  WFlags f )
    : KitaDockWidgetBase( dockManager, name, pixmap, parent, strCaption, strTabPageLabel, f ) {}


KitaSubjectDock::~KitaSubjectDock()
{
    if ( m_subjectTab ) delete m_subjectTab;
}


/* setup & create KitaSubjectDock */ /* public */
KitaSubjectTabWidget* KitaSubjectDock::setup()
{
    /* create KitaNaviTabWidget */
    m_subjectTab = new KitaSubjectTabWidget( this, "Subject" );
    setWidget( m_subjectTab );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( switchToSubject() ),
             SLOT( slotShowDock() ) );
    connect( signalCollection, SIGNAL( openBoardRequested( const QString&, bool ) ),
             SLOT( slotLoadBoard( const QString&, bool ) ) );

    return m_subjectTab;
}

/* public slot */
void KitaSubjectDock::slotLoadBoard( const QString& boardURL, bool withNewTab )
{
    slotShowDock();

    if ( m_subjectTab ) m_subjectTab->loadBoard( boardURL, withNewTab );
}
