/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kita.h"
#include "prefs/prefs.h"
#include "kitathreadview.h"
#include "kitasubjectview.h"
#include "kitaboardview.h"
#include "libkita/threadinfo.h"
#include "libkita/favoriteboards.h"
#include "kitaboardtabwidget.h"
#include "kitathreadtabwidget.h"
#include "kitasubjecttabwidget.h"
#include "kitaimgtabwidget.h"
#include "kitawritetabwidget.h"

#include "libkita/favoritethreads.h"
#include "libkita/kita_misc.h"
#include "libkita/kitaconfig.h"
#include "libkita/signalcollection.h"
#include "libkita/parsemisc.h"
#include "libkita/account.h"
#include "libkita/imgmanager.h"

#include <qdragobject.h>
#include <qtextcodec.h>
#include <qevent.h>
#include <qregexp.h>
#include <qmessagebox.h>

#include <klineedit.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kuserprofile.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <ksimpleconfig.h>
#include <kedittoolbar.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kshortcut.h>
#include <kaboutdata.h>
#include <kparts/partmanager.h>
#include <kdeversion.h>

KitaMainWindow::KitaMainWindow()
        : KParts::DockMainWindow( 0, "Kita" ),
        m_printer( 0 )
{
    // accept dnd
    setAcceptDrops( true );

    // setup imgmanager
    Kita::ImgManager::setup( this );

    // setup view, dock
    setupView();

    // then, setup our actions
    setupActions();

    // load cache
    loadCache();

    // load favorites
    loadFavorites();

    // load favorite boards;
    loadFavoriteBoards();

    // load completion
    loadCompletion();

    // load style sheet
    loadStyleSheet();

    // load ascii art
    loadAsciiArt();

    // load abone lists
    loadAboneIDList();
    loadAboneNameList();
    loadAboneWordList();

    // and a status bar
    statusBar() ->show();

    // read dock status
    m_boardDock->loadSession();
    m_subjectDock->loadSession();
    m_threadDock->loadSession();
    m_naviDock->loadSession();
    m_imgDock->loadSession();
    m_writeDock->loadSession();
    QString dockConfigPath = locateLocal( "appdata", "dock.conf" );
    KConfig dockConfig( dockConfigPath );
    readDockConfig( &dockConfig );

    // show subject dock
    m_subjectDock->slotShowDock();

    // show thread view
    m_threadDock->slotShowDock();

    // hide writedock
    makeDockInvisible( m_writeDock );
    m_showImgViewAction->setChecked( FALSE );

    // hide kitanavi
    makeDockInvisible( m_naviDock );
    m_showNaviAction->setChecked( FALSE );

    // hide imageviewer
    makeDockInvisible( m_imgDock );
    m_showImgViewAction->setChecked( FALSE );


    KitaConfig::readConfig( KGlobal::config() );

    // apply the saved mainwindow settings, if any, and ask the mainwindow
    // to automatically save settings if changed: window size, toolbar
    // position, icon size, etc.
    setAutoSaveSettings();

    // font
    setFont( KitaConfig::listFont() );

    // allow the view to change the statusbar and caption
    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();

    connect( signalCollection, SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ),
             SLOT( slotOpenURLRequest( const KURL&, const KParts::URLArgs& ) ) );

    connect( signalCollection, SIGNAL( openURLRequestExt(
                                           const KURL&, const KParts::URLArgs&, QString, int, int,
                                           const KURL&, const KURL&, const QString&, const QString& ) ),
             SLOT( slotOpenURLRequestExt(
                       const KURL& , const KParts::URLArgs&, QString, int, int,
                       const KURL&, const KURL&, const QString&, const QString& ) ) );

    connect( signalCollection, SIGNAL( setMainCaption( const QString& ) ),
             SLOT( slotSetMainCaption( const QString& ) ) );

    connect( signalCollection, SIGNAL( setMainStatusbar( const QString& ) ),
             SLOT( slotSetMainStatusbar( const QString& ) ) );

    connect( signalCollection, SIGNAL( setMainURLLine( const KURL& ) ),
             SLOT( setUrl( const KURL& ) ) );

    connect( signalCollection, SIGNAL( bookmarked( const QString&, bool ) ),
             SLOT( bookmark( const QString&, bool ) ) );

    connect( this, SIGNAL( favoritesUpdated() ),
             signalCollection, SIGNAL( favoritesUpdated() ) );

    connect( m_urlLine, SIGNAL( returnPressed() ),
             SLOT( slotURLLine() ) );

    connect( m_subjectTab, SIGNAL( showThreadRequested( const QString&, bool ) ),
             m_threadDock, SLOT( slotShowThread( const QString&, bool ) ) );

    QPopupMenu* settingsPopup = static_cast<QPopupMenu *>( factory() ->container( "settings", this ) );
    connect( settingsPopup, SIGNAL( aboutToShow() ),
             SLOT( settingsMenuAboutToShow() ) );

    // load boad list & show board dock
    m_boardTab->loadBoardList();
    m_boardDock->slotShowDock();

    // update favorite list
    emit favoritesUpdated();

    if ( KitaConfig::autoLogin() ) {
        login();
    }
}

KitaMainWindow::~KitaMainWindow()
{
    /* save dock status */
    m_boardDock->saveSession();
    m_subjectDock->saveSession();
    m_threadDock->saveSession();
    m_naviDock->saveSession();
    m_imgDock->saveSession();
    m_writeDock->saveSession();

    saveAboneIDList();
    saveAboneNameList();
    saveAboneWordList();

    saveStyleSheet();

    saveAsciiArt();

    saveCompletion();

    saveFavoriteBoards();

    saveFavorites();

    saveCache();

    saveMainWindowSettings( KGlobal::config(), "MainWindow" );

    KitaConfig::writeConfig( KGlobal::config() );

    QString dockConfigPath = locateLocal( "appdata", "dock.conf" );
    KConfig dockConfig( dockConfigPath );
    writeDockConfig( &dockConfig );

    delete m_boardDock;
    delete m_threadDock;
    delete m_naviDock;
    delete m_imgDock;
    delete m_writeDock;
}

void KitaMainWindow::load( const KURL& url )
{
    QString target;
    // the below code is what you should normally do.  in this
    // example case, we want the url to our own.  you probably
    // want to use this code instead for your app

#if 0
    // download the contents
    if ( KIO::NetAccess::download( url, target ) ) {
        // set our caption
        setCaption( url );

        // load in the file (target is always local)
        loadFile( target );

        // and remove the temp file
        KIO::NetAccess::removeTempFile( target );
    }
#endif

    setCaption( url.url() );
    //    m_view->openURL(url);
}

void KitaMainWindow::setupActions()
{
    KStdAction::quit( this, SLOT( close() ), actionCollection() );
    KStdAction::copy( this, SLOT( slotEditCopy() ), actionCollection() );

    m_toolbarAction = KStdAction::showToolbar( this,
                      SLOT( optionsShowToolbar() ),
                      actionCollection() );

    m_statusbarAction = KStdAction::showStatusbar( this,
                        SLOT( optionsShowStatusbar() ),
                        actionCollection() );

    m_boardListAction = new KToggleAction( i18n( "Show board list" ),
                                           0,
                                           m_boardDock,
                                           SLOT( slotToggleShowHide() ),
                                           actionCollection(),
                                           "window_show_board_list" );

    m_subjectListAction = new KToggleAction( i18n( "Show subject list" ),
                          0,
                          m_subjectDock,
                          SLOT( slotToggleShowHide() ),
                          actionCollection(),
                          "window_show_subject_list" );

    m_showImgViewAction = new KToggleAction( i18n( "Show Image Viewer" ),
                          0,
                          m_imgDock,
                          SLOT( slotToggleShowHide() ),
                          actionCollection(),
                          "window_show_imgviewer" );

    m_showNaviAction = new KToggleAction( i18n( "Show KitaNavi" ),
                                          0,
                                          m_naviDock,
                                          SLOT( slotToggleShowHide() ),
                                          actionCollection(),
                                          "window_show_kitanavi" );

    m_showWriteDockAction = new KToggleAction( i18n( "Show Write Dock" ),
                            0,
                            m_writeDock,
                            SLOT( slotToggleShowHide() ),
                            actionCollection(),
                            "window_show_writedock" );

    m_urlLine = new KLineEdit( "", 0 );

    new KWidgetAction( m_urlLine,
                       i18n( "URL Line" ),
                       0,
                       this,
                       SLOT( slotURLLine() ),
                       actionCollection(), "url_line_action" );

    KStdAction::keyBindings( this, SLOT( optionsConfigureKeys() ), actionCollection() );
    KStdAction::configureToolbars( this, SLOT( optionsConfigureToolbars() ), actionCollection() );
    KStdAction::preferences( this, SLOT( optionsPreferences() ), actionCollection() );

    new KAction( i18n( "Load BoardList" ),
                 0,
                 m_boardTab,
                 SLOT( loadBoardList() ),
                 actionCollection(),
                 "load_board_list" );

    new KAction( i18n( "reset windows" ),
                 0,
                 this,
                 SLOT( resetWindowsCust() ),
                 actionCollection(),
                 "reset_wincust" );

    new KAction( i18n( "default 1" ),
                 0,
                 this,
                 SLOT( resetWindowsDef1() ),
                 actionCollection(),
                 "reset_windef1" );

    new KAction( i18n( "default 2" ),
                 0,
                 this,
                 SLOT( resetWindowsDef2() ),
                 actionCollection(),
                 "reset_windef2" );

    new KAction( i18n( "default 3" ),
                 0,
                 this,
                 SLOT( resetWindowsDef3() ),
                 actionCollection(),
                 "reset_windef3" );

    m_toggleViewAction = new KAction(
                             i18n( "&Toggle subject view and thread view" ),
                             KShortcut( Qt::Key_F7 ),
                             this,
                             SLOT( windowToggleView() ),
                             actionCollection(),
                             "window_toggle_view" );

    new KAction( i18n( "Login" ),
                 0,
                 this,
                 SLOT( login() ),
                 actionCollection(),
                 "login" );

    setXMLFile( "kitaui.rc" );
    KMainWindow::createGUI();
    factory() ->addClient( m_boardTab );
    factory() ->addClient( m_subjectTab );
    factory() ->addClient( m_threadTab );
    factory() ->addClient( m_imgTab );
    factory() ->addClient( m_naviTab );
    factory() ->addClient( m_writeTab );
}

void KitaMainWindow::slotURLLine()
{
    KURL url = m_urlLine->text();
    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    m_threadDock->slotShowThread( datURL.prettyURL(), KitaConfig::alwaysUseTab() );
}

void KitaMainWindow::saveProperties( KConfig* )
{
    // the 'config' object points to the session managed
    // config file.  anything you write here will be available
    // later when this app is restored

    //    if (m_view->currentURL() != QString::null)
    //        config->writeEntry("lastURL", m_view->currentURL());
    //    KitaConfig::writeConfig( config );
}

void KitaMainWindow::readProperties( KConfig* )
{
    // the 'config' object points to the session managed
    // config file.  this function is automatically called whenever
    // the app is being restored.  read in here whatever you wrote
    // in 'saveProperties'

    //    QString url = config->readEntry("lastURL");

    //    if (url != QString::null)
    //        m_view->openURL(KURL(url));
    //    KitaConfig::readConfig( config );
}

void KitaMainWindow::dragEnterEvent( QDragEnterEvent* event )
{
    // accept uri drops only
    event->accept( QUriDrag::canDecode( event ) );
}

void KitaMainWindow::dropEvent( QDropEvent* event )
{
    // this is a very simplistic implementation of a drop event.  we
    // will only accept a dropped URL.  the Qt dnd code can do *much*
    // much more, so please read the docs there
    QStrList uri;

    // see if we can decode a URI.. if not, just ignore it
    if ( QUriDrag::decode( event, uri ) ) {
        // okay, we have a URI.. process it
        QString url, target;
        url = uri.first();

        // load in the file
        load( KURL( url ) );
    }
}

void KitaMainWindow::optionsShowToolbar()
{
    // this is all very cut and paste code for showing/hiding the
    // toolbar
    if ( m_toolbarAction->isChecked() ) {
        toolBar() ->show();
    } else {
        toolBar() ->hide();
    }
}

void KitaMainWindow::optionsShowStatusbar()
{
    // this is all very cut and paste code for showing/hiding the
    // statusbar
    if ( m_statusbarAction->isChecked() ) {
        statusBar() ->show();
    } else {
        statusBar() ->hide();
    }
}

void KitaMainWindow::optionsConfigureKeys()
{
    KKeyDialog dlg( TRUE, this );
    dlg.insert( actionCollection(), "Kita" );
    dlg.configure();
}

void KitaMainWindow::optionsConfigureToolbars()
{
    // use the standard toolbar editor
    KEditToolbar dlg( factory() );
    connect( &dlg, SIGNAL( newToolbarConfig() ),
             SLOT( newToolbarConfig() ) );
    dlg.exec();
}

void KitaMainWindow::newToolbarConfig()
{
    // this slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
    // recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
    applyMainWindowSettings( KGlobal::config(), "MainWindow" );
}

void KitaMainWindow::resetWindowsCust() { resetWindows( 0 ); }
void KitaMainWindow::resetWindowsDef1() { resetWindows( 1 ); }
void KitaMainWindow::resetWindowsDef2() { resetWindows( 2 ); }
void KitaMainWindow::resetWindowsDef3() { resetWindows( 3 ); }

void KitaMainWindow::resetWindows( int pattern )
{
    if ( !m_dummyDock || !m_threadDock || !m_boardDock
            || !m_subjectDock || !m_naviDock || !m_imgDock || !m_writeDock ) return ;

    makeDockVisible( m_threadDock );
    makeDockVisible( m_boardDock );
    makeDockVisible( m_subjectDock );
    makeDockVisible( m_naviDock );
    makeDockVisible( m_imgDock );
    makeDockVisible( m_writeDock );

    makeDockInvisible( m_threadDock );
    makeDockInvisible( m_boardDock );
    makeDockInvisible( m_subjectDock );
    makeDockInvisible( m_naviDock );
    makeDockInvisible( m_imgDock );
    makeDockInvisible( m_writeDock );

    m_threadDock->manualDock( m_dummyDock, KDockWidget::DockTop, 100 );
    m_boardDock->manualDock( m_threadDock, KDockWidget::DockLeft, 20 );
    m_subjectDock->manualDock( m_threadDock, KDockWidget::DockTop, 25 );
    m_imgDock->manualDock( m_threadDock, KDockWidget::DockTop, 25 );
    m_writeDock->manualDock( m_threadDock, KDockWidget::DockBottom, 75 );
    m_naviDock->manualDock( m_threadDock, KDockWidget::DockBottom, 50 );

    switch ( pattern ) {

        /*----------------------------------------*/
    case 0:

        /* setSession( docked, tabbed ) */
        m_threadDock ->setSession( TRUE, FALSE );
        m_boardDock ->setSession( TRUE, FALSE );
        m_subjectDock->setSession( TRUE, FALSE );
        m_imgDock ->setSession( TRUE, FALSE );
        m_writeDock ->setSession( TRUE, FALSE );
        m_naviDock ->setSession( TRUE, FALSE );

        if ( m_boardListAction ) m_boardListAction->setChecked( TRUE );
        if ( m_subjectListAction ) m_subjectListAction->setChecked( TRUE );
        if ( m_showImgViewAction ) m_showImgViewAction->setChecked( TRUE );
        if ( m_showNaviAction ) m_showNaviAction->setChecked( TRUE );
        if ( m_showWriteDockAction ) m_showWriteDockAction->setChecked( TRUE );

        break;

        /*----------------------------------------*/
    case 1:

        m_imgDock->toDesktop();
        m_writeDock->toDesktop();
        m_naviDock->toDesktop();

        makeDockInvisible( m_imgDock );
        makeDockInvisible( m_writeDock );
        makeDockInvisible( m_naviDock );

        /* setSession( docked, tabbed ) */
        m_threadDock ->setSession( TRUE, FALSE );
        m_boardDock ->setSession( TRUE, FALSE );
        m_subjectDock->setSession( TRUE, FALSE );
        m_imgDock ->setSession( FALSE, FALSE );
        m_writeDock ->setSession( FALSE, FALSE );
        m_naviDock ->setSession( FALSE, FALSE );

        if ( m_boardListAction ) m_boardListAction->setChecked( TRUE );
        if ( m_subjectListAction ) m_subjectListAction->setChecked( TRUE );
        if ( m_showImgViewAction ) m_showImgViewAction->setChecked( FALSE );
        if ( m_showNaviAction ) m_showNaviAction->setChecked( FALSE );
        if ( m_showWriteDockAction ) m_showWriteDockAction->setChecked( FALSE );

        break;

        /*----------------------------------------*/
    case 2:

        m_subjectDock->manualDock( m_threadDock, KDockWidget::DockCenter, 0 );

        m_imgDock->toDesktop();
        m_writeDock->toDesktop();
        m_naviDock->toDesktop();

        makeDockInvisible( m_imgDock );
        makeDockInvisible( m_writeDock );
        makeDockInvisible( m_naviDock );

        /* setSession( docked, tabbed ) */
        m_threadDock ->setSession( TRUE, TRUE );
        m_boardDock ->setSession( TRUE, FALSE );
        m_subjectDock->setSession( TRUE, TRUE );
        m_imgDock ->setSession( FALSE, FALSE );
        m_writeDock ->setSession( FALSE, FALSE );
        m_naviDock ->setSession( FALSE, FALSE );

        if ( m_boardListAction ) m_boardListAction->setChecked( TRUE );
        if ( m_subjectListAction ) m_subjectListAction->setChecked( TRUE );
        if ( m_showImgViewAction ) m_showImgViewAction->setChecked( FALSE );
        if ( m_showNaviAction ) m_showNaviAction->setChecked( FALSE );
        if ( m_showWriteDockAction ) m_showWriteDockAction->setChecked( FALSE );

        break;

        /*----------------------------------------*/
    case 3:

        m_subjectDock->manualDock( m_threadDock, KDockWidget::DockCenter, 0 );
        m_naviDock->manualDock( m_threadDock, KDockWidget::DockCenter, 0 );
        m_imgDock->manualDock( m_threadDock, KDockWidget::DockCenter, 0 );

        makeDockInvisible( m_imgDock );
        makeDockInvisible( m_writeDock );
        makeDockInvisible( m_naviDock );

        /* setSession( docked, tabbed ) */
        m_threadDock ->setSession( TRUE, TRUE );
        m_boardDock ->setSession( TRUE, FALSE );
        m_subjectDock->setSession( TRUE, TRUE );
        m_imgDock ->setSession( TRUE, TRUE );
        m_writeDock ->setSession( TRUE, FALSE );
        m_naviDock ->setSession( TRUE, TRUE );

        if ( m_boardListAction ) m_boardListAction->setChecked( TRUE );
        if ( m_subjectListAction ) m_subjectListAction->setChecked( TRUE );
        if ( m_showImgViewAction ) m_showImgViewAction->setChecked( FALSE );
        if ( m_showNaviAction ) m_showNaviAction->setChecked( FALSE );
        if ( m_showWriteDockAction ) m_showWriteDockAction->setChecked( FALSE );

        break;
    }

    m_threadDock->slotShowDock();
}


void KitaMainWindow::windowToggleView()
{
    //FIXME: m_subjectDock->dockSite() & KDockWidget::DockCenter != 0 ʳΤȤϤɤ롩
    kdDebug() << "Do toggle" << endl;
    if ( !m_subjectDock->isVisible()
            || !m_subjectDock->isActiveWindow() ) m_subjectDock->slotShowDock();
    else {
        m_threadDock->slotShowDock();
    }
}

void KitaMainWindow::settingsMenuAboutToShow()
{
    m_toolbarAction->setChecked( toolBar() ->isVisible() );
    m_statusbarAction->setChecked( statusBar() ->isVisible() );
}

void KitaMainWindow::optionsPreferences()
{
    // popup some sort of preference dialog, here
    KitaPreferences dlg;

    connect( &dlg, SIGNAL( fontChanged( const QFont& ) ),
             SLOT( setFont( const QFont& ) ) );

    // XXX
    dlg.setInitialSize( QSize( 600, 400 ) );
    if ( dlg.exec() ) {
        // redo your settings
    }
}

void KitaMainWindow::slotSetMainStatusbar( const QString& statusStr )
{
    // display the text on the statusbar
    statusBar() ->message( statusStr );
}

void KitaMainWindow::setupView()
{
    /* dummy Main Dock Widget */
    m_dummyDock = createDockWidget( "dummy", 0L, 0L, "dummy", "dummy" );
    m_dummyDock->setDockSite( KDockWidget::DockFullSite );
    m_dummyDock->setEnableDocking( KDockWidget::DockNone );
    setView( m_dummyDock );
    setMainDockWidget( m_dummyDock );

    /* Subject View */
    m_subjectDock = new KitaSubjectDock( manager(), "subject", 0L, 0L, i18n( "subject" ), i18n( "subject" ) );
    m_subjectTab = m_subjectDock->setup();
    connect( m_subjectDock, SIGNAL( checkToggleAction( bool ) ), SLOT( slotCheckSubjectToggleAction( bool ) ) );

    /* Board View */
    m_boardDock = new KitaBoardDock( manager(), "board", 0L, 0L, i18n( "board" ), i18n( "board" ) );
    m_boardTab = m_boardDock->setup();
    connect( m_boardDock, SIGNAL( checkToggleAction( bool ) ), SLOT( slotCheckBoardToggleAction( bool ) ) );

    /* Thread View */
    m_threadDock = new KitaThreadDock( manager(), "Thread", 0L, 0L, i18n( "Thread" ), i18n( "Thread" ) );
    m_threadTab = m_threadDock->setupAsMainView();
    m_threadDock->setDockSite( KDockWidget::DockFullSite );

    /* KitaNavi */
    m_naviDock = new KitaThreadDock( manager(), "kitanavi", 0L, 0L, i18n( "KitaNavi" ), i18n( "Navi" ) );
    m_naviTab = m_naviDock->setupAsKitaNavi();
    connect( m_naviDock, SIGNAL( checkToggleAction( bool ) ), SLOT( slotCheckNaviToggleAction( bool ) ) );

    if ( !KitaConfig::useKitaNavi() ) m_threadDock ->connectNaviSignals();
    else m_naviDock->connectNaviSignals();

    /* KitaImgViewer */
    m_imgDock = new KitaImgDock( manager(), "imgviewer", 0L, 0L, i18n( "Kita Image Viewer" ), i18n( "Image" ) );
    m_imgTab = m_imgDock->setup();
    connect( m_imgDock, SIGNAL( checkToggleAction( bool ) ), SLOT( slotCheckImgToggleAction( bool ) ) );

    /* write dock */
    m_writeDock = new KitaWriteDock( manager(), "Write", 0L, 0L, i18n( "Write" ), i18n( "Write" ) );
    m_writeTab = m_writeDock->setup();
    connect( m_writeDock, SIGNAL( checkToggleAction( bool ) ), SLOT( slotCheckWriteToggleAction( bool ) ) );

    /* reset dock status */
    m_boardListAction = NULL;
    m_subjectListAction = NULL;
    m_showImgViewAction = NULL;
    m_showNaviAction = NULL;
    m_showWriteDockAction = NULL;
    resetWindowsDef1();
}

void KitaMainWindow::closeSubjectView()
{
    m_subjectListAction->setChecked( FALSE );
}

void KitaMainWindow::slotCheckBoardToggleAction( bool check )
{
    if ( m_boardListAction ) m_boardListAction->setChecked( check );
}

void KitaMainWindow::slotCheckSubjectToggleAction( bool check )
{
    if ( m_subjectListAction ) m_subjectListAction->setChecked( check );
}

void KitaMainWindow::slotCheckNaviToggleAction( bool check )
{
    if ( m_showNaviAction ) m_showNaviAction->setChecked( check );
}

void KitaMainWindow::slotCheckImgToggleAction( bool check )
{
    if ( m_showImgViewAction ) m_showImgViewAction->setChecked( check );
}


void KitaMainWindow::slotCheckWriteToggleAction( bool check )
{
    if ( m_showWriteDockAction ) m_showWriteDockAction->setChecked( check );
}


void KitaMainWindow::slotOpenURLRequest( const KURL& url, const KParts::URLArgs& args )
{
    slotOpenURLRequestExt( url, args );
}


/* open url */
/*
  If mimetype == "text/html", open url with web browser.
  if mimetype == QString::null, use KIO::NetAccess::mimetype to decide mime type of url.
 
*/  /* public slot */
void KitaMainWindow::slotOpenURLRequestExt(
    const KURL& url, const KParts::URLArgs& args,
    QString mimetype,
    int usr1, int usr2,
    const KURL& usrUrl1, const KURL& usrUrl2,
    const QString& usrStr1, const QString& usrStr2 )
{
    kdDebug() << QString( "Requested OpenURL: \"%1\"" ).arg( url.url() ) << endl;
    kdDebug() << QString( "content type: \"%1\"" ).arg( args.contentType() ) << endl;
    kdDebug() << QString( "service type: \"%1\"" ).arg( args.serviceType ) << endl;
    kdDebug() << QString( "NewTab: %1, reload: %2" )
    .arg( args.newTab() ? "yes" : "no" )
    .arg( args.reload ? "yes" : "no" ) << endl;
    KURL datURL = Kita::ParseMisc::parseURLonly( url ); /* convert "thread URL" into "dat URL" */

    /*  open url with web browser, and return. */
    if ( mimetype == "text/html" ) {
        KRun::runURL( url, mimetype );
        return ;
    }

    QRegExp regexp( "http://([^/]*)/([^/]*)/dat/(.*)\\.dat$" );
    if ( regexp.search( datURL.url() ) == -1 ) { /* Is this URL 2ch? */

        /* bmp file */
        if ( Kita::ImgManager::isBMP( url ) ) {
            QMessageBox::warning( this,
                                  "Kita",
                                  i18n( "This is a BMP file. It might contain a Troy." ) );
            return ;
        }


        /* open with image viewer */
        /* if usr1 == 0, image viewer is hidden. */
        /* usrUrl1 is datURL.                    */
        if ( KitaConfig::useImageViewer() ) {
            if ( Kita::ImgManager::isImgFile( url, FALSE ) ) {

                if ( !usr1 ) m_imgDock->slotHideDock();
                m_imgDock->slotOpenURL( url, usrUrl1, usr1 );
                return ;
            }
        }

        if ( mimetype == QString::null ) {
#if KDE_IS_VERSION( 3, 2, 0 )
            mimetype = KIO::NetAccess::mimetype( url , this );
#else
            mimetype = KIO::NetAccess::mimetype( url );
#endif

        }
        kdDebug() << QString( "queried MIME type: \"%1\"" ).arg( mimetype ) << endl;

        /* open with kpart */
        if ( KitaConfig::usePart() ) {
            if ( KitaConfig::partMimeList().find( mimetype ) != KitaConfig::partMimeList().end() ) {
                KService::Ptr service = KServiceTypeProfile::preferredService( mimetype, "KParts/ReadOnlyPart" );
                if ( service ) {
                    QString libName = QFile::encodeName( service->library() );

                    if ( !KitaConfig::useKitaNavi() ) m_threadDock->slotShowPart( url, libName, mimetype );
                    else m_naviDock ->slotShowPart( url, libName, mimetype );
                    return ;
                }
            }
        }

        KRun::runURL( url, mimetype );

    } else {
        m_threadDock->slotShowThread( url.prettyURL(), true );
    }
}

void KitaMainWindow::loadCache()
{
    KitaThreadInfo * cache = KitaThreadInfo::getInstance();
    QString cacheConfigPath = locateLocal( "appdata", "cache" );
    QFile file( cacheConfigPath );
    if ( file.open( IO_ReadOnly ) ) {
        QDataStream stream( &file );
        stream >> *cache;
    }
}

void KitaMainWindow::saveCache()
{
    KitaThreadInfo * cache = KitaThreadInfo::getInstance();
    QString cacheConfigPath = locateLocal( "appdata", "cache" );
    QFile file( cacheConfigPath );
    if ( file.open( IO_WriteOnly ) ) {
        QDataStream stream( &file );
        stream << *cache;
    }
}

/*!
    \fn KitaMainWindow::setFont(QFont& font)
 */

void KitaMainWindow::setFont( const QFont& font )
{
    m_subjectTab->setFont( font );
    m_boardTab->setFont( font );
}


void KitaMainWindow::setUrl( const KURL& url )
{
    m_urlLine->setText( url.url() );
}

void KitaMainWindow::bookmark( const QString& datURL, bool on )
{
    FavoriteThreads * favorite = FavoriteThreads::getInstance();

    if ( on ) {
        favorite->insert( datURL );
    } else {
        favorite->remove( datURL );
    }
    emit favoritesUpdated();
}

void KitaMainWindow::loadFavorites()
{
    QString favoritesConfigPath = locateLocal( "appdata", "favorites.xml" );
    QFile file( favoritesConfigPath );
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QString xml = stream.read();
        FavoriteThreads::readFromXML( xml );
    }
}

void KitaMainWindow::saveFavorites()
{
    QString favoritesConfigPath = locateLocal( "appdata", "favorites.xml" );
    QFile file( favoritesConfigPath );
    if ( file.open( IO_WriteOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );
        stream << FavoriteThreads::getInstance() ->toXML();
    }
}

void KitaMainWindow::loadCompletion()
{
    QString configPath = locateLocal( "appdata", "completion" );
    KSimpleConfig config( configPath );

    KitaConfig::setNameCompletionItems( config.readListEntry( "name" ) );
}

void KitaMainWindow::saveCompletion()
{
    QString configPath = locateLocal( "appdata", "completion" );
    KSimpleConfig config( configPath );

    config.writeEntry( "name", KitaConfig::nameCompletion() ->items() );
    config.sync();
}

void KitaMainWindow::loadStyleSheet()
{
    QString configPath = locateLocal( "appdata", "stylesheet" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null ) {
            if ( ! str.isEmpty() ) {
                list << str;
            }
        }
        KitaConfig::setStyleSheetText( list.join( "\n" ) );
    }
}

void KitaMainWindow::saveStyleSheet()
{
    QString configPath = locateLocal( "appdata", "stylesheet" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) ) {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        stream << KitaConfig::styleSheetText();
    }
}

void KitaMainWindow::loadAsciiArt()
{
    QString configPath = locateLocal( "appdata", "asciiart" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null ) {
            if ( ! str.isEmpty() ) {
                list << str;
            }
        }
        KitaConfig::setAsciiArtList( list );
    }
}

void KitaMainWindow::saveAsciiArt()
{
    QString configPath = locateLocal( "appdata", "asciiart" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) ) {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list = KitaConfig::asciiArtList();
        QStringList::iterator it;
        for ( it = list.begin(); it != list.end(); ++it ) {
            stream << ( *it ) << endl;
        }
    }
}

void KitaMainWindow::loadFavoriteBoards()
{
    QString configPath = locateLocal( "appdata", "favorite_boards.xml" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QString xml = stream.read();
        Kita::FavoriteBoards::readFromXML( xml );
    }
}

void KitaMainWindow::saveFavoriteBoards()
{
    QString configPath = locateLocal( "appdata", "favorite_boards.xml" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );
        stream << Kita::FavoriteBoards::toXML();
    }
}

void KitaMainWindow::loadAboneIDList()
{
    QString configPath = locateLocal( "appdata", "abone_id" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null ) {
            if ( ! str.isEmpty() ) {
                list << str;
            }
        }
        KitaConfig::setAboneIDList( list );
    }
}

void KitaMainWindow::saveAboneIDList()
{
    QString configPath = locateLocal( "appdata", "abone_id" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) ) {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list = KitaConfig::aboneIDList();
        QStringList::iterator it;
        for ( it = list.begin(); it != list.end(); ++it ) {
            stream << ( *it ) << endl;
        }
    }
}

void KitaMainWindow::loadAboneNameList()
{
    QString configPath = locateLocal( "appdata", "abone_name" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null ) {
            if ( ! str.isEmpty() ) {
                list << str;
            }
        }
        KitaConfig::setAboneNameList( list );
    }
}

void KitaMainWindow::saveAboneNameList()
{
    QString configPath = locateLocal( "appdata", "abone_name" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) ) {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list = KitaConfig::aboneNameList();
        QStringList::iterator it;
        for ( it = list.begin(); it != list.end(); ++it ) {
            stream << ( *it ) << endl;
        }
    }
}

void KitaMainWindow::loadAboneWordList()
{
    QString configPath = locateLocal( "appdata", "abone_word" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null ) {
            if ( ! str.isEmpty() ) {
                list << str;
            }
        }
        KitaConfig::setAboneWordList( list );
    }
}

void KitaMainWindow::saveAboneWordList()
{
    QString configPath = locateLocal( "appdata", "abone_word" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) ) {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list = KitaConfig::aboneWordList();
        QStringList::iterator it;
        for ( it = list.begin(); it != list.end(); ++it ) {
            stream << ( *it ) << endl;
        }
    }
}

void KitaMainWindow::slotSetMainCaption( const QString& captionStr )
{
    setCaption( captionStr );
}

void KitaMainWindow::slotEditCopy()
{
    QWidget * widget = kapp->focusWidget();

    if ( widget ) {
        QKeyEvent e( QEvent::KeyPress, Key_C, 'C', ControlButton );
        QApplication::sendEvent( widget, &e );
    }
}

void KitaMainWindow::login()
{
    if ( Kita::Account::login( KitaConfig::userID(), KitaConfig::password() ) ) {
        slotSetMainStatusbar( i18n( "Login succeeded." ) );
    } else {
        slotSetMainStatusbar( i18n( "Login failed." ) );
    }
}

#include "kita.moc"
