#pragma once

#include <cassert>
#include <boost/mpl/eval_if.hpp>
#include <boost/mpl/identity.hpp>

#include "../utility/ignore_unused_variables_warning.hpp"


///////////////////////////////////////////////////////////////////////////////
// local macros
//
#define KETCHUP_process_params \
	Derived& derived, HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam, LRESULT& lResult, DWORD dwMsgMapID, BOOL& bHandled

#define KETCHUP_process_args \
	derived, hWnd, uMsg, wParam, lParam, lResult, dwMsgMapID, bHandled


namespace ketchup
{

///////////////////////////////////////////////////////////////////////////////
// empty_entry
//
struct empty_entry
{
	static bool process(KETCHUP_process_params)
	{
		ignore_unused_variables_warning(KETCHUP_process_args);
		return false;
	}
};

///////////////////////////////////////////////////////////////////////////////
// assert_entry
//
struct assert_entry
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		ignore_unused_variables_warning(KETCHUP_process_args);
		assert(false);
		return false;
	}
};

///////////////////////////////////////////////////////////////////////////////
// debug_entry
//
template< class Entry >
struct debug_entry
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		ignore_unused_variables_warning(KETCHUP_process_args);
		
		typedef boost::mpl::eval_if<utility::is_debug,
			boost::mpl::identity<Entry>,
			boost::mpl::identity<empty_entry>
		>::type entry_t;
		
		return entry_t::process(KETCHUP_process_args);
	}
};

///////////////////////////////////////////////////////////////////////////////
// chain_msg_map
//
template< class ChainClass, DWORD msgMapID >
struct chain_msg_map_alt
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		ignore_unused_variables_warning(KETCHUP_process_args);
		return derived.ChainClass::ProcessWindowMessage(hWnd, uMsg, wParam, lParam, lResult, msgMapID) == TRUE;
	}
};

template< class ChainClass >
struct chain_msg_map : chain_msg_map_alt<ChainClass, 0>
{
};

///////////////////////////////////////////////////////////////////////////////
// alt_msg_map
//
template< DWORD msgMapID, class Entry >
struct alt_msg_map
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		ignore_unused_variables_warning(KETCHUP_process_args);
		if (dwMsgMapID == msgMapID) {
				return derived.process_window_message<Entry>(hWnd, uMsg, wParam, lParam, lResult, msgMapID) == TRUE;
		}
		return false;
	}
};

///////////////////////////////////////////////////////////////////////////////
// chain_msg_map_dynamic
//
template< DWORD dynaChainID >
struct chain_msg_map_dynamic
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		ignore_unused_variables_warning(KETCHUP_process_args);
		return CDynamicChain::CallChain(dynaChainID, hWnd, uMsg, wParam, lParam, lResult) == TRUE;
	}
};

///////////////////////////////////////////////////////////////////////////////
// forward_notifications
//
struct forward_notifications
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		bHandled = TRUE;
		lResult = derived.ForwardNotifications(uMsg, wParam, lParam, bHandled);
		if(bHandled)
			return true;
	}
};

///////////////////////////////////////////////////////////////////////////////
// default_reflection_handler
//
struct default_reflection_handler
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		return Derived::DefaultReflectionHandler(hWnd, uMsg, wParam, lParam, lResult) == TRUE;
	}
};

///////////////////////////////////////////////////////////////////////////////
// chain_client_commands
//
struct chain_client_commands
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		ignore_unused_variables_warning(KETCHUP_process_args);

		if (uMsg == WM_COMMAND)
		{
			HWND hWnd = derived.m_hWndClient;
			if (hWnd != NULL)
				::SendMessage(hWnd, uMsg, wParam, lParam);
			return false;
		}

		return false;
	}
};

///////////////////////////////////////////////////////////////////////////////
// chain_mdi_child_commands
//
struct chain_mdi_child_commands
{
	template< class Derived >
	static bool process(KETCHUP_process_params)
	{
		ignore_unused_variables_warning(KETCHUP_process_args);

		if (uMsg == WM_COMMAND)
		{
			HWND hWndChild = derived.MDIGetActive();
			if (hWndChild != NULL)
				::SendMessage(hWndChild, uMsg, wParam, lParam);
			return false;
		}
		return false;
	}
};

///////////////////////////////////////////////////////////////////////////////
// cleaning up local macros
// 
#undef KETCHUP_process_params
#undef KETCHUP_process_args

} // namespace ketchup
