///////////////////////////////////////////////////////////////////////////////
// cmd_ui_updater.hpp
//
//  Concepts:
//   CmdUIMap is...
//    MPL Sequence of Entry.
//   Entry has...
//    a static member function, process
//

#pragma once

#include <boost/call_traits.hpp>
#include <boost/mpl/fold.hpp>
#include <boost/spirit/fusion/sequence/tuple.hpp>
#include <boost/spirit/fusion/sequence/get.hpp>
#include <cassert>

// #include "message_processor.hpp"
#include "cmd_ui.hpp"
#include "init_cbSize.hpp"
#include "detail/gen_process_statement.hpp"
#include "detail/menu_cmd_ui.hpp"
#include "detail/toolbar_cmd_ui.hpp"
#include "detail/childwindow_cmd_ui.hpp"
#include "detail/multipanestatusbar_cmd_ui.hpp"

namespace ketchup {

namespace detail {

	UINT update_cmd_ui_message_id()
	{
		return ::RegisterWindowMessage(_T("ketchup_update_cmd_ui"));
	}

	bool is_cmd_ui_handled_by_window(LRESULT lResult)
	{
		return static_cast<BOOL>(lResult)
			== TRUE;
	}
	
	cmd_ui& get_cmd_ui_from_wParam(WPARAM wParam)
	{
		cmd_ui* p_cmd_ui = reinterpret_cast<cmd_ui*>(wParam);
		return *p_cmd_ui;
	}
	
	///////////////////////////////////////////////////////////////////////////////
	// send_cmd_ui_to_window
	//  throws the cmd_ui to window
	//
	bool send_cmd_ui_to_window(HWND hWnd, cmd_ui& ui)
	{
		CWindow wnd(hWnd);
		assert(wnd.IsWindow());
		cmd_ui* p_cmd_ui = &ui;
		return is_cmd_ui_handled_by_window(
			wnd.SendMessage(update_cmd_ui_message_id(), reinterpret_cast<WPARAM>(p_cmd_ui))
		);
	}

	struct cmd_ui_updater_base
	{
	private:
		bool b_cmd_ui_handled;

	protected:
		bool is_cmd_ui_handled() const { return b_cmd_ui_handled; }
		void set_cmd_ui_handled(bool b_) { b_cmd_ui_handled = b_; }
	};

} // detail

template< class T >
struct get_cmd_ui_map;

///////////////////////////////////////////////////////////////////////////////
// cmd_ui_updater
//
template< class Derived >
struct cmd_ui_updater : detail::cmd_ui_updater_base
{
protected:
	typedef boost::fusion::tuple<Derived&, cmd_ui&> process_args_type;
	typedef typename boost::call_traits<process_args_type>::reference process_args_ref;
	typedef typename boost::call_traits<process_args_type>::param_type process_args_param_type;

	// shortcuts
	static Derived& get_derived(process_args_param_type xs)
	{
		return boost::fusion::get<0>(xs);
	}

	static cmd_ui& get_cmd_ui(process_args_param_type xs)
	{
		return boost::fusion::get<1>(xs);
	}

	template< class CmdUIMap >
	bool update_cmd_ui(cmd_ui& ui)
	{
		Derived& derived = *static_cast<Derived*>(this);
		bool old = derived.is_cmd_ui_handled(); // for recursive call
		typedef typename detail::gen_process_statement<CmdUIMap, process_args_ref>::type statement;
		process_args_type xs(derived, ui);
		bool ret = statement::process(xs);
		derived.set_cmd_ui_handled(old);
		return ret;
	}

	///////////////////////////////////////////////////////////////////////////////
	// update_menu_cmd_ui
	//
	template< class CmdUIMap >
	void update_menu_cmd_ui(CMenuHandle menu)
	{
		assert(menu.IsMenu());
		// See: MFC7::CFrameWnd::OnInitMenuPopup
		int count = menu.GetMenuItemCount();
		for (int i = 0; i < count; ++i)
		{
			UINT nID = menu.GetMenuItemID(i);
			if (nID == 0)
				continue; // menu separator or invalid cmd - ignore it

			if (nID == (UINT)-1)
			{
				// possibly a popup menu, route to first item of that popup
				CMenuHandle sub_menu = menu.GetSubMenu(i);
				if (sub_menu.m_hMenu == NULL ||
					(nID = sub_menu.GetMenuItemID(0)) == 0 ||
					nID == (UINT)-1)
				{
					continue;       // first item of popup can't be routed to
				}
				detail::menu_cmd_ui ui(nID, menu, i, true);
				update_cmd_ui<CmdUIMap>(ui); // popups
			}
			else
			{
				// normal menu item
				detail::menu_cmd_ui ui(nID, menu, i, false);
				update_cmd_ui<CmdUIMap>(ui);
			}

			// adjust for menu deletions and additions
			int new_count = menu.GetMenuItemCount();
			if (new_count < count)
			{
				i -= (count - new_count);
				while (i < new_count &&
					menu.GetMenuItemID(i) == nID)
				{
					++i;
				}
			}
			count = new_count;
		} // for
	}

	///////////////////////////////////////////////////////////////////////////////
	// update_menubar_cmd_ui
	//
	// who needs?

	///////////////////////////////////////////////////////////////////////////////
	// update_commandbar_cmd_ui
	//
	// who needs?

	///////////////////////////////////////////////////////////////////////////////
	// update_toolbar_cmd_ui
	//
	template< class CmdUIMap >
	void update_toolbar_cmd_ui(HWND hWndToolBar)
	{
		// See: MFC7::CToolBar::OnUpdateCmdUI
		CToolBarCtrl wndToolBar(hWndToolBar);
		assert(wndToolBar.IsWindow());

		int nIndexMax = wndToolBar.GetButtonCount();
		for (int i = 0; i < nIndexMax; ++i)
		{
			// get buttons state
			TBBUTTON button;
			wndToolBar.GetButton(i, &button);
			UINT nID = button.idCommand;
			detail::toolbar_cmd_ui ui(nID, wndToolBar);

			// ignore separators
			if (!(button.fsStyle & TBSTYLE_SEP))
				update_cmd_ui<CmdUIMap>(ui);
		}
	}

	///////////////////////////////////////////////////////////////////////////////
	// update_childwindow_cmd_ui
	//
	template< class CmdUIMap >
	void update_childwindow_cmd_ui(HWND hWndParent, UINT nID)
	{
		CWindow wndParent(hWndParent);
		assert(wndParent.IsWindow());
		CWindow wndChild = wndParent.GetDlgItem(nID);
		assert(wndChild.IsWindow());
		detail::childwindow_cmd_ui ui(nID, wndChild);
		update_cmd_ui<CmdUIMap>(ui);
	}

	///////////////////////////////////////////////////////////////////////////////
	// update_multipanestatusbar_cmd_ui
	//
	template< class CmdUIMap, class MultiPaneStatusBarCtrl >
	void update_multipanestatusbar_cmd_ui(MultiPaneStatusBarCtrl& wndStatusBar)
	{
		assert(wndStatusBar.IsWindow());

		int nIndexMax = wndStatusBar.m_nPanes;
		for (int i = 0; i < nIndexMax; ++i)
		{
			// get buttons state
			int nID = wndStatusBar.m_pPane[i];
			detail::multipanestatusbar_cmd_ui<MultiPaneStatusBarCtrl> ui(nID, wndStatusBar, i);
			update_cmd_ui<CmdUIMap>(ui);
		}
	}

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler
	//
	template< UINT id, void (Derived::*func)(cmd_ui&) >
	struct cmd_ui_handler
	{
		static bool process(process_args_ref xs)
		{
			if (get_cmd_ui(xs).is_clingy())
				return false;

			if (get_cmd_ui(xs).get_id() == id) {
				get_derived(xs).set_cmd_ui_handled(true);
				(get_derived(xs).*func)(get_cmd_ui(xs));
				if (get_derived(xs).is_cmd_ui_handled())
					return true;
			}
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler_popup
	//
	// a menu example...
	//
	//  File>   (can't have ID..., plz give up...)
	//   New>   (can't have ID..., so use ID_NEW_SHEET)
	//    Sheet (ID_NEW_SHEET)
	//    Chart (ID_NEW_CHART)
	//
	template< UINT id, void (Derived::*func)(cmd_ui&) >
	struct cmd_ui_handler_popup
	{
		static bool process(process_args_ref xs)
		{
			if (!get_cmd_ui(xs).is_clingy())
				return false;

			if (get_cmd_ui(xs).get_id() == id) {
				get_derived(xs).set_cmd_ui_handled(true);
				(get_derived(xs).*func)(get_cmd_ui(xs));
				if (get_derived(xs).is_cmd_ui_handled())
					return true;
			}
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler_empty
	//
	struct cmd_ui_handler_empty
	{
		static bool process(process_args_ref)
		{
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler_set_default
	//
	template< UINT id >
	struct cmd_ui_handler_set_default
	{
		static bool process(process_args_ref xs)
		{
			if (get_cmd_ui(xs).is_clingy())
				return false;

			if (get_cmd_ui(xs).get_id() == id) {
				get_cmd_ui(xs).set_default(true);
			}
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler_auto_enable_ids
	//  if Entry does not handle, enable the cmd_ui.
	//  if Entry handles, disable the cmd_ui
	//
	template< class nIDs, class Entry >
	struct cmd_ui_handler_auto_enable_ids
	{
		struct fold_init
		{
			static bool nID_equals(UINT) { return false; }
		};

		template< class State, class nID >
		struct fold_op
		{
			static bool nID_equals(UINT cmd_ui_id)
			{
				if (cmd_ui_id == nID::value)
					return true;
				else
					return State::nID_equals(cmd_ui_id);
			}
		};

		static bool process(process_args_ref xs)
		{
			typedef typename boost::mpl::fold<nIDs,
				fold_init,
				fold_op<boost::mpl::_1, boost::mpl::_2>
			>::type statement;

			bool bHandled = Entry::process(xs);
			if (statement::nID_equals(get_cmd_ui(xs).get_id())) {
					get_cmd_ui(xs).enable(bHandled);
			}
			return bHandled;
		}
	};

	/*
	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler_disable
	//  nIDs is MPL Sequence of IntegralConstants
	template< class nIDs, bool bOn, bool bHandled >
	struct cmd_ui_handler_enable_set
	{
		struct fold_init
		{
			static bool nID_equals(UINT) { return false; }
		};

		template< class State, class nID >
		struct fold_op
		{
			static bool nID_equals(UINT cmd_ui_id)
			{
				if (cmd_ui_id == nID::value)
					return true;
				else
					return State::nID_equals(cmd_ui_id);
			}
		};

		static bool process(process_args_ref xs)
		{
			if (get_cmd_ui(xs).is_clingy())
				return false;

			typedef typename boost::mpl::fold<nIDs,
				fold_init,
				fold_op<boost::mpl::_1, boost::mpl::_2>
			>::type statement;

			if (statement::nID_equals(get_cmd_ui(xs).get_id())) {
				get_cmd_ui(xs).enable(bOn);
				if (bHandled)
					return true;
			}
			return false;
		}
	};
	*/
	///////////////////////////////////////////////////////////////////////////////
	// chain_cmd_ui_map_cmd_ui
	//  throws the cmd_ui to cmd_ui_map
	//
	template< class CmdUIMap >
	struct chain_cmd_ui_map_cmd_ui
	{
		static bool process(process_args_type xs)
		{
				return get_derived(xs).update_cmd_ui<CmdUIMap>(get_cmd_ui(xs));
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// chain_client_cmd_ui
  //  throws the cmd_ui to a client window
	//
	struct chain_client_cmd_ui
	{
		static bool process(process_args_ref xs)
		{
			HWND hWnd = get_derived(xs).m_hWndClient;
			return hWnd != NULL
				&& detail::send_cmd_ui_to_window(hWnd, get_cmd_ui(xs));
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// chain_mdi_child_cmd_ui
  //  throws the cmd_ui to an active mdi child.
	//
	struct chain_mdi_child_cmd_ui
	{
		static bool process(process_args_ref xs)
		{
			HWND hWndActive = get_derived(xs).MDIGetActive();
			return hWndActive != NULL
				&& detail::send_cmd_ui_to_window(hWndActive, get_cmd_ui(xs));
		}
	};
	
	///////////////////////////////////////////////////////////////////////////////
	// future issue
	//
	template< class T >
	struct gen_chain_cmd_ui_map_base
	{
	};
	
	template< class T >
	struct chain_cmd_ui_map : gen_chain_cmd_ui_map_base<T>
	{
	};

	///////////////////////////////////////////////////////////////////////////////
	// future issue
	//
	/*
	template<>
	struct chain_msg_map<cmd_ui_updater>
	{
		typedef typename Derived::arguments_type args_type;
		static bool process(args_type xs)
		{
			if (Derived::get_uMsg(xs) == WM_INITMENUPOPUP) {
				Derived::get_derived(xs).update_menu_cmd_ui<UpdateCmdUIMap>((HMENU)Derived::get_wParam(xs));
			}
			return false;
		}
	};
	*/
};

} // namespace ketchup

///////////////////////////////////////////////////////////////////////////////
// KETCHUP_UPDATE_MENU_CMD_UI
// 
#define KETCHUP_UPDATE_MENU_CMD_UI(CmdUIMap) \
	if (uMsg == WM_INITMENUPOPUP)\
  	update_menu_cmd_ui<CmdUIMap>((HMENU)wParam);\
/**/

///////////////////////////////////////////////////////////////////////////////
// KETCHUP_CHAIN_CMD_UI_MAP_CMD_UI
//  throws the cmd_ui to a cmd_ui_map.
#define KETCHUP_CHAIN_CMD_UI_MAP_CMD_UI(CmdUIMap) \
	if (uMsg == ketchup::detail::update_cmd_ui_message_id()) {\
		lResult = update_cmd_ui<CmdUIMap>(ketchup::detail::get_cmd_ui_from_wParam(wParam));\
		if (lResult)\
			return TRUE;\
	}\
/**/

///////////////////////////////////////////////////////////////////////////////
// KETCHUP_CHAIN_CLIENT_CMD_UI
//  throw the cmd_ui to a clint window.
#define KETCHUP_CHAIN_CLIENT_CMD_UI() \
	if(m_hWndClient != NULL && uMsg == ketchup::detail::update_cmd_ui_message_id()) {\
		lResult = ::SendMessage(m_hWndClient, uMsg, wParam, lParam);\
		if (lResult)\
			return TRUE;\
	}\
/**/

