# -*-perl-*-
# Keitairc::IrcCallback
# $Id: IrcCallback.pm,v 1.5 2008/02/23 18:58:22 morimoto Exp $
# $Source: /cvsroot/keitairc/lib/Keitairc/IrcCallback.pm,v $
#
# Copyright (c) 2008 Jun Morimoto <morimoto@mrmt.net>
# This program is covered by the GNU General Public License 2

package Keitairc::IrcCallback;
use POE;
use Encode;
use strict;

################################################################
sub irc_start{
	my ($heap) = @_[HEAP];
	$heap->{Irc}->yield(register => 'all');
	$heap->{Irc}->yield(connect => {});
}

################################################################
sub irc_001{
	my ($kernel, $heap, $sender) = @_[KERNEL, HEAP, SENDER];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};
	for my $cid ($ib->channels()){
		$ib->add_message($cid, 'Connected to irc server!');
		$ib->part($cid);
	}
	$heap->{disconnect_msg} = 1;
	$kernel->delay(autoping => $cf->ping_delay());
}

################################################################
sub irc_public{
	my ($kernel, $heap, $who, $channel, $msg) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	my $ib = $heap->{IrcBuffer};
	$who =~ s/!.*//;
	Encode::from_to($msg, 'jis', 'euc-jp');
	my $cid = $ib->name2cid($channel->[0]);
	$heap->{IrcBuffer}->add_message($cid, $msg, $who);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_join{
	my ($kernel, $heap, $who, $channel) = @_[KERNEL, HEAP, ARG0, ARG1];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};

	$who =~ s/!.*//;

	# chop off after the gap to workaround madoka's bug
	$channel =~ s/ .*//;

	$ib->join($channel);
	if($who ne $cf->irc_nick()){
		my $cid = $ib->name2cid($channel);
		$ib->add_message($cid, "$who joined");
	}

	$heap->{Irc}->yield(who => $channel);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_part{
	my ($kernel, $heap, $who, $channel) = @_[KERNEL, HEAP, ARG0, ARG1];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};

	$who =~ s/!.*//;
	# chop off after the gap (bug workaround of POE::Filter::IRC)
	$channel =~ s/ .*//;

	my $cid = $ib->name2cid($channel);
	if($who eq $cf->irc_nick()){
		$ib->part($cid);
	} else {
		$ib->add_message($cid, "$who leaves");
	}
	$ib->remove_nick($cid, $who);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_whoreply{
	my ($kernel, $heap, $arg) = @_[KERNEL, HEAP, ARG1];
	my ($channel, $user, $host, $server, $nick, $hg, $hop, @realname) =
		split(/\s/, $arg);
	my $ib = $heap->{IrcBuffer};
	my $op = ($hg =~ /@/ ? 1 : 0);
	my $cid = $ib->name2cid($channel);
	# @realname may be iso-2022-jp
	$ib->add_nick($cid, $nick, $op, join(' ', @realname));
}

################################################################
sub irc_mode{
	my ($kernel, $heap, $from, $channel, $mode, @args) =
		@_[KERNEL, HEAP, ARG0, ARG1, ARG2, ARG3 .. $#_];

	my $ib = $heap->{IrcBuffer};
	my $cid = $ib->name2cid($channel);
	if($mode =~ s/^\+//){
		while($mode =~ s/^o//){
			my $nick = shift @args;
			$ib->op_nick($cid, $nick);
		}
	}elsif($mode =~ s/^-//){
		while($mode =~ s/^o//){
			my $nick = shift @args;
			$ib->deop_nick($cid, $nick);
		}
	}
}

################################################################
sub irc_notice{
	my ($kernel, $heap, $who, $channel, $msg) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	my $ib = $heap->{IrcBuffer};

	$who =~ s/!.*//;
	my $cid = $ib->name2cid($channel->[0]);
	Encode::from_to($msg, 'jis', 'euc-jp');
	$ib->add_message($cid, $msg, $who);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_topic{
	my ($kernel, $heap, $who, $channel, $topic) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	my $ib = $heap->{IrcBuffer};
	my $cid = $ib->name2cid($channel);
	$who =~ s/!.*//;
	Encode::from_to($topic, 'jis', 'euc-jp');
	$ib->add_message($cid, "$who set topic: $topic");
	$ib->topic($cid, $topic);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_topicraw{
	my ($kernel, $heap, $raw) = @_[KERNEL, HEAP, ARG1];
	my ($channel, $topic) = split(/ :/, $raw, 2);
	my $ib = $heap->{IrcBuffer};
	my $cid = $ib->name2cid($channel);
	$ib->topic($cid, $topic);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_msg{
	my ($kernel, $heap, $who, $channel, $msg) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};
	my $buddy;

	$who =~ s/!.*//;

	# chop off after the gap to workaround madoka's bug
	$channel =~ s/ .*//;

	if ($who eq $cf->irc_nick()) {
		# XXX: $channel may contain two or more nicks.
		$buddy = $channel->[0];
	} else {
		$buddy = $who;
	}
	$ib->join($buddy); # XXX: join $buddy if not joined
	my $cid = $ib->name2cid($buddy);

	Encode::from_to($msg, 'jis', 'euc-jp');
	$ib->add_message($cid, $msg, $who);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_ctcp_action{
	my ($kernel, $heap, $who, $channel, $msg) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	my $ib = $heap->{IrcBuffer};

	$who =~ s/!.*//;
	my $cid = $ib->name2cid($channel->[0]);
	Encode::from_to($msg, 'jis', 'euc-jp');
	$ib->add_message($cid, sprintf('* %s %s', $who, $msg));
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_reconnect{
	my ($kernel, $heap) = @_[KERNEL, HEAP];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};

	if ($heap->{disconnect_msg}) {
		for my $cid ($ib->channels()){
			$ib->add_message($cid, 'Disconnected from irc server, trying to reconnect...');
		}
	}
	$heap->{disconnect_msg} = 0;
	$kernel->delay(connect => $cf->reconnect_delay());
}

################################################################
sub irc_connect{
	my ($kernel, $heap) = @_[KERNEL, HEAP];
	$kernel->post(keitairc_irc => connect => {});
}

################################################################
sub irc_autoping{
	my ($kernel, $heap) = @_[KERNEL, HEAP];
	my $cf = $heap->{Config};

	$kernel->post(keitairc_irc => time) unless $heap->{seen_traffic};
	$heap->{seen_traffic} = 0;
	$kernel->delay(autoping => $cf->ping_delay());
}

1;
