/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-location-entry-action.h"

#include <gdk/gdkkeysyms.h>

#include "kazehakase.h"
#include "intl.h"
#include "utils/utils.h"
#include "gobject-utils.h"
#include "egg-toolbutton.h"
#include "kz-mozembed.h"


enum {
	PROP_0,
	PROP_KZ_WINDOW,
};


static void kz_location_entry_action_class_init       (KzLocationEntryActionClass *class);
static void kz_location_entry_action_init             (KzLocationEntryAction      *action);
static void kz_location_entry_action_dispose          (GObject *obj);
static void kz_location_entry_action_set_property     (GObject           *object,
						       guint              prop_id,
						       const GValue      *value,
						       GParamSpec        *pspec);
static void kz_location_entry_action_get_property     (GObject           *object,
						       guint              prop_id,
						       GValue            *value,
						       GParamSpec        *pspec);

static void kz_location_entry_action_activate         (EggAction         *action);
static void kz_location_entry_action_connect_proxy    (EggAction         *action,
						       GtkWidget         *proxy);
static void kz_location_entry_action_disconnect_proxy (EggAction         *action,
						       GtkWidget         *proxy);

static void     kz_location_entry_action_history_changed (EggHistoryAction *action);
static void     kz_location_entry_action_sync_history    (KzLocationEntryAction *action);
static gboolean cb_entry_key_press                       (GtkWidget *widget,
							  GdkEventKey *event,
							  KzLocationEntryAction *action);


static EggHistoryActionClass *parent_class = NULL;


KZ_OBJECT_GET_TYPE(kz_location_entry_action,
		   "KzLocationEntryAction",
		   KzLocationEntryAction,
		   kz_location_entry_action_class_init,
		   kz_location_entry_action_init,
		   EGG_TYPE_HISTORY_ACTION)


static void
kz_location_entry_action_class_init (KzLocationEntryActionClass *klass)
{
	GObjectClass *object_class;
	EggActionClass *action_class;
	EggHistoryActionClass *history_class;

	parent_class  = g_type_class_peek_parent(klass);
	object_class  = G_OBJECT_CLASS(klass);
	action_class  = EGG_ACTION_CLASS(klass);
	history_class = EGG_HISTORY_ACTION_CLASS(klass);

	object_class->set_property     = kz_location_entry_action_set_property;
	object_class->get_property     = kz_location_entry_action_get_property;
	object_class->dispose          = kz_location_entry_action_dispose;

	action_class->activate         = kz_location_entry_action_activate;
	action_class->connect_proxy    = kz_location_entry_action_connect_proxy;
	action_class->disconnect_proxy = kz_location_entry_action_disconnect_proxy;

	history_class->history_changed = kz_location_entry_action_history_changed;

	g_object_class_install_property
		(object_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
				      _("KzWindow"),
				      _("The KzWindow to add a home button"),
				      KZ_TYPE_WINDOW,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_location_entry_action_init (KzLocationEntryAction *action)
{
	action->kz = NULL;
	action->activating = FALSE;
	action->synchronizing = FALSE;
	kz_location_entry_action_restore_history(action);
}


static void
kz_location_entry_action_dispose (GObject *obj)
{
	KzLocationEntryAction *action = KZ_LOCATION_ENTRY_ACTION(obj);

	if (action->kz)
	{
		kz_location_entry_action_store_history(action);
		g_object_unref(action->kz);
		action->kz = NULL;
	}

	if (G_OBJECT_CLASS(parent_class)->dispose)
		G_OBJECT_CLASS(parent_class)->dispose(obj);
}


static void
kz_location_entry_action_set_property (GObject         *object,
				       guint            prop_id,
				       const GValue    *value,
				       GParamSpec      *pspec)
{
	KzLocationEntryAction *action = KZ_LOCATION_ENTRY_ACTION(object);
  
	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		action->kz = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_location_entry_action_get_property (GObject         *object,
				       guint            prop_id,
				       GValue          *value,
				       GParamSpec      *pspec)
{
	KzLocationEntryAction *action = KZ_LOCATION_ENTRY_ACTION(object);

	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		g_value_set_object(value, action->kz);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_location_entry_action_activate (EggAction *action)
{
	KzLocationEntryAction *entry_action;
	const gchar *uri;

	g_return_if_fail(KZ_IS_LOCATION_ENTRY_ACTION(action));

	KZ_LOCATION_ENTRY_ACTION(action)->activating = TRUE;
	if (EGG_ACTION_CLASS(parent_class)->activate)
		EGG_ACTION_CLASS(parent_class)->activate(action);
	KZ_LOCATION_ENTRY_ACTION(action)->activating = FALSE;

	entry_action = KZ_LOCATION_ENTRY_ACTION(action);

	uri = egg_entry_action_get_text(EGG_ENTRY_ACTION(action));
	kz_window_load_url(entry_action->kz, uri);
}


static void
kz_location_entry_action_connect_proxy (EggAction *action, GtkWidget *proxy)
{
	GtkEntry *entry;

	EGG_ACTION_CLASS (parent_class)->connect_proxy (action, proxy);

	entry = egg_entry_action_get_entry_widget(EGG_ENTRY_ACTION(action), proxy);
	if (entry)
	{
		g_signal_connect(G_OBJECT(entry), "key-press-event",
				 G_CALLBACK(cb_entry_key_press), action);
	}
}


static void
kz_location_entry_action_disconnect_proxy (EggAction *action, GtkWidget *proxy)
{
	GtkEntry *entry;

	entry = egg_entry_action_get_entry_widget(EGG_ENTRY_ACTION(action), proxy);
	if (entry)
	{
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(entry),
			 G_CALLBACK(cb_entry_key_press),
			 action);
	}

	EGG_ACTION_CLASS (parent_class)->disconnect_proxy (action, proxy);
}


static void
kz_location_entry_action_history_changed (EggHistoryAction *action)
{
	g_return_if_fail(KZ_IS_LOCATION_ENTRY_ACTION(action));

	kz_location_entry_action_sync_history(KZ_LOCATION_ENTRY_ACTION(action));

	if (EGG_HISTORY_ACTION_CLASS(parent_class)->history_changed)
		EGG_HISTORY_ACTION_CLASS(parent_class)->history_changed(action);
}


KzLocationEntryAction *
kz_location_entry_action_new (KzWindow *kz)
{
	KzLocationEntryAction *action;

	action = KZ_LOCATION_ENTRY_ACTION(
			g_object_new(KZ_TYPE_LOCATION_ENTRY_ACTION,
				     "name",      "LocationEntry",
				     "label",     _("Location Entry"),
				     "tooltip",   NULL,
				     "stock_id",  GTK_STOCK_NEW,
				     "kz-window", kz,
				     NULL));
	return action;
}


void
kz_location_entry_action_store_history (KzLocationEntryAction *action)
{
	const GList *list, *node;
	gint i = 0;

	g_return_if_fail(KZ_IS_LOCATION_ENTRY_ACTION(action));

	list = kz_profile_enum_key(kz_global_profile, "LocationEntry");
	for (node = list; node; node = g_list_next(node))
	{
		const gchar *key = node->data;
		if (key && *key && key_seems_sequential(key, "history"))
			kz_profile_delete_key(kz_global_profile,
					      "LocationEntry", key);
	}

	list = egg_history_action_get_history(EGG_HISTORY_ACTION(action));
	for (node = list; node; node = g_list_next((GList *) node))
	{
		const gchar *uri = node->data;
		gchar key[256];

		if (!uri || !*uri) continue;

		g_snprintf(key, G_N_ELEMENTS(key), "history%d", i);

		KZ_CONF_SET("LocationEntry", key, uri, STRING);
		i++;
	}
}


void
kz_location_entry_action_restore_history (KzLocationEntryAction *action)
{
	GList *list, *node, *history = NULL;
	gint max_history = 32;
	gboolean success;
	gchar *text;

	g_return_if_fail(KZ_IS_LOCATION_ENTRY_ACTION(action));

	if (action->activating) return;

	text = g_strdup(egg_entry_action_get_text(EGG_ENTRY_ACTION(action)));

	list = kz_profile_enum_key(kz_global_profile, "LocationEntry");
	for (node = list; node; node = g_list_next(node))
	{
		const gchar *key = node->data;
		gchar *value;

		if (!key_seems_sequential(key, "history")) continue;

		value = KZ_CONF_GET_STR("LocationEntry", key);
		if (value && *value)
			history = g_list_append(history, value);
	}

	success = KZ_CONF_GET("LocationEntry", "max_history",
			      max_history, INT);
	if (success)
		egg_history_action_set_max_history
			(EGG_HISTORY_ACTION(action), max_history);

	if (history)
	{
		egg_history_action_set_history(EGG_HISTORY_ACTION(action), history);
	}

	g_list_foreach(history, (GFunc) g_free, NULL);
	g_list_free(history);

	egg_entry_action_set_text(EGG_ENTRY_ACTION(action), text);
	g_free(text);
}


static void
kz_location_entry_action_sync_history (KzLocationEntryAction *action)
{
	GList *list, *node;

	g_return_if_fail(KZ_IS_LOCATION_ENTRY_ACTION(action));

	if (action->synchronizing) return;
	action->synchronizing = TRUE;

	kz_location_entry_action_store_history
		(KZ_LOCATION_ENTRY_ACTION(action));

	list = kz_window_get_window_list();
	for (node = list; node; node = g_list_next(node))
	{
		KzWindow *tmpkz = node->data;
		EggAction *act;

		if (tmpkz == action->kz) continue;

		act = egg_action_group_get_action(tmpkz->actions,
						  "LocationEntry");
		if (KZ_LOCATION_ENTRY_ACTION(act)->synchronizing) continue;
		kz_location_entry_action_restore_history
			(KZ_LOCATION_ENTRY_ACTION(act));
	}

	action->synchronizing = FALSE;
}


static gboolean
cb_entry_key_press (GtkWidget *widget, GdkEventKey *event,
		    KzLocationEntryAction *action)
{
	if ((event->keyval == GDK_Return || event->keyval == GDK_ISO_Enter) &&
	    (event->state & GDK_CONTROL_MASK))
	{
		const gchar *uri;

		uri = egg_entry_action_get_text(EGG_ENTRY_ACTION(action));
		kz_window_open_new_tab(action->kz, uri);

		return TRUE;
	}

	return FALSE;
}

