/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2007 Kouhei Sutou <kou@cozmixng.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copyED of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdlib.h>
#include <string.h>

#include <gmodule.h>

#include "kz-module.h"
#include "kz-module-impl.h"

#define KZ_MODULE_GET_PRIVATE(obj)					\
	(G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_MODULE, KzModulePrivate))

typedef struct _KzModulePrivate	KzModulePrivate;
struct _KzModulePrivate
{
	GModule      *library;
	gchar        *mod_path;
	GList        *registered_types;
	gboolean      exited;

	KzModuleInitFunc         init;
	KzModuleExitFunc         exit;
	KzModuleInstantiateFunc  instantiate;
	KzModuleGetNameFunc      get_name;
};

G_DEFINE_TYPE (KzModule, kz_module, G_TYPE_TYPE_MODULE)

static void     finalize        (GObject     *object);
static gboolean load            (GTypeModule *module);
static void     unload          (GTypeModule *module);

static void     _kz_module_show_error   (GModule     *module);
static GModule *_kz_module_open         (const gchar *mod_path);
static void     _kz_module_close        (GModule     *module);
static gboolean _kz_module_load_func    (GModule     *module,
                                         const gchar *func_name,
                                         gpointer    *symbol);
static gboolean _kz_module_match_name   (const gchar *mod_path,
                                         const gchar *name);

static void
kz_module_class_init (KzModuleClass *klass)
{
	GObjectClass *gobject_class;
	GTypeModuleClass *type_module_class;

	gobject_class = G_OBJECT_CLASS(klass);
	gobject_class->finalize     = finalize;

	type_module_class = G_TYPE_MODULE_CLASS(klass);
	type_module_class->load     = load;
	type_module_class->unload   = unload;

	g_type_class_add_private(gobject_class, sizeof(KzModulePrivate));
}

static void
kz_module_init (KzModule *module)
{
	KzModulePrivate *priv = KZ_MODULE_GET_PRIVATE(module);

	priv->library          = NULL;
	priv->mod_path         = NULL;
	priv->registered_types = NULL;
	priv->exited           = FALSE;
}

static void
finalize (GObject *object)
{
	KzModulePrivate *priv = KZ_MODULE_GET_PRIVATE(object);

	g_free(priv->mod_path);
	priv->mod_path = NULL;
	g_list_free(priv->registered_types);
	priv->registered_types = NULL;

	G_OBJECT_CLASS(kz_module_parent_class)->finalize(object);
}

static gboolean
load (GTypeModule *module)
{
	KzModulePrivate *priv = KZ_MODULE_GET_PRIVATE(module);

	priv->library = _kz_module_open(priv->mod_path);
	if (!priv->library)
		return FALSE;

	if (!_kz_module_load_func(priv->library,
				  G_STRINGIFY(KZ_MODULE_IMPL_INIT),
				  (gpointer)&priv->init) ||
	    !_kz_module_load_func(priv->library,
				  G_STRINGIFY(KZ_MODULE_IMPL_EXIT),
				  (gpointer)&priv->exit) ||
	    !_kz_module_load_func(priv->library,
				  G_STRINGIFY(KZ_MODULE_IMPL_INSTANTIATE),
				  (gpointer)&priv->instantiate)) {
		_kz_module_close(priv->library);
		priv->library = NULL;
		return FALSE;
	}

	_kz_module_load_func(priv->library,
			     G_STRINGIFY(KZ_MODULE_IMPL_GET_NAME),
			     (gpointer)&priv->get_name);

	priv->exited = FALSE;

	priv->init(module);

	return TRUE;
}

static void
unload (GTypeModule *module)
{
	KzModulePrivate *priv = KZ_MODULE_GET_PRIVATE(module);

	if (!priv->exited)
	{
		priv->exit();
		priv->exited = TRUE;
	}

	_kz_module_close(priv->library);
	priv->library  = NULL;

	priv->init = NULL;
	priv->exit = NULL;
	priv->instantiate = NULL;
	priv->get_name = NULL;
}


static void
_kz_module_show_error (GModule *module)
{
	gchar *message;

	if (!g_module_error()) return;
	message = g_locale_to_utf8(g_module_error(), -1, NULL, NULL, NULL);

	if (module) {
		gchar *name;
		name = g_strdup(g_module_name(module));
		g_warning("%s: %s", name, message);
		g_free(name);
	} else {
		g_warning("%s", message);
	}

	g_free(message);
}

KzModule *
kz_module_find (GList *modules, const gchar *name)
{
	GList *node;

	for (node = modules; node; node = g_list_next(node)) {
		KzModule *module = node->data;
		KzModulePrivate *priv;

		priv = KZ_MODULE_GET_PRIVATE(module);
		if (_kz_module_match_name(priv->mod_path, name))
			return module;
	}

	return NULL;
}

GObject *
kz_module_instantiate (KzModule *module,
		       const gchar *first_property, va_list var_args)
{
	GObject *object = NULL;
	KzModulePrivate *priv;

	priv = KZ_MODULE_GET_PRIVATE(module);
	if (g_type_module_use(G_TYPE_MODULE(module)))
	{
		object = priv->instantiate(first_property, var_args);
		g_type_module_unuse(G_TYPE_MODULE(module));
	}

	return object;
}

const gchar *
kz_module_get_name (KzModule *module)
{
	KzModulePrivate *priv;
	const gchar *name = NULL;

	priv = KZ_MODULE_GET_PRIVATE(module);
	if (priv->get_name && g_type_module_use(G_TYPE_MODULE(module)))
	{
		name = priv->get_name();
		g_type_module_unuse(G_TYPE_MODULE(module));
	}

	if (!name)
		name = G_TYPE_MODULE(module)->name;

	return name;
}

static GModule *
_kz_module_open (const gchar *mod_path)
{
	GModule *module;

	module = g_module_open(mod_path, G_MODULE_BIND_LAZY);
	if (!module) {
		_kz_module_show_error(NULL);
	}

	return module;
}

static void
_kz_module_close (GModule *module)
{
	if (!module)
		return;

	if (!g_module_close(module))
	    _kz_module_show_error(module);
}

static gchar *
_kz_module_module_file_name (const gchar *name)
{
	return g_strconcat(name, "." G_MODULE_SUFFIX, NULL);
}

static gboolean
_kz_module_load_func (GModule *module, const gchar *func_name,
		      gpointer *symbol)
{
	g_return_val_if_fail(module, FALSE);

	if (g_module_symbol(module, func_name, symbol)) {
		return TRUE;
	} else {
		_kz_module_show_error(module);
		return FALSE;
	}
}

KzModule *
kz_module_load_module (const gchar *base_dir, const gchar *name)
{
	gchar *mod_base_name, *mod_path;
	KzModule *module = NULL;

	mod_base_name = g_build_filename(base_dir, name, NULL);
	if (g_str_has_suffix(mod_base_name, G_MODULE_SUFFIX)) {
		mod_path = mod_base_name;
	} else {
		mod_path = _kz_module_module_file_name(mod_base_name);
		g_free(mod_base_name);
	}

	if (g_file_test(mod_path, G_FILE_TEST_EXISTS)) {
		KzModulePrivate *priv;
		gchar *mod_name;

		module = g_object_new(KZ_TYPE_MODULE, NULL);
		priv = KZ_MODULE_GET_PRIVATE(module);
		priv->mod_path = g_strdup(mod_path);

		mod_name = g_strdup(name);
		if (g_str_has_suffix(mod_name, "."G_MODULE_SUFFIX)) {
			guint last_index;
			last_index =
				strlen(mod_name) - strlen("."G_MODULE_SUFFIX);
			mod_name[last_index] = '\0';
		}
		g_type_module_set_name(G_TYPE_MODULE(module), mod_name);
		g_free(mod_name);
	}
	g_free(mod_path);

	return module;
}

GList *
kz_module_load_modules (const gchar *base_dir)
{
	return kz_module_load_modules_unique(base_dir, NULL);
}

GList *
kz_module_load_modules_unique (const gchar *base_dir, GList *exist_modules)
{
	GDir *dir;
	GList *modules = NULL;
	const gchar *entry;

	dir = g_dir_open(base_dir, 0, NULL);
	if (!dir)
		return modules;

	while ((entry = g_dir_read_name(dir))) {
		KzModule *module;

		module = kz_module_load_module(base_dir, entry);
		if (module)
		{
			GTypeModule *g_module;

			g_module = G_TYPE_MODULE(module);
			if (kz_module_find(exist_modules, g_module->name))
				kz_module_unload(module);
			else
				modules = g_list_prepend(modules, module);
		}
	}
	g_dir_close(dir);

	return modules;
}


static gboolean
_kz_module_match_name (const gchar *mod_path, const gchar *name)
{
	gboolean matched;
	gchar *module_base_name, *normalized_matched_name;

	module_base_name = g_path_get_basename(mod_path);
	normalized_matched_name = _kz_module_module_file_name(name);

	matched = (0 == strcmp(module_base_name, normalized_matched_name));

	g_free(module_base_name);
	g_free(normalized_matched_name);

	return matched;
}

void
kz_module_exit (KzModule *module)
{
	KzModulePrivate *priv;

	g_return_if_fail(KZ_IS_MODULE(module));

	priv = KZ_MODULE_GET_PRIVATE(module);
	if (priv->exited)
		return;

	if (g_type_module_use(G_TYPE_MODULE(module))) {
		priv->exit();
		priv->exited = TRUE;
		g_type_module_unuse(G_TYPE_MODULE(module));
	}

	g_type_module_unuse(G_TYPE_MODULE(module));
}

void
kz_module_unload (KzModule *module)
{
	GTypeModule *type_module;

	g_return_if_fail(KZ_IS_MODULE(module));

	type_module = G_TYPE_MODULE(module);
	if (type_module->type_infos || type_module->interface_infos)
		return;

	g_object_unref(module);
}
