/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2002 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-window.h"

#include <stdio.h>
#include <gtk/gtk.h>
#include <gtkmozembed.h>

#include "gobject-utils.h"
#include "gtk-utils.h"
#include "kz-bookmarkbar.h"
#include "kz-mozembed.h"
#include "kz-tablabel.h"
#include "kz-home.h"
#include "kazehakase.h"
#include "mozilla.h"
#include "prefs.h"

static void     kz_window_class_init     (KzWindowClass *class);
static void     kz_window_init           (KzWindow *kz);
static gboolean kz_window_delete_event   (GtkWidget *widget,
					  GdkEventAny *event);
static void     kz_window_destroy        (GtkObject *object);

static void     kz_window_update_nav_buttons (KzWindow *kz,
					      KzMozEmbed *kzembed);
static GtkWidget *create_toolbar         (KzWindow *kz);

/* callbacks */
static void     back_clicked_cb          (GtkButton   *button, KzWindow *kz);
static void     stop_clicked_cb          (GtkButton   *button, KzWindow *kz);
static void     forward_clicked_cb       (GtkButton   *button, KzWindow *kz);
static void     reload_clicked_cb        (GtkButton   *button, KzWindow *kz);
static void     url_activate_cb          (GtkEntry    *widget, KzWindow *kz);
static void     notebook_switch_page_cb  (GtkNotebook *notebook,
					  GtkNotebookPage *page,
					  guint page_num, KzWindow *kz);
static void     find_keyword_cb          (GtkWidget *widget, KzWindow *kz);

/* callbacks for mozilla */
static void     title_changed_cb         (GtkMozEmbed *embed, KzWindow *kz);
static void     location_changed_cb      (GtkMozEmbed *embed, KzWindow *kz);
static void     link_message_cb          (GtkMozEmbed *embed, KzWindow *kz);
static void     visibility_cb            (GtkMozEmbed *embed,
					  gboolean visibility,
					  KzWindow *kz);
static void     load_started_cb          (GtkMozEmbed *embed, KzWindow *kz);
static void     load_finished_cb         (GtkMozEmbed *embed, KzWindow *kz);
static void     new_window_cb            (GtkMozEmbed *embed,
					  GtkMozEmbed **newEmbed,
					  guint chromemask,
					  KzWindow *kz);
static void     close_tab_cb             (GtkObject *obj, KzWindow *kz);
static gint     dom_mouse_click_cb       (GtkMozEmbed *embed, gpointer event,
					  KzWindow *kz);
static gint     dom_mouse_dbl_click_cb   (GtkMozEmbed *embed, gpointer event,
					  KzWindow *kz);
static gint     dom_mouse_down_cb       (GtkMozEmbed *embed,
					 gpointer event,
					 KzWindow *kz);

/*
 * mozilla doesn't accept these signals, so we connect these funtions to
 * KzWindow instead of KzMozEmbed.
 */
static gboolean motion_notify_event_cb  (GtkWidget *widget,
					 GdkEventMotion *event,
					 KzMozEmbed *kzembed);
static gboolean button_release_event_cb (GtkWidget *widget,
					 GdkEventButton *event,
					 KzMozEmbed *kzembed);

#warning FIXME!!
/* action */
static void kz_window_go_back                (KzWindow *kz);
static void kz_window_go_forward             (KzWindow *kz);
static void kz_window_close_current_tab_idle (KzWindow *kz);
static void kz_window_new_reload             (KzWindow *kz);
/*static void kz_window_home                   (KzWindow *kz);*/
KzGestureMotion gesture_go_back[]    = { KZ_GESTURE_MOTION_LEFT,  KZ_GESTURE_MOTION_TERMINATOR };
KzGestureMotion gesture_go_forward[] = { KZ_GESTURE_MOTION_RIGHT, KZ_GESTURE_MOTION_TERMINATOR };
KzGestureMotion gesture_close_tab[]  = { KZ_GESTURE_MOTION_UP,    KZ_GESTURE_MOTION_TERMINATOR };
KzGestureMotion gesture_new_reload[] = { KZ_GESTURE_MOTION_DOWN,  KZ_GESTURE_MOTION_TERMINATOR };
KzGestureMotion gesture_home[]       = { KZ_GESTURE_MOTION_DOWN,  KZ_GESTURE_MOTION_UP, KZ_GESTURE_MOTION_TERMINATOR };
static KzGestureDefTable gesture_table[] = {
	{"Back",      gesture_go_back,    (GFunc) kz_window_go_back,                NULL},
	{"Forward",   gesture_go_forward, (GFunc) kz_window_go_forward,             NULL},
	{"Close tab", gesture_close_tab,  (GFunc) kz_window_close_current_tab_idle, NULL},
	{"Reload",    gesture_new_reload, (GFunc) kz_window_new_reload,             NULL},
/*	{"Home",      gesture_home,       (GFunc) kz_window_home,                   NULL},*/
};
static gint gesture_table_rows = sizeof(gesture_table) / sizeof(KzGestureDefTable);

static GtkWindowClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(kz_window, "KzWindow", KzWindow,
		   kz_window_class_init, kz_window_init,
		   GTK_TYPE_WINDOW)

static void
kz_window_class_init (KzWindowClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (class);

	object_class = (GtkObjectClass *) class;
	widget_class = (GtkWidgetClass *) class;

	/* GtkObject signals */
	object_class->destroy = kz_window_destroy;

	/* GtkWidget signals */
	widget_class->delete_event = kz_window_delete_event;
}

static void
kz_window_init (KzWindow *kz)
{
	kz->topLevelVBox       = NULL;
	kz->menuBar            = NULL;
	kz->fileMenuItem       = NULL;
	kz->fileMenu           = NULL;
	kz->fileClose          = NULL;
	kz->fileQuit           = NULL;
	kz->toolbarHBox        = NULL;
	kz->toolbar            = NULL;
	kz->backButton         = NULL;
	kz->stopButton         = NULL;
	kz->forwardButton      = NULL;
	kz->reloadButton       = NULL;
	kz->homeButton         = NULL;
        kz->urlEntry           = NULL;
	kz->progressAreaHBox   = NULL;
        kz->progressBar        = NULL;
	kz->statusAlign        = NULL;
	kz->statusBar          = NULL;
	kz->notebook           = NULL;
	kz->findAreaHBox       = NULL;
	kz->findArea           = NULL;
	kz->kztoolBar          = NULL; /* Kazehakase tool bar */

	kz->statusMessage      = NULL;
	kz->tempMessage        = NULL;

	kz->menuBarOn          = TRUE;
	kz->toolBarOn          = TRUE;
	kz->locationBarOn      = TRUE;
	kz->statusBarOn        = TRUE;
	kz->didFind            = FALSE; /* for find keyword */
}

GtkWidget *
kz_window_new(const gchar *url)
{
	KzWindow *kz = KZ_WINDOW(g_object_new(KZ_TYPE_WINDOW, NULL));
	GtkWidget *hbox, *toolbar;

	hbox = gtk_hbox_new(FALSE, 0);

	GTK_WINDOW(kz)->type = GTK_WINDOW_TOPLEVEL;

	/* new vbox */
	kz->topLevelVBox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(kz),
			  kz->topLevelVBox);
	gtk_widget_show(kz->topLevelVBox);

	/* new horiz toolbar with buttons + icons */
	kz->toolbar = create_toolbar (kz);
	gtk_box_pack_start(GTK_BOX(hbox), kz->toolbar,
			   FALSE,
			   FALSE,
			   0);   
	gtk_widget_show(kz->toolbar);

	/* create the url text entry */
	kz->urlEntry = gtk_combo_new();
	gtk_combo_disable_activate(GTK_COMBO(kz->urlEntry));
	toolbar = gtk_toolbar_new();
	gtk_toolbar_set_orientation(GTK_TOOLBAR(toolbar),
				    GTK_ORIENTATION_HORIZONTAL);
	gtk_widget_show(toolbar);
	gtk_box_pack_start(GTK_BOX(hbox), kz->urlEntry,
			   TRUE, 
			   TRUE, 
			   0);
	gtk_widget_show(kz->urlEntry);

	/* hook up the activate signal to the right callback */
	g_signal_connect(G_OBJECT(GTK_COMBO(kz->urlEntry)->entry), "activate",
			 G_CALLBACK(url_activate_cb), kz);

	/* add that hbox to the vbox */
	gtk_box_pack_start(GTK_BOX(kz->topLevelVBox), 
			   hbox,
			   FALSE, 
			   FALSE, 
			   0);    
	gtk_widget_show(hbox);

	/* KzWindow tool bar */
	kz->kztoolBar = GTK_WIDGET(kz_bookmark_bar_new(kz));
	gtk_box_pack_start(GTK_BOX(kz->topLevelVBox), 
			   kz->kztoolBar,
			   FALSE, 
			   FALSE, 
			   0);    
	gtk_widget_show(kz->kztoolBar);

	/* create notebook widget */
	kz->notebook = gtk_notebook_new();
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(kz->notebook), TRUE);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(kz->notebook), TRUE);
	g_signal_connect(G_OBJECT(kz->notebook), "switch_page", 
			 G_CALLBACK(notebook_switch_page_cb), kz);

	gtk_container_add(GTK_CONTAINER(kz->topLevelVBox), kz->notebook);
	gtk_widget_show(kz->notebook);

	/* create new mozembed tab */
	kz_window_open_new_tab(kz, url);

	/* create the new hbox for the search area */
	kz->findAreaHBox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(kz->topLevelVBox), kz->findAreaHBox,
			   FALSE, 
			   FALSE, 
			   0);   
	gtk_widget_show(kz->findAreaHBox);

	/* create our new progress bar */
	kz->findArea = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(kz->findAreaHBox), kz->findArea,
			   FALSE,
			   FALSE,
			   0); 
	g_signal_connect(G_OBJECT(GTK_ENTRY(kz->findArea)), "activate",
			 G_CALLBACK(find_keyword_cb), kz);
	gtk_widget_show(kz->findArea);

	/* create our status area and the alignment object that will keep it */
	/* from expanding */
	kz->statusAlign = gtk_alignment_new(0, 0, 1, 1);
	gtk_widget_set_size_request(kz->statusAlign, 1, -1);
	gtk_widget_show(kz->statusAlign);
	/* create the status bar */
	kz->statusBar = gtk_statusbar_new();
	gtk_container_add(GTK_CONTAINER(kz->statusAlign), kz->statusBar);
	gtk_box_pack_start(GTK_BOX(kz->findAreaHBox), kz->statusAlign,
			   TRUE, 
			   TRUE, 
			   0);
	gtk_widget_show(kz->statusBar);
	/* by default none of the buttons are marked as sensitive. */
	gtk_widget_set_sensitive(kz->backButton, FALSE);
/*	gtk_widget_set_sensitive(kz->stopButton, FALSE); */
	gtk_widget_set_sensitive(kz->forwardButton, FALSE);
/*	gtk_widget_set_sensitive(kz->reloadButton, FALSE); */

	/* FIX ME! set last window size! */ 
	gtk_window_set_default_size(GTK_WINDOW(kz), 
				    prefs.width, prefs.height);

	gtk_widget_show(GTK_WIDGET(kz));

	return GTK_WIDGET (kz);
}

static void
cb_gesture_stack_motion(KzGesture *gesture, KzGestureMotion motion, KzWindow *kz)
{
	const gchar *label;
	gchar buf1[256], buf2[256];

	g_return_if_fail(KZ_IS_GESTURE(gesture));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_gesture_create_gesture_string(gesture, buf1,
					 sizeof(buf1) / sizeof(gchar));
	label = kz_gesture_get_matched_label(gesture);
	if (label)
		g_snprintf(buf2, sizeof(buf2) / sizeof(gchar),
			   "Gesture: %s(Action: %s)", buf1, label);
	else
		g_snprintf(buf2, sizeof(buf2) / sizeof(gchar),
			   "Gesture: %s", buf1);


	gtk_statusbar_push(GTK_STATUSBAR(kz->statusBar), 1, buf2);
}

GtkWidget *
kz_window_open_new_tab (KzWindow *kz, const gchar *url)
{
	KzMozEmbed *kzembed;
	KzTabLabel *kztab;
	KzGestureDefTable *table;
	gint i;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);

	kzembed = KZ_MOZ_EMBED(kz_moz_embed_new(kz, url));
	kztab = KZ_TAB_LABEL(kz_tab_label_new(kz, kzembed));

#if 1 /* FIXME */
	table = g_memdup(gesture_table, sizeof(gesture_table));
	for (i = 0; i < gesture_table_rows; i++)
	{
		table[i].data = kz;
	}
	kzembed->gesture->table      = table;
	kzembed->gesture->table_rows = gesture_table_rows;
	g_signal_connect(G_OBJECT(kzembed->gesture), "stack_motion",
			 G_CALLBACK(cb_gesture_stack_motion), kz);
#endif /* FIXME */

	g_signal_connect(G_OBJECT(kzembed), "title",
			 G_CALLBACK(title_changed_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "location",
			 G_CALLBACK(location_changed_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "link_message",
			 G_CALLBACK(link_message_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "net_start",
			 G_CALLBACK(load_started_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "net_stop",
			 G_CALLBACK(load_finished_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "new_window",
			 G_CALLBACK(new_window_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "dom_mouse_click",
			 G_CALLBACK(dom_mouse_click_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "dom_mouse_dbl_click",
			 G_CALLBACK(dom_mouse_dbl_click_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "destroy",
			 G_CALLBACK(close_tab_cb), kz);

	/* mouse event signal */
	g_signal_connect(G_OBJECT(kzembed),
			 "dom_mouse_down",
			 G_CALLBACK(dom_mouse_down_cb), kz);
	/* FIXME!!: these callback should be disconnected when kzembed is destroyed */
	g_signal_connect(G_OBJECT(kz),
			 "motion_notify_event",
			 G_CALLBACK(motion_notify_event_cb), kzembed);
	g_signal_connect(G_OBJECT(kz),
			 "button_release_event",
			 G_CALLBACK(button_release_event_cb), kzembed);

	gtk_widget_show(GTK_WIDGET(kzembed));
	gtk_widget_show(GTK_WIDGET(kztab));

	gtk_notebook_append_page(GTK_NOTEBOOK(kz->notebook),
				 GTK_WIDGET(kzembed),
				 GTK_WIDGET (kztab));

	return GTK_WIDGET(kzembed);
}

void
kz_window_close_tab (KzWindow *kz, KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	g_signal_handlers_disconnect_by_func(kz, motion_notify_event_cb, kzembed);
	g_signal_handlers_disconnect_by_func(kz, button_release_event_cb, kzembed);

	gtk_widget_destroy(GTK_WIDGET(kzembed));
}

static gboolean
idle_close_tab (gpointer data)
{
	KzMozEmbed *kzembed = data;
	kz_window_close_tab(kzembed->parent_kazehakase, kzembed);
	return FALSE;
}

void
kz_window_close_tab_idle (KzWindow *kz, KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	gtk_idle_add(idle_close_tab, kzembed);
}

static gboolean
kz_window_delete_event (GtkWidget *widget, GdkEventAny *event)
{
	KzWindow *kz;
	gint n_pages, i;

	g_return_val_if_fail(KZ_IS_WINDOW(widget), FALSE);

	kz = KZ_WINDOW(widget);

	n_pages = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	for (i = n_pages - 1; i >= 0; i--) {
		GtkWidget *widget = GTK_WIDGET(KZ_WINDOW_NTH_PAGE(kz, i));
		gtk_widget_destroy(widget);
	}

	return FALSE;
}

static void
kz_window_destroy (GtkObject *object)
{
	KzWindow *kz = KZ_WINDOW(object);

	g_return_if_fail (KZ_WINDOW(kz));

	g_free(kz->tempMessage);
        kz->tempMessage = NULL;

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy(object);

	gtk_main_quit();
}

static void
kz_window_update_nav_buttons (KzWindow *kz, KzMozEmbed *kzembed)
{
	GtkMozEmbed *embed;
	gboolean can_go_back;
	gboolean can_go_forward;
	gboolean can_stop;

	g_return_if_fail (kz);
	g_return_if_fail (KZ_IS_MOZ_EMBED (kzembed));

	embed = GTK_MOZ_EMBED(kzembed);

	can_go_back = gtk_moz_embed_can_go_back(embed);
	can_go_forward = gtk_moz_embed_can_go_forward(embed);
	can_stop = kz_moz_embed_is_loading(KZ_MOZ_EMBED(embed));

	gtk_widget_set_sensitive(kz->backButton, can_go_back);
	gtk_widget_set_sensitive(kz->forwardButton, can_go_forward);
	if (can_stop) {
		gtk_widget_show(kz->stopButton);
		gtk_widget_hide(kz->reloadButton);
		gtk_widget_set_sensitive(kz->stopButton, TRUE);
		gtk_widget_set_sensitive(kz->reloadButton, FALSE);
	} else {
		gtk_widget_hide(kz->stopButton);
		gtk_widget_show(kz->reloadButton);
		gtk_widget_set_sensitive(kz->stopButton, FALSE);
		gtk_widget_set_sensitive(kz->reloadButton, TRUE);
	}
}


static GtkWidget *
create_toolbar (KzWindow *kz)
{
	GtkWidget *toolbar;

	toolbar = gtk_toolbar_new();
	gtk_toolbar_set_orientation(GTK_TOOLBAR(toolbar),
				    GTK_ORIENTATION_HORIZONTAL);
	gtk_toolbar_set_style(GTK_TOOLBAR(toolbar),
			      GTK_TOOLBAR_ICONS);
	gtk_toolbar_set_icon_size(GTK_TOOLBAR(toolbar),
				  GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_toolbar_set_tooltips(GTK_TOOLBAR(toolbar), TRUE);

	/* new back button */
	kz->backButton =
		gtk_toolbar_insert_stock(GTK_TOOLBAR(toolbar),
					 GTK_STOCK_GO_BACK,
					 "Go Back",
					 "Go Back",
					 G_CALLBACK(back_clicked_cb),
					 kz,1);

	/* new forward button */
	kz->forwardButton =
		gtk_toolbar_insert_stock(GTK_TOOLBAR(toolbar),
					 GTK_STOCK_GO_FORWARD,
					 "Forward",
					 "Forward",
					 G_CALLBACK(forward_clicked_cb),
					 kz, 2);
	/* new stop button */
	kz->stopButton = 
		gtk_toolbar_insert_stock(GTK_TOOLBAR(toolbar),
					 GTK_STOCK_STOP,
					 "Stop",
					 "Stop",
					 G_CALLBACK(stop_clicked_cb),
					 kz, 3);
	/* new reload button */
	kz->reloadButton = 
		gtk_toolbar_insert_stock(GTK_TOOLBAR(toolbar),
					 GTK_STOCK_REFRESH,
					 "Reload",
					 "Reload",
					 G_CALLBACK(reload_clicked_cb),
					 kz, 4);

	/* new home button */
	kz->homeButton = kz_home_button_new(kz);
	gtk_toolbar_insert_widget(GTK_TOOLBAR(toolbar),
				  kz->homeButton,
				 "Home",
				 "Home",
				  5);
	return toolbar;
}

void
kz_window_set_visibility (KzWindow *kz, gboolean visibility)
{
	if (!visibility)
	{
		gtk_widget_hide(GTK_WIDGET (kz));
		return;
	}

#if 0
	/*
	 * since they are on the same line here...
	 */
	if (kz->toolBarOn || kz->locationBarOn)
		gtk_widget_show(kz->toolbarHBox);
	else 
		gtk_widget_hide_all(kz->toolbarHBox);
#endif

	if (kz->statusBarOn)
	  gtk_widget_show(kz->findAreaHBox);
	else
	  gtk_widget_hide(kz->findAreaHBox);
}

void
kz_window_load_url (KzWindow *kz, const gchar *url)
{
	gtk_moz_embed_load_url(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)), url);
}

gchar *
kz_window_get_title (KzWindow *kz)
{
	return mozilla_get_title(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)));
}

gchar *
kz_window_get_uri (KzWindow *kz)
{
	return gtk_moz_embed_get_location(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)));
}

static void
back_clicked_cb (GtkButton *button, KzWindow *kz)
{
	gtk_moz_embed_go_back(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)));
}

static void 
stop_clicked_cb (GtkButton *button, KzWindow *kz)
{
	gtk_moz_embed_stop_load(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)));
}

static void
forward_clicked_cb (GtkButton *button, KzWindow *kz)
{
	gtk_moz_embed_go_forward(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)));
}

static void
reload_clicked_cb  (GtkButton *button, KzWindow *kz)
{
	GdkModifierType state = (GdkModifierType)0;
	gint x, y;
	gdk_window_get_pointer(NULL, &x, &y, &state);

	gtk_moz_embed_reload(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)),
			     (state & GDK_SHIFT_MASK) ?
			     GTK_MOZ_EMBED_FLAG_RELOADBYPASSCACHE : 
			     GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}

static void
url_activate_cb    (GtkEntry *widget, KzWindow *kz)
{
	const gchar *text = gtk_entry_get_text(GTK_ENTRY(widget));

	gtk_moz_embed_load_url(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)), text);
}

static void
notebook_switch_page_cb (GtkNotebook *notebook, GtkNotebookPage *page,
			 guint page_num, KzWindow *kz)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(KZ_WINDOW_NTH_PAGE(kz, page_num));
	const gchar *newTitle, *newLocation;

	newTitle = kz_moz_embed_get_title(kzembed);
	newLocation = kz_moz_embed_get_location(kzembed);
	if (newTitle)
	{
		gtk_window_set_title(GTK_WINDOW(kz), newTitle);
	}
	if (newLocation)
	{
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(kz->urlEntry)->entry),
				   newLocation);
	}
	kz_window_update_nav_buttons(kz, kzembed);
}

static void
find_keyword_cb(GtkWidget *widget, KzWindow *kz)
{
	GtkMozEmbed *embed = GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz));
	const gchar *text = gtk_entry_get_text(GTK_ENTRY(widget));

	gtk_widget_grab_focus(GTK_BIN(embed)->child);
	mozilla_find(embed, text, &kz->didFind);
	g_print("find=%d, page=%s\n",
		kz->didFind,
		kz_moz_embed_get_title(KZ_MOZ_EMBED(embed)));
}

static void
title_changed_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (embed == GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz))) {
		gchar *title = (gchar *) kz_moz_embed_get_title(KZ_MOZ_EMBED(embed));
		gtk_window_set_title(GTK_WINDOW(kz), title);
	}
}

static void
location_changed_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	const char *newLocation;

	g_return_if_fail (KZ_IS_MOZ_EMBED(embed));
	g_return_if_fail (KZ_IS_WINDOW(kz));

	newLocation = kz_moz_embed_get_location(KZ_MOZ_EMBED(embed));
	if (newLocation)
	{
		if (embed == GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)))
			gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(kz->urlEntry)->entry),
					   newLocation);
	}
	else
	{
		g_print("failed to get location!\n");
	}

	kz_window_update_nav_buttons(kz, KZ_MOZ_EMBED(embed));
}

static void
link_message_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	gchar *newMessage;

	g_return_if_fail (KZ_IS_MOZ_EMBED(embed));
	g_return_if_fail (KZ_WINDOW(kz));

	newMessage = mozilla_get_link_message(embed);
	if(kz->statusBar)
		gtk_statusbar_push(GTK_STATUSBAR(kz->statusBar), 1, newMessage);

#if 0
	if (KZ_MOZ_EMBED(embed)->popup_window)
	{
		/* destroy popup window*/
		gtk_widget_destroy(kzembed->popup_window);
		kzembed->popup_window = NULL;
	}
	else if (newMessage != NULL)
	{
		/* fetch image */ 
		/* create popup window */
		kzembed->popup_window = gtk_window_new(GTK_WINDOW_POPUP);
		gtk_window_set_transient_for(GTK_WINDOW(kzembed->popup_window), 
					     GTK_WINDOW(kz->topLevelWindow));
		gtk_widget_show(kzembed->popup_window);
	}
#endif

	g_free(newMessage);
}

static void
visibility_cb (GtkMozEmbed *embed, gboolean visibility, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	kz_window_set_visibility(kz, visibility);
}

static void
load_started_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	if (embed != GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz))) return;

	gtk_widget_show(kz->stopButton);
	gtk_widget_hide(kz->reloadButton);
	gtk_widget_set_sensitive(kz->stopButton, TRUE);
	gtk_widget_set_sensitive(kz->reloadButton, FALSE);
}

static void
load_finished_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	if (embed != GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz))) return;

	gtk_widget_hide(kz->stopButton);
	gtk_widget_show(kz->reloadButton);
	gtk_widget_set_sensitive(kz->stopButton, FALSE);
	gtk_widget_set_sensitive(kz->reloadButton, TRUE);
}

static void 
destroy_browser_cb (GtkMozEmbed *embed, GtkWidget *transient_window)
{
	gtk_widget_destroy(GTK_WIDGET(transient_window));
}

static void
new_window_cb (GtkMozEmbed *embed,
	       GtkMozEmbed **newEmbed, guint chromemask,
	       KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	if ((chromemask & GTK_MOZ_EMBED_FLAG_OPENASCHROME) != 0)
	{
		/* FIXME! this is ad hoc. */
		GtkWidget *newWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);

		gtk_window_set_transient_for(GTK_WINDOW(newWindow), GTK_WINDOW(kz));
		*newEmbed = GTK_MOZ_EMBED(kz_moz_embed_new(NULL, NULL));
		g_signal_connect(G_OBJECT(*newEmbed),"destroy_browser",
				 G_CALLBACK(destroy_browser_cb),newWindow);
		gtk_container_add(GTK_CONTAINER(newWindow),GTK_WIDGET(*newEmbed));
		gtk_widget_show_all(newWindow);
	}
	else
	{
		*newEmbed = GTK_MOZ_EMBED(kz_window_open_new_tab(kz, NULL));
		gtk_widget_show (GTK_WIDGET(*newEmbed));
	}
}

static void
close_tab_cb (GtkObject *obj, KzWindow *kz)
{
	KzMozEmbed *kzembed;
	gint n, n_pages;

	g_return_if_fail(KZ_IS_MOZ_EMBED(obj));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kzembed = KZ_MOZ_EMBED(obj);
	n = gtk_notebook_page_num(GTK_NOTEBOOK(kz->notebook),
				  GTK_WIDGET(kzembed));
	n_pages = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	if (n_pages <= 0)
	{
		gtk_moz_embed_pop_startup();
		gtk_widget_destroy(GTK_WIDGET(kz));
	}
	else if (n_pages >= n)
	{
		gtk_notebook_set_current_page(GTK_NOTEBOOK(kz->notebook), n);
	}
}

#warning FIXME
static gint
dom_mouse_click_cb (GtkMozEmbed *embed, gpointer event, KzWindow *kz)
{
	gint button;
	guint32 time;
     	glong type;
     	MouseEventInfo info;

	g_return_val_if_fail(KZ_WINDOW(kz), FALSE);

	time = GDK_CURRENT_TIME;
	
     	type = mozilla_get_mouse_event_info(embed, event, &info);
	button = mozilla_get_button(event);

	switch (button) {
	 case 0:
		 if ((info.modifier & SHIFT_KEY) && (type & CONTEXT_LINK) && info.cinfo.link)
		 {
			 kz_window_open_new_tab(kz, info.cinfo.link);
			 return TRUE;
		 }
		 break;
	 case 1:
		 if ((type & CONTEXT_LINK) && info.cinfo.link) {
			 kz_window_open_new_tab(kz, info.cinfo.link);
		 }
		 break;
	}

	return FALSE;
}

static gint
dom_mouse_dbl_click_cb (GtkMozEmbed *embed, gpointer event, KzWindow *kz)
{
	gint button;

	g_return_val_if_fail(KZ_WINDOW(kz), FALSE);

	button = mozilla_get_button(event);
	switch (button) {
	 case 0:
		 gtk_notebook_prev_page(GTK_NOTEBOOK(kz->notebook));
		 break;
	 case 2:	
		 gtk_notebook_next_page(GTK_NOTEBOOK(kz->notebook));
		 break;
	}

	return FALSE;
}

static gint
dom_mouse_down_cb (GtkMozEmbed *embed, gpointer event, KzWindow *kz)
{
	KzMozEmbed *kzembed;
	gint button;
     	glong type;
     	MouseEventInfo info;

	g_return_val_if_fail(KZ_IS_MOZ_EMBED(embed), FALSE);

	kzembed = KZ_MOZ_EMBED(embed);

     	type = mozilla_get_mouse_event_info(embed, event, &info);	
	button = mozilla_get_button(event);
	
	if (button == 2) 
	{
		static GdkCursor *cursor = NULL;
		glong x_pos, y_pos;
		gint x, y, win_x, win_y, win_x_pos, win_y_pos;

 		mozilla_get_mouse_position(event, &x_pos, &y_pos);
		gdk_window_get_root_origin (GTK_WIDGET(kzembed)->window,
					    &win_x, &win_y);
		gdk_window_get_position (GTK_WIDGET(kzembed)->window,
					 &win_x_pos, &win_y_pos);

		x = win_x + win_x_pos + x_pos;
		y = win_y + win_y_pos + y_pos;

		kz_gesture_start(kzembed->gesture, 0, x, y);

		if ((type & CONTEXT_LINK) && info.cinfo.link)
			kzembed->gesture_location = g_strdup(info.cinfo.link);

		if (!cursor) cursor = gdk_cursor_new (GDK_HAND1);
		gdk_pointer_grab (GTK_WIDGET(kzembed->parent_kazehakase)->window, FALSE,
				  GDK_POINTER_MOTION_MASK |
				  GDK_BUTTON_RELEASE_MASK |
				  GDK_BUTTON_PRESS_MASK,
				  NULL, cursor, gtk_get_current_event_time ());
		gtk_statusbar_push(GTK_STATUSBAR(kz->statusBar), 1, "Gesture:");
 	}
	
	return FALSE;
}

static gboolean
motion_notify_event_cb (GtkWidget *widget, GdkEventMotion *event, KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);

	if (kz_gesture_is_started(kzembed->gesture)) {
		gint win_x, win_y, win_x_pos, win_y_pos, x, y;

		gdk_window_get_root_origin (GTK_WIDGET(kzembed)->window,
					    &win_x, &win_y);
		gdk_window_get_position (GTK_WIDGET(kzembed)->window,
					 &win_x_pos, &win_y_pos);
		x = win_x + event->x;
		y = win_y + event->y;

		kz_gesture_update_position(kzembed->gesture, x, y);
	}

	return FALSE;
}

static gboolean
button_release_event_cb (GtkWidget *widget, GdkEventButton *event, KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);

	if (kz_gesture_is_started(kzembed->gesture))
	{
		if (event->button == 3)
			kz_gesture_perform(kzembed->gesture);
		else
			kz_gesture_cancel(kzembed->gesture);
		gtk_statusbar_push(GTK_STATUSBAR(KZ_WINDOW(widget)->statusBar), 1, "");
	}

        if (gdk_pointer_is_grabbed ())
        {
                gdk_pointer_ungrab (gtk_get_current_event_time ());
        }

	g_free(kzembed->gesture_location);
	kzembed->gesture_location = NULL;

	return FALSE;
}





#warning FIXME!!
static void
kz_window_go_back (KzWindow *kz)
{
	KzMozEmbed *kzembed = KZ_WINDOW_CURRENT_PAGE(kz);
	if (kzembed)
		gtk_moz_embed_go_back(GTK_MOZ_EMBED(kzembed));
}
static void
kz_window_go_forward (KzWindow *kz)
{
	KzMozEmbed *kzembed = KZ_WINDOW_CURRENT_PAGE(kz);
	if (kzembed)
		gtk_moz_embed_go_forward(GTK_MOZ_EMBED(kzembed));
}
static void
kz_window_close_current_tab_idle (KzWindow *kz)
{
	KzMozEmbed *kzembed = KZ_WINDOW_CURRENT_PAGE(kz);
	if (kzembed)
		kz_window_close_tab_idle(kz, kzembed);
}
static void
kz_window_new_reload (KzWindow *kz)
{
	KzMozEmbed *kzembed = KZ_WINDOW_CURRENT_PAGE(kz);

	if (!kzembed) return;

	if (kzembed->gesture_location)
		kz_window_open_new_tab(kzembed->parent_kazehakase,
				       kzembed->gesture_location);
	else
		gtk_moz_embed_reload(GTK_MOZ_EMBED(kzembed),
				     GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}
/*
static void
kz_window_home (KzWindow *kz)
{
	KzMozEmbed *kzembed = KZ_WINDOW_CURRENT_PAGE(kz);
	if (kzembed) {
		gtk_moz_embed_load_url(GTK_MOZ_EMBED(kzembed),
				       prefs.home_uri);
	}
}
*/
