/*
 *  Copyright (C) 2006 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "rd-site-cord.h"
#include "rendako.h"
#include "rd-image-factory.h"
#include "rd-image.h"

#include <glib/gi18n.h>
#include <string.h>

typedef struct _RdSiteCordPrivate RdSiteCordPrivate;
struct _RdSiteCordPrivate
{
	GtkListStore *list;
	RdImageFactory *factory;
};

#define RD_SITE_CORD_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), RD_TYPE_SITE_CORD, RdSiteCordPrivate))

static void     rd_site_cord_class_init     (RdSiteCordClass *klass);
static void     rd_site_cord_init           (RdSiteCord      *cord);
static void     rd_site_cord_dispose        (GObject         *object);
static void     rd_site_cord_finalize       (GObject         *object);

static void cb_image_factory_image_ready (RdImageFactory *factory, gpointer image_data, const gchar *uri, gpointer data);

static GObjectClass *parent_class = NULL;

G_DEFINE_TYPE (RdSiteCord, rd_site_cord, G_TYPE_OBJECT)

static void
rd_site_cord_class_init (RdSiteCordClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *)   klass;
  
	gobject_class->dispose  = rd_site_cord_dispose;
	gobject_class->finalize = rd_site_cord_finalize;

	g_type_class_add_private (gobject_class, sizeof(RdSiteCordPrivate));
}

static void
rd_site_cord_init (RdSiteCord *cord)
{
	RdSiteCordPrivate *priv = RD_SITE_CORD_GET_PRIVATE (cord);

	priv->factory = rd_image_factory_new ();
	g_signal_connect (priv->factory, "ready",
			  G_CALLBACK (cb_image_factory_image_ready), cord);

	priv->list = gtk_list_store_new (N_COLUMNS,
					 G_TYPE_STRING,
					 G_TYPE_STRING,
					 GTK_TYPE_WIDGET);
}

static void
rd_site_cord_dispose (GObject *object)
{
	RdSiteCordPrivate *priv = RD_SITE_CORD_GET_PRIVATE (object);

	if (priv->factory)
	{
		g_signal_handlers_disconnect_by_func (priv->factory,
						      G_CALLBACK (cb_image_factory_image_ready),
						      object);
		g_object_unref (priv->factory);
		priv->factory = NULL;
	}

	if (priv->list)
	{
		g_object_unref (priv->list);
		priv->list = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
rd_site_cord_finalize (GObject *object)
{
	if (G_OBJECT_CLASS (parent_class)->finalize)
		G_OBJECT_CLASS (parent_class)->finalize (object);
}

RdSiteCord *
rd_site_cord_new (void)
{
	RdSiteCord *cord;

	cord = g_object_new (RD_TYPE_SITE_CORD, 
				  NULL);

	return cord;
}

const GtkListStore *
rd_site_cord_get_list_store (RdSiteCord *cord)
{
	RdSiteCordPrivate *priv = RD_SITE_CORD_GET_PRIVATE (cord);
	return priv->list;
}

void
rd_site_cord_append_uri (RdSiteCord *cord, const gchar *uri)
{
	GtkWidget *image;
	RdSiteCordPrivate *priv = RD_SITE_CORD_GET_PRIVATE (cord);

	image = rd_image_new ();

	gtk_list_store_insert_with_values (priv->list, NULL,
					   G_MAXINT32,
					   COLUMN_URI, uri,
					   COLUMN_IMAGE, image,
					   -1);

	rd_image_factory_request_uri (priv->factory, uri);
}

typedef struct _FindRow
{
	const gchar *uri;
	GtkTreePath *path;
} FindRow;

static gboolean
find_row_func (GtkTreeModel *model, GtkTreePath *path,
	       GtkTreeIter *iter, gpointer data)
{
	FindRow *findrow = data;
	const gchar *uri;

	gtk_tree_model_get (model, iter,
			    COLUMN_URI, &uri,
			    -1);

	if (uri && !strcmp (findrow->uri, uri))
	{
		findrow->path = gtk_tree_path_copy (path);
		return TRUE;
	}

	return FALSE;
}

static GtkTreePath *
find_path_with_uri (GtkTreeModel *model, const gchar *uri)
{
	FindRow findrow;

	findrow.uri = uri;
	findrow.path = NULL;

	gtk_tree_model_foreach (model, find_row_func, &findrow);

	return findrow.path;
}

static void
cb_image_factory_image_ready (RdImageFactory *factory, gpointer image_data, const gchar *uri, gpointer data)
{
	RdSiteCordPrivate *priv = RD_SITE_CORD_GET_PRIVATE (data);
	GtkTreePath *path;
	
	path = find_path_with_uri (GTK_TREE_MODEL (priv->list), uri);
	if (path)
	{
		GtkTreeIter iter;
		GtkWidget *image = NULL;
		gtk_tree_model_get_iter(GTK_TREE_MODEL (priv->list),
					&iter, path);
		gtk_tree_model_get (GTK_TREE_MODEL (priv->list), &iter,
				    COLUMN_IMAGE, &image,
				    -1);
		if (image)
		{
			rd_image_set_cairo_surface (RD_IMAGE (image), 
						    (cairo_surface_t *) image_data);
			gtk_widget_show (image);
			g_object_unref (image);
		}
		gtk_tree_path_free (path);
	}
}

