/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

#include <string.h>
#include <glib/gi18n.h>

#include "gobject-utils.h"
#include "gtk-utils.h"
#include "kz-feed-info.h"

enum {
	PROP_0,
	PROP_KZ_WINDOW
};

static void kz_feed_info_class_init (KzFeedInfoClass *klass);
static void kz_feed_info_init       (KzFeedInfo      *info);
static void kz_feed_info_dispose    (GObject         *object);
static void kz_feed_info_set_property (GObject            *object,
				       guint               prop_id,
				       const GValue       *value,
				       GParamSpec         *pspec);
static void kz_feed_info_get_property (GObject            *object,
				       guint               prop_id,
				       GValue             *value,
				       GParamSpec         *pspec);

static void kz_feed_info_setup_for_with_feed    (KzFeedInfo *info);
static void kz_feed_info_setup_for_without_feed (KzFeedInfo *info);


static void cb_add_feed_menu_activate (GtkWidget      *widget,
				       KzNavi         *navi);
static void cb_feed_info_button_press (GtkWidget      *widget,
				       GdkEventButton *event,
				       KzFeedInfo     *info);


static GtkStatusbarClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE (kz_feed_info, "KzFeedInfo", KzFeedInfo,
		    kz_feed_info_class_init, kz_feed_info_init,
		    GTK_TYPE_STATUSBAR)

static void
kz_feed_info_class_init (KzFeedInfoClass *klass)
{
	GObjectClass *gobject_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = (GObjectClass *)klass;
	widget_class  = (GtkWidgetClass *)klass;

	/* GtkObject signals */
	gobject_class->dispose = kz_feed_info_dispose;
	gobject_class->set_property = kz_feed_info_set_property;
	gobject_class->get_property = kz_feed_info_get_property;

	g_object_class_install_property
		(gobject_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object("kz-window",
				     _("KzWindow"),
				     _("The parent kazehakase window"),
				     KZ_TYPE_WINDOW,
				     G_PARAM_READWRITE |
				     G_PARAM_CONSTRUCT_ONLY));
}

static void
kz_feed_info_init (KzFeedInfo *info)
{
	/* widgets */
	info->kz = NULL;
	
	info->feed_list = NULL;
}


static void
kz_feed_info_set_property (GObject *object, guint prop_id,
			   const GValue *value, GParamSpec *pspec)
{
	KzFeedInfo *info = KZ_FEED_INFO(object);
  
	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		info->kz = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
kz_feed_info_get_property (GObject *object, guint prop_id,
			   GValue *value, GParamSpec *pspec)
{
	KzFeedInfo *info = KZ_FEED_INFO(object);

	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		g_value_set_object(value, info->kz);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


GtkWidget *
kz_feed_info_new (KzWindow *kz)
{
	KzFeedInfo *info;

	info = g_object_new(KZ_TYPE_FEED_INFO,
			    "kz-window", kz,
			    NULL);

	info->event_box = gtk_event_box_new();
	info->label = gtk_label_new(NULL);

	gtk_widget_set_size_request(info->event_box, 32, -1);
	
	g_signal_connect(info->event_box, "button_press_event",
			 G_CALLBACK(cb_feed_info_button_press), info);

	gtk_container_add(GTK_CONTAINER(info->event_box), info->label);

	gtk_widget_show(info->label);
	gtk_widget_show(info->event_box);

	gtk_box_pack_start(GTK_BOX(info), info->event_box, FALSE, FALSE, 0);

	kz_feed_info_setup_for_without_feed(info);
	
	return GTK_WIDGET(info);
}

void
kz_feed_info_change_state (KzFeedInfo *info)
{
	if (kz_embed_get_nav_links(KZ_EMBED(KZ_WINDOW_CURRENT_PAGE(info->kz)),
				   KZ_EMBED_LINK_RSS))
		kz_feed_info_setup_for_with_feed(info);
	else
		kz_feed_info_setup_for_without_feed(info);
}

static void
kz_feed_info_dispose (GObject *object)
{
	KzFeedInfo *info = KZ_FEED_INFO(object);

	if (info->kz)
	{
		g_object_unref(info->kz);
		info->kz = NULL;
	}

	if (info->feed_list)
	{
		g_slist_free(info->feed_list);
		info->feed_list = NULL;
	}
	
	g_signal_handlers_disconnect_by_func(info,
					     G_CALLBACK(cb_feed_info_button_press),
					     NULL);

	if (G_OBJECT_CLASS(parent_class)->dispose)
		(*G_OBJECT_CLASS(parent_class)->dispose) (object);
}

static void
kz_feed_info_setup_for_with_feed (KzFeedInfo *info)
{
	gtk_label_set_text(GTK_LABEL(info->label), "F");
}

static void
kz_feed_info_setup_for_without_feed (KzFeedInfo *info)
{
	gtk_label_set_text(GTK_LABEL(info->label), "N");
}

static void
cb_add_feed_menu_activate (GtkWidget *widget, KzNavi *navi)
{
	kz_navi_add_bookmark(navi);
}

static void
cb_feed_info_button_press (GtkWidget *widget,
			   GdkEventButton *event,
			   KzFeedInfo *info)
{
	GList *nav_links, *nav_link;
	GtkWidget *menu, *menuitem;
	KzEmbed *embed;
	gchar *label;

	embed = KZ_EMBED(KZ_WINDOW_CURRENT_PAGE(info->kz));
	if (!embed) return;
	
	switch (event->button) {
	case 1:
		break;
	case 2:
		break;
	case 3:
		nav_links = kz_embed_get_nav_links(embed, KZ_EMBED_LINK_RSS);
		if (!nav_links) return;
		
		menu = gtk_menu_new();

		for (nav_link = nav_links;
		     nav_link;
		     nav_link = g_list_next(nav_link))
		{
			KzNavi *navi = nav_link->data;
			if (!(navi && navi->uri))
				continue;

			if (navi->title)
				label = g_strdup_printf(_("Add '%s' feed to bookmarks"),
							navi->title);
			else
				label = g_strdup_printf(_("Add the feed to bookmarks"));
			
			menuitem = gtk_menu_item_new_with_label(label);
			g_free(label);

			g_signal_connect(G_OBJECT(menuitem), "activate",
					 G_CALLBACK(cb_add_feed_menu_activate),
					 navi);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);
			gtk_widget_show(menuitem);
		}
		
		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			       event->button, event->time);
		break;
	default:
		break;
	}

}
