/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Takuro Ashie
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "kz-thumbnails-sidebar.h"

#include <glib/gi18n.h>
#include "kazehakase.h"
#include "gobject-utils.h"
#include "kz-thumbnails-view.h"
#include "kz-paned.h"
#include "kz-tab-label.h"

#define KZ_THUMBNAILS_SIDEBAR_TAB_BOOKMARK_KEY "KzThumbnailsSidebar::TabBookmark"

enum {
	PROP_0,
	PROP_SIDEBAR
};


/* Object class methods */
static void     kz_thumbnails_sidebar_class_init   (KzThumbnailsSidebarClass *klass);
static void     kz_thumbnails_sidebar_init         (KzThumbnailsSidebar *sidebar);
static GObject *kz_thumbnails_sidebar_constructor  (GType type,
						    guint n_props,
						    GObjectConstructParam *props);
static void     kz_thumbnails_sidebar_dispose      (GObject      *obj);
static void     kz_thumbnails_sidebar_set_property (GObject      *object,
						    guint         prop_id,
						    const GValue *value,
						    GParamSpec   *pspec);
static void     kz_thumbnails_sidebar_get_property (GObject      *object,
						    guint         prop_id,
						    GValue       *value,
						    GParamSpec   *pspec);
static void     cb_paned_changed_position          (KzPaned *pane,
						    GtkPositionType position,
						    KzThumbnailsView *view);
static void	cb_thumbnail_activate		   (KzThumbnailsView *view,
						    KzBookmark *bookmark,
						    KzWindow *kz);

#warning FIXME! We should create TabFolderBookmark Class?
static void       connect_folder_signals         (KzBookmark     *folder,
						  KzBookmark     *proxy);
static void       disconnect_folder_signals      (KzBookmark     *folder,
						  KzBookmark     *proxy);
static KzBookmark *create_current_page_bookmark  (KzBookmark     *tab);
static KzBookmark *create_proxy_folder           (KzBookmark     *tabs);

static GtkWidget *kz_thumbnails_sidebar_closed_tabs_new 
						   (KzSidebar *sidebar);
#if 0
static GtkWidget *kz_thumbnails_sidebar_history_search_new 
						   (KzSidebar *sidebar);
#endif
static KzSidebarEntry kz_sidebar_thumbnails[] = 
{
	{
		priority_hint: 0,
		label:         N_("Closed tabs"),
		icon:          NULL,
		create:        kz_thumbnails_sidebar_closed_tabs_new,
	}
/*
	{
		priority_hint: 0,
		label:         N_("History search"),
		icon:          NULL,
		create:        kz_thumbnails_sidebar_history_search_new,
	}
*/
};

static GtkVBoxClass *parent_class = NULL;


KzSidebarEntry *
kz_thumbnails_sidebar_get_entry (gint idx)
{
	if (idx > 0) return NULL;

	return &kz_sidebar_thumbnails[idx];
}


KZ_OBJECT_GET_TYPE(kz_thumbnails_sidebar, "KzThumbnailsSidebar", KzThumbnailsSidebar,
		   kz_thumbnails_sidebar_class_init, kz_thumbnails_sidebar_init,
		   GTK_TYPE_VBOX)


static void
kz_thumbnails_sidebar_class_init (KzThumbnailsSidebarClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent(klass);
	gobject_class = (GObjectClass *) klass;

	gobject_class->constructor  = kz_thumbnails_sidebar_constructor;
	gobject_class->dispose      = kz_thumbnails_sidebar_dispose;
	gobject_class->set_property = kz_thumbnails_sidebar_set_property;
	gobject_class->get_property = kz_thumbnails_sidebar_get_property;

	g_object_class_install_property
		(gobject_class,
		 PROP_SIDEBAR,
		 g_param_spec_object ("sidebar",
				      _("Sidebar"),
				      _("The parent sidebar"),
				      KZ_TYPE_SIDEBAR,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_thumbnails_sidebar_init (KzThumbnailsSidebar *sidebar)
{
	GtkWidget *scrwin, *view;

	/* create scrolled window */
	scrwin = gtk_scrolled_window_new(NULL, NULL);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(sidebar), scrwin, TRUE, TRUE, 0);
	gtk_widget_show(scrwin);

	view = kz_thumbnails_view_new();
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrwin),
					      view);
	gtk_widget_show(view);

	sidebar->view = view;
}


static GObject *
kz_thumbnails_sidebar_constructor (GType                  type,
			  	   guint                  n_props,
			  	   GObjectConstructParam *props)
{
	KzThumbnailsSidebar *sidebar;
	GObject *object;
	GObjectClass *klass = G_OBJECT_CLASS(parent_class);

	object = klass->constructor(type, n_props, props);

	sidebar = KZ_THUMBNAILS_SIDEBAR(object); 

	g_signal_connect(G_OBJECT(sidebar->sidebar->kz->pane),
			 "changed_position",
			 G_CALLBACK(cb_paned_changed_position),
			 sidebar->view);
	g_signal_connect(G_OBJECT(sidebar->view),
			 "activate",
			 G_CALLBACK(cb_thumbnail_activate),
			 sidebar->sidebar->kz);
	return object;
}
static void
kz_thumbnails_sidebar_dispose (GObject *obj)
{
	KzThumbnailsSidebar *sidebar = KZ_THUMBNAILS_SIDEBAR(obj);

	if (sidebar->sidebar)
		g_object_unref(sidebar->sidebar);

	sidebar->sidebar = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(obj);
}


static void
kz_thumbnails_sidebar_set_property (GObject         *object,
				    guint            prop_id,
				    const GValue    *value,
				    GParamSpec      *pspec)
{
	KzThumbnailsSidebar *sidebar = KZ_THUMBNAILS_SIDEBAR(object);
  
	switch (prop_id)
	{
	case PROP_SIDEBAR:
		sidebar->sidebar = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_thumbnails_sidebar_get_property (GObject         *object,
				    guint            prop_id,
				    GValue          *value,
				    GParamSpec      *pspec)
{
	KzThumbnailsSidebar *sidebar = KZ_THUMBNAILS_SIDEBAR(object);

	switch (prop_id)
	{
	case PROP_SIDEBAR:
		g_value_set_object(value, sidebar->sidebar);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static GtkWidget *
kz_thumbnails_sidebar_closed_tabs_new (KzSidebar *sidebar)
{
	KzThumbnailsSidebar *tsidebar;
	KzBookmark *proxy;

	tsidebar = g_object_new(KZ_TYPE_THUMBNAILS_SIDEBAR,
			        "sidebar", sidebar,
			        NULL);

	/* 
	 * create a new bookmark folder which included each current page of  
	 * closed tab.
	 */
	proxy = create_proxy_folder(sidebar->kz->closed_tabs);
	
	/* FIXME! these connections are leaks! */
	connect_folder_signals(sidebar->kz->closed_tabs, proxy);
	
	kz_thumbnails_view_set_folder(KZ_THUMBNAILS_VIEW(tsidebar->view),
				      proxy);

	return GTK_WIDGET(tsidebar);
}

#if 0
static GtkWidget *
kz_thumbnails_sidebar_history_search_new (KzSidebar *sidebar)
{
	KzThumbnailsSidebar *tsidebar;

	tsidebar = KZ_THUMBNAILS_SIDEBAR(g_object_new(KZ_TYPE_THUMBNAILS_SIDEBAR,
						   "sidebar", sidebar,
						   NULL));

	kz_thumbnails_view_set_folder(KZ_THUMBNAILS_VIEW(tsidebar->view),
				      sidebar->kz->history_search);

	return GTK_WIDGET(tsidebar);
}
#endif

static void     
cb_paned_changed_position (KzPaned *pane,
			   GtkPositionType position,
			   KzThumbnailsView *view)
{
	switch(position)
	{
	 case GTK_POS_LEFT:
	 case GTK_POS_RIGHT:
		kz_thumbnails_view_set_mode(view, KZ_THUMBNAILS_VIEW_VERTICAL);	
		break;
	 case GTK_POS_TOP:
	 case GTK_POS_BOTTOM:
		kz_thumbnails_view_set_mode(view, KZ_THUMBNAILS_VIEW_HORIZONTAL);	
		break;
	 default:
		break;
	}	
}


static KzBookmark *
create_current_page_bookmark (KzBookmark *tab)
{
	KzBookmark *cur_page;
	gint cur_pos;
	GList *pages;
	KzBookmark *new;

	cur_pos = kz_bookmark_get_current(tab);

	pages = kz_bookmark_get_children(tab);
	cur_page = KZ_BOOKMARK(g_list_nth_data(pages, cur_pos));
	g_list_free(pages);

	new = kz_bookmark_new_with_attrs(kz_bookmark_get_title(cur_page),
					 kz_bookmark_get_link(cur_page),
					 NULL);
	g_object_set_data(G_OBJECT(new),
			  KZ_THUMBNAILS_SIDEBAR_TAB_BOOKMARK_KEY,
			  tab);

	return new;
}


static KzBookmark *
create_proxy_folder (KzBookmark *tabs)
{
	GList *children, *node;
	KzBookmark *ret_bookmark;
	
	children = kz_bookmark_get_children(tabs);
	ret_bookmark = kz_bookmark_pure_folder_new();
	
	for (node = children; node; node = g_list_next(node))
	{
		KzBookmark *cur_page;
		cur_page = create_current_page_bookmark(KZ_BOOKMARK(node->data));

		kz_bookmark_append(ret_bookmark, cur_page);
		g_object_unref(cur_page);
	}

	g_list_free(children);

	return ret_bookmark;
}


static void
cb_thumbnail_activate (KzThumbnailsView *view, KzBookmark *bookmark,
		       KzWindow *kz)
{
	KzBookmark *tab;
	GtkWidget *kzembed;
	KzTabLabel *kztab;

	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	tab = g_object_get_data(G_OBJECT(bookmark),
				KZ_THUMBNAILS_SIDEBAR_TAB_BOOKMARK_KEY);
	g_return_if_fail(tab);

	kzembed = kz_window_open_new_tab_at_tail(kz, NULL);
	kztab = KZ_TAB_LABEL(gtk_notebook_get_tab_label(
			     GTK_NOTEBOOK(kz->notebook),
			     kzembed));
	kz_tab_label_set_history(KZ_TAB_LABEL(kztab), tab);

	/* FIXME! should use parent bookmark */ 
	kz_bookmark_remove(kz->closed_tabs, tab);
}


static gint
find_proxy_item (gconstpointer a, gconstpointer b)
{
	const KzBookmark *bookmark1, *bookmark2 = b;

	bookmark1 = g_object_get_data(G_OBJECT(a),
				      KZ_THUMBNAILS_SIDEBAR_TAB_BOOKMARK_KEY);

	return bookmark1 - bookmark2;
}

static void
cb_folder_insert_child (KzBookmark *folder,
			KzBookmark *child, KzBookmark *sibling,
			KzBookmark *proxy)
{
	KzBookmark *new;
	new = create_current_page_bookmark(child);

	if (sibling)
	{
		GList *list, *node;

		list = kz_bookmark_get_children(proxy);
		node = g_list_find_custom(list, sibling, find_proxy_item);
		if (node)
		{
			kz_bookmark_insert_before(proxy, new, 
					          KZ_BOOKMARK(node->data));
		}
	}
	else
	{

		kz_bookmark_prepend(proxy, new);
	}
	g_object_unref(new);
}


static void
cb_folder_remove_child (KzBookmark *folder,
			KzBookmark *child, KzBookmark *proxy) 
{
	GList *list, *node;

	list = kz_bookmark_get_children(proxy);
	node = g_list_find_custom(list, child, find_proxy_item);
	g_return_if_fail(node);
	
	kz_bookmark_remove(proxy, KZ_BOOKMARK(node->data));
}

static void
connect_folder_signals (KzBookmark *folder, KzBookmark *proxy)
{
	g_signal_connect_after(G_OBJECT(folder), "insert-child",
			       G_CALLBACK(cb_folder_insert_child),
			       proxy);
	g_signal_connect_after(G_OBJECT(folder), "remove-child",
			       G_CALLBACK(cb_folder_remove_child),
			       proxy);
}


static void
disconnect_folder_signals (KzBookmark *folder, KzBookmark *proxy)
{
	g_signal_handlers_disconnect_by_func
		(G_OBJECT(folder),
		 G_CALLBACK(cb_folder_insert_child),
		 proxy);
	g_signal_handlers_disconnect_by_func
		(G_OBJECT(folder),
		 G_CALLBACK(cb_folder_remove_child),
		 proxy);
}

