/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-actions-bookmark.c,v 1.25 2004/08/28 04:04:19 ikezoe Exp $
 */

#include "kz-actions-bookmark.h"

#include "gtk24backports.h"
#include "intl.h"
#include "kz-icons.h"
#include "kz-bookmark-edit-win.h"

static GtkActionEntry kz_actions_bookmark[];
static const gint     kz_actions_bookmark_len;


static void
act_new_window (GtkAction *action, KzBookmarkEditor *editor)
{
	GtkWidget *widget;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	widget = kz_bookmark_editor_new(editor->root_folder);
	gtk_widget_show(widget);
}


static void
act_close_window (GtkAction *action, KzBookmarkEditor *editor)
{
	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	kz_bookmark_editor_store_state(editor);
	gtk_widget_destroy(GTK_WIDGET(editor));
}


static void
act_edit_bookmark (GtkAction *action, KzBookmarkEditor *editor)
{
	GList *node;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	for (node = kz_bookmark_editor_get_selected_list(editor);
	     node;
	     node = g_list_next(node))

	{
		GtkWidget *widget = kz_bookmark_edit_win_new(node->data);

		gtk_widget_show(widget);
	}
}


static void
insert_bookmark_item (KzBookmarkEditor *editor, KzBookmark *bookmark)
{
	KzBookmark *parent, *sibling = NULL;
	GList *list;

	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));

	list = kz_bookmark_editor_get_selected_list(editor);

	if (list)
	{
		sibling = list->data;
		parent = kz_bookmark_get_parent(sibling);
	}
	else
	{
		parent = editor->current_folder;
		g_return_if_fail(KZ_IS_BOOKMARK(parent));
		sibling = NULL;
	}

	kz_bookmark_insert_before(parent, bookmark, sibling);

	g_list_free(list);
	/* focus to new bookmark item */
	kz_bookmark_editor_bookmarks_view_select(editor, bookmark);
}


static void
act_insert_bookmark (GtkAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *bookmark;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	bookmark = kz_bookmark_new_with_attrs(_("New bookmark"), NULL, NULL);
	insert_bookmark_item(editor, bookmark);
	g_object_unref(G_OBJECT(bookmark));
}


static void
act_insert_folder (GtkAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *folder;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	folder = kz_bookmark_pure_folder_new();
	kz_bookmark_set_title(folder, _("New folder"));
	insert_bookmark_item(editor, folder);
	g_object_unref(G_OBJECT(folder));
}


static void
act_insert_remote_bookmark (GtkAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *file;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	file = kz_bookmark_file_new(NULL, NULL, NULL);
	kz_bookmark_set_title(file, _("New Remote Bookmark"));
	insert_bookmark_item(editor, file);
	g_object_unref(G_OBJECT(file));
}


static void
act_insert_smart_bookmark (GtkAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *smart;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	smart = kz_bookmark_new_with_attrs(_("New Smart bookmark"), 
					   NULL, NULL);
	kz_bookmark_append_smart_property(smart,
					  "(.*)", "%s",
					  NULL, FALSE);
	insert_bookmark_item(editor, smart);
	g_object_unref(G_OBJECT(smart));
}


static void
act_insert_separator (GtkAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *sep;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	sep = kz_bookmark_separator_new();
	insert_bookmark_item(editor, sep);
	g_object_unref(G_OBJECT(sep));
}


static void
act_remove_selected_item (GtkAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *parent, *bookmark;
	GList *list, *node;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	list = kz_bookmark_editor_get_selected_list(editor);

	parent = editor->current_folder;
	g_return_if_fail(KZ_IS_BOOKMARK(parent));

	/* FIXME!: show dialog */

	for (node = list; node; node = g_list_next(node))
	{
		bookmark = node->data;
		if (!KZ_IS_BOOKMARK(bookmark)) continue;
		kz_bookmark_remove(parent, bookmark);
	}

	g_list_free(list);
}


static void
act_show_hide_folder_view (GtkToggleAction *action, KzBookmarkEditor *editor)
{
	g_return_if_fail(GTK_IS_TOGGLE_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));
	g_return_if_fail(GTK_IS_WIDGET(editor->scrolledwin[0]));

	if (gtk_toggle_action_get_active (action))
		gtk_widget_show(editor->scrolledwin[0]);
	else
		gtk_widget_hide(editor->scrolledwin[0]);
}


static void
act_show_hide_content_view (GtkToggleAction *action, KzBookmarkEditor *editor)
{
	g_return_if_fail(GTK_IS_TOGGLE_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));
	g_return_if_fail(GTK_IS_WIDGET(editor->content_view));

	if (gtk_toggle_action_get_active (action))
		gtk_widget_show(editor->content_view);
	else
		gtk_widget_hide(editor->content_view);
}


static void
act_change_list_mode (GtkRadioAction *action, GtkRadioAction *cur_action, KzBookmarkEditor *editor)
{
	gint value;

	g_return_if_fail(GTK_IS_RADIO_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	value = gtk_radio_action_get_current_value(action);
	switch (value) {
	case 0:
		kz_bookmark_editor_set_tree_mode(editor, FALSE);
		break;
	case 1:
		kz_bookmark_editor_set_tree_mode(editor, TRUE);
		break;
	default:
		break;
	}
}


static void
act_go_up (GtkAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *parent;

	g_return_if_fail(GTK_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	if (!editor->current_folder) return;

	parent = kz_bookmark_get_parent(editor->current_folder);
	if (!parent) return;

	kz_bookmark_editor_set_current(editor, parent);
}


static void
cb_popup_menu_hide (void)
{
	gtk_main_quit();
}


static GtkActionEntry kz_actions_bookmark[] =
{
  {"StockFileMenu",     NULL, N_("_File"),       NULL, NULL, NULL},
  {"StockEditMenu",     NULL, N_("_Edit"),       NULL, NULL, NULL},
  {"StockViewMenu",     NULL, N_("_View"),       NULL, NULL, NULL},
  {"StockGoMenu",       NULL, N_("_Go"),         NULL, NULL, NULL},
  {"StockModeMenu",     NULL, N_("_Mode"),       NULL, NULL, NULL},

  {"NewWindow",            KZ_STOCK_NEW_WINDOW,      N_("_New Window"),             "<control>N", N_("Open a new window"), G_CALLBACK(act_new_window)},
  {"CloseWindow",          KZ_STOCK_CLOSE_WINDOW,    N_("_Close Window"),           "<control>W", N_("Close the current window"), G_CALLBACK(act_close_window)},

  {"EditBookmarkItem",     NULL,                     N_("_Edit selected item"),     NULL,         N_("Edit the selected item"), G_CALLBACK(act_edit_bookmark)},
  {"InsertBookmark",       KZ_STOCK_BOOKMARK,        N_("Insert _bookmark"),        "<control>B", N_("Insert a new bookmark"), G_CALLBACK(act_insert_bookmark)},
  {"InsertFolder",         KZ_STOCK_FOLDER,          N_("Insert _folder"),          "<control>F", N_("Insert a new folder"), G_CALLBACK(act_insert_folder)},
  {"InsertRemoteBookmark", KZ_STOCK_REMOTE_BOOKMARK, N_("Insert _remote bookmark"), "<control>R", N_("Insert a new remote bookmark"), G_CALLBACK(act_insert_remote_bookmark)},
  {"InsertSmartBookmark", KZ_STOCK_SMART_BOOKMARK, N_("Insert _smart bookmark"), "<control>S", N_("Insert a new smart bookmark"), G_CALLBACK(act_insert_smart_bookmark)},
  {"InsertSeparator",      KZ_STOCK_SEPARATOR,       N_("Insert _separator"),       "<control>P", N_("Insert a separator"), G_CALLBACK(act_insert_separator)},
  {"RemoveBookmarkItem",   GTK_STOCK_DELETE,         N_("_Remove selected item"),   "<control>Delete",     N_("Remove the selected item"), G_CALLBACK(act_remove_selected_item)},

  {"GoUp",  GTK_STOCK_GO_UP, N_("_Up"), NULL, NULL, G_CALLBACK(act_go_up)},
};
static const gint kz_actions_bookmark_len = G_N_ELEMENTS(kz_actions_bookmark);

static GtkToggleActionEntry kz_toggle_actions_bookmark[] =
{
  {"ShowHideFolderView",  NULL, N_("Folder view"),  NULL, NULL, G_CALLBACK(act_show_hide_folder_view)},
  {"ShowHideContentView", NULL, N_("Content view"), NULL, NULL, G_CALLBACK(act_show_hide_content_view)},
};
static const gint kz_toggle_actions_bookmark_len = G_N_ELEMENTS(kz_toggle_actions_bookmark);


static GtkRadioActionEntry kz_radio_actions_bookmark[] =
{
  {"ListMode", NULL, N_("_List Mode"), NULL, NULL, 0},
  {"TreeMode", NULL, N_("_Tree Mode"), NULL, NULL, 1},
};
static const gint kz_radio_actions_bookmark_len = G_N_ELEMENTS(kz_radio_actions_bookmark);


GtkActionGroup *
kz_actions_bookmark_create_group (KzBookmarkEditor *editor, 
				  GtkAccelGroup *accel_group)
{
	GtkActionGroup *action_group;
	GList *node, *action_list;

	action_group = gtk_action_group_new("KzBookmarkEditor");

	gtk_action_group_set_translation_domain(action_group, NULL);

	gtk_action_group_add_actions (action_group,
				      kz_actions_bookmark,
				      kz_actions_bookmark_len,
				      editor);
	gtk_action_group_add_toggle_actions (action_group,
					     kz_toggle_actions_bookmark,
					     kz_toggle_actions_bookmark_len,
					     editor);
	gtk_action_group_add_radio_actions (action_group,
					    kz_radio_actions_bookmark,
					    kz_radio_actions_bookmark_len,
					    0,
					    G_CALLBACK(act_change_list_mode),
					    editor);

	action_list = gtk_action_group_list_actions(action_group);
	
	for (node = action_list; node; node = g_list_next(node))
	{
		gtk_action_set_accel_group(GTK_ACTION(node->data),
					   accel_group);
		gtk_action_connect_accelerator(GTK_ACTION(node->data));
	}
	g_list_free(action_list);

	return action_group;
}


void
kz_actions_bookmark_popup_menu_modal (KzBookmarkEditor *editor,
				      guint button, guint time)
{
	GtkWidget *popup_menu = NULL;

	popup_menu = gtk_ui_manager_get_widget(editor->menu_merge,
					       "/BookmarksViewPopup");
	if (!popup_menu) return;

	g_signal_connect(G_OBJECT(popup_menu), "hide",
			 G_CALLBACK(cb_popup_menu_hide), editor);
	gtk_menu_popup(GTK_MENU(popup_menu), NULL, NULL,
		       NULL, NULL, button, time);
	gtk_main();

	g_signal_handlers_disconnect_by_func(G_OBJECT(popup_menu),
					     G_CALLBACK(cb_popup_menu_hide),
					     editor);
}
