/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-history-action.h"

#include <string.h>

#include "gobject-utils.h"
#include "intl.h"
#include "gtk24backports.h"


enum {
	HISTORY_CHANGED_SIGNAL,
	LAST_SIGNAL
};


static void       kz_history_action_class_init       (KzHistoryActionClass *klass);
static void       kz_history_action_init             (KzHistoryAction *action);
static void       kz_history_action_dispose          (GObject *object);

static void       kz_history_action_activate         (GtkAction *action);
static GtkWidget *kz_history_action_create_menu_item (GtkAction *action);
static GtkWidget *kz_history_action_create_tool_item (GtkAction *action);

static GtkCombo  *kz_history_action_real_get_combo_widget
						     (KzHistoryAction *action,
						      GtkWidget        *proxy);
static void       kz_history_action_real_set_history (KzHistoryAction *action,
						      GList            *list);

static void       kz_history_action_connect_proxy    (GtkAction        *action,
						      GtkWidget        *proxy);

static KzEntryActionClass *parent_class = NULL;

static gint history_action_signals[LAST_SIGNAL] = { 0 };

KZ_OBJECT_GET_TYPE(kz_history_action, "KzHistoryAction", KzHistoryAction,
		   kz_history_action_class_init, kz_history_action_init,
		   KZ_TYPE_ENTRY_ACTION)

static void
kz_history_action_class_init (KzHistoryActionClass *klass)
{
	GObjectClass *object_class;
	GtkActionClass *action_class;
	KzEntryActionClass *entry_action_class;

	parent_class = g_type_class_peek_parent(klass);
	object_class = G_OBJECT_CLASS(klass);
	action_class = GTK_ACTION_CLASS(klass);
	entry_action_class = KZ_ENTRY_ACTION_CLASS(klass);

	object_class->dispose      = kz_history_action_dispose;
	/*
	object_class->set_property = kz_history_action_set_property;
	object_class->get_property = kz_history_action_get_property;
	*/

	action_class->activate         = kz_history_action_activate;
	action_class->create_menu_item = kz_history_action_create_menu_item;
	action_class->create_tool_item = kz_history_action_create_tool_item;
	action_class->connect_proxy    = kz_history_action_connect_proxy;

	action_class->toolbar_item_type = GTK_TYPE_TOOL_ITEM;

	klass->get_combo_widget = kz_history_action_real_get_combo_widget;
	klass->set_history      = kz_history_action_real_set_history;
	klass->history_changed  = NULL;

	history_action_signals[HISTORY_CHANGED_SIGNAL] =
		g_signal_new("history-changed",
			     G_OBJECT_CLASS_TYPE(klass),
			     G_SIGNAL_RUN_FIRST | G_SIGNAL_NO_RECURSE,
			     G_STRUCT_OFFSET(KzHistoryActionClass, history_changed),
			     NULL,
			     NULL,
			     g_cclosure_marshal_VOID__VOID,
			     G_TYPE_NONE, 0);
}


static void
kz_history_action_init (KzHistoryAction *action)
{
	action->max_history = 32;
	action->history     = NULL;
}


static void
kz_history_action_dispose (GObject *object)
{
	KzHistoryAction *action = KZ_HISTORY_ACTION(object);

	if (action->history)
	{
		g_list_foreach (action->history, (GFunc) g_free, NULL);
		g_list_free(action->history);
		action->history = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


static void
kz_history_action_activate (GtkAction *action)
{
	KzHistoryAction *hist_act = KZ_HISTORY_ACTION(action);
	const gchar *text;

	text = kz_entry_action_get_text(KZ_ENTRY_ACTION(action));

	if (text && *text)
	{
		GList *node;
		GSList *snode, *proxies;
		GList *prev;

		/* append, or reorder history */
		prev = g_list_find_custom(hist_act->history, text,
					  (GCompareFunc)strcmp);

		if (prev)
		{
			text = prev->data;
			hist_act->history = g_list_remove(hist_act->history,
							  text);
		}
		else
		{
			text = g_strdup(text);
		}

		hist_act->history = g_list_prepend(hist_act->history,
						   (gpointer) text);

		/* check max history */
		if (g_list_length(hist_act->history) > hist_act->max_history)
		{
			node = g_list_nth(hist_act->history, hist_act->max_history);
			while (node)
			{
				GList *tmp;
				gchar *str = node->data;

				tmp = g_list_next(node);
				hist_act->history = g_list_remove(hist_act->history, str);
				g_free(str);
				node = tmp;
			}
		}

		proxies = gtk_action_get_proxies(action);
		/* set history to all proxy widgets */
		for (snode = proxies; snode; snode = g_slist_next(snode))
		{
			GtkCombo *combo;
			GtkWidget *proxy = snode->data;

			combo = kz_history_action_get_combo_widget
					(KZ_HISTORY_ACTION(action),
					 proxy);
			if (GTK_IS_COMBO(combo))
				gtk_combo_set_popdown_strings(combo,
							      hist_act->history);
		}
	}

	g_signal_emit(G_OBJECT(action),
		      history_action_signals[HISTORY_CHANGED_SIGNAL], 0);

	/* not needed */
	if (GTK_ACTION_CLASS (parent_class)->activate)
		GTK_ACTION_CLASS (parent_class)->activate(action);
}


static gboolean
cb_menu_item_combo_button_press (GtkWidget *widget)
{
	/* don't propagate to parent */
	return TRUE;
}


static gboolean
cb_menu_item_combo_button_release (GtkWidget *widget)
{
	/* don't propagate to parent */
	return TRUE;
}


static gboolean
cb_menu_item_combo_enter (GtkWidget *widget, GdkEventCrossing *event)
{
	gtk_grab_add(GTK_BIN(widget)->child);
	gtk_widget_grab_focus(GTK_BIN(widget)->child);

	return TRUE;
}


static gboolean
cb_menu_item_combo_leave (GtkWidget *widget, GdkEventCrossing *event)
{
	gtk_grab_remove(GTK_BIN(widget)->child);

	return TRUE;
}


static GtkWidget *
kz_history_action_create_menu_item (GtkAction *action)
{
	GtkWidget *widget, *combo;

#warning FIXME! implement as our original widget and force blink cursor.
	widget = gtk_menu_item_new();
	combo = gtk_combo_new();
	gtk_widget_show(combo);
	gtk_container_add(GTK_CONTAINER(widget), combo);

	g_signal_connect_after(G_OBJECT(widget), "button-press-event",
			       G_CALLBACK(cb_menu_item_combo_button_press),
			       NULL);
	g_signal_connect_after(G_OBJECT(widget), "button-release-event",
			       G_CALLBACK(cb_menu_item_combo_button_release),
			       NULL);

	g_signal_connect(G_OBJECT(widget), "enter-notify-event",
			 G_CALLBACK(cb_menu_item_combo_enter), NULL);
	g_signal_connect(G_OBJECT(widget), "leave-notify-event",
			 G_CALLBACK(cb_menu_item_combo_leave), NULL);

	return widget;
}


static GtkWidget *
kz_history_action_create_tool_item (GtkAction *action)
{
	GType toolbar_item_type;
	GtkWidget *widget, *combo;

	toolbar_item_type = GTK_ACTION_GET_CLASS(action)->toolbar_item_type;
	widget = GTK_WIDGET(g_object_new(toolbar_item_type, NULL));
	gtk_tool_item_set_expand (GTK_TOOL_ITEM(widget), TRUE);

	combo = gtk_combo_new();
	gtk_container_add(GTK_CONTAINER(widget), combo);
	gtk_container_set_border_width(GTK_CONTAINER(widget), 4);
	gtk_combo_disable_activate(GTK_COMBO(combo));
	gtk_widget_show(combo);

	return widget;
}


static void
kz_history_action_connect_proxy (GtkAction *action, GtkWidget *proxy)
{
	GtkCombo *combo;

	g_return_if_fail(proxy);


	if (KZ_HISTORY_ACTION(action)->history)
	{
		combo = kz_history_action_get_combo_widget(KZ_HISTORY_ACTION(action),
							    proxy);
		if (GTK_IS_COMBO(combo))
		{
			gtk_combo_set_popdown_strings
				(GTK_COMBO(combo), KZ_HISTORY_ACTION(action)->history);
			gtk_entry_set_text(GTK_ENTRY(combo->entry), "");
		}
	}
	GTK_ACTION_CLASS(parent_class)->connect_proxy(action, proxy);
}


static GtkCombo *
kz_history_action_real_get_combo_widget (KzHistoryAction *action, GtkWidget *proxy)
{
	GtkCombo *combo = NULL;

	g_return_val_if_fail(proxy, NULL);

	if (GTK_IS_BIN(proxy))
		combo = GTK_COMBO(GTK_BIN(proxy)->child);
	else
		combo = GTK_COMBO(proxy);

	if (GTK_IS_COMBO(combo))
		return combo;

	return NULL;
}


GtkCombo *
kz_history_action_get_combo_widget (KzHistoryAction *action, GtkWidget *proxy)
{
	KzHistoryActionClass *klass;

	g_return_val_if_fail(KZ_IS_ENTRY_ACTION(action), NULL);

	klass = KZ_HISTORY_ACTION_GET_CLASS(action);

	if (klass->get_combo_widget)
		return klass->get_combo_widget(action, proxy);

	return NULL;
}


static void
kz_history_action_real_set_history (KzHistoryAction *action, GList *list)
{
	GList *node, *new_list = NULL;
	GSList *snode, *proxies;
	gint num = 0;

	g_return_if_fail(KZ_IS_ENTRY_ACTION(action));

	/* dup the list */
	for (node = list, num = 0;
	     node && num < action->max_history;
	     node = g_list_next(node), num++)
	{
		const gchar *text = node->data;

		if (text && *text)
			new_list = g_list_append(new_list, g_strdup(text));
	}

	/* free old list */
	g_list_foreach(action->history, (GFunc) g_free, NULL);
	g_list_free(action->history);

	/* set history */
	action->history = new_list;

	proxies = gtk_action_get_proxies(GTK_ACTION(action));
	/* set history to all proxy widgets */
	for (snode = proxies;
	     snode;
	     snode = g_slist_next(snode))
	{
		GtkCombo *combo;
		GtkWidget *proxy = snode->data;

		combo = kz_history_action_get_combo_widget
				(KZ_HISTORY_ACTION(action),
				 proxy);
		if (GTK_IS_COMBO(combo))
			gtk_combo_set_popdown_strings(combo,
						      action->history);
	}

	g_signal_emit(G_OBJECT(action),
		      history_action_signals[HISTORY_CHANGED_SIGNAL], 0);
}


void
kz_history_action_set_history (KzHistoryAction *action, GList *list)
{
	KzHistoryActionClass *klass;

	g_return_if_fail(KZ_IS_ENTRY_ACTION(action));

	klass = KZ_HISTORY_ACTION_GET_CLASS(action);

	if (klass->set_history)
		return klass->set_history(action, list);
}


const GList *
kz_history_action_get_history (KzHistoryAction *action)
{
	g_return_val_if_fail(KZ_IS_ENTRY_ACTION(action), NULL);

	return action->history;
}


void
kz_history_action_set_max_history (KzHistoryAction *action, guint max_history)
{
	g_return_if_fail(KZ_IS_ENTRY_ACTION(action));

	action->max_history = max_history;
}


guint
kz_history_action_get_max_history (KzHistoryAction *action)
{
	g_return_val_if_fail(KZ_IS_ENTRY_ACTION(action), 0);

	return action->max_history;
}
