unit FullTextSearch;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
	Dialogs, ExtCtrls, StdCtrls, ComCtrls, ToolWin, Katjusha, FavTypes,
  ImgList, ActnList, Menus, Contnrs;

type
	TFullTextSearchThread = class;
	PFullTextSearchInfo = ^TFullTextSearchInfo;
	TFullTextSearchInfo = record
		Title: string;
		Text: string;
		Line: string;
		Number: Integer;
		ThreadIdx: T2chThreadIndex;
	end;
	TFullTextSearchURLStringType = (ustTitle, ustURL, ustMessage);
	TFullTextSearchURLStringTypes = set of TFullTextSearchURLStringType;
	TFullTextSearchAbornType = (atNormal, atInvisible);
	TFullTextSearchForm = class(TForm)
    ListView: TListView;
    StatusBar: TStatusBar;
    Memo: TMemo;
    Splitter1: TSplitter;
    ImageList: TImageList;
    ToolBar: TToolBar;
    CloseButton: TToolButton;
    ResearchButton: TToolButton;
    CancelButton: TToolButton;
    ActionList: TActionList;
    ToolBarClose: TAction;
    ToolBarResearch: TAction;
    ToolBarCancel: TAction;
    PopupMenu: TPopupMenu;
    ToolBarNextItem: TAction;
    NextItemButton: TToolButton;
    MenuImageList: TImageList;
    MenuActionList: TActionList;
    BrowserCallKatjusha: TAction;
    BrowserCallStandard: TAction;
    K1: TMenuItem;
    E1: TMenuItem;
    EditCopyURL: TAction;
    EditCopyTitleWithURL: TAction;
    N1: TMenuItem;
    URLU1: TMenuItem;
    URLT1: TMenuItem;
    EditCopyMessageWithTitleAndURL: TAction;
    URLM1: TMenuItem;
    BrowserShowSimpleBrowser: TAction;
    W1: TMenuItem;
    EditAborn: TAction;
    EditInvisibleAborn: TAction;
    N2: TMenuItem;
    A1: TMenuItem;
    B1: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ListViewClick(Sender: TObject);
    procedure ToolBarCloseExecute(Sender: TObject);
    procedure ToolBarCancelUpdate(Sender: TObject);
    procedure ToolBarResearchExecute(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ToolBarCancelExecute(Sender: TObject);
    procedure ToolBarNextItemExecute(Sender: TObject);
    procedure BrowserCallKatjushaExecute(Sender: TObject);
    procedure BrowserCallStandardExecute(Sender: TObject);
    procedure EditCopyURLExecute(Sender: TObject);
    procedure EditCopyTitleWithURLExecute(Sender: TObject);
    procedure EditCopyMessageWithTitleAndURLExecute(Sender: TObject);
    procedure BrowserShowSimpleBrowserExecute(Sender: TObject);
    procedure ListViewDblClick(Sender: TObject);
    procedure EditAbornExecute(Sender: TObject);
    procedure EditInvisibleAbornExecute(Sender: TObject);
  private
		{ Private 錾 }
		FProgressBar: TProgressBar;
		FThread: TFullTextSearchThread;
		FIndexList: T2chIndexList;
		FConfig: TFullTextSearchConfig;
		FInSearching: Boolean;	{  }
		procedure TerminateThread;
		procedure ClearAll;
		procedure CatchResult(AInfo: PFullTextSearchInfo);
		procedure NotifyProgress(Index, Max: Integer; const Text: string);
		procedure SetStatusText(const Value: string);
		procedure SearchComplete(AMatchCount: Integer);
		procedure RefreshColumn;
		function ArrangeMessage(AInfo: PFullTextSearchInfo): string;
		function MakeURLString(ATypes: TFullTextSearchURLStringTypes): string;
		function AbornSelected(AType: TFullTextSearchAbornType): Integer;
	public
		{ Public 錾 }
    constructor Create(AOwner: TComponent); override;
		destructor Destroy; override;
		procedure StartSearch(List: T2chIndexList = nil; Config: TFullTextSearchConfig = nil);
		property StatusText: string write SetStatusText;
	end;
{ SXbh }
	TFullTextSearchThread = class(TThread)
	private
		FOwner: TFullTextSearchForm;
		FIndexList: T2chIndexList;
		FConfig: TFullTextSearchConfig;
		FProgressIndex: Integer;
		FProgressMax: Integer;
		FProgressText: string;
		FMatchCount: Integer;
		FInfo: TFullTextSearchInfo;
		procedure SendResult;
		procedure Notify;
		procedure Complete;
		procedure Hit(Text: string; Index, Number: Integer; const Line: string; ThreadIdx: T2chThreadIndex);
	protected
		procedure Execute; override;
	public
		constructor Create(AOwner: TFullTextSearchForm; List: T2chIndexList; Config: TFullTextSearchConfig);
    destructor Destroy; override;
	end;

var
	FullTextSearchForm: TFullTextSearchForm;

implementation

{$R *.dfm}

uses Math, StrUtils, regex, AWKFunc, FavUtils, bmRExp, FullTextSearchConfig,
  FavMain, Clipbrd;

procedure TFullTextSearchForm.CatchResult(AInfo: PFullTextSearchInfo);
var
	PInfo: PFullTextSearchInfo;
	Refresh: Boolean;
begin
	PInfo := nil;
	Refresh := False;
	try
		New(PInfo);
		PInfo^ := AInfo^;

		if ListView.VisibleRowCount = ListView.Items.Count then
			Refresh := True;

		with ListView.Items.Add do
		begin
			Caption := PInfo.Title;
			SubItems.Add(IntToStr(PInfo.Number));
			SubItems.Add(PInfo.Text);
			Data := PInfo;
		end;

		if Refresh then RefreshColumn;
	except
		if PInfo <> nil then Dispose(PInfo);
		raise;
	end;
end;

procedure TFullTextSearchForm.ClearAll;
var
	I: Integer;
begin
	for I := 0 to ListView.Items.Count - 1 do
		Dispose(PFullTextSearchInfo(ListView.Items[I].Data));
	ListView.Clear;
	Memo.Lines.Text := '';
end;

constructor TFullTextSearchForm.Create(AOwner: TComponent);
begin
	inherited;
	FThread := nil;
	FInSearching := False;
	FIndexList := T2chIndexList.Create;
	FConfig := TFullTextSearchConfig.Create;
end;

destructor TFullTextSearchForm.Destroy;
begin
	inherited;
	FreeAndNil(FConfig);
	FreeAndNil(FIndexList);
end;

procedure TFullTextSearchForm.FormCreate(Sender: TObject);
begin
	{ vOXo[̍쐬 }
	FProgressBar := TProgressBar.Create(StatusBar);
	with FProgressBar do
	begin
		Parent := StatusBar;
		Top := 2;
		Height := StatusBar.Height - 2;
		Left := 0;
		Width := 100;
	end;

	(Application.MainForm as TKatfavMainForm).AddSubForm(Self);
end;

procedure TFullTextSearchForm.StartSearch(List: T2chIndexList;
  Config: TFullTextSearchConfig);
begin
	if List <> nil then FIndexList.Assign(List);
	if Config <> nil then FConfig.Assign(Config);

	ClearAll;
	Caption := Format('''%s''ŃXbhSc', [FConfig.SearchText]);

	TerminateThread;

	Show;

	FInSearching := True;
	FThread := TFullTextSearchThread.Create(Self, FIndexList, FConfig);
end;

procedure TFullTextSearchForm.TerminateThread;
begin
	if Assigned(FThread) then
	begin
		FThread.Terminate;
		if FThread.Suspended then FThread.Resume;
		FThread.WaitFor;
		FreeAndNil(FThread);
	end;
	FInSearching := False;
end;

procedure TFullTextSearchForm.FormDestroy(Sender: TObject);
begin
	(Application.MainForm as TKatfavMainForm).RemoveSubForm(Self);

	TerminateThread;
	ClearAll;
end;

procedure TFullTextSearchForm.ListViewClick(Sender: TObject);
var
	PInfo: PFullTextSearchInfo;
begin
	if ListView.Selected = nil then Exit;
	PInfo := PFullTextSearchInfo(ListView.Selected.Data);
	Memo.Text := ArrangeMessage(PInfo);
end;

procedure TFullTextSearchForm.NotifyProgress(Index, Max: Integer; const Text: string);
begin
	FProgressBar.Min := 1;
	FProgressBar.Max := Max;
	FProgressBar.Position := Index;
	StatusText := Text;
end;

procedure TFullTextSearchForm.SetStatusText(const Value: string);
begin
	StatusBar.Panels[1].Text := Value;
end;

procedure TFullTextSearchForm.SearchComplete(AMatchCount: Integer);
begin
	Caption := Format('''%s''̃XbhS', [FConfig.SearchText]);
	StatusText :=
		Format('XbhS܂(%dX/%dqbg)', [FIndexList.Count, AMatchCount]);
	FInSearching := False;
end;

procedure TFullTextSearchForm.RefreshColumn;
begin
	ListView.Height := ListView.Height - 1;
end;

procedure TFullTextSearchForm.ToolBarCloseExecute(Sender: TObject);
begin
	Close;
end;

procedure TFullTextSearchForm.ToolBarResearchExecute(Sender: TObject);
begin
	FConfig.TargetThread := fttFixed;
	if TFullTextSearchConfigDlg.Execute(Self, FConfig) then
		StartSearch;
end;

procedure TFullTextSearchForm.ToolBarCancelUpdate(Sender: TObject);
begin
	(Sender as TAction).Enabled := FInSearching;
end;

procedure TFullTextSearchForm.ToolBarCancelExecute(Sender: TObject);
begin
	TerminateThread;

	StatusText := Format('𒆎~܂(%dqbg)B',
		[ListView.Items.Count]);
	FProgressBar.Position := 1;
end;

procedure TFullTextSearchForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
	{ EBhEƃ }
	Action := caFree;
end;

procedure TFullTextSearchForm.ToolBarNextItemExecute(Sender: TObject);
var
	Index: Integer;
begin
	Index := -1;

	if ListView.Selected = nil then
	begin
		if ListView.Items.Count > 0 then
			Index := 0;
	end else
	begin
		Index := ListView.ItemIndex;
		Inc(Index);
		if Index >= ListView.Items.Count then Index := 0;
		ListView.Selected := nil;
	end;

	if Index >= 0 then
	begin
		ListView.ItemIndex := Index;
		ListView.Items[Index].MakeVisible(False);
		ListViewClick(Self);
		Memo.SetFocus;
	end;
end;

{ TFullTextSearchThread }

procedure TFullTextSearchThread.Complete;
begin
	FOwner.SearchComplete(FMatchCount);
end;

constructor TFullTextSearchThread.Create(AOwner: TFullTextSearchForm; List: T2chIndexList; Config: TFullTextSearchConfig);
begin
	FConfig := Config;
	FIndexList := List;
	FOwner := AOwner;

	inherited Create(False);
end;

destructor TFullTextSearchThread.Destroy;
begin
	inherited;
end;

procedure TFullTextSearchThread.Execute;
var
	br, Tag: regex_t;
	DatList: T2chThreadDat;
	I, J: Integer;
	RIndex, RLength: Integer;
	SearchRegExp: TbmRExp;
	SearchText, Text: string;
	ThreadIdx: T2chThreadIndex;
begin
	DatList := nil;
	SearchRegExp := nil;
	AWKClearPattern(br);
	AWKClearPattern(Tag);

	try
		DatList := T2chThreadDat.Create;
		SearchRegExp := TbmRExp.Create;
		
		AWKCompilePattern(br, '<[Bb][Rr]>');
		AWKCompilePattern(Tag, '<[^>]+>');

		if FConfig.UseRegularExpression then
		begin
			SearchRegExp.AsciiIgnoreCase := not FConfig.CaseInsensitive;
			SearchRegExp.RegExp := FConfig.SearchText;
		end else
		begin
			SearchText := FConfig.SearchText;
			if not FConfig.CaseInsensitive then
				SearchText := AnsiLowerCase(SearchText);
		end;
			
		for I := 0 to FIndexList.Count - 1 do
		begin
			if Terminated then Exit;

			ThreadIdx := FIndexList[I];
			DatList.LoadFromFile(TKatjusha.GetKatjushaBaseDir + ChangeFileExt('log\' + ThreadIdx.LogPath, '.dat'));

			FProgressIndex := I + 1;
			FProgressMax := FIndexList.Count;
			FProgressText := Format('[%d/%d]''%s''c',
				[I + 1, FIndexList.Count, ThreadIdx.ThreadName]);
			Synchronize(Notify);

			for J := 0 to DatList.Count - 1 do
			begin
				if Terminated then Exit;

				case FConfig.Target of
					ftsAll: Text := DatList[J];
					ftsName: Text := DatList.Names[J];
					ftsMail: Text := DatList.Mails[J];
					ftsDate, ftsID: Text := DatList.Dates[J];
					ftsMessage: Text := DatList.Messages[J];
				end;

				if FConfig.IgnoreSpace then
				begin
					Text := StringReplace(Text, '@', '', [rfReplaceAll]);
					Text := StringReplace(Text, ' ', '', [rfReplaceAll]);
					Text := AWKGSub(br, '', Text);
				end;

				if not FConfig.IgnoreSpace then
					Text := AWKGSub(br, #13#10, Text);

				if FConfig.IgnoreTag then
					Text := AWKGSub(Tag, '', Text);

				if Terminated then Exit;

				if FConfig.UseRegularExpression then
				begin
					if bmREMatch(SearchRegExp, Text, 1, RIndex, RLength) <> 0 then
						Hit(Text, RIndex, J + 1, DatList[J], ThreadIdx);
				end else
				begin
					if not FConfig.CaseInsensitive
						then RIndex := AnsiPos(SearchText, AnsiLowerCase(Text))
						else RIndex := AnsiPos(SearchText, Text);
					if RIndex > 0 then
						Hit(Text, RIndex, J + 1, DatList[J], ThreadIdx);
				end;
			end;
		end;

		Synchronize(Complete);
	finally
		AWKFreePattern(br);
		AWKFreePattern(Tag);
		DatList.Free;
		SearchRegExp.Free;
	end;
end;

procedure TFullTextSearchThread.Hit(Text: string; Index, Number: Integer;
	const Line: string; ThreadIdx: T2chThreadIndex);
var
	SubStr: string;
	I: Integer;
{ RPos }
	function RPos(const SubStr: string; Str: string): Integer;
	var
		I: Integer;
	begin
		Result := 0;
		repeat
			I := Pos(SubStr, Str);
			if I > 0 then
			begin
				if Result > 0 then Result := Result + Length(SubStr) - 1;
				Result := Result + I;
				Str := Copy(Str, I + Length(SubStr), MaxInt);
			end;
		until I <= 0;
	end;
{ MinNotZero }
	function MinNotZero(const A: array of Integer): Integer;
	var
		I: Integer;
	begin
		Result := MaxInt;
		for I := 0 to High(A) do
			if A[I] <> 0 then Result := Min(Result, A[I]);
		if Result = MaxInt then Result := 0;
	end;
begin
  { qbg }
	Inc(FMatchCount);
	
	{ ʕ\p̕ }
	SubStr := Copy(Text, 1, Index);
	I := Max(RPos('<>', SubStr), RPos(#13#10, SubStr));
	if 0 < I then
		SubStr := Copy(SubStr, I + 2, MaxInt);
	Text := Copy(Text, Index + 1, MaxInt);
	I := MinNotZero([Pos('<>', Text), Pos(#13#10, Text)]);
	if 0 < I then
		Text := Copy(Text, 1, I - 1);
	Text := SubStr + Text;
	
	Text := HTML2String(Text);
	Text := AWKGSub('^[ ]+', '', Text);
	Text := AWKGSub('[ ]+$', '', Text);

	FInfo.Title := StringReplace(ThreadIdx.ThreadName, 'M', ',', [rfReplaceAll]);
	FInfo.Text := Text;
	FInfo.Line := Line;
	FInfo.Number := Number;
	FInfo.ThreadIdx := ThreadIdx;

	Synchronize(SendResult);
end;

procedure TFullTextSearchThread.Notify;
begin
	FOwner.NotifyProgress(FProgressIndex, FProgressMax, FProgressText);
end;

procedure TFullTextSearchThread.SendResult;
begin
	FOwner.CatchResult(@FInfo);
end;

procedure TFullTextSearchForm.BrowserCallKatjushaExecute(Sender: TObject);
var
	List: TStringList;
	I: Integer;
begin
	List := nil;
	try
		List := TStringList.Create;
		List.Text := MakeURLString([ustURL]);
		for I := 0 to List.Count - 1 do
			CallKatjusha(List[I]);
	finally
		List.Free;
	end;
end;

procedure TFullTextSearchForm.BrowserCallStandardExecute(Sender: TObject);
var
	List: TStringList;
	I: Integer;
begin
	List := nil;
	try
		List := TStringList.Create;
		List.Text := MakeURLString([ustURL]);
		for I := 0 to List.Count - 1 do
			CallBrowser(List[I]);
	finally
		List.Free;
	end;
end;

procedure TFullTextSearchForm.EditCopyURLExecute(Sender: TObject);
var
	URL: string;
	List: TStringList;
begin
	List := nil;
	try
		List := TStringList.Create;
		URL := MakeURLString([ustURL]);
		List.Text := URL;
		if List.Count = 1 then URL := List[0];
		if URL <> '' then
			Clipbrd.Clipboard.AsText := URL;
	finally
		List.Free;
	end;
end;

procedure TFullTextSearchForm.EditCopyTitleWithURLExecute(Sender: TObject);
var
	URL: string;
begin
	URL := MakeURLString([ustTitle, ustURL]);
	if URL <> '' then
		Clipbrd.Clipboard.AsText := URL;
end;

function TFullTextSearchForm.ArrangeMessage(
  AInfo: PFullTextSearchInfo): string;
var
	Text: string;
	DatArray: array [0 .. 4] of string;
	List: TStringList;
	I, L: Integer;
	LastSpace: Boolean;
begin
	Result := '';
	List := nil;

	try
		List := TStringList.Create;

		StringSplit(AInfo.Line, '<>', DatArray);
		Text := Format('%d O:%s [%s] %s',
			[AInfo.Number, HTML2String(DatArray[0]), DatArray[1], AWKGSub('<[^>]+>', '', DatArray[2])]);
		Text := Text + #13#10 + HTML2String(DatArray[3]);

		List.Text := Text;
		LastSpace := True;
			
		for I := List.Count - 1 downto 0 do
		begin
			Text := List[I];
			L := Length(Text);
			{ 擪Ɩ̋󔒂̍폜 }
			if (L > 0) and (Text[1] = ' ') then
			begin
				Text := Copy(Text, 2, MaxInt);
				Dec(L);
			end;
			if (L > 0) and (Text[L] = ' ') then
				Text := Copy(Text, 1, L - 1);
			if LastSpace and (Text = '') then
				List.Delete(I)
			else
			begin
				List[I] := Text;
				LastSpace := False;
			end;
		end;

		Result := List.Text;
	finally
		List.Free;
	end;
end;

procedure TFullTextSearchForm.EditCopyMessageWithTitleAndURLExecute(
  Sender: TObject);
var
	Text: string;
begin
	Text := MakeURLString([ustTitle, ustURL, ustMessage]);
	if Text <> '' then
		Clipbrd.Clipboard.AsText := Text;
end;

function TFullTextSearchForm.MakeURLString(
	ATypes: TFullTextSearchURLStringTypes): string;
var
	Item: TListItem;
	PInfo: PFullTextSearchInfo;
	Text: string;
	LogPath: string;
	List: TList;
	I, J, K, L: Integer;
	FirstTime: Boolean;
begin
	Result := '';
	List := nil;

	try
		List := TList.Create;
		
		with ListView do
		begin
			if Selected = nil then Exit;
			Item := Selected;
			repeat
				PInfo := PFullTextSearchInfo(Item.Data);
				List.Add(PInfo);
				Item := GetNextItem(Item, sdAll, [isSelected]);
			until Item = nil;
		end;

		LogPath := '';
		I := 0;
		while I < List.Count do
		begin
			PInfo := PFullTextSearchInfo(List[I]);
			LogPath := PInfo.ThreadIdx.LogPath;

			if (ustMessage in ATypes) and (I > 0) then
				Text := Text + #13#10;
				
			if ustTitle in ATypes then
				Text := Text + PInfo.Title + #13#10;

			J := I + 1;
			while J < List.Count do
			begin
				if LogPath <> PFullTextSearchInfo(List[J]).ThreadIdx.LogPath then
					Break;
				Inc(J);
			end;
			Dec(J);

			if ustMessage in ATypes then
			begin
				if ustURL in ATypes then
					if (J - I) > 0 then
					begin
						Text := Text + PInfo.ThreadIdx.URL + PInfo.ThreadIdx.GetResRangePrefix(PInfo.Number,
							PFullTextSearchInfo(List[J]).Number) + #13#10;
					end else
					begin
						Text := Text + PInfo.ThreadIdx.URL +
							PInfo.ThreadIdx.GetResDirectPrefix(PInfo.Number) + #13#10;
					end;
			end;

			FirstTime := True;
			K := I;
			while K <= J do
			begin
				if ustMessage in ATypes then
				begin
					if not FirstTime then
						Text := Text + #13#10;
					Text := Text + ArrangeMessage(PFullTextSearchInfo(List[K]));
					Inc(K);
				end else if ustURL in ATypes then
				begin
					L := K;
					while L < J do
					begin
						if PFullTextSearchInfo(List[L]).Number <> (PFullTextSearchInfo(List[L + 1]).Number - 1) then
							Break;
						Inc(L);
					end;

					if K < L then
					begin
						Text := Text + PInfo.ThreadIdx.URL + PInfo.ThreadIdx.GetResRangePrefix(PFullTextSearchInfo(List[K]).Number,
							PFullTextSearchInfo(List[L]).Number) + #13#10;
						K := L + 1;
					end else
					begin
						Text := Text + PInfo.ThreadIdx.URL + PInfo.ThreadIdx.GetResDirectPrefix(PFullTextSearchInfo(List[K]).Number) + #13#10;
						Inc(K);
					end;
				end;

				FirstTime := False;
			end;

			I := J + 1;
		end;
	finally
		List.Free;
	end;

	Result := Text;
end;

procedure TFullTextSearchForm.BrowserShowSimpleBrowserExecute(
  Sender: TObject);
var
	PInfo: PFullTextSearchInfo;
begin
	if ListView.Selected = nil then Exit;
	PInfo := PFullTextSearchInfo(ListView.Selected.Data);
	Application.MainForm.BringToFront;
	Application.MainForm.Perform(WM_FAV_SHOWSIMPLEBROWSER, Integer(PInfo.ThreadIdx), PInfo.Number);
end;

procedure TFullTextSearchForm.ListViewDblClick(Sender: TObject);
begin
	BrowserShowSimpleBrowser.Execute;
end;

function TFullTextSearchForm.AbornSelected(
	AType: TFullTextSearchAbornType): Integer;
const
	AbornTypeString: array [atNormal .. atInvisible] of string =
		('ځ`', 'ځ`');
var
	Item: TListItem;
	PInfo: PFullTextSearchInfo;
	DatList: T2chThreadDat;
	LogPath: string;
	List: TList;
	ThreadList: TObjectList;
	I, Count: Integer;
begin
	DatList := nil;
	List := nil;
	ThreadList := nil;
	Count := 0;
	Result := 0;
	
	try
		List := TList.Create;
		ThreadList := TObjectList.Create(False);
		
		with ListView do
		begin
			if Selected = nil then Exit;
			Item := Selected;
			repeat
				PInfo := PFullTextSearchInfo(Item.Data);
				List.Add(PInfo);
				if ThreadList.IndexOf(PInfo.ThreadIdx) < 0 then
					ThreadList.Add(PInfo.ThreadIdx);
				Inc(Count);
				Item := GetNextItem(Item, sdAll, [isSelected]);
			until Item = nil;
		end;

		Result := Count;
		if Count <= 0 then Exit;
		if MessageDlg(Format('%dXbh/%dX%sĂ낵łH',
				[ThreadList.Count, Count, AbornTypeString[AType]]), mtWarning, [mbYes, mbNo], -1) = mrNo then
		begin
			Result := 0;
			Exit;
		end;

		for I := 0 to List.Count - 1 do
		begin
			PInfo := PFullTextSearchInfo(List[I]);
			if LogPath <> PInfo.ThreadIdx.LogPath then
			begin
				if DatList <> nil then
				begin
					if DatList.Modified then DatList.Save;
					FreeAndNil(DatList);
				end;
				LogPath := PInfo.ThreadIdx.LogPath;
				DatList := PInfo.ThreadIdx.GetThreadDat;
			end;
			DatList.Aborn(PInfo.Number - 1, (AType = atInvisible));
		end;

		if (DatList <> nil) and DatList.Modified then
			DatList.Save;
	finally
		FreeAndNil(DatList);
		ThreadList.Free;
		List.Free;
	end;
end;

procedure TFullTextSearchForm.EditAbornExecute(Sender: TObject);
var
	Count: Integer;
begin
	Count := AbornSelected(atNormal);
	if Count > 0 then
		ShowMessage(Format('%d̃Xځ`񂵂܂B', [Count]));
end;

procedure TFullTextSearchForm.EditInvisibleAbornExecute(Sender: TObject);
var
	Count: Integer;
begin
	Count := AbornSelected(atInvisible);
	if Count > 0 then
		ShowMessage(Format('%d̃X𓧖ځ`񂵂܂B', [Count]));
end;

end.

