require 'PageParser.rb'

module Wiki

class PageDataSource
	
	def initialize(data_dir)
		self.data_dir = data_dir
		unless File.exist?(data_dir) then
			Dir.mkdir(data_dir)
		end
		self.use_cache = true
		@parser = nil;
		@document_cache = {}
	end

	attr_reader 'data_dir'
	attr 'use_cache', true
	attr 'use_memory_cache', true

	def load_page_document(page_name, lang, version = nil)
		document = nil
		page_id = make_page_id(page_name, lang, version)

		property = page_property(page_name, lang, version)
		if use_memory_cache then
			doc, cache_time = @document_cache[page_id]
			if property and cache_time and cache_time >= property['source_time'] then
				document = doc 
			end
		end
		if document == nil and use_cache then 
			if property and property['cache_time'] >= property['source_time'] then
				document = load_cached_document(page_name, lang, version)
			end
		end
		if document == nil then
			page_source = load_page_source(page_name, lang, version)
			document =  page_parser.parse(page_source);
			if use_cache and document then
				save_cached_document(page_name, lang, version, document) 
			end
		end

		if use_memory_cache then
			@document_cache[page_id] = [document, Time.now]
		end

		document
	end

	def load_page_source(page_name, lang, version = nil)
	end

	def save_page_source(page_name, lang, version, source)
		page_id = make_page_id(page_name, lang)
		@document_cache[page_id] = nil
	end

	def delete_page(page_name, lang, version)
	end

	def exist?(page_name, lang, version = nil)
	end

	def exist_page?(page_name)
	end

	def page_property(page_name, lang, version, &block)
	end

	def site_property(&block)
	end

	def page_list()
		[]
	end

	def parse_page_id(page_id)
		version = nil
		page_name = page_id
		if page_name =~ /(.*)@(\d\d\d\d\d\d\d\d)$/ then
			page_name = $1
			version = $2
		end
		lang = nil
		if page_name =~ /(.*)\.(..)$/ then
			page_name = $1
			lang = $2
		end
		return [ page_name, lang, version]
	end

	def make_page_id(page_name, page_lang, page_version = nil)
		page_id = page_name;
		page_id = "#{page_id}.#{page_lang}" if page_lang
		if page_version then
			version_str = sprintf("%08d", page_version.to_s.to_i)
			page_id = "#{page_id}@#{version_str}" 
		end
		return page_id;
	end

	def suitable_page_id(page_name, ordered_lang_list, version = nil)
		page_id = nil
		ordered_lang_list.each(){ |lang|
			if self.exist?(page_name, lang, version) then
				page_id = self.make_page_id(page_name, lang, version)
				break
			else
				if self.exist?(page_name, lang) then
					page_id = self.make_page_id(page_name, lang)
					break
				end
			end
		}

		unless page_id then
			page_list = self.page_list()
			page_list.each(){ |page_info|
				id = page_info[0]
				n, l, v = parse_page_id(id)
				if n == page_name and v == nil then
					page_id = id
					break
				end
			}
		end

		return page_id;
	end

protected

	attr_writer 'data_dir'

	def default_page_property
		prop = {}
		prop['page_name'] = 'page_name'
		prop['page_title'] = 'page_title'
		prop['source_time'] = Time.now()
		prop['cache_time'] = Time.at(0)
		prop['count_int'] = 0
		prop['version_int'] = 1
		prop['passwd'] = ''
		prop['last_modified_time'] = Time.at(0);
		prop['language'] = 'en'
		prop['referers'] = ''
		return prop
	end

	def default_site_property
		prop = {}
		prop['site_name'] = 'SiteName'
		prop['admin_passwd'] = ''
		prop['admin_name'] = 'Administrator'
		prop['admin_email'] = ''
		prop['theme'] = 'default'
		prop['sidebar_control'] = 'none'
		prop['last_modified_time'] = Time.at(0);
		prop['default_language'] = 'en'
		prop['languages'] = 'en,ja'
		prop['is_frozen'] = false
		return prop
	end


	def page_parser()
		unless @page_parser then
			@page_parser = Wiki::PageParser.new()
		end
		@page_parser
	end

	def load_cached_document(page_name, lang, version)
	end

	def save_cached_document(page_name, lang, version, document)
	end
end

end # module Wiki

