// $Id: BS2SecsCmd.cpp,v 1.1.1.1 2002/08/31 04:47:24 fukasawa Exp $

//=============================================================================
/**
 *  @file    BS2SecsCmd.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 1998-2001 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#include "BS2Stream.h"
#include "BS2TclDevice.h"
#include "BS2Message.h"
#include "BS2MessageDictionary.h"
#include "BS2ItemDictionary.h"
#include "BS2MsgControl.h"
#include "tcl.h"


#define BUF_SIZE  256

Tcl_Encoding _encoding_sjis;
Tcl_Encoding _encoding_iso2022jp;
static int _equipmentCount = 0;
/*
 * Initialized
 */
enum {SECSTCL_UNKNOWN, SECSTCL_INITIALIZED};
static int _status = SECSTCL_UNKNOWN;
#define INITIALIZED_CHECK(x)                                      \
    if (_status != SECSTCL_INITIALIZED) {                         \
        Tcl_AppendResult(interp, (x), ": not initialized", NULL); \
        return TCL_ERROR; \
    }

/*
 * Received SECS message queue.
 */
static BS2MessageQueue _msg_queue;
/* 
 * Access to the list of threads and to the thread send results is
 * guarded by this mutex. 
 */
TCL_DECLARE_MUTEX(messageMutex)

//
static string _xmlName("secs.xml");

// Forward declaration
extern "C" {
int TclParseSecs(Tcl_Interp * interp, char * script, int size,
                 void ** msgptr);
}
extern int  parseSecsXMLFile(char * filename);

int BS2QueueMessage(BS2Message * msg);
BS2Message * BS2DequeueMessage();
static int  _getSECSParameter(CommParameter& comm_parm, Tcl_Interp *interp,
                              int argc, Tcl_Obj *CONST objv[]);
static int  _getHSMSParameter(SocketParameter& socket_parm, Tcl_Interp *interp,
                              int argc, Tcl_Obj *CONST objv[]);
static int  _strToRole(char * varstr);
static int  _strToProtocol(char * varstr);
static int  _strToBaudrate(char * varstr);
static int  _strToFlow(char * varstr);
static char * _strToParity(char * varstr);
static int  _checkCommPort(char * varstr);
static int  ParseStreamFunctionNum(Tcl_Interp * interp, char * sfptr,
                                  int * sfnum);
static int  parse_bytes(Tcl_Interp * interp, char * datastr, BYTE ** bufpp,
                        int * size);
static int  parse_words(Tcl_Interp * interp, char * datastr, BYTE ** bufpp,
                        int * size);

/*
 *----------------------------------------------------------------------
 *
 * Secs_initializeCmd --
 *
 *  This procedure is invoked to process the "secs::init" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_initializeCmd(ClientData cdata, Tcl_Interp * interp, int objc,
                       Tcl_Obj *CONST objv[])
{
    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;
    if (objc > 2)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "init ?fileName?");
        return TCL_ERROR;
    }
    else if (objc == 2)
    {
        _xmlName = Tcl_GetString(objv[1]);
    }

    equipInfo->m_tcl.interp(interp);
    _status = SECSTCL_INITIALIZED;
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Secs_createCmd --
 *
 *  This procedure is invoked to process the "secs::create" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_createCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                   Tcl_Obj *CONST objv[])
{
    static char *createopts[] = {
        "-role",
        "-protocol",
        "-devid",
        "-srcid",
        "-name",
        "-global",
        "-parms",
        NULL
    };
    enum createopts {
        CREAT_ROLE,
        CREAT_PROTOCOL,
        CREAT_DEVID,
        CREAT_SRCID,
        CREAT_NAME,
        CREAT_GLOBAL,
        CREAT_PARMS
    };

    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;
    int elemc, end, i, optindex, result;
    int role = 0, protocol = 0, deviceNum = 0, sourceNum = 0;
    char * equipName = NULL;
    char * varName = NULL;
    BS2TclDevice * device;
    SocketParameter socket_parm;
    CommParameter   comm_parm;
    DeviceParameter * dev_parm;
    char *arg, buf[BUF_SIZE];
    struct Tcl_Obj **elemv;

    INITIALIZED_CHECK("secs::create");

    if (objc < 3)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "?role? ?protocol? ?args?");
        return TCL_ERROR;
    }

    /*
     * Get the command name index from the object based on the options
     * defined above.
     */
    result = TCL_OK;
    end = objc;
    i = 1;
    while (i < end)
    {
        if (Tcl_GetIndexFromObj(interp, objv[i], createopts, "option",
                                TCL_EXACT, &optindex) != TCL_OK)
        {
            return TCL_ERROR;
        }
        i++;
        switch ((enum createopts)optindex)
        {
        case CREAT_ROLE:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-role host|equipment?");
                result = TCL_ERROR;
                break;
            }
            arg = Tcl_GetString(objv[i++]);
            role = _strToRole(arg);
            if (role < 0)
            {
                sprintf(buf, "Create: Invalid role: %s\n", arg);
                Tcl_SetResult(interp, buf, TCL_VOLATILE);
                result = TCL_ERROR;
            }
            break;
        case CREAT_PROTOCOL:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-protocol serial|socket?");
                result = TCL_ERROR;
                break;
            }
            arg = Tcl_GetString(objv[i++]);
            protocol = _strToProtocol(arg);
            if (protocol < 0)
            {
                sprintf(buf, "Create: Invalid protocol: %s\n", arg);
                Tcl_SetResult(interp, buf, TCL_VOLATILE);
                result = TCL_ERROR;
            }
            break;
        case CREAT_DEVID:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-devid id?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetIntFromObj(interp, objv[i++], &deviceNum);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid device id\n",
                              TCL_STATIC);
            }
            break;
        case CREAT_SRCID:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-srcid id?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetIntFromObj(interp, objv[i++], &sourceNum);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid source id\n",
                              TCL_STATIC);
            }
            break;
        case CREAT_NAME:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-name deviceName?");
                result = TCL_ERROR;
                break;
            }
            equipName = Tcl_GetString(objv[i++]);
            if (equipName == NULL)
            {
                sprintf(buf, "Create: Invalid equipment name: %s\n", arg);
                Tcl_SetResult(interp, buf, TCL_VOLATILE);
                result = TCL_ERROR;
            }
            break;
        case CREAT_GLOBAL:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-global varName?");
                result = TCL_ERROR;
                break;
            }
            varName = Tcl_GetString(objv[i++]);
            if (varName == NULL)
            {
                sprintf(buf, "Create: Invalid variable name: %s\n", arg);
                Tcl_SetResult(interp, buf, TCL_VOLATILE);
                result = TCL_ERROR;
            }
            break;
        case CREAT_PARMS:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, 
                                 "?-parms {?pname? ?value?}?");
                result = TCL_ERROR;
                break;
            }
            /*
             * Get sublist as {?pname? ?value?}
             */
            result = Tcl_ListObjGetElements(interp, objv[i++], &elemc,
                                                               &elemv);
            if (is_odd(elemc))
            {
                Tcl_SetResult(interp,
                    "Parameter list must be {?pname? ?value?}", TCL_STATIC);
                result = TCL_ERROR;
            }
            break;

        }
        if (result != TCL_OK)
            break;
    }

    if (result == TCL_ERROR)
        return (result);

    if (role == BS2Device::EQUIPMENT)
    {   // Check equipment count
        if (_equipmentCount > 0)
        {
            Tcl_SetResult(interp, "Create: Equipment is created.\n",
                          TCL_STATIC);
            return TCL_ERROR;
        }
        _equipmentCount = 1;
    }

    //
    // Set parameter to the device
    //
    if (protocol == DRIVER_SERIAL)
    {   // serial
        comm_parm.m_slave = (role == BS2Device::EQUIPMENT) ? 0 : 1;
        comm_parm.m_deviceId = deviceNum;
        comm_parm.m_sourceId = sourceNum;
        // initial communication parameters.
        if (_getSECSParameter(comm_parm, interp, elemc, elemv) != TCL_OK)
        {
            return TCL_ERROR;
        }
        dev_parm = &comm_parm;
    }
    else if (protocol == DRIVER_SOCKET)
    {   // socket
        socket_parm.m_mode = (role == BS2Device::EQUIPMENT) ? 1 : 0;
        socket_parm.m_slave = (role == BS2Device::EQUIPMENT) ? 0 : 1;
        socket_parm.m_deviceId = deviceNum;
        socket_parm.m_sourceId = sourceNum;
        if (_getHSMSParameter(socket_parm, interp, elemc, elemv) != TCL_OK)
        {
            return TCL_ERROR;
        }
        dev_parm = &socket_parm;
    }
    else
    {
        Tcl_SetResult(interp,
                "Create: Illegal driver type, type is serial or socket.\n",
                TCL_STATIC);
        return TCL_ERROR;
    }

    device = new BS2TclDevice(equipName);
    device->setGlobalName(varName);
    device->setInterp(interp);
    Tcl_ThreadId id = Tcl_GetCurrentThread();
    device->setThreadId(id);
    strcpy(dev_parm->m_xmlname, _xmlName.c_str());   // set xml file name

    if (device->open(dev_parm) < 0)
    {
        sprintf(buf, "Create: Abort initialed device: %s\n", equipName);
        Tcl_SetResult(interp, buf, TCL_VOLATILE);
        delete device;
        _equipmentCount = 0;
        return TCL_ERROR;
    }

    ACE_OS::sleep(1);   // must ! wait for establish connection.

    equipInfo->m_device = device;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Secs_viewCmd --
 *
 *  This procedure is invoked to process the "secs::view" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_viewCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                 Tcl_Obj *CONST objv[])
{
    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;

    INITIALIZED_CHECK("secs::view");

    if (objc >= 2)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "");
        return TCL_ERROR;
    }


    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Secs_deleteCmd --
 *
 *  This procedure is invoked to process the "secs::delete" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_deleteCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                 Tcl_Obj *CONST objv[])
{
    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;
    INITIALIZED_CHECK("secs::delete");
    if (objc >= 2)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "");
        return TCL_ERROR;
    }

    BS2TclDevice * device = equipInfo->m_device;
    if (device == NULL)
    {
        Tcl_SetResult(interp,
                      "secs::delete : object not found \n", TCL_STATIC);
        return TCL_ERROR;
    }

    device->stopLoop();    // stop receive command

    ACE_OS::sleep(1);

    device->close();
    delete device;

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Secs_sendCmd --
 *
 *  This procedure is invoked to process the "secs::send" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_sendCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                 Tcl_Obj *CONST objv[])
{
    int result;
    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;
    BS2Message * msg;

    INITIALIZED_CHECK("secs::send");

    if (objc != 2)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "send message");
        return TCL_ERROR;
    }

    BS2TclDevice * device = equipInfo->m_device;
    if (device == NULL)
    {
        Tcl_SetResult(interp, "Send: object not found \n", TCL_STATIC);
        return TCL_ERROR;
    }

    char * messageStr = Tcl_GetString(objv[1]);
    result = TclParseSecs(interp, messageStr, strlen(messageStr), (void **)&msg);
    if (result != TCL_OK)
    {
        Tcl_SetResult(interp, "Send : Illegal message data.\n", TCL_STATIC);
        return result;
    }

    // Send secs message.
    result = device->send(msg);
    if (result < 0)
    {
        Tcl_SetResult(interp, "Send: i/o error.\n", TCL_STATIC);
        delete msg;
        return TCL_ERROR;
    }

    delete msg;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Secs_receiveCmd --
 *
 *  This procedure is invoked to process the "secs::receive" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_receiveCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                    Tcl_Obj *CONST objv[])
{
    int result;
    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;
    BS2Message * msg;

    INITIALIZED_CHECK("secs::receive");
    if (objc > 3)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "receive ?args ?");
        return TCL_ERROR;
    }
    BS2TclDevice * device = equipInfo->m_device;
    if (device == NULL)
    {
        Tcl_SetResult(interp, "Receive: object not found \n", TCL_STATIC);
        return TCL_ERROR;
    }

    if (objc == 2 && EQUALSTR(Tcl_GetString(objv[1]), "-stop"))
    {
        // device->m_device->stop_receive();
        Tcl_SetResult(interp, "Receive: Not supported \n", TCL_STATIC);
        return TCL_ERROR;
    }
    else if (objc == 3 && EQUALSTR(Tcl_GetString(objv[1]), "-limit"))
    {
        int sec;
        result = Tcl_GetIntFromObj(interp, objv[2], &sec);
        if (result != TCL_OK)
        {
            Tcl_SetResult(interp, "Receive: Illegal limit time.\n", TCL_STATIC);
            return TCL_ERROR;
        }
        if (sec > 0)
        {   // check while received message
            msg = BS2DequeueMessage();
            while (msg == NULL && sec > 0)
            {
                ACE_OS::sleep(1);
                msg = BS2DequeueMessage();
                sec--;
            }
        }
        else
        {
            msg = BS2DequeueMessage();
        }
    }
    else if (objc == 1)
    {
        msg = BS2DequeueMessage();
    }
    else
    {
        Tcl_SetResult(interp, "Receive: Illegal arguments.\n", TCL_STATIC);
        return TCL_ERROR;
    }

    // Convert received message to string.
    if (msg != NULL)
    {
        msg->print(&equipInfo->m_tcl);
        delete msg;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Secs_convertCmd --
 *
 *  This procedure is invoked to process the "secs::convert" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_convertCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                    Tcl_Obj *CONST objv[])
{
    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;
    int  result = TCL_ERROR;
    BS2Message * msg = NULL;
    BS2OStream * buff;
    BYTE * data_ptr;
    int  data_size;
    int  sfnum;
    BYTE trandata[10];

    INITIALIZED_CHECK("secs::convert");

    if (objc != 4)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "convert SxxFxx type hex-dump");
        return TCL_ERROR;
    }

    result = ParseStreamFunctionNum(interp, Tcl_GetString(objv[1]), &sfnum);
    if (result != TCL_OK)
        return TCL_ERROR;

    trandata[2] = sfnum >> 8;
    trandata[3] = sfnum & 0xFF;
    trandata[4] = 0x80;              // end bit

    // convert hex data to secs message
    if (EQUALSTR(Tcl_GetString(objv[2]), "byte"))
    {
        result = parse_bytes(interp, Tcl_GetString(objv[3]), &data_ptr,
                                                             &data_size);
        if (result != TCL_OK)
            return TCL_ERROR;
    }
    else if (EQUALSTR(Tcl_GetString(objv[2]), "word"))
    {
        result = parse_words(interp, Tcl_GetString(objv[3]), &data_ptr,
                                                             &data_size);
        if (result != TCL_OK)
            return TCL_ERROR;
    }
    else
    {
        Tcl_SetResult(interp,
                      "Illegal data format: format is \"byte\" or \"word\".\n",
                      TCL_STATIC);
        return TCL_ERROR;
    }

    buff = new BS2OStream;
    buff->write((const char *)trandata, sizeof(trandata));
    buff->write((const char *)data_ptr, data_size);

    msg = BS2MessageDictionary::instance()->make(buff);
    msg->print(&equipInfo->m_tcl);
    delete msg;
    delete buff;
    free(data_ptr);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Secs_testCmd --
 *
 *  This procedure is invoked to process the "secs::test" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_testCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                 Tcl_Obj *CONST objv[])
{
    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;
    int  result = TCL_ERROR;
    BS2Message * msg = NULL;

    INITIALIZED_CHECK("secs::test");

    if (objc != 2)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "test message");
        return TCL_ERROR;
    }

    char * messageStr = Tcl_GetString(objv[1]);
    result = TclParseSecs(interp, messageStr, strlen(messageStr), (void **)&msg);
    if (result != TCL_OK)
    {
        Tcl_SetResult(interp, "Test: Illegal message data.\n", TCL_STATIC);
        return result;
    }

    // convert secs message to stream data
    BS2OStream * buff = new BS2OStream;
    if (buff->set(msg) == true)
    {
        buff->print(interp);
    }
    delete msg;

    // Convert stream data to secs message.
    BS2Message * newmsg = BS2MessageDictionary::instance()->make(buff);
    if (newmsg == NULL)
    {
        Tcl_SetResult(interp, " Test: Illegal I/O data format.\n", TCL_STATIC);
        return TCL_ERROR;
    }

    newmsg->print(&equipInfo->m_tcl);     // MESSAGE {... WAIT} SxFx ....
    delete newmsg;
    delete buff;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Secs_configCmd --
 *
 *  This procedure is invoked to process the "secs::config" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_configCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                   Tcl_Obj *CONST objv[])
{
    static char *confopts[] = {
        "-item",
        "-xml",
        "-tcl",
        NULL
    };
    enum confopts {
        CONF_ITEM,
        CONF_XML,
        CONF_TCL,
    };

    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;
    int i, end, optindex, result;
    int sw_item, sw_xml;

    INITIALIZED_CHECK("secs::config");

    if (objc < 3)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "parm ?args?");
        return TCL_ERROR;
    }

    /*
     * Get the command name index from the object based on the options
     * defined above.
     */
    result = TCL_OK;
    end = objc;
    i = 1;
    while (i < end)
    {
        if (Tcl_GetIndexFromObj(interp, objv[1], confopts, "option", TCL_EXACT,
                                &optindex) != TCL_OK)
        {
            return TCL_ERROR;
        }
        i++;
        switch ((enum confpts)optindex)
        {
        case CONF_ITEM:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-item true|false?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetBooleanFromObj(interp, objv[i++], &sw_item);
            if (result == TCL_OK)
            {
                equipInfo->m_tcl.set_tag((sw_item != 0) ? true : false);
            }
            else
            {
                Tcl_SetResult(interp, "-item true|false.\n", TCL_STATIC);
            }
            break;
        case CONF_XML:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-xml true|false?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetBooleanFromObj(interp, objv[i++], &sw_xml);
            if (result == TCL_OK)
            {
                equipInfo->m_tcl.set_xml((sw_xml != 0) ? true : false);
            }
            else
            {
                Tcl_SetResult(interp, "-xml true|false.\n", TCL_STATIC);
            }
            break;
        case CONF_TCL:
            Tcl_SetResult(interp, "config -tcl: Not supported, yet.\n",
                                  TCL_STATIC);
            result = TCL_ERROR;
            break;
        }

        if (result != TCL_OK)
            break;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Secs_dictionaryCmd --
 *
 *  This procedure is invoked to process the "secs::dictionary" Tcl
 *  command. See the user documentation for details on what it does.
 *
 * Results:
 *  A standard Tcl result.
 *
 * Side effects:
 *  See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Secs_dictionaryCmd(ClientData cdata, Tcl_Interp *interp, int objc,
                    Tcl_Obj *CONST objv[])
{
    static char *dictopts[] =
    {
        "dump", "delitem", "delmsg", "add",
        NULL
    };
    enum dictopts
    {
        DICT_DUMP,  DICT_DELITEM,  DICT_DELMSG,  DICT_ADD
    };

    int  result, sfnum, index;
    char * arg;
    Tcl_Equipment * equipInfo = (Tcl_Equipment *)cdata;

    INITIALIZED_CHECK("secs::dictionary");

    if (objc == 1 || objc > 4)
    {
        Tcl_WrongNumArgs(interp, 1, objv, "dictionary sub-command ?args?");
        return TCL_ERROR;
    }
    if (Tcl_GetIndexFromObj(interp, objv[1], dictopts, "option", 0, &index) 
            != TCL_OK) {
        return TCL_ERROR;
    }
    switch (index)
    {
    case DICT_DUMP: 
        Tcl_AppendResult(interp, "# Item Dictionary\n", NULL);
        BS2ItemDictionary::instance()->print(&equipInfo->m_tcl);
        Tcl_AppendResult(interp, "\n# Message Dictionary\n", NULL);
        BS2MessageDictionary::instance()->print(&equipInfo->m_tcl);
        break;
    case DICT_DELITEM: 
        arg = Tcl_GetString(objv[3]);
        if (EQUALSTR(arg, "all"))
        {
            BS2ItemDictionary::instance()->eraseAll();
        }
        else
        {
            BS2ItemDictionary::instance()->erase(arg);
        }
        break;
    case DICT_DELMSG: 
        arg = Tcl_GetString(objv[3]);
        if (EQUALSTR(arg, "all"))
        {
            BS2MessageDictionary::instance()->eraseAll();
        }
        else
        {
            if ((result = ParseStreamFunctionNum(interp, arg, &sfnum)) != TCL_OK)
            {
                return result;
            }
            BS2MessageDictionary::instance()->erase(sfnum);
        }
        break;
    case DICT_ADD: 
        Tcl_SetResult(interp, "secs::dictionary: add is not suported yet.\n",
                              TCL_STATIC);
        return TCL_ERROR;
        break;

    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * BS2QueueMessage --
 *
 *  Queue message to the receipt message table.
 *
 * Results:
 *  None
 *
 * Side effects:
 *  The list of threads is queued to message the current thread.
 *
 *----------------------------------------------------------------------
 */

int BS2QueueMessage(BS2Message * msg)
{
    Tcl_MutexLock(&messageMutex);

    _msg_queue.push_back(msg);

    Tcl_MutexUnlock(&messageMutex);
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * BS2DequeueMessage --
 *
 *  Queue message to the receipt message table.
 *
 * Results:
 *  Receipt message
 *
 * Side effects:
 *  The list of threads is queued to message the current thread.
 *
 *----------------------------------------------------------------------
 */

BS2Message * BS2DequeueMessage()
{
    Tcl_MutexLock(&messageMutex);

    BS2Message * msg = _msg_queue.front();
    _msg_queue.pop_front();

    Tcl_MutexUnlock(&messageMutex);
    return msg;
}

//-----------------------------------------------------------------------------
// Get parameter for SECS-I
//-----------------------------------------------------------------------------
int _getSECSParameter(CommParameter& comm_parm, Tcl_Interp *interp,
                       int objc, Tcl_Obj *CONST objv[])
{
    static char * secsopts[] = {
        "-baudrate",
        "-port",
        "-t2",
        "-t3",
        "-t4",
        "-retry",
        "-parity",
        "-databit",
        "-stopbit",
        "-flow",
        "-readtimeout",
        NULL
    };
    enum secsopts {
        SECS_BAUDRATE,
        SECS_PORT,
        SECS_T2,
        SECS_T3,
        SECS_T4,
        SECS_RETRY,
        SECS_PARITY,
        SECS_DATABIT,
        SECS_STOPBIT,
        SECS_FLOW,
        SECS_READTIMEOUT
    };

    int  end, i, optindex, result;
    int  baud_rate = 9600;
    long t2_timeout = 10*1000;
    long t3_timeout = 45*1000;
    long t4_timeout = 45*1000;
    int  rty_count = 3;
    int  data_bit = 8;
    char * parity = "none";
    int  stop_bit = 1;
    int  flow_control = 0;
    int  read_timeout = 30;
#ifdef WIN32
    char * port_id = "COM1";
#else
    char * port_id = "/dev/ttyS1";
#endif
    char *arg, buf[BUF_SIZE];

    end = objc;
    i = 0;
    while (i < end)
    {
        if (Tcl_GetIndexFromObj(interp, objv[i], secsopts, "parameter",
                                TCL_EXACT, &optindex) != TCL_OK)
        {
            return TCL_ERROR;
        }
        i++;
        switch ((enum secsopts)optindex)
        {
        case SECS_BAUDRATE:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-baudrate rate?");
                result = TCL_ERROR;
                break;
            }
            arg = Tcl_GetString(objv[i++]);
            baud_rate = _strToBaudrate(arg);
            if (baud_rate < 0)
            {
                sprintf(buf, "Create: Invalid baudrate: %s\n", arg);
                Tcl_SetResult(interp, buf, TCL_VOLATILE);
                result = TCL_ERROR;
            }
            break;
        case SECS_PORT:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-port num?");
                result = TCL_ERROR;
                break;
            }
            port_id = Tcl_GetString(objv[i++]);
            result = _checkCommPort(port_id);
            if (result != TCL_ERROR)
            {
                sprintf(buf, "Create: Invalid port: %s\n", port_id);
                Tcl_SetResult(interp, buf, TCL_VOLATILE);
                result = TCL_ERROR;
            }
            break;
        case SECS_T2:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-t2 100msec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetLongFromObj(interp, objv[i++], &t2_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid T2 timeout value\n",
                              TCL_STATIC);
            }

            if (t2_timeout < 2 || t2_timeout > 250)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid T2 timeout value (2 .. 250)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            t2_timeout *= 100;           // Convert unit from 100msec to msec
            break;
        case SECS_T3:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-t3 sec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetLongFromObj(interp, objv[i++], &t3_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid T3 timeout value\n",
                              TCL_STATIC);
            }

            if (t3_timeout < 1 || t3_timeout > 120)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid T3 timeout value (1 .. 120)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            t3_timeout *= 1000;           // Convert unit to msec
            break;
        case SECS_T4:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-t4 sec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetLongFromObj(interp, objv[i++], &t4_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid T4 timeout value\n",
                              TCL_STATIC);
            }

            if (t4_timeout < 1 || t4_timeout > 120)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid T4 timeout value (1 .. 120)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            t4_timeout *= 1000;           // Convert unit to msec
            break;
        case SECS_RETRY:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?retry num?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetIntFromObj(interp, objv[i++], &rty_count);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid retry count\n",
                              TCL_STATIC);
            }

            if (rty_count < 0 || rty_count > 31)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid retry count (0 .. 31)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            break;
        case SECS_DATABIT:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?databit num?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetIntFromObj(interp, objv[i++], &data_bit);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid databit\n",
                              TCL_STATIC);
            }

            if (data_bit < 3 || data_bit > 8)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid databit (4 .. 8)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            break;
        case SECS_PARITY:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?parity even|odd|none?");
                result = TCL_ERROR;
                break;
            }
            arg = Tcl_GetString(objv[i++]);
            parity = _strToParity(arg);
            if (parity != NULL)
            {
                sprintf(buf, "Create: Invalid parity: %s\n", arg);
                Tcl_SetResult(interp, buf, TCL_VOLATILE);
                result = TCL_ERROR;
            }
            break;
        case SECS_STOPBIT:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?stopbit num?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetIntFromObj(interp, objv[i++], &stop_bit);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid stopbit\n",
                              TCL_STATIC);
            }

            if (stop_bit != 1 && stop_bit != 2)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid stopbit (1 or 2)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            break;
        case SECS_FLOW:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?flow none|hard|xon/xoff?");
                result = TCL_ERROR;
                break;
            }
            arg = Tcl_GetString(objv[i++]);
            flow_control = _strToFlow(arg);
            if (flow_control < 0)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid flow (none, hard or xon/xoff)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            break;
        case SECS_READTIMEOUT:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-readtimeout msec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetIntFromObj(interp, objv[i++], &read_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid readtimeout value\n",
                              TCL_STATIC);
            }

            if (read_timeout < 0 || read_timeout > 2000)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid readtimeout value (0 .. 2000)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            break;
        }

        if (result != TCL_OK)
            break;
    }

    //////////////////////////////////////////////////////
    // set parameters to communication device
    strncpy(comm_parm.m_port, port_id, sizeof(comm_parm.m_port)-1);
    comm_parm.m_port[sizeof(comm_parm.m_port)-1] = '\0';
    comm_parm.m_baudrate = baud_rate;
    strncpy(comm_parm.m_parity, parity, sizeof(comm_parm.m_parity)-1);
    comm_parm.m_parity[sizeof(comm_parm.m_parity)-1] = '\0';
    comm_parm.m_databit = data_bit;
    comm_parm.m_stopbit = stop_bit;
    comm_parm.m_read_timeout = read_timeout;
    comm_parm.m_t2timeout = t2_timeout;
    comm_parm.m_retry = rty_count;
    comm_parm.m_t3timeout = t3_timeout;
    comm_parm.m_t4timeout = t4_timeout;

    return TCL_OK;
}

//-----------------------------------------------------------------------------
// Get Parameter for HSMS
//-----------------------------------------------------------------------------
int _getHSMSParameter(SocketParameter& socket_parm, Tcl_Interp *interp,
                      int objc, Tcl_Obj *CONST objv[])
{
    static char * hsmsopts[] = {
        "-hostname",
        "-port",
        "-t3",
        "-t5",
        "-t6",
        "-t7",
        "-t8",
#ifdef SUPPORT_HEARTBEAT
        "-heartbeat",
#endif
        NULL
    };
    enum hsmsopts {
        HSMS_HOSTNAME,
        HSMS_PORT,
        HSMS_T3,
        HSMS_T5,
        HSMS_T6,
        HSMS_T7,
        HSMS_T8,
#ifdef SUPPORT_HEARTBEAT
        HSMS_HEARTBEAT
#endif
    };

    int  end, i, optindex, result;
    int  port_num = 5000;
    char * hostname = "localhost";
    long t3_timeout = 45*1000;
    long t5_timeout = 10*1000;
    long t6_timeout = 5*1000;
    long t7_timeout = 10*1000;
    long t8_timeout = 5*1000;
    long hb_timeout = 0;
    char buf[BUF_SIZE];

    end = objc;
    i = 0;
	result = TCL_OK;
    while (i < end)
    {
        if (Tcl_GetIndexFromObj(interp, objv[i], hsmsopts, "parameter",
                                TCL_EXACT, &optindex) != TCL_OK)
        {
            return TCL_ERROR;
        }
        i++;
        switch ((enum hsmsopts)optindex)
        {
        case HSMS_HOSTNAME:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-hostname ip?");
                result = TCL_ERROR;
                break;
            }
            hostname = Tcl_GetString(objv[i++]);
            if (hostname == NULL)
            {
                sprintf(buf, "Create: Invalid hostname: %s\n", hostname);
                Tcl_SetResult(interp, buf, TCL_VOLATILE);
                result = TCL_ERROR;
            }
            break;
        case HSMS_PORT:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-port num?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetIntFromObj(interp, objv[i++], &port_num);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid port number\n",
                              TCL_STATIC);
            }

            if (port_num < 5000)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid port number (>=5000)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            break;
        case HSMS_T3:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-t3 sec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetLongFromObj(interp, objv[i++], &t3_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid T3 timeout value\n",
                              TCL_STATIC);
            }

            if (t3_timeout < 1 || t3_timeout > 120)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid T3 timeout value (1 .. 120)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            t3_timeout *= 1000;           // Convert unit to msec
            break;
        case HSMS_T5:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-t5 sec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetLongFromObj(interp, objv[i++], &t5_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid T5 timeout value\n",
                              TCL_STATIC);
            }

            if (t5_timeout < 1 || t5_timeout > 240)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid T5 timeout value (1 .. 240)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            t5_timeout *= 1000;           // Convert unit to msec
            break;
        case HSMS_T6:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-t6 sec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetLongFromObj(interp, objv[i++], &t6_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid T6 timeout value\n",
                              TCL_STATIC);
            }

            if (t6_timeout < 1 || t6_timeout > 240)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid T6 timeout value (1 .. 240)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            t6_timeout *= 1000;           // Convert unit to msec
            break;
        case HSMS_T7:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 2, objv, "?-t7 sec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetLongFromObj(interp, objv[i++], &t7_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid T7 timeout value\n",
                              TCL_STATIC);
            }

            if (t7_timeout < 1 || t7_timeout > 240)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid T7 timeout value (1 .. 240)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            t7_timeout *= 1000;           // Convert unit to msec
            break;
        case HSMS_T8:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 0, objv, "?-t8 sec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetLongFromObj(interp, objv[i++], &t8_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid T8 timeout value\n",
                              TCL_STATIC);
            }

            if (t8_timeout < 1 || t8_timeout > 120)
            {
                Tcl_SetResult(interp,
                              "Create: Invalid T8 timeout value (1 .. 120)\n",
                              TCL_STATIC);
                result = TCL_ERROR;
            }
            t8_timeout *= 1000;           // Convert unit to msec
            break;
#ifdef SUPPORT_HEARTBEAT
        case HSMS_HEARTBEAT:
            if (i > (end - 1))
            {
                Tcl_WrongNumArgs(interp, 0, objv, "?-heartbeat msec?");
                result = TCL_ERROR;
                break;
            }
            result = Tcl_GetIntFromObj(interp, objv[i++], &hb_timeout);
            if (result != TCL_OK)
            {
                Tcl_SetResult(interp, "Create: Invalid heart beat timeout value\n",
                              TCL_STATIC);
            }
            break;
#endif
        }
        if (result != TCL_OK)
            break;
    }


    //////////////////////////////////////////////////////
    // set parameter for communication device
    socket_parm.m_port = port_num;
    strncpy(socket_parm.m_hostname, hostname,
            sizeof(socket_parm.m_hostname)-1);
    socket_parm.m_hostname[sizeof(socket_parm.m_hostname)-1] = '\0';
    socket_parm.m_t3timeout = t3_timeout;
    socket_parm.m_t5timeout = t5_timeout;
    socket_parm.m_t6timeout = t6_timeout;
    socket_parm.m_t7timeout = t7_timeout;
    socket_parm.m_t8timeout = t8_timeout;
    socket_parm.m_hbtimeout = hb_timeout;

    return TCL_OK;
}

//-----------------------------------------------------------------------------
// Convert string to baudrate code
//-----------------------------------------------------------------------------
struct BaudConvert
{
    char * baudstr;
    int  baudnum;
};

static BaudConvert _baudConvert[] =
{
    {"110",    110},     {"300",    300},     {"600",    600},
    {"1200",   1200},    {"2400",   2400},    {"4800",   4800},
    {"9600",   9600},
    {"14400",  14400},   {"19200",  19200},
    {"38400",  38400},   {"56000",  56000},   {"57600",  57600},
    {"115200", 115200},  {"128000", 128000},  {"256000", 256000},
    {NULL,     0},
};

//
static int _strToBaudrate(char * varstr)
{
    int result = -1;
    if (varstr == NULL)
        return result;

    BaudConvert * baudTable = _baudConvert;
    while (baudTable->baudstr != NULL)
    {
        if (EQUALSTR(varstr, baudTable->baudstr))
            break;
        baudTable++;
    }
    if (baudTable->baudstr != NULL)
    {
        result = baudTable->baudnum;
    }

    return result;
}

//-----------------------------------------------------------------------------
// Check comm(serial) port name
//-----------------------------------------------------------------------------
static int _checkCommPort(char * varstr)
{
    int result = TCL_ERROR;
    if (varstr == NULL)
        return result;

#ifdef WIN32
    result = strncmp(varstr, "COM", 3);
#else
    result = strncmp(varstr, "/dev/ttyS", 9);
#endif
    return (result == 0) ? TCL_OK : TCL_ERROR ;
}


//-----------------------------------------------------------------------------
// Convert string to parity token
//-----------------------------------------------------------------------------
static char * _noparity = "NONE";
static char * _evenparity = "EVEN";
static char * _oddparity = "ODD";
// static char * _markparity ="MARK";
// static char * _spaceparity = "SPACE";

static char * _strToParity(char * varstr)
{
    char * result = NULL;
    if (varstr != NULL)
        return result;

    if (EQUALSTR(varstr, "none") || EQUALSTR(varstr, "NONE"))
        result = _noparity;
    else if (EQUALSTR(varstr, "even") || EQUALSTR(varstr, "EVEN"))
        result = _evenparity;
    else if (EQUALSTR(varstr, "odd") || EQUALSTR(varstr, "ODD"))
        result = _oddparity;
    else
        result = NULL;

    return result;
}

//-----------------------------------------------------------------------------
// Convert string to flow code
//-----------------------------------------------------------------------------
static int _strToFlow(char * varstr)
{
    int result = -1;
    if (varstr != NULL)
        return result;

    if (EQUALSTR(varstr, "none"))
        result = 0;
    else if (EQUALSTR(varstr, "hard"))
        result = 1;
    else if (EQUALSTR(varstr, "xon/xoff"))
        result = 2;
    else
        result = -1;

    return result;
}

//-----------------------------------------------------------------------------
// Convert string to role code.
//-----------------------------------------------------------------------------
static int _strToRole(char * varstr)
{
    int result = -1;
    if (varstr == NULL)
        return result;

    if (EQUALSTR(varstr, "host"))
    {
        result = BS2Device::HOST;
    }
    else if (EQUALSTR(varstr, "equipment"))
    {
        result = BS2Device::EQUIPMENT;
    }
    return result;
}

//-----------------------------------------------------------------------------
// Convert string to protocol code.
//-----------------------------------------------------------------------------
static int _strToProtocol(char * varstr)
{
    int result = -1;
    if (varstr == NULL)
        return result;

    if (EQUALSTR(varstr, "serial"))
    {
        result = DRIVER_SERIAL;
    }
    else if (EQUALSTR(varstr, "socket"))
    {
        result = DRIVER_SOCKET;
    }
    return result;
}

//-----------------------------------------------------------------------------
//  Convert stream function number
//-----------------------------------------------------------------------------
static int ParseStreamFunctionNum(Tcl_Interp * interp, char * sfptr,
                                  int * sfnum)
{
    int  snum, fnum;
    char * stop;
    char * ftop;

    stop = ftop = sfptr + 1;
    while (*ftop != '\0' && *ftop != 'f' && *ftop != 'F')
        ftop++;
    if (*ftop == '\0')
    {
        Tcl_AppendResult(interp, "Illegal stream function number", NULL);
        return TCL_ERROR;
    }
    ftop++;
    snum = strtol(stop, NULL, 10);
    fnum = strtol(ftop, NULL, 10);
    if (snum > 255 || fnum > 255)
    {
        Tcl_AppendResult(interp, "illegal stream function number", NULL);
        return TCL_ERROR;
    }
    *sfnum = ((snum << 8) + fnum);
    return TCL_OK;
}

//------------------------------------------------------------------------------
//  Parse text which is dumped memory
//------------------------------------------------------------------------------
//
// Dumped format is byte
//
static int parse_bytes(Tcl_Interp * interp, char * datastr, BYTE ** bufpp,
                       int * size)
{
    int  result;
    int  count, i, j;
    char **lists;
    ULONG hex;
    BYTE * databuf;

    result = Tcl_SplitList(interp, datastr, &count, &lists);
    if (result != TCL_OK)
        return result;

    if ((databuf = (BYTE *)malloc(count)) == NULL)
    {
        Tcl_AppendResult(interp, " empty memory ", (char *)NULL);
        return TCL_ERROR;
    }

    for (i = j = 0; i < count; i++)
    {
        hex = strtoul(lists[i], NULL, 16);
        databuf[j++] = (BYTE)(hex & 0xFF);
    }
    Tcl_Free((char *)lists);            // free splitted list
    *bufpp = databuf;
    *size = j;                          // set byte size
    return TCL_OK;
}

//-----------------------------------------------------------------------------
// Dumped format is word
//
static int parse_words(Tcl_Interp * interp, char * datastr, BYTE ** bufpp,
                       int * size)
{
    int  result;
    int  count, i, j;
    char **lists;
    ULONG hex;
    BYTE * databuf;

    result = Tcl_SplitList(interp, datastr, &count, &lists);
    if (result != TCL_OK)
        return result;

    if ((databuf = (BYTE *)malloc(count * sizeof(USHORT))) == NULL)
    {
        Tcl_AppendResult(interp, " empty memory ", (char *)NULL);
        return TCL_ERROR;
    }

    for (i = j = 0; i < count; i++)
    {
        hex = strtoul(lists[i], NULL, 16);
        databuf[j++] = (BYTE)((hex >> 8) & 0xFF);
        databuf[j++] = (BYTE)(hex & 0xFF);
    }
    Tcl_Free((char *)lists);            // free splitted list
    *bufpp = databuf;
    *size = j;                          // set word size
    return TCL_OK;
}


