/*
 * Copyright (c) 1991-2004 Kyoto University
 * Copyright (c) 2000-2004 NAIST
 * All rights reserved
 */

/* j_printf.c --- generic message output and error exit for libsent function */

/* $Id: j_printf.c,v 1.6 2004/03/23 03:00:16 ri Exp $ */

/* user can register callback for j_error() and j_exit() by
   j_error_reggister_exitfunc() and j_exit_register_func() */

#include <sent/stddefs.h>
#include <sent/tcpip.h>
#include <stdarg.h>

/* for Japanese character encoding conversion (ANY->SJIS) */
#include <jlib.h>

#if defined(_WIN32) || defined(__CYGWIN32__) /* Windows env. */
static boolean convert_enabled = TRUE;        /* default: on */
#else
static boolean convert_enabled = FALSE;       /* default: off */
#endif

static void (*j_print_func)(char *) = NULL;
static void (*j_printerr_func)(char *) = NULL;
static void (*j_flush_func)(void) = NULL;

void
j_printf_enable_jconv()
{
  convert_enabled = TRUE;
}
void
j_printf_disable_jconv()
{
  convert_enabled = FALSE;
}

void set_print_func(void (*print)(char *), void (*printerr)(char *), void (*flush)(void))
{
  j_print_func = print;
  j_printerr_func = printerr;
  j_flush_func = flush;
  return;
}

/* output message (to stdout) */
static char jbuf[4096];
int
j_printf(char *fmt, ...)
{
  va_list ap;
  int ret;
  
  va_start(ap,fmt);
  ret = vsnprintf(jbuf, 4096, fmt, ap);
  va_end(ap);
  if (convert_enabled) {
    if (j_print_func != NULL) {
      (*j_print_func)(toStringSJIS(jbuf));
    } else {
      ret = printf("%s", toStringSJIS(jbuf));
    }
  } else {
    if (j_print_func != NULL) {
      (*j_print_func)(jbuf);
    } else {
      ret = printf("%s", jbuf);
    }
  }   

  return(ret);
}

/* output error message (to stderr) */
int
j_printerr(char *fmt, ...)
{
  va_list ap;
  int ret;
  va_start(ap,fmt);
  ret = vsnprintf(jbuf, 4096, fmt, ap);
  va_end(ap);
  if (convert_enabled) {
    if (j_printerr_func != NULL) {
      (*j_printerr_func)(toStringSJIS(jbuf));
    } else {
      ret = fprintf(stderr, "%s", toStringSJIS(jbuf));
    }
  } else {
    if (j_printerr_func != NULL) {
      (*j_printerr_func)(jbuf);
    } else {
      ret = fprintf(stderr, "%s", jbuf);
    }
  }
  return(ret);
}

/* flush output */
int
j_flushprint()
{
  if (j_flush_func != NULL) {
    (*j_flush_func)();
    return 0;
  } else {
    return(fflush(stdout));
  }
  return(fflush(stdout));
}


/* exit program with error */
static void (*error_func)() = NULL;
/* register callback function for j_error() */
void
j_error_register_exitfunc(void (*f)())
{
  error_func = f;
}
/* output error and exit */
void
j_error(char *fmt, ...)
{
  va_list ap;
  int ret;
  va_start(ap,fmt);
  ret = vfprintf(stderr, fmt, ap);
  va_end(ap);
  /* clean up socket if already opened */
  cleanup_socket();
  if (error_func != NULL) (*error_func)(); /* call external error function */
  else exit(1);
  return(ret);
}

/* normal exit end  */
static void (*exit_func)() = NULL;
/* register callback function for j_exit() */
void
j_exit_register_exitfunc(void (*f)())
{
  exit_func = f;
}
/* output message and exit */
void
j_exit()
{
  /* clean up socket if already opened */
  cleanup_socket();
  if (exit_func != NULL) (*exit_func)(); /* call external error function */
  else exit(0);
}
