/*
 * Copyright (c) 1991-2004 Kyoto University
 * Copyright (c) 2000-2004 NAIST
 * All rights reserved
 */

/* rdhmmdef_dens.c --- read in hmmdefs (density) */

/* $Id: rdhmmdef_dens.c,v 1.7 2004/03/22 04:14:31 ri Exp $ */

/* currently cannot treat macros below this level except ~v (~u,~i,~x) */

#include <sent/stddefs.h>
#include <sent/htk_hmm.h>

extern char *rdhmmdef_token;	/* defined in rdhmmdef.c */

/* calculate and set GCONST */
/* GCONST = log((2*PI)^n|\sigma|) */
static void
update_gconst(HTK_HMM_Dens *d)
{
  LOGPROB gconst;
  int i;

  gconst = d->var->len * LOGTPI;
  for (i=0;i<d->var->len;i++) {
    gconst += log(d->var->vec[i]);
  }
  d->gconst = gconst;
}

/* malloc new */
static HTK_HMM_Dens *
dens_new()
{
  HTK_HMM_Dens *new;

  new = (HTK_HMM_Dens *)mybmalloc(sizeof(HTK_HMM_Dens));

  new->name = NULL;
  new->meanlen = 0;
  new->mean = NULL;
  new->var = NULL;
  new->gconst = 0.0;
  new->next = NULL;

  return(new);
}

/* add new to global structure */
void
dens_add(HTK_HMM_INFO *hmm, HTK_HMM_Dens *new)
{
  HTK_HMM_Dens *match;

  /* link data structure */
  new->next = hmm->dnstart;
  hmm->dnstart = new;

  if (new->name != NULL) {
    /* add index to search index tree */
    if (hmm->dn_root == NULL) {
      hmm->dn_root = aptree_make_root_node(new);
    } else {
      match = aptree_search_data(new->name, hmm->dn_root);
      if (strmatch(match->name, new->name)) {
	j_printerr("Error: ~m \"%s\" is already defined\n", new->name);
	rderr(NULL);
      } else {
	aptree_add_entry(new->name, new, match->name, &(hmm->dn_root));
      }
    }
  }
}

/* search HMM_Dens of the name, and return the data */
/* return NULL on failure */
HTK_HMM_Dens *
dens_lookup(HTK_HMM_INFO *hmm, char *keyname)
{
  HTK_HMM_Dens *d;

  d = aptree_search_data(keyname, hmm->dn_root);
  if (strmatch(d->name, keyname)) {
    return d;
  } else {
    return NULL;
  }
}

/* read in dens data and return new (malloced) HTK_HMM_Dens */
static HTK_HMM_Dens *
dens_read( FILE *fp, HTK_HMM_INFO *hmm)
{
  HTK_HMM_Dens *new;
  int i;

  new = dens_new();

  /* read regression class ID (just skip) */
  if (currentis("RCLASS")) {
    read_token(fp);
    NoTokErr("no RCLASS arg");
    read_token(fp);
  }
  /* read mean vector */
  if (!currentis("MEAN")) rderr("<MEAN> not found");
  read_token(fp); NoTokErr("MEAN vector length not found");
  new->meanlen = atoi(rdhmmdef_token);
  read_token(fp);
  new->mean = (VECT *)mybmalloc(sizeof(VECT) * new->meanlen);
  /* needs comversion if integerized */
  for (i=0;i<new->meanlen;i++) {
    NoTokErr("missing MEAN element");
    new->mean[i] = atof(rdhmmdef_token);
    read_token(fp);
  }

  /* read covariance matrix data */
  new->var = get_var_data(fp, hmm);
  if ((new->var)->len != new->meanlen) {
    rderr("mean vector length != variance vector len");
  }

  /* read GCONST if any */
  if (currentis("GCONST")) {
    read_token(fp);
    NoTokErr("GCONST found but no value");
    new->gconst = atof(rdhmmdef_token);
    read_token(fp);
  } else {
    /* calc */
    update_gconst(new);
  }

  return (new);
}

/* read in dens data at the current point, and
   return pointer to the data */
HTK_HMM_Dens *
get_dens_data(FILE *fp, HTK_HMM_INFO *hmm)
{
  HTK_HMM_Dens *tmp = NULL;

  if (currentis("~m")) {
    /* macro reference: lookup and return the pointer */
    read_token(fp);
    NoTokErr("missing macro name");
    tmp = dens_lookup(hmm, rdhmmdef_token);
    if (tmp == NULL) {
      j_printerr("~m \"%s\" not defined\n", rdhmmdef_token);
      rderr(NULL);
    }
    read_token(fp);
  } else if (currentis("MEAN") || currentis("RCLASS")) {
    /* definition: define density data, and return the pointer */
    tmp = dens_read(fp, hmm);
    tmp->name = NULL; /* no name */
    dens_add(hmm, tmp);
  } else {
    rderr("no density data");
  }
  return tmp;
}


/* define density data */
void
def_dens_macro(char *name, FILE *fp, HTK_HMM_INFO *hmm)
{
  HTK_HMM_Dens *new;

  /* read in data and return newly malloced data */
  new = dens_read(fp, hmm);

  /* register it to the grobal HMM structure */
  new->name = name;
  dens_add(hmm, new);
}
