/*
 * Copyright (c) 2003 NAIST
 * All rights reserved
 */

/* read_binhmm.c --- read HTK_HMM_INFO from disk in binary format */

/* $Id: read_binhmm.c,v 1.2 2003/12/08 03:44:38 ri Exp $ */

#include <sent/stddefs.h>
#include <sent/htk_param.h>
#include <sent/htk_hmm.h>

#undef DMES			/* define to enable debug message */

/* binary read function with byte swap (assume file is BIG ENDIAN) */
static void
rdn(FILE *fp, void *buf, size_t unitbyte, int unitnum)
{
  size_t tmp, count;
  if ((tmp = myfread(buf, unitbyte, unitnum, fp)) < unitnum) {
    perror("ngram_read_bin");
    j_error("read failed\n");
  }
#ifndef WORDS_BIGENDIAN
  if (unitbyte != 1) {
    swap_bytes(buf, unitbyte, unitnum);
  }
#endif
}

/* read in a string (new buffer allocated) */
static char buf[MAXLINELEN];
static char *
rdn_str(FILE *fp)
{
  int c;
  int len;
  char *p;

  len = 0;
  while ((c = fgetc(fp)) != EOF) {
    if (len >= MAXLINELEN) j_error("Error: string len exceeded %d bytes\n", len);
    buf[len++] = c;
    if (c == '\0') break;
  }
  if (len == 1) {
    p = NULL;
  } else {
    p = (char *)mybmalloc(len);
    strcpy(p, buf);
  }
  return(p);
}


/* read header */
static char *binhmm_header = BINHMM_HEADER;
static boolean
rd_header(FILE *fp)
{
  char *p;
  
  p = rdn_str(fp);
  if (strmatch(p, binhmm_header)) {
    return TRUE;
  } else {
    return FALSE;
  }
}


/* read option data */
static void
rd_opt(FILE *fp, HTK_HMM_Options *opt)
{
  rdn(fp, &(opt->stream_info.num), sizeof(short), 1);
  rdn(fp, opt->stream_info.vsize, sizeof(short), 50);
  rdn(fp, &(opt->vec_size), sizeof(short), 1);
  rdn(fp, &(opt->cov_type), sizeof(short), 1);
  rdn(fp, &(opt->dur_type), sizeof(short), 1);
  rdn(fp, &(opt->param_type), sizeof(short), 1);
}

/* read type data */
static void
rd_type(FILE *fp, HTK_HMM_INFO *hmm)
{
  rdn(fp, &(hmm->is_tied_mixture), sizeof(boolean), 1);
  rdn(fp, &(hmm->maxmixturenum), sizeof(int), 1);
}


/* read transition data */
static HTK_HMM_Trans **tr_index;
static unsigned int tr_num;

static void
rd_trans(FILE *fp, HTK_HMM_INFO *hmm)
{
  HTK_HMM_Trans *t;
  unsigned int idx;
  int i;
  PROB *atmp;

  rdn(fp, &tr_num, sizeof(unsigned int), 1);
  tr_index = (HTK_HMM_Trans **)mymalloc(sizeof(HTK_HMM_Trans *) * tr_num);

  hmm->trstart = NULL;
  hmm->tr_root = NULL;
  for (idx = 0; idx < tr_num; idx++) {
    t = (HTK_HMM_Trans *)mybmalloc(sizeof(HTK_HMM_Trans));
    t->name = rdn_str(fp);
    rdn(fp, &(t->statenum), sizeof(short), 1);
    t->a = (PROB **)mybmalloc(sizeof(PROB *) * t->statenum);
    atmp = (PROB *)mybmalloc(sizeof(PROB) * t->statenum * t->statenum);
    for (i=0;i<t->statenum;i++) {
      t->a[i] = &(atmp[i*t->statenum]);
      rdn(fp, t->a[i], sizeof(PROB), t->statenum);
    }
    trans_add(hmm, t);
    tr_index[idx] = t;
  }

#ifdef DMES
  j_printf("%d transition maxtix read\n", tr_num);
#endif
}


/* write variance data */
static HTK_HMM_Var **vr_index;
static unsigned int vr_num;

static void
rd_var(FILE *fp, HTK_HMM_INFO *hmm)
{
  HTK_HMM_Var *v;
  unsigned int idx;

  rdn(fp, &vr_num, sizeof(unsigned int), 1);
  vr_index = (HTK_HMM_Var **)mymalloc(sizeof(HTK_HMM_Var *) * vr_num);
  
  hmm->vrstart = NULL;
  hmm->vr_root = NULL;
  for (idx = 0; idx < vr_num; idx++) {
    v = (HTK_HMM_Var *)mybmalloc(sizeof(HTK_HMM_Var));
    v->name = rdn_str(fp);
    rdn(fp, &(v->len), sizeof(short), 1);
    v->vec = (VECT *)mybmalloc(sizeof(VECT) * v->len);
    rdn(fp, v->vec, sizeof(VECT), v->len);
    vr_index[idx] = v;
    var_add(hmm, v);
  }
#ifdef DMES
  j_printf("%d variance read\n", vr_num);
#endif
}


/* read density data */
static HTK_HMM_Dens **dens_index;
static unsigned int dens_num;

static void
rd_dens(FILE *fp, HTK_HMM_INFO *hmm)
{
  HTK_HMM_Dens *d;
  unsigned int idx;
  unsigned int vid;

  rdn(fp, &dens_num, sizeof(unsigned int), 1);
  hmm->totalmixnum = dens_num;
  dens_index = (HTK_HMM_Dens **)mymalloc(sizeof(HTK_HMM_Dens *) * dens_num);

  hmm->dnstart = NULL;
  hmm->dn_root = NULL;
  for (idx = 0; idx < dens_num; idx++) {
    d = (HTK_HMM_Dens *)mybmalloc(sizeof(HTK_HMM_Dens));
    d->name = rdn_str(fp);
    rdn(fp, &(d->meanlen), sizeof(short), 1);
    d->mean = (VECT *)mybmalloc(sizeof(VECT) * d->meanlen);
    rdn(fp, d->mean, sizeof(VECT), d->meanlen);
    rdn(fp, &vid, sizeof(unsigned int), 1);
    d->var = vr_index[vid];
    rdn(fp, &(d->gconst), sizeof(LOGPROB), 1);
    dens_index[idx] = d;
    dens_add(hmm, d);
  }
#ifdef DMES
  j_printf("%d gaussian densities read\n", dens_num);
#endif
}


/* read tmix data */
static GCODEBOOK **tm_index;
static unsigned int tm_num;

static void
rd_tmix(FILE *fp, HTK_HMM_INFO *hmm)
{
  GCODEBOOK *tm, *match;
  unsigned int idx;
  unsigned int did;
  int i;

  rdn(fp, &tm_num, sizeof(unsigned int), 1);
  hmm->codebooknum = tm_num;
  tm_index = (GCODEBOOK **)mymalloc(sizeof(GCODEBOOK *) * tm_num);

  hmm->codebook_root = NULL;
  for (idx = 0; idx < tm_num; idx++) {
    tm = (GCODEBOOK *)mybmalloc(sizeof(GCODEBOOK));
    tm->name = rdn_str(fp);
    rdn(fp, &(tm->num), sizeof(int), 1);
    tm->d = (HTK_HMM_Dens **)mybmalloc(sizeof(HTK_HMM_Dens *) * tm->num);
    for(i=0;i<tm->num;i++) {
      rdn(fp, &did, sizeof(unsigned int), 1);
      if (did >= dens_num) {
	tm->d[i] = NULL;
      } else {
	tm->d[i] = dens_index[did];
      }
    }
    tm->id = idx;
    tm_index[idx] = tm;
    codebook_add(hmm, tm);
  }
#ifdef DMES
  j_printf("%d tied-mixture codebooks read\n", tm_num);
#endif  
}

/* read state data */
static HTK_HMM_State **st_index;
static unsigned int st_num;

static void
rd_state(FILE *fp, HTK_HMM_INFO *hmm)
{
  HTK_HMM_State *s;
  unsigned int idx;
  unsigned int did;
  int i;
  short dummy;

  rdn(fp, &st_num, sizeof(unsigned int), 1);
  hmm->totalstatenum = st_num;
  st_index = (HTK_HMM_State **)mymalloc(sizeof(HTK_HMM_State *) * st_num);

  hmm->ststart = NULL;
  hmm->st_root = NULL;
  for (idx = 0; idx < st_num; idx++) {
    s = (HTK_HMM_State *)mybmalloc(sizeof(HTK_HMM_State));
    s->name = rdn_str(fp);
    rdn(fp, &(s->mix_num), sizeof(short), 1);
    if (s->mix_num == -1) {
      /* tmix */
      rdn(fp, &did, sizeof(unsigned int), 1);
      s->b = (HTK_HMM_Dens **)tm_index[did];
      s->mix_num = (tm_index[did])->num;
    } else {
      /* mixture */
      s->b = (HTK_HMM_Dens **)mybmalloc(sizeof(HTK_HMM_Dens *) * s->mix_num);
      for (i=0;i<s->mix_num;i++) {
	rdn(fp, &did, sizeof(unsigned int), 1);
	if (did >= dens_num) {
	  s->b[i] = NULL;
	} else {
	  s->b[i] = dens_index[did];
	}
      }
    }
    s->bweight = (PROB *)mybmalloc(sizeof(PROB) * s->mix_num);
    rdn(fp, s->bweight, sizeof(PROB), s->mix_num);
    s->id = idx;
    st_index[idx] = s;
    state_add(hmm, s);
  }
#ifdef DMES
  j_printf("%d states read\n", st_num);
#endif
}

/* read toplevel model data */
static void
rd_data(FILE *fp, HTK_HMM_INFO *hmm)
{
  HTK_HMM_Data *d;
  unsigned int md_num;
  unsigned int sid, tid;
  unsigned int idx;
  int i;

  rdn(fp, &(md_num), sizeof(unsigned int), 1);
  hmm->totalhmmnum = md_num;

  hmm->start = NULL;
  hmm->physical_root = NULL;
  for (idx = 0; idx < md_num; idx++) {
    d = (HTK_HMM_Data *)mybmalloc(sizeof(HTK_HMM_Data));
    d->name = rdn_str(fp);
    rdn(fp, &(d->state_num), sizeof(short), 1);
    d->s = (HTK_HMM_State **)mybmalloc(sizeof(HTK_HMM_State *) * d->state_num);
    for (i=0;i<d->state_num;i++) {
      rdn(fp, &sid, sizeof(unsigned int), 1);
      if (sid > hmm->totalstatenum) {
	d->s[i] = NULL;
      } else {
	d->s[i] = st_index[sid];
      }
    }
    rdn(fp, &tid, sizeof(unsigned int), 1);
    d->tr = tr_index[tid];
    htk_hmmdata_add(hmm, d);
  }
#ifdef DMES
  j_printf("%d HMM model definition read\n", md_num);
#endif
}


boolean
read_binhmm(FILE *fp, HTK_HMM_INFO *hmm)
{

  char *p;

  /* read header */
  if (rd_header(fp) == FALSE) {
    return FALSE;
  }

  j_printerr("(binary)...");
  
  /* read option data */
  rd_opt(fp, &(hmm->opt));

  /* read type data */
  rd_type(fp, hmm);

  /* read transition data */
  rd_trans(fp, hmm);

  /* read variance data */
  rd_var(fp, hmm);

  /* read density data */
  rd_dens(fp, hmm);

  /* read tmix data */
  if (hmm->is_tied_mixture) {
    rd_tmix(fp, hmm);
  }

  /* read state data */
  rd_state(fp, hmm);

  /* read model data */
  rd_data(fp, hmm);

  /* free pointer->index work area */
  free(tr_index);
  free(vr_index);
  free(dens_index);
  if (hmm->is_tied_mixture) free(tm_index);
  free(st_index);

  j_printf("finished\n");

  return (TRUE);
}
