/*
 * Copyright (c) 1991-2003 Kyoto University
 * Copyright (c) 2000-2003 NAIST
 * All rights reserved
 */

/* adin-cut.c --- read audio input from device with skipping silence */

/* $Id: adin-cut.c,v 1.18 2004/02/09 05:24:19 ri Exp $ */

/* use zerocross & level threshold for silence detection */
/* process one speech segment till next silence at a call */

/* for real-time processing, it does not (basically) store speech samples
   to a buffer.  Instead, an external function that directly processes the
   incoming speech segment should be specified.  The incoming speech segments
   are sequencially processed by the function. */

#ifdef HAVE_PTHREAD
/*
 *  threading is enabled.  Two threads share one buffer for sending/receiving
 *  incoming speech:
 *    Thread 1 (created): A/D-in thread with cutting
 *          read in speech data from device, detect sil, and if triggered,
 *          keep appending the segment to buffer.
 *    Thread 2 (original): processing thread
 *          keep watching the buffer, and process the incoming speech
 *          segment as soon as they appear.  Then it shrinks the buffer.
 *
 */
#endif

#include <sent/stddefs.h>
#include <sent/speech.h>
#include <sent/adin.h>

#undef THREAD_DEBUG

/* zero-cross params (local) */
static int c_length  = 5000;	/* length of cycle buffer for zero-cross */
static int c_offset  = 0;	/* data DC offset */
static int wstep   = DEFAULT_WSTEP;	/* data window size */
static int thres;		/* level threshold */
static int noise_zerocross;	/* threshold of zerocross num in a period */
static int nc_max;		/* tail margin in number of wstep segment */

/* for delayed tail silence processing */
static SP16 *swapbuf;		/* buffer for re-triggering in tail margin */
static int sbsize, sblen;	/* size and current length of swapbuf[] */
static int rest_tail;		/* unprocessed samples in swap buffer */

/* device-dependent configuration */
static boolean adin_cut_on;	/* if TRUE, do input segmentation by silence */
static boolean silence_cut_default; /* device-dependent default value of adin_cut_on */
static boolean strip_flag;	/* if TRUE, do invalid sample skipping */
static boolean enable_thread = FALSE;	/* TRUE if device needs threading */
static boolean ignore_speech_while_recog = TRUE; /* if TRUE, ignore speech input while recognition process */
static boolean need_zmean;	/* if TRUE, do zmensource for incoming input */

/* device handling function */
static boolean (*ad_resume)();	/* restart input */
static boolean (*ad_pause)();	/* stop input */
static int (*ad_read)(SP16 *, int); /* read in samples */

#ifdef HAVE_PTHREAD
static void adin_thread_create(); /* create and start input thread */
#endif

/* store device-dependent configurations to local */
/* (only called from adin_select() via adin_register_func() ) */
void
adin_setup_func(int (*cad_read)(SP16 *, int),
		boolean (*cad_pause)(),
		boolean (*cad_resume)(),
		boolean use_cut_def,
		boolean need_thread
		)
{
  ad_read = cad_read;
  ad_pause = cad_pause;
  ad_resume = cad_resume;
  silence_cut_default = use_cut_def;
#ifdef HAVE_PTHREAD
  enable_thread = need_thread;
#else
  if (need_thread == TRUE) {
    j_printerr("Warning: thread not supported, input may be corrupted on slow machines\n");
  }
#endif
}

/* setup silence detection parameters (should be called after adin_select()) */
void
adin_setup_param(int silence_cut, boolean strip_zero, int cthres, int czc, int head_margin, int tail_margin, int sample_freq, boolean ignore_speech, boolean need_zeromean)
{
  float samples_in_msec;
  /* whether to perform silence cutting */
  /* 0...force off  1...force on  2...keep device-specific default */
  if (silence_cut < 2) {
    adin_cut_on = (silence_cut == 1) ? TRUE : FALSE;
  } else {
    adin_cut_on = silence_cut_default;
  }
  /* whether to strip zero samples */
  strip_flag = strip_zero;
  /* set level threshold */
  thres = cthres;
  /* calc & set internal parameter from configuration */
  samples_in_msec = (float) sample_freq / 1000.0;
  /* cycle buffer length = head margin length */
  c_length = head_margin * samples_in_msec;	/* in msec. */
  /* compute zerocross trigger count threshold in the cycle buffer */
  noise_zerocross = czc * c_length / sample_freq;
  /* process step */
  wstep = DEFAULT_WSTEP;
  /* tail margin length (in wstep) */
  nc_max = (tail_margin * samples_in_msec / wstep) + 2;
  sbsize = tail_margin * samples_in_msec + (c_length * czc / 200);
  /*printf("c_length=%d, noise_zerocross=%d nc_max=%d\n", c_length, noise_zerocross, nc_max);*/

  ignore_speech_while_recog = ignore_speech;

  need_zmean = need_zeromean;

#ifdef HAVE_PTHREAD
  if (enable_thread) {
    /* create A/D-in thread here */
    adin_thread_create();
  }
#endif
}

/* query functions for the caller to know the resulting configurations */
boolean
query_segment_on()
{
  return adin_cut_on;
}
boolean
query_thread_on()
{
  return enable_thread;
}

/* reset zero mean if required */
void
adin_reset_zmean()
{
  if (need_zmean) zmean_reset();
}


#ifdef HAVE_PTHREAD
#include <pthread.h>
static pthread_t adin_thread;	/* thread info */
static pthread_mutex_t mutex;	/* lock primitive */
static int dummy;		/* dummy arg */
/* variables shared among adin thread and process thread */
static boolean transfer_online = FALSE;	/* semaphore to start/stop recognition.  if TRUE, adin thread will store samples to speech[], and process thread will take them and process.  If FALSE, adin thread will get input and check trigger, but does not store them to speech[]  */
static boolean adinthread_buffer_overflowed = FALSE; /* will be set to TRUE if speech[], where the adin thread will store triggered samples, has been overflowed */
#endif

/* temporary input data buffer */
static SP16 *buffer = NULL;	/* temporary buffer to hold input samples */
static int bpmax;		/* maximum length of buffer[] */
static int bp;			/* current point to store data */
static int current_len;		/* current length of stored samples */
static SP16 *cbuf;		/* buffer for flushing cycle buffer just after detecting trigger  */


/* purge processed samples in temporary buffer */
static void
adin_purge(int from)
{
  if (from > 0 && current_len-from > 0) {
    memmove(buffer, &(buffer[from]), (current_len - from) * sizeof(SP16));
  }
  bp = current_len - from;
}

/* main adin function */
/* In threaded mode, this function will detach and loop forever in ad-in
   thread,  storing triggered samples in speech[], and telling the status
   to another process thread via boolean variable transfer_online.
   The process thread, called from adin_go(), polls the length of speech[]
   and transfer_online, and if there are stored samples, process it.
   
   In non-threaded mode, this function will be called directly from adin_go(),
   and triggered samples are immediately processed within here.
 */
/* In module mode, the argument "ad_check" is specified to poll the status of
   incoming command from client while recognition */
/* return: -1 on error, 0 on end of stream, >0 paused by external process */

static int
adin_cut(
	 int (*ad_process)(SP16 *, int), /* function to process the triggered samples */
	 int (*ad_check)())	/* function periodically called while input processing */
{
  static int i;
  static boolean is_valid_data;	/* TRUE if we are now triggered */
  int ad_process_ret;
  int imax, len, cnt;
  static boolean end_of_stream;	/* will be set to TRUE if current input stream has reached the end (in case of file input or adinnet input).  If TRUE, no more input will be got by ad_read, but just process the already stored samples until it becomes empty */
  static int need_init = TRUE;	/* if TRUE, initialize buffer on startup */
  static int end_status;	/* return value */
  static boolean transfer_online_local;	/* local repository of transfer_online */
  
  static int zc;		/* count of zero cross */
  static int nc;		/* count of current tail silence segments */

  /*
   * there are 3 buffers:
   *   temporary storage queue: buffer[]
   *   cycle buffer for zero-cross counting: (in zc_e)
   *   swap buffer for re-starting after short tail silence
   *
   * Each samples are first read to buffer[], then passed to count_zc_e()
   * to find trigger.  Samples between trigger and end of speech are 
   * passed to (*ad_process) with pointer to the first sample and its length.
   *
   */

  /**********************/
  /* initialize buffers */
  /**********************/
  if (buffer == NULL) {		/* beginning of stream */
    buffer = (SP16 *)mymalloc(sizeof(SP16) * MAXSPEECHLEN);
    cbuf = (SP16 *)mymalloc(sizeof(SP16) * c_length);
    swapbuf = (SP16 *)mymalloc(sizeof(SP16) * sbsize);
  }
  if (need_init) {
    bpmax = MAXSPEECHLEN;
    bp = 0;
    /* reset zero-cross status */
    if (adin_cut_on) {
      init_count_zc_e(thres, c_length, c_offset);
      is_valid_data = FALSE;
    }
    end_of_stream = FALSE;
    nc = 0;
    sblen = 0;
    need_init = FALSE;		/* for next call */
  }
      
  /****************/
  /* resume input */
  /****************/
  /* restart speech input if paused on the last call */
  if (ad_resume != NULL) {
    if ((*ad_resume)() == FALSE)  return(-1);
  }

  /*************/
  /* main loop */
  /*************/
  for (;;) {

    /****************************/
    /* read in new speech input */
    /****************************/
    if (end_of_stream) {
      /* already reaches end of stream, just process the rest */
      current_len = bp;
    } else {
      /*****************************************************/
      /* get samples from input device to temporary buffer */
      /*****************************************************/
      /* buffer[0..bp] is the current remaining samples */
      /*
	mic input - samples exist in a device buffer
        tcpip input - samples exist in a socket
        file input - samples in a file
	   
	Return value is the number of read samples.
	If no data exists in the device (in case of mic input), ad_read()
	will return 0.  If reached end of stream (in case end of file or
	receive end ack from tcpip client), it will return -1.
	If error, returns -2.
      */
      cnt = (*ad_read)(&(buffer[bp]), bpmax - bp);
      if (cnt < 0) {		/* end of stream or error */
	/* set the end status */
	if (cnt == -2) end_status = -1; /* end by error */
	else if (cnt == -1) end_status = 0; /* end by normal end of stream */
	/* now the input has been ended, 
	   we should not get further speech input in the next loop, 
	   instead just process the samples in the temporary buffer until
	   the entire data is processed. */
	end_of_stream = TRUE;		
	cnt = 0;			/* no new input */
	/* in case the first trial of ad_read() fails, exit this loop */
	if (bp == 0) break;
      }

      /*************************************************/
      /* some speech processing for the incoming input */
      /*************************************************/
      if (cnt > 0) {
	if (strip_flag) {
	  /* strip off successive zero samples */
	  len = strip_zero(&(buffer[bp]), cnt);
	  if (len != cnt) cnt = len;
	}
	if (need_zmean) {
	  /* remove DC offset */
	  sub_zmean(&(buffer[bp]), cnt);
	}
      }
      
      /* current len = current samples in buffer */
      current_len = bp + cnt;
    }
#ifdef THREAD_DEBUG
    if (end_of_stream) {
      printf("stream already ended\n");
    }
    printf("input: get %d samples [%d-%d]\n", current_len - bp, bp, current_len);
#endif

    /**************************************************/
    /* call the periodic callback (non threaded mode) */
    /*************************************************/
    /* this function is mainly for periodic checking of incoming command
       in module mode */
    /* in threaded mode, this will be done in process thread, not here in adin thread */
    if (ad_check != NULL
#ifdef HAVE_PTHREAD
	&& !enable_thread
#endif
	) {
      /* if ad_check() returns value < 0, termination of speech input is required */
      if ((i = (*ad_check)()) < 0) { /* -1: soft termination -2: hard termination */
	if ((i == -1 && current_len == 0) || i == -2) {
	  end_status = -2;	/* recognition terminated by outer function */
	  goto break_input;
	}
      }
    }

    /***********************************************************************/
    /* if no data has got but not end of stream, repeat next input samples */
    /***********************************************************************/
    if (current_len == 0) continue;


    /******************************************************/
    /* prepare for processing samples in temporary buffer */
    /******************************************************/
    
    wstep = DEFAULT_WSTEP;	/* process unit (should be smaller than cycle buffer) */

    /* imax: total length that should be processed at one ad_read() call */
    /* if in real-time mode and not threaded, recognition process 
       will be called and executed as the ad_process() callback within
       this function.  If the recognition speed is over the real time,
       processing all the input samples at the loop below may result in the
       significant delay of getting next input, that may result in the buffer
       overflow of the device (namely a microphone device will suffer from
       this). So, in non-threaded mode, in order to avoid buffer overflow and
       input frame dropping, we will leave here by processing 
       only one segment [0..wstep], and leave the rest in the temporary buffer.
    */
#ifdef HAVE_PTHREAD
    if (enable_thread) imax = current_len; /* process whole */
    else imax = (current_len < wstep) ? current_len : wstep; /* one step */
#else
    imax = (current_len < wstep) ? current_len : wstep;	/* one step */
#endif
    
    /* wstep: unit length for the loop below */
    if (wstep > current_len) wstep = current_len;

#ifdef THREAD_DEBUG
    printf("process %d samples by %d step\n", imax, wstep);
#endif

#ifdef HAVE_PTHREAD
    /* get transfer status to local */
    pthread_mutex_lock(&mutex);
    transfer_online_local = transfer_online;
    pthread_mutex_unlock(&mutex);
#endif

    /*********************************************************/
    /* start processing buffer[0..current_len] by wstep step */
    /*********************************************************/
    i = 0;
    while (i + wstep <= imax) {
      
      if (adin_cut_on) {

	/********************/
	/* check triggering */
	/********************/
	/* the cycle buffer in count_zc_e() holds the last
	   samples of (head_margin) miliseconds, and the zerocross
	   over the threshold level are counted within the cycle buffer */
	
	/* store the new data to cycle buffer and update the count */
	/* return zero-cross num in the cycle buffer */
	zc = count_zc_e(&(buffer[i]), wstep);
	
	if (zc > noise_zerocross) { /* now triggering */
	  
	  if (is_valid_data == FALSE) {
	    /*****************************************************/
	    /* process off, trigger on: detect speech triggering */
	    /*****************************************************/
	    
	    is_valid_data = TRUE;   /* start processing */
	    nc = 0;
#ifdef THREAD_DEBUG
	    printf("detect on\n");
#endif
	    /****************************************/
	    /* flush samples stored in cycle buffer */
	    /****************************************/
	    /* (last (head_margin) msec samples */
	    /* if threaded mode, processing means storing them to speech[].
	       if ignore_speech_while_recog is on (default), ignore the data
	       if transfer is offline (=while processing second pass).
	       Else, datas are stored even if transfer is offline */
	    if ( ad_process != NULL
#ifdef HAVE_PTHREAD
		 && (!enable_thread || !ignore_speech_while_recog || transfer_online_local)
#endif
		 ) {
	      /* copy content of cycle buffer to cbuf */
	      zc_copy_buffer(cbuf, &len);
	      /* Note that the last 'wstep' samples are the same as
		 the current samples 'buffer[i..i+wstep]', and
		 they will be processed later.  So, here only the samples
		 cbuf[0...len-wstep] will be processed
	      */
	      if (len - wstep > 0) {
#ifdef THREAD_DEBUG
		printf("callback for buffered samples (%d bytes)\n", len - wstep);
#endif
		ad_process_ret = (*ad_process)(cbuf, len - wstep);
		switch(ad_process_ret) {
		case 1:		/* segmentation notification from process callback */
#ifdef HAVE_PTHREAD
		  if (enable_thread) {
		    /* in threaded mode, just stop transfer */
		    pthread_mutex_lock(&mutex);
		    transfer_online = transfer_online_local = FALSE;
		    pthread_mutex_unlock(&mutex);
		  } else {
		    /* in non-threaded mode, set end status and exit loop */
		    end_status = 1;
		    adin_purge(i);
		    goto break_input;
		  }
		  break;
#else
		  /* in non-threaded mode, set end status and exit loop */
		  end_status = 1;
		  adin_purge(i);
		  goto break_input;
#endif
		case -1:		/* error occured in callback */
		  /* set end status and exit loop */
		  end_status = -1;
		  goto break_input;
		}
	      }
	    }
	    
	  } else {		/* is_valid_data == TRUE */
	    /******************************************************/
	    /* process on, trigger on: we are in a speech segment */
	    /******************************************************/
	    
	    if (nc > 0) {
	      
	      /*************************************/
	      /* re-triggering in trailing silence */
	      /*************************************/
	      
#ifdef THREAD_DEBUG
	      printf("re-triggered\n");
#endif
	      /* reset noise counter */
	      nc = 0;
	      
	      /*************************************************/
	      /* process swap buffer stored while tail silence */
	      /*************************************************/
	      /* In trailing silence, the samples within the tail margin length
		 will be processed immediately, but samples after the tail
		 margin will not be processed, instead stored in swapbuf[].
		 If re-triggering occurs while in the trailing silence,
		 the swapped samples should be processed now to catch up
		 with current input
	      */
	      if (sblen > 0) {
#ifdef THREAD_DEBUG
		printf("callback for swapped %d samples\n", sblen);
#endif
		ad_process_ret = (*ad_process)(swapbuf, sblen);
		sblen = 0;
		switch(ad_process_ret) {
		case 1:		/* segmentation notification from process callback */
#ifdef HAVE_PTHREAD
		  if (enable_thread) {
		    /* in threaded mode, just stop transfer */
		    pthread_mutex_lock(&mutex);
		    transfer_online = transfer_online_local = FALSE;
		    pthread_mutex_unlock(&mutex);
		  } else {
		    /* in non-threaded mode, set end status and exit loop */
		    end_status = 1;
		    adin_purge(i);
		    goto break_input;
		  }
		  break;
#else
		  /* in non-threaded mode, set end status and exit loop */
		  end_status = 1;
		  adin_purge(i);
		  goto break_input;
#endif
		case -1:		/* error occured in callback */
		  /* set end status and exit loop */
		  end_status = -1;
		  goto break_input;
		}
	      }
	    }
	  } 
	} else if (is_valid_data == TRUE) {
	  
	  /*******************************************************/
	  /* process on, trigger off: processing tailing silence */
	  /*******************************************************/
	  
#ifdef THREAD_DEBUG
	  printf("TRAILING SILENCE\n");
#endif
	  if (nc == 0) {
	    /* start of tail silence: prepare valiables for start swapbuf[] */
	    rest_tail = sbsize - c_length;
	    sblen = 0;
#ifdef THREAD_DEBUG
	    printf("start tail silence, rest_tail = %d\n", rest_tail);
#endif
	  }

	  /* increment noise counter */
	  nc++;
	}
      }	/* end of triggering handlers */
      
      
      /********************************************************************/
      /* process the current segment buffer[i...i+wstep] if process == on */
      /********************************************************************/
      
      if (adin_cut_on && is_valid_data && nc > 0 && rest_tail == 0) {
	
	/* The current trailing silence is now longer than the user-
	   specified tail margin length, so the current samples
	   should not be processed now.  But if 're-triggering'
	   occurs in the trailing silence later, they should be processed
	   then.  So we just store the overed samples in swapbuf[] and
	   not process them now */
	
#ifdef THREAD_DEBUG
	printf("tail silence over, store to swap buffer (nc=%d, rest_tail=%d, sblen=%d-%d)\n", nc, rest_tail, sblen, sblen+wstep);
#endif
	if (sblen + wstep > sbsize) {
	  j_printerr("Error: swapbuf exceeded!\n");
	}
	memcpy(&(swapbuf[sblen]), &(buffer[i]), wstep * sizeof(SP16));
	sblen += wstep;
	
      } else {

	/* we are in a normal speech segment (nc == 0), or
	   trailing silence (shorter than tail margin length) (nc>0,rest_tail>0)
	   The current trailing silence is shorter than the user-
	   specified tail margin length, so the current samples
	   should be processed now as same as the normal speech segment */
	
	if(
	   (!adin_cut_on || is_valid_data == TRUE)
#ifdef HAVE_PTHREAD
	   && (!enable_thread || !ignore_speech_while_recog || transfer_online_local)
#endif
	   ) {
	  if (nc > 0) {
	    /* if we are in a trailing silence, decrease the counter to detect
	     start of swapbuf[] above */
	    if (rest_tail < wstep) rest_tail = 0;
	    else rest_tail -= wstep;
#ifdef THREAD_DEBUG
	    printf("%d processed, rest_tail=%d\n", wstep, rest_tail);
#endif
	  }
	  if ( ad_process != NULL ) {
#ifdef THREAD_DEBUG
	    printf("callback for input sample [%d-%d]\n", i, i+wstep);
#endif
	    /* call external function */
	    ad_process_ret = (*ad_process)(&(buffer[i]),  wstep);
	    switch(ad_process_ret) {
	    case 1:		/* segmentation notification from process callback */
#ifdef HAVE_PTHREAD
	      if (enable_thread) {
		/* in threaded mode, just stop transfer */
		pthread_mutex_lock(&mutex);
		transfer_online = transfer_online_local = FALSE;
		pthread_mutex_unlock(&mutex);
	      } else {
		/* in non-threaded mode, set end status and exit loop */
		adin_purge(i+wstep);
		end_status = 1;
		goto break_input;
	      }
	      break;
#else
	      /* in non-threaded mode, set end status and exit loop */
	      adin_purge(i+wstep);
	      end_status = 1;
	      goto break_input;
#endif
	    case -1:		/* error occured in callback */
	      /* set end status and exit loop */
	      end_status = -1;
	      goto break_input;
	    }
	  }
	}
      }	/* end of current segment processing */

      
      if (adin_cut_on && is_valid_data && nc >= nc_max) {
	/*************************************/
	/* process on, trailing silence over */
	/* = end of input segment            */
	/*************************************/
#ifdef THREAD_DEBUG
	printf("detect off\n");
#endif
	/* end input by silence */
	is_valid_data = FALSE;	/* turn off processing */
	sblen = 0;
#ifdef HAVE_PTHREAD
	if (enable_thread) { /* just stop transfer */
	  pthread_mutex_lock(&mutex);
	  transfer_online = transfer_online_local = FALSE;
	  pthread_mutex_unlock(&mutex);
	} else {
	  adin_purge(i+wstep);
	  end_status = 1;
	  goto break_input;
	}
#else
	adin_purge(i+wstep);
	end_status = 1;
	goto break_input;
#endif
      }

      /*********************************************************/
      /* end of processing buffer[0..current_len] by wstep step */
      /*********************************************************/
      i += wstep;		/* increment to next wstep samples */
    }
    
    /* purge processed samples and update queue */
    adin_purge(i);

    /* end of input by end of stream */
    if (end_of_stream && bp == 0) break;
  }

break_input:

  /****************/
  /* pause input */
  /****************/
  /* stop speech input */
  if (ad_pause != NULL) {
    if ((*ad_pause)() == FALSE) {
      j_printerr("Error: failed to pause recording\n");
      end_status = -1;
    }
  }

  if (end_of_stream) {			/* input already ends */
    if (bp == 0) {		/* rest buffer successfully flushed */
      /* reset status */
      if (adin_cut_on) end_count_zc_e();
      need_init = TRUE;		/* bufer status shoule be reset at next call */
    }
    end_status = (bp) ? 1 : 0;
  }
  
  return(end_status);
}








#ifdef HAVE_PTHREAD
/***********************/
/* threading functions */
/***********************/

/* adin-thread store data to speech[], and process thread read it */
static SP16 *speech;		/* stored samples by adin-cut() */
static int speechlen;		/* length of above */

/*************************/
/* adin thread functions */
/*************************/
/* call-back for storing triggered samples in adin thread */
static int
adin_store_buffer(SP16 *now, int len)
{
  if (speechlen + len > MAXSPEECHLEN) {
    /* just mark as overflowed, and continue this thread */
    pthread_mutex_lock(&mutex);
    adinthread_buffer_overflowed = TRUE;
    pthread_mutex_unlock(&mutex);
    return(0);
  }
  pthread_mutex_lock(&mutex);
  memcpy(&(speech[speechlen]), now, len * sizeof(SP16));
  speechlen += len;
  pthread_mutex_unlock(&mutex);
#ifdef THREAD_DEBUG
  printf("input: stored %d samples, total=%d\n", len, speechlen);
#endif
  /* output progress bar in dots */
  /*if ((++dotcount) % 3 == 1) j_printerr(".");*/
  return(0);			/* continue */
}

/* A/D-in thread main function: only call adin_cut() with storing function */
void
adin_thread_input_main(void *dummy)
{
  adin_cut(adin_store_buffer, NULL);
}

/* create A/D-in thread */
static void
adin_thread_create()
{
  /* init storing buffer */
  speechlen = 0;
  speech = (SP16 *)mymalloc(sizeof(SP16) * MAXSPEECHLEN);

  transfer_online = FALSE; /* tell adin-mic thread to wait at initial */
  adinthread_buffer_overflowed = FALSE;

  if (pthread_mutex_init(&(mutex), NULL) != 0) { /* error */
    j_error("Error: pthread: cannot initialize mutex\n");
  }
  if (pthread_create(&adin_thread, NULL, (void *)adin_thread_input_main, NULL) != 0) {
    j_error("Error: pthread: failed to create AD-in thread\n");
  }
  if (pthread_detach(adin_thread) != 0) { /* not join, run forever */
    j_error("Error: pthread: failed to detach AD-in thread\n");
  }
  j_printerr("AD-in thread created\n");
}

/****************************/
/* process thread functions */
/****************************/
/* used for module mode: return value: -2 = input cancellation forced by control module */
static int
adin_thread_process(int (*ad_process)(SP16 *, int), int (*ad_check)())
{
  int prev_len, nowlen;
  int ad_process_ret;
  int end_status;
  int i;
  boolean overflowed_p;
  boolean transfer_online_local;

  /* reset storing buffer --- input while recognition will be ignored */
  pthread_mutex_lock(&mutex);
  /*if (speechlen == 0) transfer_online = TRUE; /* tell adin-mic thread to start recording */
  transfer_online = TRUE;
#ifdef THREAD_DEBUG
  printf("process: reset, speechlen = %d, online=%d\n", speechlen, transfer_online);
#endif
  pthread_mutex_unlock(&mutex);

  j_printerr("<<< please speak >>>");

  /* main processing loop */
  prev_len = 0;
  for(;;) {
    /* get current length (locking) */
    pthread_mutex_lock(&mutex);
    nowlen = speechlen;
    overflowed_p = adinthread_buffer_overflowed;
    transfer_online_local = transfer_online;
    pthread_mutex_unlock(&mutex);
    /* check if other input thread has overflowed */
    if (overflowed_p) {
      j_printerr("Warning: too long input (> %d samples), segmented now\n", MAXSPEECHLEN);
      /* segment input here */
      pthread_mutex_lock(&mutex);
      adinthread_buffer_overflowed = FALSE;
      speechlen = 0;
      transfer_online = transfer_online_local = FALSE;
      pthread_mutex_unlock(&mutex);
      return(1);		/* return with segmented status */
    }
    /* callback poll */
    if (ad_check != NULL) {
      if ((i = (*ad_check)()) < 0) {
	if ((i == -1 && nowlen == 0) || i == -2) {
	  pthread_mutex_lock(&mutex);
	  transfer_online = transfer_online_local = FALSE;
	  speechlen = 0;
	  pthread_mutex_unlock(&mutex);
	  return(-2);
	}
      }
    }
    if (prev_len < nowlen) {
#ifdef THREAD_DEBUG
      printf("process: proceed [%d-%d]\n",prev_len, nowlen);
#endif
      if (prev_len == 0) {	/* first trigger */
	/* flush prompt */
	j_printerr("\r                    \r");
      }
      /* got new sample, process */
      /* As the speech[] buffer is monotonously increase,
	 content of speech buffer [prev_len..nowlen] would not alter
	 in both threads
	 So locking is not needed while processing.
       */
      /*printf("main: read %d-%d\n", prev_len, nowlen);*/
      if (ad_process != NULL) {
	ad_process_ret = (*ad_process)(&(speech[prev_len]),  nowlen - prev_len);
#ifdef THREAD_DEBUG
	printf("ad_process_ret=%d\n",ad_process_ret);
#endif
	switch(ad_process_ret) {
	case 1:			/* segmented */
	  /* segmented by callback function */
	  /* purge processed samples and keep transfering */
	  pthread_mutex_lock(&mutex);
	  if(speechlen > nowlen) {
	    memmove(buffer, &(buffer[nowlen]), (speechlen - nowlen) * sizeof(SP16));
	    speechlen = speechlen - nowlen;
	  } else {
	    speechlen = 0;
	  }
	  transfer_online = transfer_online_local = FALSE;
	  pthread_mutex_unlock(&mutex);
	  /* keep transfering */
	  return(1);		/* return with segmented status */
	case -1:		/* error */
	  pthread_mutex_lock(&mutex);
	  transfer_online = transfer_online_local = FALSE;
	  pthread_mutex_unlock(&mutex);
	  return(-1);		/* return with error */
	}
      }
      prev_len = nowlen;
    } else {
      if (transfer_online_local == FALSE) {
	/* segmented by zero-cross */
	/* reset storing buffer for next input */
	pthread_mutex_lock(&mutex);
	speechlen = 0;
	pthread_mutex_unlock(&mutex);
        break;
      }
      usleep(100000);   /* wait = 0.1sec*/            
    }
  }

  /* as threading assumes infinite input */
  /* return value should be 1 (segmented) */
  return(1);
}
#endif /* HAVE_PTHREAD */


/* global entry point */
int
adin_go(int (*ad_process)(SP16 *, int), int (*ad_check)())
{
#ifdef HAVE_PTHREAD
  if (enable_thread) {
    return(adin_thread_process(ad_process, ad_check));
  }
#endif
  return(adin_cut(ad_process, ad_check));
}
