/*
 * Copyright (c) 1991-2003 Kyoto University
 * Copyright (c) 2000-2003 NAIST
 * All rights reserved
 */

/* search_bestfirst_main.c --- 2nd pass: stack decoding (main) */

/* $Id: search_bestfirst_main.c,v 1.19 2003/12/22 07:00:08 ri Exp $ */


/* NODE = hypothesis */

#include <julius.h>

static NODE *get_best_from_stack(NODE **start, int *stacknum);
static int put_to_stack(NODE *new, NODE **start, NODE **bottom, int *stacknum, int stacksize);
static void put_all_in_stack(NODE **start, int *stacknum);
static void free_all_nodes(NODE *node);
static void put_hypo_woutput(NODE *hypo);
static void put_hypo_wname(NODE *hypo);
static void put_hypo_phoneme(NODE *hypo);


/**********************************************************************/
/********** $B<!C18l3JG<NN0h$N3d$jEv$F(B          *************************/
/********** allocate memory for nextword data *************************/
/**********************************************************************/

/* $BM=B,<!C18l3JG<NN0h(B NEXTWORD $BG[Ns$r3d$jIU$1$F%]%$%s%?$rJV$9(B */
/* allocate NEXTWORD array and return the newly allocated pointer */
static NEXTWORD **
nw_malloc(int *maxlen, NEXTWORD **root)
{
  NEXTWORD *nwtmp;
  NEXTWORD **nw;
  int i;
  int max;

  /* the initial maximum number of nextwords is the size of vocabulary */
  max = winfo->num;
  
  nw = (NEXTWORD **)mymalloc(max * sizeof(NEXTWORD *));
  nwtmp = (NEXTWORD *)mymalloc(max * sizeof(NEXTWORD));
  for (i=0;i<max; i++) {
    nw[i] = &(nwtmp[i]);
  }
  *maxlen = max;
  *root = nwtmp;
  return nw;
}

/* $BM=B,<!C18l3JG<NN0h$r2rJ|(B */
/* free the given nextword data 'nw' */
static void
nw_free(NEXTWORD **nw, NEXTWORD *root)
{
  free(root);
  free(nw);
}

#ifdef USE_DFA
/* DFA $B%b!<%I$G$O<!C18l=89g$NBg$-$5$,8lWC%5%$%:$r1[$($k$3$H$,$"$k!%(B
   $B$3$l$O%7%g!<%H%]!<%:$N%9%-%C%W=hM}$K$h$C$FJ#?t>uBV(B(sp$B$"$j(B/$B$J$7(B)$B$,(B
   $BF1;~$KE83+$5$l$k>l9g$K5/$3$j$&$k!%0J2<$N4X?t$O$=$N>l9g$K8F$P$l(B,
   $B$5$i$K8lWC?tJ,<!C18l=89g%(%j%"$r?-D9$9$k!%(B*/
/* In DFA mode, the number of nextwords can exceed the vocabulary size when
   more than one DFA states are expanded by short-pause skipping.
   In such case, the nextword data area should expanded */
static NEXTWORD **
nw_expand(NEXTWORD **nwold, int *maxlen, NEXTWORD **root)
{
  NEXTWORD *nwtmp;
  NEXTWORD **nw;
  int i;
  int nwmaxlen;

  nwmaxlen = *maxlen + winfo->num;

  nwtmp = (NEXTWORD *)myrealloc(*root, nwmaxlen * sizeof(NEXTWORD));
  nw = (NEXTWORD **)myrealloc(nwold, nwmaxlen * sizeof(NEXTWORD *));
  nw[0] = nwtmp;
  for (i=1;i<nwmaxlen; i++) {
    nw[i] = &(nwtmp[i]);
  }
  *maxlen = nwmaxlen;
  *root = nwtmp;
  return nw;
}
#endif



/**********************************************************************/
/********** $B2>@b%9%?%C%/$NA`:n(B         ********************************/
/********** hypothesis stack operation ********************************/
/**********************************************************************/

/* $BJ82>@b%9%?%C%/$O(B double linked list $B$G$"$k!%$^$?>o$K%9%3%"=g$,J]$?$l$k(B */
/* the hypothesis stack is double-linked list, and holds entries in sorted order */

/* $B%9%?%C%/%H%C%W$N2>@b$r<h$j=P$9(B */
/* pop the top hypothesis from the stack */
static NODE *
get_best_from_stack(NODE **start, /* pointer to stack top node */
		    int *stacknum /* pointer to stack size */
		    )
{
  NODE *tmp;

  /* return top */
  tmp=(*start);
  if ((*start)!=NULL) {		/* delete it from stack */
    (*start)=(*start)->next;
    if ((*start)!=NULL) (*start)->prev=NULL;
    (*stacknum)--;
    return(tmp);
  }
  else {
    return(NULL);
  }
}

/* $B%9%?%C%/$K2>@b(B new $B$rA^F~$9$k(B   $BJV$jCM(B: $B@.8y(B=0, $B<:GT(B=-1 */
/* insert 'new' hypothesis to stack (keep order)
   retrun 0 on success, -1 on failure */
static int
put_to_stack(
     NODE *new,			/* new hypothesis */
     NODE **start,		/* pointer to stack top node */
     NODE **bottom,		/* pointer to stack bottom node */
     int *stacknum,		/* pointer to stack size */
     int stacksize)		/* maximum stack size */
{
  NODE *tmp;

  (*stacknum)++;

  /* stack size check */
  if ((*stacknum)>=stacksize) {
    /* stack size overflow */
    (*stacknum)--;
    if ((*bottom)->score < new->score) {
      /* new node will be inserted in the stack: free the bottom */
      tmp=(*bottom);
      (*bottom)->prev->next=NULL;
      (*bottom)=(*bottom)->prev;
      free_node(tmp);
    } else {
      /* new node is below the bottom: discard it */
      free_node(new);
      return(-1);
    }
  }

  /* insert new node on edge */
  if ((*start)==NULL) {		/* no node in stack */
    /* new node is the only node */
    (*start)=new;
    (*bottom)=new;
    new->next=NULL;
    new->prev=NULL;
    return(0);
  }
  if ((*start)->score <= new->score) {
    /* insert on the top */
    new->next = (*start);
    new->next->prev = new;
    (*start)=new;
    new->prev=NULL;
    return(0);
  }
  if ((*bottom)->score >= new->score) {
    /* insert on the bottom */
    new->prev = (*bottom);
    new->prev->next = new;
    (*bottom)=new;
    new->next=NULL;
    return(0);
  }
    
  /* now the new node is between (*start) and (*bottom) */
  if (((*start)->score + (*bottom)->score) / 2 > new->score) {
    /* search from bottom */
    tmp=(*bottom);
    while(tmp->score < new->score) tmp=tmp->prev;
    new->prev=tmp;
    new->next=tmp->next;
    tmp->next->prev=new;
    tmp->next=new;
  } else {
    /* search from start */
    tmp=(*start);
    while(tmp->score > new->score) tmp=tmp->next;
    new->next=tmp;
    new->prev=tmp->prev;
    tmp->prev->next=new;
    tmp->prev=new;
  }
  return(0);
}

/* $B%9%?%C%/$NCf?H$rA4$F=PNO$9$k!%%9%?%C%/$NCf?H$O<:$o$l$k!%(B($B%G%P%C%0MQ(B) */
/* output all nodes in the stack.  all nodes will be lost (for debug) */
static void
put_all_in_stack(NODE **start, int *stacknum)
{
  NODE *ntmp;
  
  j_printf("stack remains::\n");
  while ((ntmp = get_best_from_stack(start, stacknum)) != NULL) {
    j_printf("%3d: s=%f ",*stacknum, ntmp->score);
    put_hypo_woutput(ntmp);
    free_node(ntmp);
  }
}

/* $B%9%?%C%/Fb$NA42>@b$r2rJ|(B */
/* free all nodes in stack */
static void
free_all_nodes(NODE *start)
{
  NODE *tmp;
  NODE *next;

  tmp=start;
  while(tmp) {
    next=tmp->next;
    free_node(tmp);
    tmp=next;
  }
}


#ifdef CONFIDENCE_MEASURE

/**********************************************************************/
/********** confidence scoring ****************************************/
/**********************************************************************/

/* for multiple alpha scoring */
#ifdef CM_MULTIPLE_ALPHA
static LOGPROB *cmsumlist = NULL; /* sum of cm score for each alpha coef. */
#endif


#ifdef CM_SEARCH
/***********************************************************/
/**** CM computation method 1(default): search score *******/
/**** no N-best computation (-n ...) needed ****************/
/***********************************************************/

/* CM of each word is calculated at hypothesis expansion time, and */
/* all scores are kept in the node till the search ends */

static LOGPROB cm_tmpbestscore;	/* temporary best score for adding score */
#ifndef CM_MULTIPLE_ALPHA
static LOGPROB cm_tmpsum;	/* sum of cm score */
#endif

/* local stack */
static int l_stacksize;
static int l_stacknum;
static NODE *l_start = NULL;
static NODE *l_bottom = NULL;

/* initialize local stack */
static void
cm_init(WORD_INFO *winfo)
{
  l_stacksize = winfo->num;
  l_start = l_bottom = NULL;
  l_stacknum = 0;
#ifdef CM_MULTIPLE_ALPHA
  if (cmsumlist == NULL) {	/* allocate if not yet */
    cmsumlist = (LOGPROB *)mymalloc(sizeof(LOGPROB) * cm_alpha_num);
  }
#endif    
}

/* store hypothesis to the local stack */
static void
cm_store(NODE *new)
{
  /* store the generated hypo into local stack */
  put_to_stack(new, &l_start, &l_bottom, &l_stacknum, l_stacksize);
}

/* compute score sum in the stack */
static void
cm_sum_score()
{
  NODE *node;
  LOGPROB sum;
#ifdef CM_MULTIPLE_ALPHA
  LOGPROB cm_alpha;
  int j;
#endif

  if (l_start == NULL) return;	/* no hypo */
  cm_tmpbestscore = l_start->score; /* best hypo is at the top of the stack */

#ifdef CM_MULTIPLE_ALPHA
  for (j = 0, cm_alpha = cm_alpha_bgn; cm_alpha <= cm_alpha_end; cm_alpha += cm_alpha_step) {
#endif
    sum = 0.0;
    for(node = l_start; node; node = node->next) {
      sum += pow(10, cm_alpha * (node->score - cm_tmpbestscore));
    }
#ifdef CM_MULTIPLE_ALPHA
    cmsumlist[j++] = sum;	/* store sums for each alpha coef. */
#else
    cm_tmpsum = sum;		/* store sum */
#endif

#ifdef CM_MULTIPLE_ALPHA
  }
#endif
}

/* compute CM scores of the new word in the local stack from the sum above */
static LOGPROB
cm_set_score(NODE *node)
{
#ifdef CM_MULTIPLE_ALPHA
  int j;
  LOGPROB cm_alpha;
#endif

#ifdef CM_MULTIPLE_ALPHA
  for (j = 0, cm_alpha = cm_alpha_bgn; cm_alpha <= cm_alpha_end; cm_alpha += cm_alpha_step) {
    node->cmscore[node->seqnum-1][j] = pow(10, cm_alpha * (node->score - cm_tmpbestscore)) / cmsumlist[j];
    j++;
  }
#else
  node->cmscore[node->seqnum-1] = pow(10, cm_alpha * (node->score - cm_tmpbestscore)) / cm_tmpsum;
#endif
}

/* pop one node from stack */
static NODE *
cm_get_node()
{
  return(get_best_from_stack(&l_start, &l_stacknum));
}

#endif /* CM_SEARCH */

#ifdef CM_NBEST
/*****************************************************************/
/**** CM computation method 2: conventional N-best scoring *******/
/**** NOTE1: enough N-best should be computed (-n 10 ~ -n 100) ***/
/**** NOTE2: same words in a sentence are not isolated ***********/
/*****************************************************************/

/* CM of each word is calculated after all search had finished, and */
/* all scores are kept in the node on the final time */

static LOGPROB *sentcm = NULL;	/* CM of each sentence hypothesis */
static LOGPROB *wordcm = NULL;	/* CM of each word voted from above */
static int sentnum;		/* allocated number of sentcm[] */

/* compute CM from N-best sentence candidates in the global stack */
static void
cm_compute_from_nbest(NODE *start, int stacknum)
{
  NODE *node;
  LOGPROB bestscore, sum, s;
  WORD_ID w;
  int i;
#ifdef CM_MULTIPLE_ALPHA
  LOGPROB cm_alpha;
  int j;
#endif

  /* prepare buffer */
#ifdef CM_MULTIPLE_ALPHA
  if (cmsumlist == NULL) {
    cmsumlist = (LOGPROB *)mymalloc(sizeof(LOGPROB) * cm_alpha_num);
  }
#endif    
  if (sentcm == NULL) {		/* not allocated yet */
    sentcm = (LOGPROB *)mymalloc(sizeof(LOGPROB)*stacknum);
    sentnum = stacknum;
  } else if (sentnum < stacknum) { /* need expanded */
    sentcm = (LOGPROB *)myrealloc(sentcm, sizeof(LOGPROB)*stacknum);
    sentnum = stacknum;
  }
  if (wordcm == NULL) {
    wordcm = (LOGPROB *)mymalloc(sizeof(LOGPROB) * winfo->num);
  }
#ifdef CM_MULTIPLE_ALPHA
  for (j = 0, cm_alpha = cm_alpha_bgn; cm_alpha <= cm_alpha_end; cm_alpha += cm_alpha_step) {
#endif
    /* clear whole word cm buffer */
    for(w=0;w<winfo->num;w++) {
      wordcm[w] = 0.0;
    }
    /* get best score */
    bestscore = start->score;
    /* compute sum score of all hypothesis */
    sum = 0.0;
    for (node = start; node != NULL; node = node->next) {
      sum += pow(10, cm_alpha * (node->score - bestscore));
    }
    /* compute sentence posteriori probabilities */
    i = 0;
    for (node = start; node != NULL; node = node->next) {
      sentcm[i] = pow(10, cm_alpha * (node->score - bestscore)) / sum;
      i++;
    }
    /* compute word posteriori probabilities */
    i = 0;
    for (node = start; node != NULL; node = node->next) {
      for (w=0;w<node->seqnum;w++) {
	wordcm[node->seq[w]] += sentcm[i];
      }
      i++;
    }
    /* store the probabilities to node */
    for (node = start; node != NULL; node = node->next) {
      for (w=0;w<node->seqnum;w++) {
#ifdef CM_MULTIPLE_ALPHA
	node->cmscore[w][j] = wordcm[node->seq[w]];
#else	
	node->cmscore[w] = wordcm[node->seq[w]];
#endif
      }
    }
#ifdef CM_MULTIPLE_ALPHA
    j++;
  }
#endif
}

#endif /* CM_NBEST */

#endif /* CONFIDENCE_MEASURE */


/**********************************************************************/
/********** enveloped best-first search *******************************/
/**********************************************************************/

/*****************/
/* word envelope */
/*****************/

/* $B0l<o$N2>@b%S!<%`I}$r@_Dj(B: $BE83+85$H$J$C$?2>@b$N?t$r$=$N2>@bD9(B($BC18l?t(B)
   $B$4$H$K%+%&%s%H$9$k!%0lDj?t$r1[$($?$i$=$l$h$jC;$$2>@b$O0J8eE83+$7$J$$(B */
/* introduce a kind of beam width: count the number of popped hypotheses
   per the length.  When a count in a certain length reaches given threshold,
   hypotheses shorter than the length will be discarded */

static int hypo_len_count[MAXSEQNUM+1];	/* count per hypothesis length */
static int maximum_filled_length;

/* initialize count */
static void
wb_init()
{
  int i;
  for(i=0;i<=MAXSEQNUM;i++) hypo_len_count[i] = 0;
  maximum_filled_length = -1;
}

/* Count number of popped hypotheses, and return FALSE if it reaches the
   limit.  Otherwise return TRUE */
static boolean
wb_ok(NODE *now)
{
  int i;

  if (now->seqnum <= maximum_filled_length) {
    if (debug2_flag) {
      j_printf("popped but pruned by word envelope: ");
      put_hypo_woutput(now);
    }
    return FALSE;
  } else {
    hypo_len_count[now->seqnum]++;
    if (hypo_len_count[now->seqnum] > enveloped_bestfirst_width) {
      if (maximum_filled_length < now->seqnum) maximum_filled_length = now->seqnum;
    }
    return TRUE;
  }
}

#ifdef SCAN_BEAM
/******************/
/* score envelope */
/******************/

/* Viterbi$B7W;;NL:o8:(B: $BA42>@b$K$o$?$k%U%l!<%`$4$H$N:GBgL`EY$r(B
   $B5-O?$7$F$*$/!%%H%l%j%9E83+;~(B($BA08~$-L`EY7W;;;~(B)$B$K$=$N:GBgL`EY7ONs(B
   ($B$9$J$o$A(Benvelope) $B$+$i0lDj%9%3%"I}0J>eN%$l$?(BViterbi$B%Q%9$O(B,
   $B$=$l0J9_$N%U%l!<%`$O7W;;$+$i=|30$9$k!%>\:Y$O(B scan_word() $B;2>H$N$3$H(B */
/* reduce computation cost in hypothesis scoring: keep the maximum
   score at each frame throughout the expanded hypotheses as 'score envelope'.
   When expanding HMM trellis for updating score, only Viterbi paths that
   are within a certain range from the maximum scores are expanded.
   see scan_word() for detail */

/* the score envelope is kept in framemaxscore[0..framelen-1] */

/* initialize score envelope */
void
envl_init(int framenum)		/* maximum frame num */
{
  int i;
  for(i=0;i<framenum;i++) framemaxscore[i] = LOG_ZERO;
}

/* update score envelope using forward score g[] of hypothesis 'n' */
static void
envl_update(NODE *n, int framenum)
{
  int t;
  for(t=framenum-1;t>=0;t--) {
    if (framemaxscore[t] < n->g[t]) framemaxscore[t] = n->g[t];
  }
}
#endif /* SCAN_BEAM */


#ifdef USE_DFA
static NEXTWORD fornoise;		/* noise dummy */
#endif


#ifdef SP_BREAK_CURRENT_FRAME
/**********************************************************************/
/********** short pause segmentation **********************************/
/**********************************************************************/
/* $B<!$N%;%0%a%s%H$NBh#1%Q%9$r3+;O$9$k:]$N=i4|C18lMzNr$r(B hypo $B$+$iEAHB$5$;$k!%(B
   $B=i4|C18lMzNr$O(B, $B2>@b$N:G=*C18l(B(hypo->seq[0..])$B$+$i?t$($F(B,
   $BC5:w$r:F3+$9$k=i4|2>@bC18l$H0[$J$j$+$DF)2a8l$G$J$$:G=i$NC18l!%(B
   $B3:EvC18lL5$7(B($B$3$N%;%0%a%s%H$,(B1$BC18l$dF)2aC18l$@$1$@$C$?(B)$B$N>l9g(B,
   $B85$NCM(B($B0JA0$N%;%0%a%s%H$NCM(B)$B$r$=$N$^$^J];}(B
*/
/* set the initial word context info of the next segment from hypothesis
   'hypo'.  the initial context word will be the last word which is not
   equal to the resuming word (initial hypo in the next segment) and
   not transparent word.  If no suitable word is found in hypothesis 'hypo',
   the previous value (given by the previous input and used in this segment)
   is kept */
void
sp_segment_set_last_nword(NODE *hypo)
{
  int i;
  WORD_ID w;
  for(i=0;i<hypo->seqnum;i++) {
    w = hypo->seq[i];
    if (w != sp_break_last_word
       && !is_sil(w, winfo, hmminfo)
       && !winfo->is_transparent[w]
       ) {
      sp_break_last_nword = w;
      break;
    }
  }
#ifdef SP_BREAK_DEBUG
  printf("sp_break_last_nword=%d[%s]\n", sp_break_last_nword, winfo->woutput[sp_break_last_nword]);
#endif
}
#endif


/**********************************************************************/
/********* debug output of hypothesis *********************************/
/**********************************************************************/

/* $BDL>o$N=PNO4X?t$O(B result_*.c $B$K$"$k(B */
/* normal output functions are in result_*.c */

static HTK_Param *tparam;	/* temporal for forced alignment */

/* output word sequence of a hypothesis */
static void
put_hypo_woutput(NODE *hypo)
{
  int i,w;

  if (hypo != NULL) {
    for (i=hypo->seqnum-1;i>=0;i--) {
      w = hypo->seq[i];
      j_printf(" %s",winfo->woutput[w]);
    }
  }
  j_printf("\n");  
}

/* output N-gram entry (or DFA ID) sequence of a hypothesis */
static void
put_hypo_wname(NODE *hypo)
{
  int i,w;

  if (hypo != NULL) {
    for (i=hypo->seqnum-1;i>=0;i--) {
      w = hypo->seq[i];
      j_printf(" %s",winfo->wname[w]);
    }
  }
  j_printf("\n");  
}

/* output phoneme sequence (with word delimiter '|') of a hypothesis */
static void
put_hypo_phoneme(NODE *hypo)
{
  int i,j,w;
  char buf[20];

  if (hypo != NULL) {
    for (i=hypo->seqnum-1;i>=0;i--) {
      w = hypo->seq[i];
      for (j=0;j<winfo->wlen[w];j++) {
	center_name(winfo->wseq[w][j]->name, buf);
	j_printf(" %s", buf);
      }
      if (i > 0) j_printf(" |");
    }
  }
  j_printf("\n");  
}

/**********************************************************************/
/******** output top 'ncan' hypotheses in a stack and free all ********/
/**********************************************************************/
static void
result_reorder_and_output(NODE **r_start, /* top node */
			  NODE **r_bottom, /* bottom node */
			  int *r_stacknum, /* total number of node in stack */
			  int ncan, /* number to pop */
			  WORD_INFO *winfo)
{
  NODE *now;
  int num;

#ifdef CM_NBEST 
  /* compute CM from the N-best sentence candidates */
  cm_compute_from_nbest(*r_start, *r_stacknum);
#endif
  num = 0;
  result_pass2_begin();
  while ((now = get_best_from_stack(r_start,r_stacknum)) != NULL && num < ncan) {
    num++;
    /* output result */
    result_pass2(now, num, winfo);
#ifdef VISUALIZE
    visual2_best(now, winfo);
#endif
#ifdef SP_BREAK_CURRENT_FRAME
    /* set the last context-aware word for next recognition */
    if (sp_break_last_nword_allow_override && num == 1) sp_segment_set_last_nword(now);
#endif
    /* do forced alignment if needed */
    if (align_result_word_flag) word_rev_align(now->seq, now->seqnum, tparam);
    if (align_result_phoneme_flag) phoneme_rev_align(now->seq, now->seqnum, tparam);
    if (align_result_state_flag) state_rev_align(now->seq, now->seqnum, tparam);
    free_node(now);
  }
  result_pass2_end();
  /* free the rest */
  if (now != NULL) free_node(now);
  free_all_nodes(*r_start);
}  



/**********************************************************************/
/********* main stack decoding function *******************************/
/**********************************************************************/

/* subfunctions called by this main function:
     for word prediction: ngram_decode.c or dfa_decode.c
     for hypothesis expansion and forward viterbi: search_bestfirst_v{1,2}.c
                                  (v1: for efficiency   v2: for accuracy)
*/

void
wchmm_fbs(
     HTK_Param *param,		/* input parameter vector */
     BACKTRELLIS *backtrellis,	/* backward word trellis from 1st pass */
     LOGPROB backmax,		/* maximum score in backward trellis */
     int stacksize,		/* stack size of hypothesis */
     int ncan,			/* num of hypothesis to get */
     int maxhypo		/* search abort when 'maxhypo' hypotheses are popped */
     )
{
  /* $BJ82>@b%9%?%C%/(B */
  /* hypothesis stack (double-linked list) */
  int stacknum;			/* current stack size */
  NODE *start = NULL;		/* top node */
  NODE *bottom = NULL;		/* bottom node */

  /* $BG'<17k2L3JG<%9%?%C%/(B($B7k2L$O$3$3$X$$$C$?$s=8$a$i$l$k(B) */
  /* result sentence stack (found results will be stored here and then re-ordered) */
  int r_stacksize = ncan + 1;
  int r_stacknum;
  NODE *r_start = NULL;
  NODE *r_bottom = NULL;

  /* $B<o!9$N%+%&%s%?!<(B */
  /* monitoring counters */
  int popctr = 0;		/* num of popped hypotheses from stack */
  int genectr = 0;		/* num of generated hypotheses */
  int pushctr = 0;		/* num of hypotheses actually pushed to stack */
  int finishnum = 0;		/* num of found sentence hypothesis */

  /* $B%o!<%/%(%j%"(B */
  /* work area */
  NEXTWORD **nextword, *nwroot;	/* buffer to store predicted words */
  int maxnwnum;			/* current allocated number of words in nextword */
  int nwnum;			/* current number of words in nextword */
  NODE *now, *new;		/* popped current hypo., expanded new hypo. */
#ifdef USE_DFA
  NODE *now_noise, *now_noise_tmp; /* for inserting/deleting noise word */
  boolean now_noise_calced;
  int t;
#endif
  int w,i,j;
  LOGPROB last_score = LOG_ZERO;

  /* copy parameter pointer beforehand for forced alignment */
  if (align_result_word_flag || align_result_phoneme_flag || align_result_state_flag) tparam = param;

  /*
   * $B=i4|2=(B
   * Initialize
   */
  peseqlen = backtrellis->framelen; /* (just for quick access) */
  /* $BM=B,C18l3JG<NN0h$r3NJ](B */
  /* malloc area for word prediction */
  nextword = nw_malloc(&maxnwnum, &nwroot);
  /* $BA08~$-%9%3%"7W;;MQ$NNN0h$r3NJ](B */
  /* malloc are for forward viterbi (scan_word()) */
  malloc_wordtrellis();		/* scan_word$BMQNN0h(B */
  /* $B2>@b%9%?%C%/=i4|2=(B */
  /* initialize hypothesis stack */
  start = bottom = NULL;
  stacknum = 0;
  /* $B7k2L3JG<%9%?%C%/=i4|2=(B */
  /* initialize result stack */
  if (result_reorder_flag) {
    r_start = r_bottom = NULL;
    r_stacknum = 0;
  }
  
#ifdef CM_SEARCH
  /* initialize local stack */
  cm_init(winfo);
#endif
  /* $B%(%s%Y%m!<%WC5:wMQ$NC18lD9JLE83+?t%+%&%s%?$r=i4|2=(B */
  /* initialize counters for envelope search */
  if (enveloped_bestfirst_width >= 0) wb_init();

  if (verbose_flag) j_printf("samplenum=%d\n", peseqlen);
  if (result_reorder_flag) {
    if (debug2_flag) VERMES("getting %d candidates...\n",ncan);
  }

#ifdef VISUALIZE
  visual2_init(maxhypo);
#endif

  /* 
   * $B=i4|2>@b(B(1$BC18l$+$i$J$k(B)$B$rF@(B, $BJ82>@b%9%?%C%/$K$$$l$k(B
   * get a set of initial words from LM function and push them as initial
   * hypotheses
   */
  /* the first words will be stored in nextword[] */
#ifdef USE_NGRAM
  nwnum = ngram_firstwords(nextword, peseqlen, maxnwnum, winfo, backtrellis);
#else  /* USE_DFA */
  nwnum = dfa_firstwords(nextword, peseqlen, maxnwnum, dfa);
  /* $B0n$l$?$i!"%P%C%U%!$rA}$d$7$F:F%A%c%l%s%8(B */
  /* If the number of nextwords can exceed the buffer size, expand the
     nextword data area */
  while (nwnum < 0) {
    nextword = nw_expand(nextword, &maxnwnum, &nwroot);
    nwnum = dfa_firstwords(nextword, peseqlen, maxnwnum, dfa);
  }
#endif

  if (debug2_flag) {
    j_printf("%d words in wordtrellis as first hypothesis\n", nwnum);
  }
  
  /* store them to stack */
  for (w = 0; w < nwnum; w++) {
    /* generate new hypothesis */
    new = newnode();
    start_word(new,nextword[w],param,backtrellis);
#ifdef USE_DFA
    if (new->score <= LOG_ZERO) { /* not on trellis */
      free_node(new);
      continue;
    }
#endif
    genectr++;
#ifdef CM_SEARCH
    /* store the local hypothesis to temporal stack */
    cm_store(new);
#else 
    /* put to stack */
    if (put_to_stack(new, &start, &bottom, &stacknum, stacksize) != -1) {
#ifdef VISUALIZE
      visual2_next_word(new, NULL, popctr);
#endif
      pushctr++;
    }
#endif
  }
#ifdef CM_SEARCH
  /* compute score sum */
  cm_sum_score();
  /* compute CM and put the generated hypotheses to global stack */
  while ((new = cm_get_node()) != NULL) {
    cm_set_score(new);
    if (put_to_stack(new, &start, &bottom, &stacknum, stacksize) != -1) {
#ifdef VISUALIZE
      visual2_next_word(new, NULL, popctr);
#endif
      pushctr++;
    }
  }
#endif

  if (debug2_flag) {
    j_printf("%d pushed\n", pushctr);
  }
  
  /********************/
  /* main search loop */
  /********************/

  for (;;) {

    if (module_mode) {
      /* if terminate signal has been received, cancel this input */
      if (module_wants_terminate()) {
	j_printf("terminated\n");
	break;
      }
    }
    
    /* 
     * $B2>@b%9%?%C%/$+$i:G$b%9%3%"$N9b$$2>@b$r<h$j=P$9(B
     * pop the top hypothesis from stack
     */
#ifdef DEBUG
    VERMES("get one hypothesis\n");
#endif
    now = get_best_from_stack(&start,&stacknum);
    if (now == NULL) {  /* stack empty ---> $BC5:w=*N;(B*/
      j_printf("stack empty, search terminate now\n");
      break;
    }
    /* (bogus score check) */
    if (now->score <= LOG_ZERO) continue;

    /* word envelope $B%A%'%C%/(B */
    /* consult word envelope */
    if (enveloped_bestfirst_width >= 0) {
      if (!wb_ok(now)) {
	/* $B$3$N2>@bD9$K$*$1$kE83+852>@b?t$NN_7W?t$O4{$KogCM$r1[$($F$$$k!%(B
	   $B$=$N$?$a!$$3$N2>@b$O<N$F$k!%(B*/
	/* the number of popped hypotheses at the length already
	   reaches its limit, so the current popped hypothesis should
	   be discarded here with no expansion */
	free_node(now);
	continue;
      }
    }
    
    popctr++;

    /* (for debug) $B<h$j=P$7$?2>@b$H$=$N%9%3%"$r=PNO(B */
    /*             output information of the popped hypothesis to stdout */
    if (debug2_flag) {
      j_printf("--- pop %d:\n", popctr);
      j_printf("  "); put_hypo_woutput(now);
      j_printf("  "); put_hypo_wname(now);
      j_printf("  %d words, f=%f, g=%f\n", now->seqnum, now->score, now->g[now->bestt]);
      j_printf("  last word on trellis: [%d-%d]\n", now->estimated_next_t + 1, now->bestt);
    }

#ifdef VISUALIZE
    visual2_popped(now, popctr);
#endif
    
    /* $B<h$j=P$7$?2>@b$N%9%3%"$r85$K(B score envelope $B$r99?7(B */
    /* update score envelope using the popped hypothesis */
    envl_update(now, peseqlen);

    /* 
     * $B<h$j=P$7$?2>@b$N<uM}%U%i%0$,4{$KN)$C$F$$$l$P!$(B
     * $B$=$N2>@b$OC5:w=*N;$H$_$J$7!$7k2L$H$7$F=PNO$7$F<!$N%k!<%W$X!%(B
     *
     * If the popped hypothesis already reached to the end, 
     * we can treat it as a recognition result.
     */
#ifdef DEBUG
    VERMES("endflag check\n");
#endif
    
    if (now->endflag) {
      if (debug2_flag) {
	j_printf("  This is a final sentence candidate\n");
      }
      /* quick, dirty hack */
      if (now->score == last_score) {
	free_node(now);
	continue;
      } else {
	last_score = now->score;
      }
      
      finishnum++;
      if (debug2_flag) {
	j_printf("  %d-th sentence found\n", finishnum);
      }

      if (result_reorder_flag) {
	/* $B0lDj?t$N2>@b$,F@$i$l$?$"$H%9%3%"$G%=!<%H$9$k$?$a!$(B
	   $B0l;~E*$KJL$N%9%?%C%/$K3JG<$7$F$*$/(B */
	/* store the result to result stack
	   after search is finished, they will be re-ordered and output */
	put_to_stack(now, &r_start, &r_bottom, &r_stacknum, r_stacksize);
	/* $B;XDj?t$NJ82>@b$,F@$i$l$?$J$iC5:w$r=*N;$9$k(B */
	/* finish search if specified number of results are found */
	if (finishnum >= ncan) {
	  if (debug2_flag) VERMES("%d\n",finishnum);
	  break;
	} else {
	  if (debug2_flag) VERMES("%d..", finishnum);
	  continue;
	}
      } else {			/* result_reorder $B$G$J$$>l9g$O(B */
	/* $B7k2L$O8+$D$+$j<!Bh$9$0$K=PNO(B */
	/* result are directly output as soon as found */
	if (finishnum == 1) result_pass2_begin(); /* first time */
	result_pass2(now, finishnum, winfo);
#ifdef VISUALIZE
	visual2_best(now, winfo);
#endif
#ifdef SP_BREAK_CURRENT_FRAME
	/* $B%7%g!<%H%]!<%:%;%0%a%s%F!<%7%g%s;~$O<!$N%;%0%a%s%HG'<13+;O;~$K(B
	   $BC18lMzNr$rEAHB$5$;$k(B */
	/* set the last context-aware word for next recognition */
	if (sp_break_last_nword_allow_override && finishnum == 1) sp_segment_set_last_nword(now);
#endif
	/* $BI,MW$G$"$l$P(B, $BG'<17k2L$rMQ$$$FF~NO$N%;%0%a%s%F!<%7%g%s$r9T$J$&(B */
	/* do forced alignment with the result if needed */
	if (align_result_word_flag) word_rev_align(now->seq, now->seqnum, tparam);
	if (align_result_phoneme_flag) phoneme_rev_align(now->seq, now->seqnum, tparam);
	if (align_result_state_flag) state_rev_align(now->seq, now->seqnum, tparam);
	/* $B2>@b$r2rJ|$7$F<!$N2>@b$X(B */
	/* free the hypothesis and continue to next */
	free_node(now);
	/* $B$3$l$G;XDj?t$NJ82>@b$,F@$i$l$?$J$i(B, $B$3$3$GC5:w$r=*N;$9$k(B */
	/* finish search if specified number of results are found */
	if (finishnum >= ncan) {
	  result_pass2_end();
	  break;		/* end of search */
	}
	else continue;
      }
      
    } /* end of now->endflag */

    
    /* 
     * $BC5:w<:GT$r8!=P$9$k!%(B
     * $B2>@b?t$,(B maxhypo $B0J>eE83+$5$l$?$i(B, $B$b$&$3$l0J>e$OC5:w$7$J$$(B
     *
     * detecting search failure:
     * if the number of expanded hypotheses reaches maxhypo, giveup further search
     */
#ifdef DEBUG
    VERMES("loop end check\n");
#endif
    if (popctr >= maxhypo) {
      j_printf("num of hypotheses overflow\n");
      /* (for debug) $BC5:w<:GT;~$K!"%9%?%C%/$K;D$C$?>pJs$rEG$-=P$9(B */
      /* (for debug) output all hypothesis remaining in the stack */
      if (debug2_flag) put_all_in_stack(&start, &stacknum);
      free_node(now);
      break;			/* end of search */
    }
    /* $B2>@bD9$,0lDjCM$r1[$($?$H$-!$$=$N2>@b$rGK4~$9$k(B */
    /* check hypothesis word length overflow */
    if (now->seqnum >= MAXSEQNUM) {
      j_printerr("sentence length exceeded ( > %d)\n",MAXSEQNUM);
      free_node(now);
      continue;
    }

    /*
     * $BA08~$-%9%3%"$r99?7$9$k!'(B $B:G8e$NC18l$NItJ,$NA08~$-%9%3%"$r7W;;$9$k!%(B
     * update forward score: compute forward trellis for the last word
     */
#ifdef DEBUG
    VERMES("scan_word\n");
#endif
    scan_word(now, param);
    if (now->score < backmax + LOG_ZERO) { /* another end-of-search detecter */
      j_printf("now->score = %f?\n",now->score);
      put_hypo_woutput(now);
      free_node(now);
      break;
    }
    
    /* 
     * $B<h$j=P$7$?2>@b$,J8$H$7$F<uM}2DG=$G$"$l$P!$(B
     * $B<uM}%U%i%0$rN)$F$F$r%9%?%C%/$K$$$lD>$7$F$*$/!%(B
     * ($B<!$K<h$j=P$5$l$?$i2r$H$J$k(B)
     *
     * if the current popped hypothesis is acceptable, set endflag
     * and return it to stack: it will become the recognition result
     * when popped again.
     */
#ifdef DEBUG
    VERMES("accept check\n");
#endif
    if (
#ifdef USE_NGRAM
	ngram_acceptable(now, winfo)
#else  /* USE_DFA */
	dfa_acceptable(now, dfa)
#endif
	&& now->estimated_next_t <= 5) {
      new = newnode();
      /* new $B$K(B now $B$NCf?H$r%3%T!<$7$F!$:G=*E*$J%9%3%"$r7W;;(B */
      /* copy content of 'now' to 'new', and compute the final score */
      last_next_word(now, new, param);
      if (debug2_flag) {
	j_printf("  This is acceptable as a sentence candidate\n");
      }
      /* $B<uM}%U%i%0$rN)$F$FF~$lD>$9(B */
      /* set endflag and push again  */
      if (debug2_flag) {
	j_printf("  This hypo itself was pushed with final score=%f\n", new->score);
      }
      new->endflag = TRUE;
      put_to_stack(new, &start, &bottom, &stacknum, stacksize);
      /* $B$3$N2>@b$O$3$3$G=*$o$i$:$K(B, $B$3$3$+$i$5$i$KC18lE83+$9$k(B */
      /* continue with the 'now' hypothesis, not terminate here */
    }

    /*
     * $B$3$N2>@b$+$i!$<!C18l=89g$r7hDj$9$k!%(B
     * $B<!C18l=89g$O(B, $B$3$N2>@b$N?dDj;OC<%U%l!<%`<~JU$KB8:_$7$?(B
     * $BBh#1%Q%9$N%H%l%j%9C18l=89g!%(B
     *
     * N-gram$B$N>l9g$O3FC18l$N(B n-gram $B@\B33NN($,4^$^$l$k!%(B
     * DFA $B$N>l9g$O(B, $B$=$NCf$G$5$i$K(B DFA $B>e$G@\B32DG=$J$b$N$N$_$,JV$C$F$/$k(B
     */
    /*
     * Determine next word set that can connect to this hypothesis.
     * They come from the trellis word that has been survived at near the
     * beginning of the last word.
     *
     * In N-gram mode, they also contain N-gram probabilities toward the
     * source hypothesis.  In DFA mode, the word set is further reduced
     * by the grammatical constraint
     */
#ifdef DEBUG
    VERMES("get next words\n");
#endif
#ifdef USE_NGRAM
    nwnum = ngram_nextwords(now, nextword, maxnwnum, ngram, winfo, backtrellis);
#else  /* USE_DFA */
    nwnum = dfa_nextwords(now, nextword, maxnwnum, dfa);
    /* nextword $B$,0n$l$?$i!"%P%C%U%!$rA}$d$7$F:F%A%c%l%s%8(B */
    /* If the number of nextwords can exceed the buffer size, expand the
       nextword data area */
    while (nwnum < 0) {
      nextword = nw_expand(nextword, &maxnwnum, &nwroot);
      nwnum = dfa_nextwords(now, nextword, maxnwnum, dfa);
    }
#endif
    if (debug2_flag) {
      j_printf("  %d words extracted from wordtrellis\n", nwnum);
    }

    /* 
     * $B2>@b$H<!C18l=89g$+$i?7$?$JJ82>@b$r@8@.$7!$%9%?%C%/$K$$$l$k!%(B
     */
    /*
     * generate new hypotheses from 'now' and 'nextword', 
     * and push them to stack
     */
#ifdef DEBUG
    VERMES("generate hypo\n");
#endif

#ifdef USE_DFA
    now_noise_calced = FALSE;	/* TRUE is noise-inserted score has been calculated */
#endif
    i = pushctr;		/* store old value */

#ifdef CM_SEARCH
    /* initialize local stack */
    cm_init(winfo);
#endif

    /* for each nextword, generate a new hypothesis */
    for (w = 0; w < nwnum; w++) {
      new = newnode();
#ifdef USE_DFA
      if (nextword[w]->can_insert_sp == TRUE) {
	/* $B%N%$%:$r64$s$@%H%l%j%9%9%3%"$r7W;;$7!$64$^$J$$>l9g$H$N:GBgCM$r<h$k(B */
	/* compute hypothesis score with noise inserted */
	
	if (now_noise_calced == FALSE) {
	  /* now $B$K(B sp $B$r$D$1$?2>@b(B now_noise $B$r:n$j(B,$B$=$N%9%3%"$r7W;;(B */
	  /* generate temporal hypothesis 'now_noise' which has short-pause
	     word after the original 'now' */
	  fornoise.id = dfa->sp_id;
	  now_noise = newnode();
	  cpy_node(now_noise, now);
#if 0
	  now_noise_tmp = newnode();
	  next_word(now, now_noise_tmp, &fornoise, param, backtrellis);
	  scan_word(now_noise_tmp, param);
	  for(t=0;t<peseqlen;t++) {
	    now_noise->g[t] = max(now_noise_tmp->g[t], now->g[t]);
	  }
	  free_node(now_noise_tmp);
#else
	  /* expand NOISE only if it exists in backward trellis */
	  /* begin patch by kashima */
	  if (looktrellis_flag) {
	    if(!dfa_look_around(&fornoise, now, backtrellis)){
	      free_node(now_noise);
	      free_node(new);
	      continue;
	    }
	  }
	  /* end patch by kashima */

	  /* now_nosie $B$N(B $B%9%3%"(B g[] $B$r7W;;$7!$85$N(B now $B$N(B g[] $B$HHf3S$7$F(B
	     $B9b$$J}$r:NMQ(B */
	  /* compute trellis score g[], and adopt the maximum score
	     for each frame compared with now->g[] */
	  next_word(now, now_noise, &fornoise, param, backtrellis);
	  scan_word(now_noise, param);
	  for(t=0;t<peseqlen;t++) {
	    now_noise->g[t] = max(now_noise->g[t], now->g[t]);
	  }
	  /* $B%N%$%:$r64$s$@:]$r9MN8$7$?%9%3%"$r7W;;$7$?$N$G!$(B
	     $B$3$3$G:G8e$N%N%$%:C18l$r(B now_noise $B$+$i>C$9(B */
	  /* now that score has been computed considering pause insertion,
	     we can delete the last noise word from now_noise here */
	  now_noise->seqnum--;
#endif
	  now_noise_calced = TRUE;
	}

	/* expand word only if it exists in backward trellis */
	/* begin patch by kashima */
	if (looktrellis_flag) {
	  if(!dfa_look_around(nextword[w], now_noise, backtrellis)){
	    free_node(new);
	    continue;
	  }
	}
	/* end patch by kashima */
	
	/* $B?7$7$$2>@b(B' new' $B$r(B 'now_noise' $B$+$i@8@.(B */
	/* generate a new hypothesis 'new' from 'now_noise' */
	next_word(now_noise, new, nextword[w], param, backtrellis);
	
      } else {
	
	/* expand word only if it exists in backward trellis */
	/* begin patch by kashima */
	if (looktrellis_flag) {
	  if(!dfa_look_around(nextword[w], now, backtrellis)){
	    free_node(new);
	    continue;
	  }
	}
	/* end patch by kashima */
	
	/* $B?7$7$$2>@b(B' new' $B$r(B 'now_noise' $B$+$i@8@.(B */
	/* generate a new hypothesis 'new' from 'now_noise' */
	next_word(now, new, nextword[w], param, backtrellis);
	
      }
#else  /* not USE_DFA */

      /* $B?7$7$$2>@b(B' new' $B$r(B 'now_noise' $B$+$i@8@.(B
	 N-gram $B$N>l9g$O%N%$%:$rFCJL07$$$7$J$$(B */
      /* generate a new hypothesis 'new' from 'now'.
	 pause insertion is treated as same as normal words in N-gram mode. */
      next_word(now, new, nextword[w], param, backtrellis);

#endif /* USE_DFA */

      if (new->score <= LOG_ZERO) { /* not on trellis */
	free_node(new);
	continue;
      }
      genectr++;

#ifdef CM_SEARCH
      /* store the local hypothesis to temporal stack */
      cm_store(new);
#else 
      /* $B@8@.$7$?2>@b(B 'new' $B$r%9%?%C%/$KF~$l$k(B */
      /* push the generated hypothesis 'new' to stack */
      if (put_to_stack(new, &start, &bottom, &stacknum, stacksize) != -1) {/* success */
	if (debug2_flag) {
	  j = new->seq[new->seqnum-1];
	  j_printf("  %15s [%15s](id=%5d)(%f) [%d-%d] pushed\n",winfo->wname[j], winfo->woutput[j], j, new->score, new->estimated_next_t + 1, new->bestt);
	}
#ifdef VISUALIZE
	visual2_next_word(new, now, popctr);
#endif
	pushctr++;
      }
#endif
    } /* end of nextword loop */

#ifdef CM_SEARCH
    /* compute score sum */
    cm_sum_score();
    /* compute CM and put the generated hypotheses to global stack */
    while ((new = cm_get_node()) != NULL) {
      cm_set_score(new);
      /*      j = new->seq[new->seqnum-1];
	      printf("  %15s [%15s](id=%5d)(%f) [%d-%d] cm=%f\n",winfo->wname[j], winfo->woutput[j], j, new->score, new->estimated_next_t + 1, new->bestt, new->cmscore[new->seqnum-1]);*/
      if (put_to_stack(new, &start, &bottom, &stacknum, stacksize) != -1) {/* success */
	if (debug2_flag) {
	  j = new->seq[new->seqnum-1];
	  j_printf("  %15s [%15s](id=%5d)(%f) [%d-%d] pushed\n",winfo->wname[j], winfo->woutput[j], j, new->score, new->estimated_next_t + 1, new->bestt);
	}
#ifdef VISUALIZE
	visual2_next_word(new, now, popctr);
#endif
	pushctr++;
      }
    }
#endif

    if (debug2_flag) {
      j_printf("%d pushed\n",pushctr-i);
    }
#ifdef USE_DFA
    if (now_noise_calced == TRUE) free_node(now_noise);
#endif

    /* 
     * $B<h$j=P$7$?2>@b$r<N$F$k(B
     * free the source hypothesis
     */
    free_node(now);

    if (catch_intr_flag) break;
    
  }
  /***************/
  /* End of Loop */
  /***************/

  /* output */
  /* $BC5:w<:GT$G0l$D$b8uJd$,F@$i$l$J$1$l$P!$Bh(B1$B%Q%9$N7k2L$r$=$N$^$^=PNO$9$k(B */
  /* If search failed and no candidate obtained in this pass, output
     the result of the previous 1st pass as a final result. */
  if (finishnum == 0) {		/* if search failed */
    if (verbose_flag) {
      j_printf("got no candidates, output 1st pass result as a final result\n",finishnum);
    }
    /* make hypothesis data from the result of previous 1st pass */
    now = newnode();
    for (i=0;i<pass1_wnum;i++) {
      now->seq[i] = pass1_wseq[pass1_wnum-1-i];
    }
    now->seqnum = pass1_wnum;
    now->score = pass1_score;
    /* output it */
    if (module_mode) {
      /* $B%b%8%e!<%k%b!<%I$G$O!$Bh(B2$B%Q%9$,<:GT$7$?$i2?$b=PNO$7$J$$(B */
      result_pass2_failed(winfo);
    } else {
      /* output the result of the first pass as the final output */
      result_pass2_begin();
      result_pass2(now, 1, winfo);
      result_pass2_end();
    }
#ifdef SP_BREAK_CURRENT_FRAME
    /* segment restart processing */
    if (sp_break_last_nword_allow_override) sp_segment_set_last_nword(now);
#endif
    /* do forced alignment if needed */
    if (align_result_word_flag) word_rev_align(now->seq, now->seqnum, tparam);
    if (align_result_phoneme_flag) phoneme_rev_align(now->seq, now->seqnum, tparam);
    if (align_result_state_flag) state_rev_align(now->seq, now->seqnum, tparam);
    free_node(now);
  } else {			/* if at least 1 candidate found */
    if (debug2_flag) {
      j_printf("got %d candidates\n",finishnum);
    }
    if (result_reorder_flag) {
      /* $B7k2L$O$^$@=PNO$5$l$F$$$J$$$N$G!$J88uJdMQ%9%?%C%/Fb$r%=!<%H$7$F(B
	 $B$3$3$G=PNO$9$k(B */
      /* As all of the found candidate are in result stack, we sort them
	 and output them here  */
      if (debug2_flag) VERMES("done\n");
      result_reorder_and_output(&r_start, &r_bottom, &r_stacknum, output_hypo_maxnum, winfo);
    }
  }

  /* $B3F<o%+%&%s%?$r=PNO(B */
  /* output counters */
  if (verbose_flag) {
    j_printf("%d generated, %d pushed, %d nodes popped in %d\n",
	     genectr, pushctr, popctr, backtrellis->framelen);
    j_flushprint();
  }
  
  /* $B=*N;=hM}(B */
  /* finalize */
  nw_free(nextword, nwroot);
  free_all_nodes(start);
  free_wordtrellis();

}
