/*
 * Copyright (c) 1991-2003 Kyoto University
 * Copyright (c) 2000-2003 NAIST
 * All rights reserved
 */

/* ngram_decode.c --- for Julius: predict next word using N-gram probability
                      and word trellis index (for 2nd pass)*/

/* $Id: ngram_decode.c,v 1.8 2003/09/29 06:01:22 ri Exp $ */

/* N-gram$B$rMQ$$$?%9%?%C%/%G%3!<%G%#%s%0(B($BBh(B2$B%Q%9(B)$B$K$*$1$k<!C18l=89g$N7hDj(B:
   $BM?$($i$l$?E83+852>@b$NM=B,$5$l$k;OC<%U%l!<%`<~JU$NC18l%H%l%j%9>e$K(B
   $B;D$C$F$$$kC18l$K$D$$$F!$$=$N(BN-gram$B3NN($H$H$b$KJV$9(B */

/* Prediction of next words on N-gram based stack decoding (2nd pass):
   survived words in word trellis around the estimated end frame of source
   hypothesis will be returned with their N-gram probabilities. */

/* Julius $B%b!<%I$G$O(B ngram_firstwords(), ngram_nextwords(),
   ngram_acceptable(), ngram_eosscore() $B$,Bh(B2$B%Q%9$N%a%$%s4X?t(B wchmm_fbs()
   $B$KEO$5$l(B, $B;HMQ$5$l$k!%(B(Julian $B%b!<%I$G$O(B "dfa_decode.c" $BFb$N(B dfa_*() $B$,(B
   $B;H$o$l$k(B */

/* In julian mode, pointers to functions ngram_firstwords(), ngram_nextwords(),
   ngram_acceptable() and ngram_eosscore() are passed to main search function
   wchmm_fbs().  (for julian, corresponding functions dfa_*() in
   "dfa_decode.c" will be used instead) */
   
#include <julius.h>

#ifdef USE_NGRAM

/* for sorting next word candidates by their word ID */
static int
compare_nw(NEXTWORD **a, NEXTWORD **b)
{
  if ((*a)->id > (*b)->id) return 1;
  if ((*a)->id < (*b)->id) return -1;
  return 0;
}
/* find next word candiate whose id 'w' */
static NEXTWORD *
search_nw(NEXTWORD **nw, WORD_ID w, int num)
{
  int left,right,mid;
  NEXTWORD *tmp;

  if (num == 0) return NULL;
  left = 0;
  right = num - 1;
  while (left < right) {
    mid = (left + right) / 2;
    if ((nw[mid])->id < w) {
      left = mid + 1;
    } else {
      right = mid;
    }
  }
  tmp = nw[left];
  if (tmp->id == w) {
    return tmp;
  } else {
    return NULL;
  }
}

/*********** transparent words handling on 2nd pass **********/
static WORD_ID cnword[2];	/* current context words */
static int cnnum;		/* num of found context words */
static int last_trans;		/* num of skipped transparent words */
/* From cseq[0..n-1], set the last context words (= non-transparent words in
   hypothesis context) to cnword[0] and cnword[1].
   Also, the num of skipped transparent words are
   counted in last_trans.
 */
static void
set_word_context(WORD_ID *cseq, int n, WORD_INFO *winfo)
{
  WORD_ID cw[2];
  int i;

  cnnum = 0;
  last_trans = 0;
  for(i=n-1;i>=0;i--) {
    if (! winfo->is_transparent[cseq[i]]) {
      cnword[cnnum++] = winfo->wton[cseq[i]];
      if (cnnum >= 2) break;
    } else {
      last_trans++;
    }
  }
}

/*************************************************************/

/* lookup survived words on specified frame in backtrellis, 
   compute their N-gram probabilities, and add them to NEXTWORD data */
static int
pick_backtrellis_words(
		       BACKTRELLIS *bt,	/* backtrellis info */
		       WORD_INFO *winfo, /* word dictionary */
		       NGRAM_INFO *ngram, /* N-gram info */
		       NEXTWORD **nw, /* next words info (return value) */
		       int oldnum, /* initial offset of nw */
		       NODE *hypo, /* source hypothesis */
		       short t	/* time frame */
		       )
{
  int i;
  WORD_ID w;
  int nword1=0, nword2=0;
  LOGPROB rawscore;
#ifdef WPAIR
  int w_old = WORD_INVALID;
#endif
  int num;

  num = oldnum;
  /* set word context to cnword[0], cnword[1] */
  set_word_context(hypo->seq, hypo->seqnum, winfo);
  /* lookup survived words in backtrellis on time frame 't' */
  for (i=0;i<bt->num[t];i++) {
    w = (bt->rw[t][i])->wid;
#ifdef WORD_GRAPH
    /* only words on the word graphs are expanded */
    if (!(bt->rw[t][i])->within_wordgraph) continue;
    /* word connection is restricted by the graph structure */
    if (w != ((hypo->tre)->last_tre)->wid) continue;
#else /* not WORD_GRAPH */
#ifdef WPAIR
    /* some word have same word ID with different previous word, so
       only one will be opened (best word will be selected later
       by next_word() */
    if (w == w_old) continue;	/* backtrellis is sorted by word ID */
    else w_old = w;
#endif /* WPAIR */
#endif /* not WORD_GRAPH */
    /* skip if already exist */
    if (search_nw(nw, w, oldnum) != NULL) continue;
    switch(cnnum) {		/* length of valid context */
    case 0:			/* unigram */
      rawscore = uni_prob(ngram, winfo->wton[w]);
      break;
    case 1:			/* bigram */
      rawscore = bi_prob_rl(ngram, winfo->wton[w], cnword[0]);
      break;
    default:			/* trigram */
      rawscore = tri_prob_rl(ngram, winfo->wton[w], cnword[0], cnword[1]);
      break;
    }
#ifdef CLASS_NGRAM
    rawscore += winfo->cprob[w];
#endif
    nw[num]->tre   = bt->rw[t][i];
    nw[num]->id    = w;
    nw[num]->lscore = lm_weight2 * rawscore + lm_penalty2;
    if (winfo->is_transparent[w]) {
      /*nw[num]->lscore -= (LOGPROB)last_trans * TRANS_RENZOKU_PENALTY;*/
      if (winfo->is_transparent[hypo->seq[hypo->seqnum-1]]) {
	nw[num]->lscore += lm_penalty_trans;
      }
    }
    
    /* j_printf("%d: %s added\n", num, winfo->wname[nw[num]->id]); */
    num++;
  }
  return num;
}

/* Look for survived backtrellis words near the specified frame, and
   make NEXTWORD list.
   Words in frame [tm-lookup_range..tm+lookup_range-1] will be picked up.
   If a word exist in several frames, only one near the center frame
   will be taken: the true connection point will be determined later at
   next_word() */
int
get_backtrellis_words(
		      BACKTRELLIS *bt,	/* backtrellis info */
		      WORD_INFO *winfo, /* word dictionary */
		      NGRAM_INFO *ngram, /* N-gram info */
		      NEXTWORD **nw, /* next words info (return value) */
		      NODE *hypo, /* source hypothesis */
		      short tm,	/* center frame to lookup */
		      short t_end /* end frame of input */
		      )
{
  int num = 0;
  int t, t_step;
  int oldnum=0;

  if (tm < 0) return(0);

#ifdef PREFER_CENTER_ON_TRELLIS_LOOKUP
  /* fix for 3.2 (01/10/18 by ri) */
  /* before and after (one near center frame has high priority) */
  for (t_step = 0; t_step < lookup_range; t_step++) {
    /* before or center */
    t = tm - t_step;
    if (t < 0 || t > bt->framelen - 1 || t >= t_end) continue;
    num = pick_backtrellis_words(bt, winfo, ngram, nw, oldnum, hypo, t);
    if (num > oldnum) {
      qsort(nw, num, sizeof(NEXTWORD *),
	    (int (*)(const void *,const void *))compare_nw);
      oldnum = num;
    }
    if (t_step == 0) continue;	/* center */
    /* after */
    t = tm + t_step;
    if (t < 0 || t > bt->framelen - 1 || t >= t_end) continue;
    num = pick_backtrellis_words(bt, winfo, ngram, nw, oldnum, hypo, t);
    if (num > oldnum) {
      qsort(nw, num, sizeof(NEXTWORD *),
	    (int (*)(const void *,const void *))compare_nw);
      oldnum = num;
    }
  }

#else

  /* before the center frame */
  for(t = tm; t >= tm - lookup_range; t--) {
    if (t < 0) break;
    num = pick_backtrellis_words(bt, winfo, ngram, nw, oldnum, hypo, t);
    if (num > oldnum) {
      qsort(nw, num, sizeof(NEXTWORD *),
	    (int (*)(const void *,const void *))compare_nw);
      oldnum = num;
    }
  }
  /* after the center frame */
  for(t = tm + 1; t < tm + lookup_range; t++) {
    if (t > bt->framelen - 1) break;
    if (t >= t_end) break;
    num = pick_backtrellis_words(bt, winfo, ngram, nw, oldnum, hypo, t);
    if (num > oldnum) {
      qsort(nw, num, sizeof(NEXTWORD *),
	    (int (*)(const void *,const void *))compare_nw);
      oldnum = num;
    }
  }
#endif

  return num;
}

/* check limitation of next word */
/* return value: nextword num */
int
limit_nw(NEXTWORD **nw, NODE *hypo, int num)
{
  int src,dst;
  int newnum;

  /* <s>$B$+$i$O2?$bE83+$7$J$$(B */
  /* no hypothesis will be generated after "<s>" */
  if (hypo->seq[hypo->seqnum-1] == winfo->head_silwid) {
    return(0);
  }

  dst = 0;
  for (src=0; src<num; src++) {
    if (nw[src]->id == winfo->tail_silwid) {
      /* </s> $B$OE83+$7$J$$(B */
      /* do not expand </s> (it only appears at start) */
      continue;
    }
    if (src != dst) memcpy(nw[dst], nw[src], sizeof(NEXTWORD));
    dst++;
  }
  newnum = dst;

  return newnum;
}
	


/* $B:G=i$NC18l72$rJV$9!%JV$jCM(B: $BC18l?t(B (-1 on error) */
/* return initial word set.  return value: num of words (-1 on error) */
int
ngram_firstwords(
		 NEXTWORD **nw,	/* next word set (return value) */
				/* (assume already malloced) */
		 int peseqlen,	/* end time of input */
		 int maxnw,	/* maximum length of nw[] */
		 WORD_INFO *winfo, /* word dictionary */
		 BACKTRELLIS *bt)	/* backtrellis info */
{
#ifdef WORD_GRAPH
  int last_time;
#endif

#ifdef SP_BREAK_CURRENT_FRAME
  if (rest_param != NULL) {
    /* $B=i4|2>@b$O(B $B:G=*%U%l!<%`$K;D$C$?C18l%H%l%j%9>e$N:GL`C18l(B */
    /* the initial hypothesis is the best word survived on the last frame of
       the segment */
    nw[0]->id = sp_break_2_begin_word;
  } else {
    /* $B:G=*%;%0%a%s%H(B: $B=i4|2>@b$O(B $BC18l$NKvHx$NL52;C18l(B(=winfo->tail_silwid) */
    /* we are in the last of sentence: initial hypothesis is word-end silence word */
    nw[0]->id = winfo->tail_silwid;
  }
#else
  /* initial hypothesis is word-end silence word */
  nw[0]->id = winfo->tail_silwid;
#endif
#ifdef FIX_PENALTY
  nw[0]->lscore = 0.0;
#else
  nw[0]->lscore = lm_penalty2;
#endif
#ifdef WORD_GRAPH
  /* $B=i4|2>@b$N(BTRELLIS_ATOM$B$r$3$3$G7hDj$9$k(B */
  /* in word-graph mode, the first trellis atom should be defined here */
  for (last_time = peseqlen - 1; last_time >= 0; last_time--) {
    nw[0]->tre = bt_binsearch_atom(&backtrellis, last_time, winfo->tail_silwid);
    if (nw[0]->tre != NULL) break;
  }
  if (nw[0]->tre == NULL) {		/* </s> not found */
    j_printerr("no wordend left in beam!\n");
    j_printerr("beam width too small or input speech too short\n");
    return 0;
  }
#endif

  return 1;			/* number of words = 1 */
}

/* $B$"$k2>@b$N<!$N@\B3C18l72$rJV$9!%5"$jCM(B: $BC18l?t(B (-1 on error) */
/* return next word set from the hypothesis.  return value:
   num of words (-1 on error) */
int
ngram_nextwords(
		NODE *hypo,
		NEXTWORD **nw,
		int maxnw,	/* hypo: source */
		NGRAM_INFO *ngram, /* N-gram info */
		WORD_INFO *winfo, /* word dictionary */
		BACKTRELLIS *bt) /* backtrellis info */
{
  int num, num2;

  if (hypo->seqnum == 0) {
    j_error("gs_get_next_words: hypo contains no word\n");
  }

  /* $B2>@b$N?dDj=*C<;~9o$K$*$$$F(B backtrellis$BFb$K;D$C$F$$$kC18l$rF@$k(B */
  /* get survived words on backtrellis at the estimated end frame */
  num = get_backtrellis_words(bt, winfo, ngram, nw, hypo, hypo->estimated_next_t, hypo->bestt);

  if (debug2_flag) j_printf("%d",num);

  /* $BE83+$G$-$J$$C18l$r%A%'%C%/$7$F30$9(B */
  /* exclude unallowed words */
  num2 = limit_nw(nw, hypo, num);

  if (debug2_flag) j_printf("-%d=%d unfolded\n",num-num2,num2);

  return(num2);
}

/* $B2>@b$,J8$H$7$F<uM}2DG=$G$"$k$+$I$&$+$rJV$9(B */
/* return if the hypothesis is "acceptable" */
boolean
ngram_acceptable(NODE *hypo, WORD_INFO *winfo)
{
  if (
#ifdef SP_BREAK_CURRENT_FRAME
      /* $B:G8e$N2>@b$,Bh#1%Q%9:GL`2>@b$N:G=i$NC18l$H0lCW$7$J$1$l$P$J$i$J$$(B */
      /* the last word should be equal to the first word on the best hypothesis on 1st pass */
      hypo->seq[hypo->seqnum-1] == sp_break_2_end_word
#else
      /* $B:G8e$N2>@b$,J8F,L52;C18l$G$J$1$l$P$J$i$J$$(B */
      /* the last word should be head silence word */
      hypo->seq[hypo->seqnum-1] == winfo->head_silwid
#endif
      ) {
    return TRUE;
  } else {
    return FALSE;
  }
}

#endif /* USE_NGRAM */ /* ------------------------------------------ */
