/* Copyright (c) 1991-2002 Doshita Lab. Speech Group, Kyoto University */
/* Copyright (c) 2000-2002 Speech and Acoustics Processing Lab., NAIST */
/*   All rights reserved   */

/* beam.c --- frame-synchronous beam search on 1st pass */
/* use (1-best|word-pair) with static tree structure */

/* $Id: beam.c,v 1.9 2002/09/11 22:02:33 ri Exp $ */

#include <julius.h>

#undef DEBUG

static boolean idc_on;
static int progout_interval_frame;


/* -------------------------------------------------------------------- */
/*                     $BBh#1%Q%9$N7k2L=PNO$H=*N;=hM}(B                     */
/*              result output and end procedure of 1st pass             */
/* -------------------------------------------------------------------- */

#ifdef WORD_GRAPH
/* $BBh#1%Q%97k2L$+$iC18l%0%i%U$r@8@.(B
   $B<B:]$K$OC18l%H%l%j%9>e$G%0%i%U>e$K;D$k$b$N$K%^!<%/$rIU$1(B,
   ($BBh(B2$B%Q%9$G$O%^!<%/$N$D$$$?C18l$N$_E83+$9$k(B) */
/* generate word graphs from result of 1st pass (=backtrellis) */

static int glevel;		/* depth of generate_lattice() call */
static int gnodes;		/* node num */
static int garcs;		/* arc (=word) num */

/* lookup word on trellis and traverse backward (recursive)
   to mark which trellis words are on the graph */
static void
generate_lattice(
     int endtime,		/* word end frame to lookup */
     WORD_ID wid,		/* word ID to lookup */
     BACKTRELLIS *bt,		/* backtrellis structure */
     WORD_INFO winfo		/* used dictionary (for word output) */
     )
{
  TRELLIS_ATOM *ta, *ta_last;
  int i,j;
  boolean new_node = FALSE;

  glevel++;

  if (endtime >= 0) {
    for (i=0;i<bt->num[endtime];i++) {
      ta = bt->rw[endtime][i];
      if (ta->wid == wid) {
	if (ta->within_wordgraph) {
	  /* already marked */
	} else {
	  /* mark  */
	  ta->within_wordgraph = TRUE;
	  garcs++;
	  new_node = TRUE;	/* new node mark */
	  ta_last = ta->last_tre;
	  if (debug2_flag) {
	    for(j=0;j<glevel;j++) j_printf(" ");
	    j_printf("%s: %d->", winfo->wname[ta->wid], ta->endtime);
	    if (ta_last->wid == WORD_INVALID) {
	      j_printf("%d(WORD_INVALID)\n", ta_last->endtime);
	    } else {
	      j_printf("%d(%s)\n", ta_last->endtime, winfo->wname[ta_last->wid]);
	    }
	  }
	  /* recursive call */
	  generate_lattice(ta_last->endtime, ta_last->wid, bt, winfo);
	}
      }
    }
  }
  if (new_node) {
    gnodes++;
  }
  glevel--;
}
#endif

/* $B%H%l%j%9C18l>pJs$r=PNO(B ($B%G%P%C%0MQ(B) */
/* output hypothesis word (= trellis atom) info (for debug) */
static void
put_atom(TRELLIS_ATOM *atom, WORD_INFO *winfo)
{
  int i;
  j_printf("%3d,%3d %f %16s (id=%5d)", atom->begintime, atom->endtime,
	 atom->backscore, winfo->wname[atom->wid], atom->wid);
  for (i=0;i<winfo->wlen[atom->wid]; i++) {
    j_printf(" %s",winfo->wseq[atom->wid][i]->name);
  }
  j_printf("\n");
}

/* $BM?$($i$l$?%H%l%j%9C18l$+$i;OC<$K8~$+$C$FC18l%H%l%j%9=89g$r(B
   $B%H%l!<%9%P%C%/$7(B, $B$=$3$+$i$N:GL`C18l7ONs8uJd$rJV$9(B */
/* backtrace the word trellis (=result of 1st pass) from specified
   trellis atom, and return the best sequence
 */

static LOGPROB			/* return value (total language score) */
trace_backptr(WORD_ID wordseq_rt[MAXSEQNUM], /* word sequence (return) */
	      int *rt_wordlen,	/* length of above (return) */
	      TRELLIS_ATOM *atom, /* start backtrace from this atom */
	      BACKTRELLIS *backtrellis,	/* whole backward trellis */
	      WORD_INFO *winfo)	/* used word info (for word output) */
{
  int wordlen = 0;		/* word length of best sentence hypothesis */
  TRELLIS_ATOM *tretmp;
  LOGPROB langscore = 0.0;
  static wordseq[MAXSEQNUM];	/* temporal: in reverse order */
  int i;
  
  /* initialize */
  wordseq[0] = atom->wid;	/* start from specified atom */
  wordlen = 1;
  tretmp = atom;
#ifdef USE_NGRAM
  langscore += tretmp->lscore;
#endif
  if (debug2_flag) {
    put_atom(tretmp, winfo);
  }
  
  /* trace the backtrellis */
  while (tretmp->begintime > 0) {/* until beginning of input */
    tretmp = tretmp->last_tre;
/*    t = tretmp->boundtime - 1;
    tretmp = bt_binsearch_atom(backtrellis, tretmp->boundtime-1, tretmp->last_wid);*/
    if (tretmp == NULL) {	/* should not happen */
      j_error("ERROR: BackTrellis Pass missing??\n");
    }
#ifdef USE_NGRAM
    langscore += tretmp->lscore;
#endif
    wordseq[wordlen] = tretmp->wid;
    wordlen++;
    if (debug2_flag) {
      put_atom(tretmp, winfo);
    }
    if (wordlen >= MAXSEQNUM) {
      j_error("sentence length exceeded ( > %d)\n",MAXSEQNUM);
    }
  }
  *rt_wordlen = wordlen;
  /* reverse order -> normal order */
  for(i=0;i<wordlen;i++) wordseq_rt[i] = wordseq[wordlen-i-1];
  return(langscore);
}

/* $BBh#1%Q%9=*N;(B: $B7k2L$+$i:GL`2>@b$N%Q%9$r7hDj(B, $BI=<($7(B, pass1_wseq[] $B$K3JG<(B */
/* $BJV$jCM(B: $B:GL`2>@b$N%9%3%"(B */
/* $BI=<($9$k4X?t(B result_pass1_*() $B$O(B result_*.c $B$GDj5A(B */

/* end of the 1st pass: determine best hypothesis from resulting trellis,
   output it, and store it to global val 'pass1_wseq[]'.
   real text output functions result_pass1_*() is in result_*.c */
/* return value: score of the hypothesis */
static LOGPROB
print_1pass_result(
     BACKTRELLIS *backtrellis,	/* word trellis */
     int framelen,		/* where 1st pass search ends */
     WORD_INFO *winfo)		/* word info (for word output) */
{
  WORD_ID wordseq[MAXSEQNUM];
  int wordlen;
  int i;
  TRELLIS_ATOM *best;
  int last_time;
  LOGPROB total_lscore;
  LOGPROB maxscore;
  TRELLIS_ATOM *tmp;

  /* look for the first trellis word */
  for (last_time = framelen - 1; last_time >= 0; last_time--) {
#ifdef USE_NGRAM
#ifdef SP_BREAK_CURRENT_FRAME
    /* $B:G=*%U%l!<%`$K;D$C$?:GBg%9%3%"$NC18l(B */
    /* it should be the best trellis word on the last frame */
    maxscore = LOG_ZERO;
    for (i=0;i<backtrellis->num[last_time];i++) {
      tmp = backtrellis->rw[last_time][i];
      if (maxscore < tmp->backscore) {
	maxscore = tmp->backscore;
	best = tmp;
      }
    }
    if (maxscore != LOG_ZERO) break;
#else
    /* $B:G=*C18l$O(B winfo->tail_silwid $B$K8GDj(B */
    /* it is fixed to the tail silence model (winfo->tail_silwid) */
    best = bt_binsearch_atom(backtrellis, last_time, winfo->tail_silwid);
    if (best != NULL) break;
#endif
#else  /* USE_DFA */
    /* $BKvHx$K;D$C$?C18l$NCf$G:GBg%9%3%"$NC18l(B(cp_end$B$O;HMQ$7$J$$(B) */
    /* the best trellis word on the last frame (not use cp_end[]) */
    maxscore = LOG_ZERO;
    for (i=0;i<backtrellis->num[last_time];i++) {
      tmp = backtrellis->rw[last_time][i];
      /*      if (dfa->cp_end[winfo->wton[tmp->wid]] == TRUE) {*/
	if (maxscore < tmp->backscore) {
	  maxscore = tmp->backscore;
	  best = tmp;
	}
	/*      }*/
    }
    if (maxscore != LOG_ZERO) break;
#endif /* USE_NGRAM */
  }
  if (last_time < 0) {		/* not found */
    j_printerr("[no sentence-end word survived on last beam]\n");
    /* print null result */
    result_pass1_final(NULL, 0, LOG_ZERO, LOG_ZERO, winfo);
    return(LOG_ZERO);
  }
  
  /* traceback word trellis from the best word */
  total_lscore = trace_backptr(wordseq, &wordlen, best, backtrellis, winfo);

  if (progout_flag) {		/* just flush last progress output */
    result_pass1_current(last_time, wordseq, wordlen, best->backscore, total_lscore, winfo);
  }

  /* output 1st pass result */    
  if (verbose_flag || !progout_flag) {
    result_pass1_final(wordseq, wordlen, best->backscore, total_lscore, winfo);
  }
  result_pass1_end();
  
  /* store the result to global val (notice: in reverse order) */
  for(i=0;i<wordlen;i++) pass1_wseq[i] = wordseq[i];
  pass1_wnum = wordlen;
  pass1_score = best->backscore;

#ifdef WORD_GRAPH
  /* $BC18l%H%l%j%9$+$i!$%i%F%#%9$r@8@.$9$k(B */
  /* $B<B:]$K$O(B within_wordgraph $B$r(B on $B$K$9$k$@$1(B */
  /* generate word graph from the word trellis */
  /* actually generate_lattice() does not construct graph:
     it only marks words in trellis that are on the word graph */
  glevel = 0;
  gnodes = 1;			/* frame = 0 $B$NJ,(B */
  garcs = 0;
  generate_lattice(last_time, winfo->tail_silwid, backtrellis, winfo);
  if (verbose_flag) j_printf("word graph generated (nodes=%d,arcs=%d)\n",gnodes, garcs);
#endif

  /* return maximum score */
  return(best->backscore);
}

/* $B;~4V(B t $B$G$N%Y%9%H%P%9$rI=<((B */
/* output best part-of-sentence hypothesis on time t */
static void
bt_current_max(BACKTRELLIS *bt, int t, WORD_INFO *winfo)
{
  static WORD_ID wordseq[MAXSEQNUM];
  int wordlen;
  TRELLIS_ATOM *tre;
  TRELLIS_ATOM *tremax;
  LOGPROB maxscore;
  LOGPROB lscore;

  /* bt->list is ordered by time frame */
  maxscore = LOG_ZERO;
  tremax = NULL;
  tre = bt->list;
  while (tre != NULL && tre->endtime == t) {
    if (maxscore < tre->backscore) {
      maxscore = tre->backscore;
      tremax = tre;
    }
    tre = tre->next;
  }

  if (maxscore != LOG_ZERO) {
    lscore = trace_backptr(wordseq, &wordlen, tremax, bt, winfo);
    result_pass1_current(t, wordseq, wordlen, tremax->backscore, lscore, winfo);
  } else {
    wordlen = 0;
    result_pass1_current(t, wordseq, wordlen, LOG_ZERO, LOG_ZERO, winfo);
  }
}

/* for debug: it only  */
static void
bt_current_max_word(BACKTRELLIS *bt, int t, WORD_INFO *winfo)
{
  TRELLIS_ATOM *tre;
  TRELLIS_ATOM *tremax;
  LOGPROB maxscore;
  WORD_ID w;

  /* bt->list $B$O;~4V=g$K3JG<$5$l$F$$$k(B */
  maxscore = LOG_ZERO;
  tremax = NULL;
  tre = bt->list;
  while (tre != NULL && tre->endtime == t) {
    if (maxscore < tre->backscore) {
      maxscore = tre->backscore;
      tremax = tre;
    }
    tre = tre->next;
  }

  if (maxscore != LOG_ZERO) {
    j_printf("%3d: ",t);
    w = tremax->wid;
    j_printf("\"%s [%s]\"(id=%d)",
	   winfo->wname[w], winfo->woutput[w], w);
    j_printf(" [%d-%d] %f <- ", tremax->begintime, t, tremax->backscore);
    w = tremax->last_tre->wid;
    if (w != WORD_INVALID) {
      j_printf("\"%s [%s]\"(id=%d)\n",
	     winfo->wname[w], winfo->woutput[w], w);
    } else {
      j_printf("bgn\n");
    }
  }
}

/* -------------------------------------------------------------------- */
/*                 $B%S!<%`C5:wCf$N%H!<%/%s$r07$&%5%V4X?t(B                 */
/*                functions to handle hypothesis tokens                  */
/* -------------------------------------------------------------------- */

/* Token structure:
   
/* How tokens are managed:
   o  tlist[][] is a token stocker.  It holds all tokens in sequencial
      buffer.  They are malloced first on startup, and refered by ID while
      Viterbi procedure.  In word-pair mode, each token also has a link to
      another token to allow a node to have more than 1 token.
      
   o  token[n] holds the current ID number of a token associated to a
      lexicon tree node 'n'.

 */

static TOKEN2 *tlist[2];	/* sequencial list of all tokens */
static TOKENID *tindex[2];	/* index for sort */
static int tnum[2];		/* current number of tokens used in tlist */
static int maxtnum;		/* maximum number of tokens */
static int expand_step;		/* step to expand token */
static TOKENID *token;		/* sequencial list per tree node */
static int totalnodenum;	/* allocated node num of above */
static int tl,tn;		/* node id flip/flop */
static int n_start, n_end;	/* (for heap sort) */

static TRELLIS_ATOM bos;

/* malloc work area */
static void
malloc_nodes(int n)
{
  totalnodenum = n;
  token        = mymalloc(sizeof(TOKENID)*totalnodenum);
  maxtnum = n;
  tlist[0]     = mymalloc(sizeof(TOKEN2)*maxtnum);
  tlist[1]     = mymalloc(sizeof(TOKEN2)*maxtnum);
  tindex[0]     = mymalloc(sizeof(TOKENID)*maxtnum);
  tindex[1]     = mymalloc(sizeof(TOKENID)*maxtnum);
  tnum[0] = tnum[1] = 0;
  expand_step = n;
}
/* expand work area */
static void
expand_tlist()
{
  maxtnum += expand_step;
  tlist[0]     = myrealloc(tlist[0],sizeof(TOKEN2)*maxtnum);
  tlist[1]     = myrealloc(tlist[1],sizeof(TOKEN2)*maxtnum);
  tindex[0]     = myrealloc(tindex[0],sizeof(TOKENID)*maxtnum);
  tindex[1]     = myrealloc(tindex[1],sizeof(TOKENID)*maxtnum);
  j_printerr("warn: token space expanded\n");
}
/* free work area */
static void
free_nodes()
{
  free(token);
  free(tlist[0]);
  free(tlist[1]);
  free(tindex[0]);
  free(tindex[1]);
}

/* initialize token buffer */
static void
clear_tlist(int tt)
{
  tnum[tt] = 0;
}
/* initialize token buffer: only clear ones used in the last call */
static void
clear_tokens(int tt)
{
  int j;
  for (j=0; j<tnum[tt]; j++) {
    token[tlist[tt][j].node] = TOKENID_UNDEFINED;
  }
}
/* return a new token ID */
static TOKENID
create_token()
{
  TOKENID newid;
  newid = tnum[tn];
  tindex[tn][newid] = newid;
  tnum[tn]++;
  if (tnum[tn]>maxtnum) expand_tlist();
#ifdef WPAIR
  tlist[tn][newid].next = TOKENID_UNDEFINED;
#endif
  return(newid);
}

/* assign token to a node */
static void
node_assign_token(int node, TOKENID tkid)
{
#ifdef WPAIR
  /* add to list */
  tlist[tn][tkid].next = token[node];
#endif
  token[node] = tkid;
  tlist[tn][tkid].node = node;
}

/* check if token with last word='wid' exist on 'node' */
/* return the to-be-replaced token ID (TOKENID_UNDEFINED if not found) */
static TOKENID
node_exist_token(int tt, int node, WORD_ID wid)
{
#ifdef WPAIR
  /* In word-pair mode, multiple tokens are assigned to a node as a list.
     so we have to search for tokens with same last word ID */
#ifdef WPAIR_KEEP_NLIMIT
  /* 1$B%N!<%I$4$H$KJ];}$9$k(Btoken$B?t$N>e8B$r@_Dj(B */
  /* token$B$,L5$$$,>e8B$KC#$7$F$$$k$H$-$O0lHV%9%3%"$NDc$$(Btoken$B$r>e=q$-$9$k(B */
  /* N-best: limit number of assigned tokens to a node */
  int i = 0;
  TOKENID lowest_token = TOKENID_UNDEFINED;
#endif
  TOKENID tmp;
  for(tmp=token[node]; tmp != TOKENID_UNDEFINED; tmp=tlist[tt][tmp].next) {
    if (tlist[tt][tmp].last_tre->wid == wid) {
      return(tmp);
    }
#ifdef WPAIR_KEEP_NLIMIT
    if (lowest_token == TOKENID_UNDEFINED ||
	tlist[tt][lowest_token].score < tlist[tt][tmp].score)
      lowest_token = tmp;
    if (++i >= wpair_keep_nlimit) break;
#endif
  }
#ifdef WPAIR_KEEP_NLIMIT
  if (i >= wpair_keep_nlimit) { /* overflow, overwrite lowest score */
    return(lowest_token);
  } else {
    return(TOKENID_UNDEFINED);
  }
#else 
  return(TOKENID_UNDEFINED);
#endif
  
#else  /* not WPAIR */
  /* 1$B$D$@$1J];}(B,$B$3$l$r>o$K>e=q$-(B */
  /* Only one token is kept in 1-best mode (default), so
     simply return the ID */
  return(token[node]);
#endif
}

#ifdef DEBUG
/* tlist $B$H(B token $B$NBP1~$r%A%'%C%/$9$k(B(debug) */
/* for debug: check tlist <-> token correspondence
   where  tlist[tt][tokenID].node = nodeID and
          token[nodeID] = tokenID
 */
static void
node_check_token(int tt)
{
  int i;
  for(i=0;i<tnum[tt];i++) {
    if (node_exist_token(tt, tlist[tt][i].node, tlist[tt][i].last_tre->wid) != i) {
      j_printerr("token %d not found on node %d\n", i, tlist[tt][i].node);
    }
  }
}
#endif



/* -------------------------------------------------------------------- */
/*       $B%H!<%/%s$r%=!<%H$7(B $B>e0L(B N $B%H!<%/%s$rH=JL$9$k(B (heap sort)       */
/*        Sort generated tokens and get N-best (use heap sort)          */
/* -------------------------------------------------------------------- */
/* $B%S!<%`$NogCM$H$7$F>e0L(B N $BHVL\$N%9%3%"$,M_$7$$$@$1$G$"$j!$<B:]$K%=!<%H(B
   $B$5$l$kI,MW$O$J$$(B */
/* we only want to know the N-th score for determining beam threshold, so
   order is not considered here */

#define SD(A) tindex[tn][A-1]
#define SCOPY(D,S) D = S
#define SVAL(A) (tlist[tn][tindex[tn][A-1]].score)
#define STVAL (tlist[tn][s].score)

/* sort tokens upward to find 'neednum'-best score */
/* $BBg$-$$=g$K%=!<%H$7$F>e0L(B neednum $B8D$,8+$D$+$C$?$i=*$o$j(B */
static void
sort_token_upward(int neednum, int totalnum)
{
  int n,root,child,parent;
  TOKENID s;
  for (root = totalnum/2; root >= 1; root--) {
    SCOPY(s, SD(root));
    parent = root;
    while ((child = parent * 2) <= totalnum) {
      if (child < totalnum && SVAL(child) < SVAL(child+1)) {
	child++;
      }
      if (STVAL >= SVAL(child)) {
	break;
      }
      SCOPY(SD(parent), SD(child));
      parent = child;
    }
    SCOPY(SD(parent), s);
  }
  n = totalnum;
  while ( n > totalnum - neednum) {
    SCOPY(s, SD(n));
    SCOPY(SD(n), SD(1));
    n--;
    parent = 1;
    while ((child = parent * 2) <= n) {
      if (child < n && SVAL(child) < SVAL(child+1)) {
	child++;
      }
      if (STVAL >= SVAL(child)) {
	break;
      }
      SCOPY(SD(parent), SD(child));
      parent = child;
    }
    SCOPY(SD(parent), s);
  }
}

/* sort tokens downward to find 'neednum'-lowest score */
/* $B>.$5$$=g$K%=!<%H$7$F2<0L(B neednum $B8D$,8+$D$+$C$?$i=*$o$j(B */
static void
sort_token_downward(int neednum, int totalnum)
{
  int n,root,child,parent;
  TOKENID s;
  for (root = totalnum/2; root >= 1; root--) {
    SCOPY(s, SD(root));
    parent = root;
    while ((child = parent * 2) <= totalnum) {
      if (child < totalnum && SVAL(child) > SVAL(child+1)) {
	child++;
      }
      if (STVAL <= SVAL(child)) {
	break;
      }
      SCOPY(SD(parent), SD(child));
      parent = child;
    }
    SCOPY(SD(parent), s);
  }
  n = totalnum;
  while ( n > totalnum - neednum) {
    SCOPY(s, SD(n));
    SCOPY(SD(n), SD(1));
    n--;
    parent = 1;
    while ((child = parent * 2) <= n) {
      if (child < n && SVAL(child) > SVAL(child+1)) {
	child++;
      }
      if (STVAL <= SVAL(child)) {
	break;
      }
      SCOPY(SD(parent), SD(child));
      parent = child;
    }
    SCOPY(SD(parent), s);
  }
}

/* $B%=!<%H$9$k!J>e8~$-!&2<8~$-$GAa$$J}$r;H$&!K(B */
/* sort tokens up to need num */
/* sort upward till need num, or sort downward tille (totalnum-neednum) */
/* start, end: return value of range where [0..neednum-1] tokens exist */
static void
sort_token_no_order(int neednum, int *start, int *end)
{
  int totalnum = tnum[tn];
  int restnum;

  restnum = totalnum - neednum;

  if (neednum >= totalnum) {
    /* no need to sort */
    *start = 0;
    *end = totalnum - 1;
  } else if (neednum < restnum)  {
    sort_token_upward(neednum,totalnum);
    *start = totalnum - neednum;
    *end = totalnum - 1;
  } else {
    sort_token_downward(restnum,totalnum);
    *start = 0;
    *end = neednum - 1;
  }
}
    

#ifdef SP_BREAK_CURRENT_FRAME
/* -------------------------------------------------------------------- */
/*                     $B%7%g!<%H%]!<%:!&%;%0%a%s%F!<%7%g%s(B               */
/*                     short pause segmentation                         */
/* -------------------------------------------------------------------- */
/* ====================================================================== */
/* sp segmentation */
/*
  |---************-----*********************-------*******--|
[input segments]
  |-------------------|
                  |-------------------------------|
		                            |---------------|

		     
  |-------------------|t-2
                       |t-1 ... token processed (= lastlen)
		        |t  ... outprob computed
		       
*/

static boolean in_sparea;	/* TRUE if we are within a pause area */
static int sparea_start;	/* start frame of pause area */
static int tmp_sparea_start;
#ifdef SP_BREAK_RESUME_WORD_BEGIN
static WORD_ID tmp_sp_break_last_word;
#else
static WORD_ID last_tre_word;
#endif
static boolean first_sparea;	/* TRUE if this is the first pause segment in a input stream */
static int sp_duration;		/* number of sp frame to be durated */

/* $B%7%g!<%H%]!<%:$NO"B3$r8+$F(B, $BF~NO$N@ZL\$r8!=P$9$k(B */
/* $BBh#1%Q%9$,#1%U%l!<%`?J$`$4$H$K8F$P$l$k(B */
/* watch duration of short-pause words and detect end of input segment */
/* called for every frame in 1st pass */
/* return value: TRUE if segment end here, FALSE if continue */
static boolean
detect_end_of_segment(BACKTRELLIS *backtrellis, int time, WORD_INFO *winfo)
{
  TRELLIS_ATOM *tre;
  LOGPROB maxscore = LOG_ZERO;
  TRELLIS_ATOM *tremax = NULL;
  int count = 0;
  char *p;
  boolean detected = FALSE;

  /* look for the best trellis word on the given time frame */
  for(tre = backtrellis->list; tre != NULL && tre->endtime == time; tre = tre->next) {
    if (maxscore < tre->backscore) {
      maxscore = tre->backscore;
      tremax = tre;
    }
    count++;
  }
  if (tremax == NULL) {	/* no word end: possible in the very beggining of input*/
    detected = TRUE;		/* assume it's in the short-pause duration */
  } else if (count > 0) {	/* many words found --- check if maximum is sp */
    p = winfo->wseq[tremax->wid][0]->name;
    if (is_sil(p)) {
      detected = TRUE;
    }
  }
  
  /* sp$B6h4V;}B3%A%'%C%/(B */
  /* check sp segment duration */
  if (in_sparea && detected) {	/* we are already in sp segment and sp continues */
    sp_duration++;		/* increment count */
#ifdef SP_BREAK_RESUME_WORD_BEGIN
    /* resume word at the "beggining" of sp segment */
    /* if this segment has triggered by (tremax == NULL) (in case the first
       several frame of input), the sp word (to be used as resuming
       word in the next segment) is not yet set. it will be detected here */
    if (tmp_sp_break_last_word == WORD_INVALID) {
      if (tremax != NULL) tmp_sp_break_last_word = tremax->wid;
    }
#else
    /* resume word at the "end" of sp segment */
    /* simply update the best sp word */
    if (tremax != NULL) last_tre_word = tremax->wid;
#endif
  }

  /* sp$B6h4V3+;O%A%'%C%/(B */
  /* check if sp segment begins at this frame */
  else if (!in_sparea && detected) {
    /* $B0l;~E*$K3+;O%U%l!<%`$H$7$F%^!<%/(B */
    /* mark this frame as "temporal" begging of short-pause segment */
    tmp_sparea_start = time;
#ifdef SP_BREAK_RESUME_WORD_BEGIN
    /* sp $B6h4V3+;O;~E@$N:GL`C18l$rJ]B8(B */
    /* store the best word in this frame as resuming word */
    tmp_sp_break_last_word = tremax ? tremax->wid : WORD_INVALID;
#endif
    in_sparea = TRUE;		/* yes, we are in sp segment */
    sp_duration = 1;		/* initialize duration count */
#ifdef SP_BREAK_DEBUG
    printf("sp start %d\n", time);
#endif /* SP_BREAK_DEBUG */
  }
  
  /* sp $B6h4V=*N;%A%'%C%/(B */
  /* check if sp segment ends at this frame */
  else if (in_sparea && !detected) {
    /* (time-1) is end frame of pause segment */
    in_sparea = FALSE;		/* we are not in sp segment */
#ifdef SP_BREAK_DEBUG
    printf("sp end %d\n", time);
#endif /* SP_BREAK_DEBUG */
    /* sp $B6h4VD9%A%'%C%/(B */
    /* check length of the duration*/
    if (sp_duration < sp_frame_duration) {
      /* $BC;$9$.$k(B: $BBh#1%Q%9$rCfCG$;$:B39T(B */
      /* too short segment: not break, continue 1st pass */
#ifdef SP_BREAK_DEBUG
      printf("too short (%d<%d), ignored\n", sp_duration, sp_frame_duration);
#endif /* SP_BREAK_DEBUG */
    } else if (first_sparea) {
      /* $B:G=i$N(Bsp$B6h4V$O(B silB $B$K$"$?$k$N$G(B,$BBh#1%Q%9$rCfCG$;$:B39T(B */
      /* do not break at first sp segment: they are silB */
      first_sparea = FALSE;
#ifdef SP_BREAK_DEBUG
      printf("first silence, ignored\n");
#endif /* SP_BREAK_DEBUG */
    } else {
      /* $B6h4V=*N;3NDj(B, $BBh#1%Q%9$rCfCG$7$FBh(B2$B%Q%9$X(B */
      /* break 1st pass */
#ifdef SP_BREAK_DEBUG
      printf(">> segment [%d..%d]\n", sparea_start, time-1);
#else
      j_printerr("|");
#endif /* SP_BREAK_DEBUG */
      /* store begging frame of the segment */
      sparea_start = tmp_sparea_start;
#ifdef SP_BREAK_RESUME_WORD_BEGIN
      /* resume word = most likely sp word on beginning frame of the segment */
      sp_break_last_word = tmp_sp_break_last_word;
#else
      /* resume word = most likely sp word on end frame of the segment */
      sp_break_last_word = last_tre_word;
#endif

      /*** segment: [sparea_start - time-1] ***/
      return(TRUE);
    }
  }
    
#ifdef SP_BREAK_EVAL
  printf("[%d %d %d]\n", time, count, (detected) ? 50 : 0);
#endif
  return (FALSE);
}

#endif /* SP_BREAK_CURRENT_FRAME */



/* -------------------------------------------------------------------- */
/*             $BBh#1%Q%9(B($B%U%l!<%`F14|%S!<%`%5!<%A(B) $B%a%$%s(B                */
/*           main routines of 1st pass (frame-synchronous beam search)  */
/* -------------------------------------------------------------------- */

/* $B=i4|2=(B */
/* initialization */
static void
init_nodescore(HTK_Param *param, WCHMM_INFO *wchmm)
{
  TOKENID newid;
  TOKEN2 *new;
  WORD_ID beginword;
  int node;
#ifdef USE_DFA
  int i;
#endif

  /* $B=i4|2>@bMQC18lMzNr(B */
  /* setup initial word context */
#ifdef SP_BREAK_CURRENT_FRAME
  /* initial word context = last non-sp word of previous 2nd pass at last segment*/
  if (sp_break_last_nword == wchmm->winfo->tail_silwid) {
    /* if end with silE, initialize as normal start of sentence */
    bos.wid = WORD_INVALID;
  } else {
    bos.wid = sp_break_last_nword;
  }
#else
  bos.wid = WORD_INVALID;	/* no context */
#endif
  bos.begintime = bos.endtime = -1;

  /* $B%N!<%I!&%H!<%/%s$r=i4|2=(B */
  /* clear tree lexicon nodes and tokens */
  for(node=0;node<totalnodenum;node++) {
    token[node] = TOKENID_UNDEFINED;
  }
  tnum[0] = tnum[1]  = 0;
  
#ifdef PASS1_IWCD
  /* $B=PNO3NN(7W;;%-%c%C%7%e$r=i4|2=(B */
  /* initialize outprob cache */
  outprob_style_cache_init(wchmm);
#endif

  /* $B=i4|2>@b$N:n@.(B: $B=i4|C18l$N7hDj$H=i4|%H!<%/%s$N@8@.(B */
  /* initial word hypothesis */
#ifdef USE_NGRAM
  
#ifdef SP_BREAK_CURRENT_FRAME
  if (sp_break_last_word != WORD_INVALID) { /* last segment exist */
    /* $B3+;OC18l!aA0$N%;%0%a%s%H7W;;;~$N:G8e$N:GL`C18l(B */
    /* $BJ8=*N;C18l(B(silE,$B6gE@(B(IPA$B%b%G%k(B))$B$J$i!$(BsilB $B$G3+;O(B */
    /* initial word = best last word hypothesis on the last segment */
    /* if silE or sp, begin with silB */
    if (sp_break_last_word == wchmm->winfo->tail_silwid ||
	sp_break_last_word == 4) { /* sp: super ad-hoc, need fix... */
      beginword = wchmm->winfo->head_silwid;
      bos.wid = WORD_INVALID;	/* reset initial word context */
    } else {
      beginword = sp_break_last_word;
    }
  } else {
    /* initial word fixed to silB */
    beginword = wchmm->winfo->head_silwid;
  }
#else
  /* initial word fixed to silB */
  beginword = wchmm->winfo->head_silwid;
#endif
#ifdef SP_BREAK_DEBUG
  printf("startword=[%s], last_nword=[%s]\n",
	 (beginword == WORD_INVALID) ? "WORD_INVALID" : wchmm->winfo->wname[beginword],
	 (bos.wid == WORD_INVALID) ? "WORD_INVALID" : wchmm->winfo->wname[bos.wid]);
#endif
  /* create the first token at the first node of initial word */
  newid = create_token();
  new = &(tlist[tn][newid]);
  node = wchmm->offset[beginword][0];
  if (wchmm->state[node].sc != NULL
#ifdef UNIGRAM_FACTORING
      || wchmm->state[node].fscore != LOG_ZERO
#endif
      ) {
    new->last_lscore = max_successor_prob(wchmm, bos.wid, node);
  } else {
    new->last_lscore = 0.0;
  }
#ifdef FIX_PENALTY
  new->last_lscore = new->last_lscore * lm_weight;
#else
  new->last_lscore = new->last_lscore * lm_weight + lm_penalty;
#endif
  new->last_tre = &bos;
  new->last_cword = bos.wid;
  new->score = outprob_style(wchmm, node, bos.wid, 0, param) + new->last_lscore;
  node_assign_token(node, newid);
  
#else  /* USE_DFA */
  
  /* $B=i4|2>@b(B: $BJ8K!>eJ8F,$K@\B3$7$&$kC18l=89g(B */
  /* initial words: all words that can be begin of sentence grammatically */
  /* $B%"%/%F%#%V$JJ8K!$KB0$9$kC18l$N$_5v$9(B */
  /* only words in active grammars are allowed to be an initial words */
  {
    MULTIGRAM *m;
    int t,tb,te;
    WORD_ID iw;

    for(m = gramlist; m; m = m->next) {
      if (m->active) {
	tb = m->cate_begin;
	te = tb + m->dfa->term_num;
	for(t=tb;t<te;t++) {
	  if (dfa_cp_begin(dfa, t) == TRUE) {
	    for (iw=0;iw<dfa->term.wnum[t];iw++) {
	      i = dfa->term.tw[t][iw];
	      /* create the first token at the first node of the beggning word */
	      node = wchmm->offset[i][0];
	      /* words in the same category share the same root node, so
		 skip if already exist */
	      if (node_exist_token(tn, node, bos.wid) != TOKENID_UNDEFINED) continue;
	      newid = create_token();
	      new = &(tlist[tn][newid]);
	      new->last_tre = &bos;
	      new->score = outprob_style(wchmm, node, bos.wid, 0, param);
	      node_assign_token(node, newid);
	    }
	  }
	}
      }
    }
  }
/* 
 *   for (i=0;i<wchmm->winfo->num;i++) {
 *     if (dfa->cp_begin[wchmm->winfo->wton[i]] == TRUE) {
 *	 node = wchmm->offset[i][0];
 *	 if (node_exist_token(tn, node, bos.wid) != TOKENID_UNDEFINED) continue;
 *	 newid = create_token();
 *	 new = &(tlist[tn][newid]);
 *	 new->last_tre = &bos;
 *	 new->score = outprob_style(wchmm, node, bos.wid, 0, param);
 *	 node_assign_token(node, newid);
 *     }
 *   }
 */
  
#endif /* USE_DFA */
}

/******************************************************/
/* $B%U%l!<%`F14|%S!<%`C5:w$N<B9T(B --- $B:G=i$N%U%l!<%`MQ(B  */
/* frame synchronous beam search --- first frame only */
/******************************************************/
void
get_back_trellis_init(HTK_Param *param,	/* parameter vectors of speech data (only t=0 used in this function) */
		      WCHMM_INFO *wchmm, /* lexicon tree */
		      BACKTRELLIS *backtrellis	/* backward trellis (return value) */
		      )
{
  
  /* Viterbi$B1i;;MQ%o!<%/%(%j%"$N%9%$%C%A%c!<(B tl,tn $B$N=i4|CM@_Dj(B */
  /* tn: $B$3$N%U%l!<%`MQ(BID   tl: $B#1%U%l!<%`A0$N(BID */
  /* initialize switch tl, tn for Viterbi computation */
  /* tn: this frame  tl: last frame */
  tn = 0;
  tl = 1;

  /* $B7k2L$NC18l%H%l%j%9$r3JG<$9$k%P%C%/%H%l%j%99=B$BN$r=i4|2=(B */
  /* initialize backtrellis structure to store resulting word trellis */
  bt_prepare(backtrellis);

  /* $B%o!<%/%(%j%"$r3NJ](B */
  /* malloc work area */
  malloc_nodes(wchmm->n);
  
  /* $B=i4|%9%3%"$r(B nodescore[tn] $B$K%;%C%H(B */
  /* set initial score to nodescore[tn] */
  init_nodescore(param, wchmm);
  sort_token_no_order(trellis_beam_width, &n_start, &n_end);

  /* $B%F%-%9%H=PNO$r=i4|2=(B */
  /* initialize message output */
  result_pass1_begin();
  /* $BA2<!=PNO$r9T$J$&>l9g$N%$%s%?!<%P%k$r7W;;(B */
  /* set interval frame for progout */
  progout_interval_frame = (float)progout_interval / ((float)param->header.wshift / 10000.0);

  /* $B?J9T>u67I=<($r=i4|2=(B */
  /* progress bar setting */
  if (!realtime_flag && verbose_flag && (!progout_flag) && isatty(1)) {
    idc_on = TRUE;
  } else { 
    idc_on = FALSE;
  }
  
#ifdef SP_BREAK_CURRENT_FRAME
  /* $B%7%g!<%H%]!<%:%;%0%a%s%F!<%7%g%sMQ%Q%i%a!<%?$N=i4|2=(B */
  /* initialize parameter for short pause segmentation */
  in_sparea = TRUE;		/* assume beginning is silence */
  sparea_start = tmp_sparea_start = 0; /* set start frame to 0 */
#ifdef SP_BREAK_RESUME_WORD_BEGIN
  tmp_sp_break_last_word = WORD_INVALID;
#endif
  sp_break_last_word = WORD_INVALID;
  /* $B:G=i$N%;%0%a%s%H(B: $B<!$NHs%]!<%:%U%l!<%`$GBh(B2$B%Q%9$X0\9T$7$J$$(B */
  /* the first end of pause segment should be always silB, so
     skip the first segment */
  first_sparea = TRUE;
  sp_break_2_begin_word = WORD_INVALID;
#endif
}

/* local work area for get_back_trellis_proceed() */
static LOGPROB maxscore, minscore;
static LOGPROB tmpprob;
static TRELLIS_ATOM *tre;
static int node, stid, isoid;
static A_CELL *ac;
static LOGPROB *iwparray;
#ifdef UNIGRAM_FACTORING
/* for wordend processing with 1-gram factoring */
static LOGPROB wordend_best_score; /* best score of wordend nodes */
static int wordend_best_node;	/* node id of the best wordend nodes */
static TRELLIS_ATOM *wordend_best_tre; /* trellis word corresponds to above */
static WORD_ID wordend_best_last_cword;	/* last_cword of above */
#endif

/*****************************************************/
/* frame synchronous beam search --- proceed 1 frame */
/* $B%U%l!<%`F14|%S!<%`C5:w$N<B9T(B --- 1$B%U%l!<%`?J$a$k(B  */
/*****************************************************/
boolean				/* return value: FALSE if segmented */
get_back_trellis_proceed(
     int t,			/* frame t*/
     HTK_Param *param,		/* parameter vector of speech data (use only t) */
     WCHMM_INFO *wchmm,		/* lexicon tree */
     BACKTRELLIS *backtrellis	/* backward trellis (return value) */
     )
{
  LOGPROB tmpsum;
  int j, next_node, sword;
  TOKEN2  *tk, *tknext;
  TOKENID  tknextid;
#ifdef USE_NGRAM
  LOGPROB ngram_score_cache;
#endif

  /*********************/
  /* 1. $B=i4|2=(B         */
  /*    initialization */
  /*********************/

  /* tl $B$H(B tn $B$rF~$lBX$($F:n6HNN0h$r@Z$jBX$((B */
  /* tl (= $BD>A0$N(B tn) $B$OD>A0%U%l!<%`$N7k2L$r;}$D(B */
  /* swap tl and tn to switch work buffer */
  /* tl (= last tn) holds result of the previous frame */
  tl = tn;
  if (tn == 0) tn = 1; else tn = 0;

  /* $B?J9T>u67I=<($r99?7(B */
  /* update progress bar */
  if (idc_on) {
#ifdef SP_BREAK_CURRENT_FRAME
    if (in_sparea) j_printerr("."); else j_printerr("-");
#else  /* normal */
    j_printerr(".");
#endif /* SP_BREAK_CURRENT_FRAME */
  }

  /* $B:GBg(B/$B:G>.%9%3%"CM$r=i4|2=(B */
  /* initialize max/min score */
  maxscore = LOG_ZERO;
  minscore = 0;

#ifdef UNIGRAM_FACTORING
  /* 1-gram factoring $B$G$OC18l@hF,$G$N8@8l3NN($,0lDj$GD>A0C18l$K0MB8$7$J$$(B
     $B$?$a!$C18l4V(B Viterbi $B$K$*$$$FA*$P$l$kD>A0C18l$O(B,$B<!C18l$K$h$i$:6&DL$G$"$k!%(B
     $B$h$C$FC18l=*C<$+$i(Bfactoring$BCM$N$"$kC18l@hF,$X$NA+0\$O#1$D$K$^$H$a$i$l$k!%(B
     $B$?$@$7!$LZ$+$iFHN)$7$?C18l$K$D$$$F$O(B, $BC18l@hF,$GMzNr$K0MB8$7$?(B2-gram$B$,(B
     $BM?$($i$l$k$?$a(B, $B:GL`$NC18l4V(B Viterbi $B%Q%9$O<!C18l$4$H$K0[$J$k!%(B
     $B$h$C$F$=$l$i$K$D$$$F$O$^$H$a$:$KJL$K7W;;$9$k(B */
  /* In 1-gram factoring, the language score on the word-head node is constant
     and independent of the previous word.  So, the same word hypothesis will
     be selected as the best previous word at the inter-word Viterbi
     processing.  So, in inter-word processing, we can (1) select only
     the best word-end hypothesis, and then (2) process viterbi from the node
     to each word-head node.  On the other hand, the isolated words,
     i.e. words not sharing any node with other word, has unique word-head
     node and the true 2-gram language score is determined at the top node.
     In such case the best word hypothesis prior to each node will differ
     according to the language scores.  So we have to deal such words separately. */
  /* initialize max value to delect best word-end hypothesis */
  wordend_best_score = LOG_ZERO;
#endif

#ifdef DEBUG
  /* debug */
  /* node_check_token(tl); */
#endif

  /* $B%H!<%/%s%P%C%U%!$r=i4|2=(B: $BD>A0%U%l!<%`$G;H$o$l$?ItJ,$@$1%/%j%"$9$l$P$h$$(B */
  /* initialize token buffer: for speedup, only ones used in the last call will be cleared */
  clear_tokens(tl);

  /**************************/
  /* 2. Viterbi$B7W;;(B         */
  /*    Viterbi computation */
  /**************************/
  /* $BD>A0%U%l!<%`$+$i$3$N%U%l!<%`$X$N(B Viterbi $B7W;;$r9T$J$&(B */
  /* tindex[tl][n_start..n_end] $B$KD>A0%U%l!<%`>e0L%N!<%I$N(BID$B$,3JG<$5$l$F$$$k(B */
  /* do one viterbi computation from last frame to this frame */
  /* tindex[tl][n_start..n_end] holds IDs of survived nodes in last frame */
  for (j=n_start;j<=n_end;j++) {

    /* tk: $BBP>]%H!<%/%s(B  node: $B$=$N%H!<%/%s$r;}$DLZ9=B$2=<-=q%N!<%I(BID */
    /* tk: token data  node: lexicon tree node ID that holds the 'tk' */
    tk = &(tlist[tl][tindex[tl][j]]);
    node = tk->node;
    if (tk->score <= LOG_ZERO) continue; /* invalid node */

    /*********************************/
    /* 2.1. $BC18lFbA+0\(B               */
    /*      word-internal transition */
    /*********************************/
    for (ac = wchmm->state[node].ac; ac; ac = ac->next) {
      next_node = ac->arc;

      /******************************************************************/
      /* 2.1.1 $BA+0\@h$X$N%9%3%"7W;;(B($BA+0\3NN(!\8@8l%9%3%"(B)               */
      /*       compute score of destination node (transition prob + LM) */
      /******************************************************************/
      tmpsum = tk->score + ac->a;
#ifdef USE_NGRAM
      ngram_score_cache = LOG_ZERO;
#endif

#ifndef CATEGORY_TREE
      /* $B8@8l%9%3%"(B factoring:
         arc$B$,<+8JA+0\$G$J$$C18lFb$NA+0\$G!$$+$DA+0\@h$K(Bsuccessor$B%j%9%H(B
	 $B$,$"$l$P!$(Blexicon tree $B$NJ,4tItJ,$NA+0\$G$"$k(B */
      /* LM factoring:
	 If arc is not a self transition and destination node has successor
	 list, this is branching transition
       */
      if (next_node != node) {
	if (wchmm->state[next_node].sc != NULL
#ifdef UNIGRAM_FACTORING
	    /* 1-gram factoring $B;HMQ;~$O(B, $BJ#?t$G6&M-$5$l$k;^$G$O(B
	       wchmm->state[node].sc $B$O(B NULL $B$H$J$j!$$=$l$r6&M-$7$F$$$k(B
	       $BC18l=89g$N(B1-gram$B$N:GBgCM$,(B wchmm->state[node].fscore $B$K3JG<(B
	       $B$5$l$F$$$k!%KvC<$N;^(B($BJ#?tC18l$G6&M-$5$l$J$$(B)$B$G$O!$$=$N(B
	       $B#1C18l$r(B sc $B$H$7$F;}$D$N$G@53N$J(B2-gram$B$r7W;;(B */
	    /* When uni-gram factoring,
	       Shared branches: has maximum uni-gram probability for sub-tree
	                        in wchmm->state[node].fscore, and
				wchmm->state[node].sc set to NULL.
	       Leaf branches (with only one successor word): has
	       the word ID in wchmm->state[node].sc.  So precise 2-gram
	       is computed in this point */
	    || wchmm->state[next_node].fscore != LOG_ZERO
#endif
	    ){
#ifdef USE_NGRAM
	  /* N-gram$B3NN($+$i(Bfactoring $BCM$r7W;;(B */
	  /* compute new factoring value from N-gram probabilities */
#ifdef FIX_PENALTY
	  if (tk->last_cword == WORD_INVALID) {
	    ngram_score_cache = max_successor_prob(wchmm, tk->last_cword, next_node) * lm_weight;
	  } else {
	    ngram_score_cache = max_successor_prob(wchmm, tk->last_cword, next_node) * lm_weight + lm_penalty;
	  }
#else
	  ngram_score_cache = max_successor_prob(wchmm, tk->last_cword, next_node) * lm_weight + lm_penalty;
#endif
	  /* $B%9%3%"$N99?7(B: tk->last_lscore $B$KC18lFb$G$N:G8e$N(Bfactoring$BCM(B */
	  /* update score: tk->last_lscore holds the last factoring value in this word */
	  tmpsum -= tk->last_lscore;
	  tmpsum += ngram_score_cache;
	  
#else  /* USE_DFA --- not CATEGORY_TREE */
	  
          /* $B7hDjE*(Bfactoring: $BD>A0C18l$KBP$7$F(B,sub-tree$BFb$K%+%F%4%jBP@)Ls>e(B
	     $B@\B3$7$&$kC18l$,#1$D$b$J$1$l$P(B, $B$3$NA+0\$OIT2D(B */
	  /* deterministic factoring in grammar mode:
	     transition disabled if there are totally no sub-tree word that can
	     grammatically (in category-pair constraint) connect
	     to the previous word.
	   */
          if (!can_succeed(wchmm, tk->last_tre->wid, next_node)) {
            tmpsum = LOG_ZERO;
          }
	  
#endif /* USE_NGRAM */
	}
      }
#endif /* CATEGORY_TREE */
      /* factoring not needed when DFA mode and uses category-tree */

      /****************************************/
      /* 2.1.2 $BA+0\@h%N!<%I$X%H!<%/%sEAHB(B     */
      /*       pass token to destination node */
      /****************************************/

      if ((tknextid = node_exist_token(tn, next_node, tk->last_tre->wid)) != TOKENID_UNDEFINED) {
	/* $BA+0\@h%N!<%I$K$O4{$KB>%N!<%I$+$iEAHB:Q$_(B: $B%9%3%"$,9b$$$[$&$r;D$9(B */
	/* the destination node already has a token: compare score */
	tknext = &(tlist[tn][tknextid]);
	if (tknext->score < tmpsum) {
	  /* $B$=$NA+0\@h%N!<%I$,;}$D%H!<%/%s$NFbMF$r>e=q$-$9$k(B($B?75,%H!<%/%s$O:n$i$J$$(B) */
	  /* overwrite the content of existing destination token: not create a new token */
	  tknext->last_tre = tk->last_tre;
#ifdef USE_NGRAM
	  tknext->last_cword = tk->last_cword;
	  tknext->last_lscore = (ngram_score_cache != LOG_ZERO) ? ngram_score_cache : tk->last_lscore;
#endif /* USE_NGRAM */
	  tknext->score = tmpsum;
	}
      } else {
	/* $BA+0\@h%N!<%I$OL$EAHB(B: $B?75,%H!<%/%s$r:n$C$F3d$jIU$1$k(B */
	/* token unassigned: create new token and assign */
	if (tmpsum > LOG_ZERO) { /* valid token */
	  tknextid = create_token();
	  tknext = &(tlist[tn][tknextid]);
	  tk = &(tlist[tl][tindex[tl][j]]);
	  tknext->last_tre = tk->last_tre;
#ifdef USE_NGRAM
	  tknext->last_cword = tk->last_cword;
	  tknext->last_lscore = (ngram_score_cache != LOG_ZERO) ? ngram_score_cache : tk->last_lscore;
#endif /* USE_NGRAM */
	  tknext->score = tmpsum;
	  node_assign_token(next_node, tknextid);
	}
      }
    }
    

    /* $BA+0\85%N!<%I$,C18l=*C<$J$i$P(B */
    /* if source node is end state of a word, */
    if (wchmm->stend[node] != WORD_INVALID) {

      sword = wchmm->stend[node];

      /**************************/
      /* 2.2. $B%H%l%j%9C18lJ]B8(B  */
      /*      save trellis word */
      /**************************/

      /* $B$3$NA+0\85$NC18l=*C<%N!<%I$O!VD>A0%U%l!<%`$G!W@8$-;D$C$?%N!<%I!%(B
	 ($B!V$3$N%U%l!<%`!W$G$J$$$3$H$KCm0U!*!*(B)
	 $B$h$C$F$3$3$G(B, $B;~4V(B(t-1) $B$rC18l=*C<$H$9$k%H%l%j%9>e$NC18l2>@b(B
	 (TRELLIS_ATOM)$B$H$7$F!$C18l%H%l%j%99=B$BN$KJ]B8$9$k!%(B*/
      /* This source node (= word end node) has been survived in the
	 "last" frame (notice: not "this" frame!!).  So this word end
	 is saved here to the word trellis structure (BACKTRELLIS) as a
	 trellis word (TRELLIS_ATOM) with end frame (t-1). */
      tre = (TRELLIS_ATOM *)mymalloc(sizeof(TRELLIS_ATOM));
      tre->wid = sword;		/* word ID */
      tre->backscore = tk->score; /* log score (AM + LM) */
      tre->begintime = tk->last_tre->endtime + 1; /* word beginning frame */
      tre->endtime   = t-1;	/* word end frame */
      tre->last_tre  = tk->last_tre; /* previous word context */
#ifdef USE_NGRAM
      tre->lscore    = tk->last_lscore;	/* log score (LM only) */
#endif
      bt_store(backtrellis, tre); /* save to backtrellis */
      
      /******************************/
      /* 2.3. $BC18l4VA+0\(B            */
      /*      cross-word transition */
      /******************************/

      /* $B@h$[$I$N%H%l%j%9C18l(Btre$B$,!$$3$3$+$i$NA+0\@h$NC18l@hF,%N!<%I0J9_$N(B
	 $BD>A0C18l>pJs$H$7$F$b;2>H$5$l$k(B */
      /* The trellis atom 'tre' will be refered as the word context
	 information for next word-beginning nodes */

#ifdef UNIGRAM_FACTORING
      /* $B$3$3$G=hM}$5$l$k$N$O(B isolated words $B$N$_!$(B
	 shared nodes $B$O$^$H$a$F$3$N%k!<%W$N30$G7W;;$9$k(B */
      /* Only the isolated words will be processed here.
	 The shared nodes with constant factoring values will be computed
	 after this loop */
#endif

#ifdef USE_NGRAM
      /* $BA+0\85C18l$,KvHxC18l$N=*C<$J$i!$$I$3$X$bA+0\$5$;$J$$(B */
      /* do not allow transition if the source word is end-of-sentence word */
      if (sword == wchmm->winfo->tail_silwid) continue;

#ifdef UNIGRAM_FACTORING
      /* $B:GBgL`EY$r;}$DC18l=*C<%N!<%I$r5-O?$7$F$*$/(B */
      /* find the max wordend node */
      if (wordend_best_score < tk->score + wchmm->wordend_a[sword]) {
	wordend_best_score = tk->score + wchmm->wordend_a[sword];
	wordend_best_node = node;
	wordend_best_tre = tre;
	wordend_best_last_cword = tk->last_cword;
      }
#endif
      
      /* N-gram$B$K$*$$$F$O>o$KA4C18l$N@\B3$r9MN8$9$kI,MW$,$"$k$?$a!$(B
	 $B$3$3$GC18l4V$N8@8l3NN(CM$r$9$Y$F7W;;$7$F$*$/!%(B
	 $B%-%c%C%7%e$O(B max_successor_prob_iw() $BFb$G9MN8!%(B*/
      /* As all words are possible to connect in N-gram, we first compute
	 all the inter-word LM probability here.
	 Cache is onsidered in max_successor_prob_iw(). */
      if (wchmm->winfo->is_transparent[sword]) {
	iwparray = max_successor_prob_iw(wchmm, tk->last_cword);
      } else {
	iwparray = max_successor_prob_iw(wchmm, sword);
      }
#endif

      /* $B$9$Y$F$NC18l;OC<%N!<%I$KBP$7$F0J2<$r<B9T(B */
      /* for all beginning-of-word nodes, */
      /* wchmm->startnode[0..stid-1] ... $BC18l;OC<%N!<%I%j%9%H(B */
      /* wchmm->startnode[0..stid-1] ... list of word start node (shared) */
      for (stid = wchmm->startnum - 1; stid >= 0; stid--) {
	next_node = wchmm->startnode[stid];

	/*****************************************/
	/* 2.3.1. $BC18l4V8@8l@)Ls$rE,MQ(B           */
	/*        apply cross-word LM constraint */
	/*****************************************/
	
#ifdef USE_NGRAM
	/* N-gram$B3NN($r7W;;(B */
	/* compute N-gram probability */
#ifdef UNIGRAM_FACTORING
	/* 1-gram factoring $B$K$*$1$kC18l4V8@8l3NN(%-%c%C%7%e$N8zN(2=(B:
	   1-gram factoring $B$OC18lMzNr$K0MB8$7$J$$$N$G!$(B
	   $B$3$3$G;2>H$9$k(B factoring $BCM$NB?$/$O(B
	   wchmm->state[node].fscore $B$K4{$K3JG<$5$l(B, $BC5:wCf$bITJQ$G$"$k!%(B
	   $B$h$C$F7W;;$,I,MW$JC18l(B($B$I$NC18l$H$b%N!<%I$r6&M-$7$J$$C18l(B)
	   $B$K$D$$$F$N$_(B iwparray[] $B$G7W;;!&%-%c%C%7%e$9$k!%(B */
	/* Efficient cross-word LM cache:
	   As 1-gram factoring values are independent of word context,
	   they remain unchanged while search.  So, in cross-word LM
	   computation, beginning-of-word states which share nodes with
	   others and has factoring value in wchmm does not need cache.
	   So only the unshared beginning-of-word states are computed and
	   cached here in iwparray[].
	 */
	/* wchmm,start2isolate[0..stid-1] ... $B%N!<%I$r6&M-$7$J$$C18l$O(B
	   $B$=$NDL$7(BID, $B6&M-$9$k(B($B%-%c%C%7%e$NI,MW$N$J$$(B)$BC18l$O(B -1 */
	/* wchmm->start2isolate[0..stid-1] ... isolate ID for
	   beginning-of-word state.  value: -1 for states that has
	   1-gram factoring value (share nodes with some other words),
	   and ID for unshared words
	 */
	isoid = wchmm->start2isolate[stid];
	/* $B7W;;$,I,MW$G$J$$C18l@hF,%N!<%I$O%Q%9$r$^$H$a$F8e$K7W;;$9$k$N$G(B
	   $B$3$3$G$O%9%-%C%W(B */
	/* the shared nodes will be computed afterward, so just skip them
	   here */
	if (isoid == -1) continue;
	tmpprob = iwparray[isoid];
#else
	tmpprob = iwparray[stid];
#endif
#endif

#ifdef USE_NGRAM
	/* $BA+0\@h$NC18l$,@hF,C18l$J$iA+0\$5$;$J$$!%(B
	   $B$3$l$O(B wchmm.c $B$G3:EvC18l$K(B stid $B$r3d$j?6$i$J$$$3$H$GBP1~(B  */
	/* do not allow transition if the destination word is
	   beginning-of-sentence word.  This limitation is realized by
	   not assigning 'stid' for the word in wchmm.c.
	*/
#endif
	
#ifdef CATEGORY_TREE
	/* $BJ8K!$N>l9g(B, $B@)Ls$O7hDjE*(B: $B%+%F%4%jBP@)Ls>e5v$5$l$J$$>l9g$OA+0\$5$;$J$$(B */
	/* In grammar mode, LM constraint is deterministic:
	   do not allow transition if the category connection is not allowed
	   (with category tree, constraint can be determined on top node) */
	if (dfa_cp(dfa, wchmm->winfo->wton[sword], wchmm->winfo->wton[wchmm->ststart[next_node]]) == FALSE) continue;
#endif

	/*******************************************************************/
	/* 2.3.2. $BA+0\@h$NC18l@hF,$X$N%9%3%"7W;;(B($BA+0\3NN(!\8@8l%9%3%"(B)     */
	/*        compute score of destination node (transition prob + LM) */
	/*******************************************************************/
	tmpsum = tk->score + wchmm->wordend_a[sword];
#ifdef USE_NGRAM
	/* $B8@8l%9%3%"$rDI2C(B */
	/* add LM score */
	ngram_score_cache = tmpprob * lm_weight + lm_penalty;
	tmpsum += ngram_score_cache;
	if (wchmm->winfo->is_transparent[sword] && wchmm->winfo->is_transparent[tk->last_cword]) {
	  
	  tmpsum += lm_penalty_trans;
	}
#else  /* USE_DFA */
	/* grammar: $BC18lA^F~%Z%J%k%F%#$rDI2C(B */
	/* grammar: add insertion penalty */
	tmpsum += penalty1;

	/* grammar: deterministic factoring (in case category-tree not enabled) */
#ifdef CATEGORY_TREE
	/* no need to factoring */
#else
	if (!can_succeed(wchmm, sword, next_node)) {
	  tmpsum = LOG_ZERO;
	}
#endif /* CATEGORY_TREE */
#endif /* USE_NGRAM */
	
	/*********************************************************************/
	/* 2.3.3. $BA+0\@h%N!<%I$X%H!<%/%sEAHB(B($BC18lMzNr>pJs$O99?7(B)             */
	/*        pass token to destination node (updating word-context info */
	/*********************************************************************/
	if ((tknextid = node_exist_token(tn, next_node, tre->wid)) != TOKENID_UNDEFINED) {
	  /* $BA+0\@h%N!<%I$K$O4{$KB>%N!<%I$+$iEAHB:Q$_(B: $B%9%3%"$,9b$$$[$&$r;D$9(B */
	  /* the destination node already has a token: compare score */
	  tknext = &(tlist[tn][tknextid]);
	  if (tknext->score < tmpsum) {
	    /* $B$=$NA+0\@h%N!<%I$,;}$D%H!<%/%s$NFbMF$r>e=q$-$9$k(B($B?75,%H!<%/%s$O:n$i$J$$(B) */
	    /* overwrite the content of existing destination token: not create a new token */
#ifdef USE_NGRAM
	    tknext->last_lscore = ngram_score_cache;
	    if (wchmm->winfo->is_transparent[sword]) {
	      tknext->last_cword = tk->last_cword;
	    } else {
	      tknext->last_cword = sword;
	    }
#endif
	    tknext->score = tmpsum;
	    /* $BA+0\@h$K!$@h$[$IJ]B8$7$?(Bsword$B$N%H%l%j%9C18l$X$N%]%$%s%?$r(B
	       $B!VD>A0C18lMzNr>pJs!W$H$7$FEAHB(B */
	    /* pass destination the pointer to the saved trellis atom
	       corresponds to 'sword' as "previous word-context info". */
	    tknext->last_tre = tre;
	  }
	} else {
	  /* $BA+0\@h%N!<%I$OL$EAHB(B: $B?75,%H!<%/%s$r:n$C$F3d$jIU$1$k(B */
	  /* token unassigned: create new token and assign */
	  if (tmpsum > LOG_ZERO) { /* valid token */
	    tknextid = create_token();
	    tknext = &(tlist[tn][tknextid]);
	    tk = &(tlist[tl][tindex[tl][j]]);
#ifdef USE_NGRAM
	    tknext->last_lscore = ngram_score_cache;
	    if (wchmm->winfo->is_transparent[sword]) {
	      tknext->last_cword = tk->last_cword;
	    } else {
	      tknext->last_cword = sword;
	    }
#endif
	    tknext->score = tmpsum;
	    tknext->last_tre = tre;
	    node_assign_token(next_node, tknextid);
	  }
	}
      }
    }
    
  } /* end of main viterbi loop */
#ifdef UNIGRAM_FACTORING
  /***********************************************************/
  /* 2.4 $BC18l=*C<$+$i(Bfactoring$BIU$-C18l@hF,$X$NA+0\(B ***********/
  /*    transition from wordend to shared (factorized) nodes */
  /***********************************************************/
  if (wordend_best_score > LOG_ZERO) {
    node = wordend_best_node;
    sword = wchmm->stend[node];
    for (stid = wchmm->startnum - 1; stid >= 0; stid--) {
      next_node = wchmm->startnode[stid];
      /* skip nodes already calculated in the above main loop */
      if (wchmm->start2isolate[stid] != -1) continue; 
      tmpprob = wchmm->state[next_node].fscore;
      ngram_score_cache = tmpprob * lm_weight + lm_penalty;
      tmpsum = wordend_best_score;
      tmpsum += ngram_score_cache;
      if (wchmm->winfo->is_transparent[sword] && wchmm->winfo->is_transparent[wordend_best_last_cword]) {
	tmpsum += lm_penalty_trans;
      }
      if ((tknextid = node_exist_token(tn, next_node, sword)) != TOKENID_UNDEFINED) {
	tknext = &(tlist[tn][tknextid]);
	if (tknext->score < tmpsum) {
	  tknext->last_lscore = ngram_score_cache;
	  if (wchmm->winfo->is_transparent[sword]) {
	    tknext->last_cword = wordend_best_last_cword;
	  } else {
	    tknext->last_cword = sword;
	  }
	  tknext->score = tmpsum;
	  tknext->last_tre = wordend_best_tre;
	}
      } else {
	if (tmpsum > LOG_ZERO) { /* valid token */
	  tknextid = create_token();
	  tknext = &(tlist[tn][tknextid]);
	  tknext->last_lscore = ngram_score_cache;
	  if (wchmm->winfo->is_transparent[sword]) {
	    tknext->last_cword = wordend_best_last_cword;
	  } else {
	    tknext->last_cword = sword;
	  }
	  tknext->score = tmpsum;
	  tknext->last_tre = wordend_best_tre;
	  node_assign_token(next_node, tknextid);
	}
      }
    }
  }
#endif /* UNIGRAM_FACTORING */

  /***************************************/
  /* 3. $B>uBV$N=PNO3NN(7W;;(B               */
  /*    compute state output probability */
  /***************************************/

  /* $B<!CJ$NM-8z%N!<%I$K$D$$$F=PNO3NN($r7W;;$7$F%9%3%"$K2C$($k(B */
  /* compute outprob for new valid (token assigned) nodes and add to score */
  for (j=0;j<tnum[tn];j++) {
    tk = &(tlist[tn][tindex[tn][j]]);
    tk->score += outprob_style(wchmm, tk->node, tk->last_tre->wid, t, param);
  }

  /*******************************************************/
  /* 4. $B%9%3%"$G%H!<%/%s$r%=!<%H$7%S!<%`I}J,$N>e0L$r7hDj(B */
  /*    sort tokens by score up to beam width            */
  /*******************************************************/

  /* tlist[tl]$B$r<!CJ$N$?$a$K%j%;%C%H(B */
  clear_tlist(tl);

  /* $B%R!<%W%=!<%H$rMQ$$$F$3$NCJ$N%N!<%I=89g$+$i>e0L(B(bwidth)$B8D$rF@$F$*$/(B */
  /* ($B>e0LFb$N=gNs$OI,MW$J$$(B) */
  sort_token_no_order(trellis_beam_width, &n_start, &n_end);

  /* check for sort result */
/* 
 *     j_printf("%d: vwidth=%d / %d\n",t, vwidth[tn], wchmm->n);
 */
  /* make sure LOG_ZERO not exist width vwidth[tn] */
/* 
 *     for (j=0;j<vwidth[tn];j++) {
 *	 if (nodescore[tn][hsindex[tn][j+1]] <= LOG_ZERO) {
 *	   j_error("dsadsadas %d %d\n",hsindex[tn][j+1], nodescore[tn][hsindex[tn][j+1]]);
 *	 }
 *     }
 */
  
  /***************/
  /* 5. $B=*N;=hM}(B */
  /*    finalize */
  /***************/
  if (progout_flag) {
    /* $BA2<!=PNO(B: $B8=%U%l!<%`$N%Y%9%H%Q%9$r0lDj;~4V$*$-$K>e=q$-=PNO(B */
    /* progressive result output: output current best path in certain time interval */
    if ((t % progout_interval_frame) == 0) {
      bt_current_max(backtrellis, t-1, wchmm->winfo);
    }
  }
  /* j_printf("%d: %d\n",t,tnum[tn]); */
  /* for debug: output current max word */
  if (debug2_flag) {
    bt_current_max_word(backtrellis, t-1, wchmm->winfo);
  }

#ifdef SP_BREAK_CURRENT_FRAME
  /* $B%7%g!<%H%]!<%:%;%0%a%s%F!<%7%g%s(B: $BD>A0%U%l!<%`$G%;%0%a%s%H$,@Z$l$?$+$I$&$+%A%'%C%/(B */
  if (detect_end_of_segment(backtrellis, t-1, wchmm->winfo)) {
    /* $B%;%0%a%s%H=*N;8!CN(B: $BBh#1%Q%9$3$3$GCfCG(B */
    return FALSE;		/* segment: [sparea_start..t-2] */
  }
#endif

  /* $B%S!<%`Fb%N!<%I?t$,(B 0 $B$K$J$C$F$7$^$C$?$i!$6/@)=*N;(B */
  if (tnum[tn] == 0) {
    j_printerr("Error: %dth frame: no nodes left in beam! model mismatch or wrong input?\n", t);
    return(FALSE);
  }

  return(TRUE);
    
}

/*************************************************/
/* frame synchronous beam search --- last frame  */
/* $B%U%l!<%`F14|%S!<%`C5:w$N<B9T(B --- $B:G=*%U%l!<%`(B */
/*************************************************/
void
get_back_trellis_end(
     HTK_Param *param,		/* parameters */
     WCHMM_INFO *wchmm,		/* lexicon tree */
     BACKTRELLIS *backtrellis	/* backward trellis (return value) */
     )
{
  int t, node, j;
  TOKEN2 *tk;

  /* $B:G8e$K%S!<%`Fb$K;D$C$?C18l=*C<%H!<%/%s$r=hM}$9$k(B */
  /* process the last wordend tokens */
  t = param->samplenum;
  tl = tn;
  if (tn == 0) tn = 1; else tn = 0;
  for (j=n_start; j<=n_end; j++) {
    tk = &(tlist[tl][tindex[tl][j]]);
    node = tk->node;
    if (wchmm->stend[node] != WORD_INVALID) {
      tre = (TRELLIS_ATOM *)mymalloc(sizeof(TRELLIS_ATOM));
      tre->wid = wchmm->stend[node];
      tre->backscore = tk->score;
      tre->begintime = tk->last_tre->endtime + 1;
      tre->endtime   = t-1;
      tre->last_tre  = tk->last_tre;
#ifdef USE_NGRAM
      tre->lscore    = tk->last_lscore;
#endif
      bt_store(backtrellis, tre);
    }
  }

#ifdef SP_BREAK_CURRENT_FRAME
  /*if (detect_end_of_segment(backtrellis, param->samplenum-1, winfo)) {
    return;
    }*/
#endif
}

/*************************/
/* $BC5:w=*N;(B --- $B=*N;=hM}(B */
/* end of search         */
/*************************/
LOGPROB
finalize_1st_pass(BACKTRELLIS *backtrellis, WORD_INFO *winfo, int len)
{
  LOGPROB lastscore;
  
  /* $BC18l%H%l%j%9(B(backtrellis) $B$r@0M}(B: $B%H%l%j%9C18l$N:FG[CV$H%=!<%H(B */
  /* re-arrange backtrellis: index them by frame, and sort by word ID */
  backtrellis->framelen = len;
  bt_relocate_rw(backtrellis);
  bt_sort_rw(backtrellis);
  
  /* $B7k2L$rI=<($9$k(B (best $B2>@b(B)*/
  /* output 1st pass result (best hypothesis) */
  if (verbose_flag && (!progout_flag)) j_printerr("\n");
  lastscore = print_1pass_result(backtrellis, len, winfo);

#ifdef USE_NGRAM
  /* free succesor cache */
  /* $B<!$NG'<1$G(Bwchmm->winfo$B$H$b$KL5JQ99$N>l9g(B free $B$9$kI,MW$J$7(B */
  /* no need to free if wchmm and wchmm are not changed in the next recognition */
  /* max_successor_cache_free(); */
#endif
  /* free nodes */
  free_nodes();

  /* return the best score */
  return(lastscore);
}

#ifdef SP_BREAK_CURRENT_FRAME
/*******************************************************************/
/* $BBh#1%Q%9%;%0%a%s%H=*N;=hM}(B ($B%7%g!<%H%]!<%:%;%0%a%s%F!<%7%g%sMQ(B) */
/* end of 1st pass for a segment (for short pause segmentation)    */
/*******************************************************************/
void
finalize_segment(BACKTRELLIS *backtrellis, HTK_Param *param, int len)
{
  int t, v;

  /* $B%H%l%j%9;O=*C<$K$*$1$k:GL`C18l$rBh(B2$B%Q%9$N;O=*C<C18l$H$7$F3JG<(B */
  /* fix initial/last word hypothesis of the next 2nd pass to the best
     word hypothesis at the first/last frame in backtrellis*/
  set_terminal_words(backtrellis);

  /* $B%Q%i%a!<%?$r(B, $B:#Bh#1%Q%9$,=*N;$7$?%;%0%a%s%H6h4V$H;D$j$N6h4V$KJ,3d$9$k!%(B
     $B$?$@$7@\B3It$N(Bsp$B6h4VItJ,(B(sparea_start..len-1)$B$O!V$N$j$7$m!W$H$7$FN>J}$K(B
     $B%3%T!<$9$k(B */
  /* Divide input parameter into two: the last segment and the rest.
     The short-pause area (sparea_start..len-1) is considered as "tab",
     copied in both parameters
   */
  /* param[sparea_start..framelen] -> rest_param
     param[0..len-1] -> param
     [sparea_start...len-1] overlapped
  */
  if (len != param->samplenum) {
    /* copy rest parameters for next process */
#ifdef SP_BREAK_DEBUG
    printf("[%d..%d]->[%d..%d] %d\n", sparea_start, param->samplenum-1,
	   0, param->samplenum-sparea_start-1, len);
#endif
    rest_param = new_param();
    memcpy(&(rest_param->header), &(param->header), sizeof(HTK_Param_Header));
    rest_param->samplenum = param->samplenum - sparea_start;
    rest_param->header.samplenum = rest_param->samplenum;
    rest_param->veclen = param->veclen;
    rest_param->parvec = (VECT **)mymalloc(sizeof(VECT *) * rest_param->samplenum);
    /* copy 1: overlap area (copy) */
    for(t=sparea_start;t<len;t++) {
      rest_param->parvec[t-sparea_start] = (VECT *)mymalloc(sizeof(VECT) * rest_param->veclen);
      memcpy(rest_param->parvec[t-sparea_start], param->parvec[t], sizeof(VECT) * rest_param->veclen);
    }
    /* copy 2: rest area (move) */
    for(t=len;t<param->samplenum;t++) {
      rest_param->parvec[t-sparea_start] = param->parvec[t];
    }

    /* shrink original param to [0..len-1] */
    /* just shrink the parvec */
    param->samplenum = len;
    param->parvec = (VECT **)myrealloc(param->parvec, sizeof(VECT *) * param->samplenum);
    sp_break_last_nword_allow_override = TRUE;
    
  } else {
    
    /* last segment is on end of input: no rest parameter */
    rest_param = NULL;
    /* reset last_word info */
    sp_break_last_word = WORD_INVALID;
    sp_break_last_nword = WORD_INVALID;
    sp_break_last_nword_allow_override = FALSE;
  }
}
#endif /* SP_BREAK_CURRENT_FRAME */
  
/********************************************************************/
/* $BBh#1%Q%9$r<B9T$9$k%a%$%s4X?t(B                                     */
/* $BF~NO$r%Q%$%W%i%$%s=hM}$9$k>l9g$O(B realtime_1stpass.c $B$r;2>H$N$3$H(B */
/* main function to execute 1st pass                                */
/* the pipeline processing is not here: see realtime_1stpass.c      */
/********************************************************************/
void
get_back_trellis(
     HTK_Param *param,		/* whole parameter */
     WCHMM_INFO *wchmm,		/* lexicon tree */
     BACKTRELLIS *backtrellis,	/* backward trellis to hold result */
     LOGPROB *backmax		/* backmax (retuen value) */
     )
{
  int t;

  /* $B=i4|2=5Z$S(B t=0 $B$r7W;;(B */
  /* initialize and compute frame = 0 */
  get_back_trellis_init(param, wchmm, backtrellis);

  /* $B%a%$%s%k!<%W(B */
  /* main loop */
  for (t = 1; t < param->samplenum; t++) {
    if (get_back_trellis_proceed(t, param, wchmm, backtrellis) == FALSE
	|| catch_intr_flag
	|| (module_mode && module_wants_terminate())) {
      /* $BC5:wCfCG(B: $B=hM}$5$l$?F~NO$O(B 0 $B$+$i(B t-2 $B$^$G(B */
      /* search terminated: processed input = [0..t-2] */
      /* $B$3$N;~E@$GBh(B1$B%Q%9$r=*N;$9$k(B */
      /* end the 1st pass at this point */
      *backmax = finalize_1st_pass(backtrellis, wchmm->winfo, t-1);
#ifdef SP_BREAK_CURRENT_FRAME
      /* $B%7%g!<%H%]!<%:%;%0%a%s%F!<%7%g%s$N>l9g(B,
	 $BF~NO%Q%i%a!<%?J,3d$J$I$N:G=*=hM}$b9T$J$&(B */
      /* When short-pause segmentation enabled */
      finalize_segment(backtrellis, param, t-1);
#endif
      /* terminate 1st pass here */
      return;
    }
  }
  /* $B:G=*%U%l!<%`$r7W;;(B */
  /* compute the last frame of input */
  get_back_trellis_end(param, wchmm, backtrellis);
  /* $BBh(B1$B%Q%9=*N;=hM}(B */
  /* end of 1st pass */
  *backmax = finalize_1st_pass(backtrellis, wchmm->winfo, param->samplenum);
#ifdef SP_BREAK_CURRENT_FRAME
  /* $B%7%g!<%H%]!<%:%;%0%a%s%F!<%7%g%s$N>l9g(B,
     $BF~NO%Q%i%a!<%?J,3d$J$I$N:G=*=hM}$b9T$J$&(B */
  /* When short-pause segmentation enabled */
  finalize_segment(backtrellis, param, param->samplenum);
#endif
}

/* end of file */
