/**
 * @file   wav2mfcc-pipe.c
 * @author Akinobu LEE
 * @date   Thu Feb 17 18:12:30 2005
 * 
 * <JA>
 * @brief  ȷ MFCC ħ̤Ѵ (ե졼ñ)
 *
 * Ǥ wav2mfcc.c δؿե졼Ʊ˽뤿Ѵ
 * ؿǼƤޤǧϤʿԤƹԤ硤
 * ؿѤޤ
 * </JA>
 * 
 * <EN>
 * @brief  Convert speech inputs into MFCC parameter vectors (per input frame)
 *
 * There are functions are derived from wav2mfcc.c, to compute
 * MFCC vectors in per-frame basis.  When performing on-line recognition,
 * these functions will be used instead of ones in wav2mfcc.c
 * </EN>
 * 
 * $Revision:$
 * 
 */
/*
 * Copyright (c) 1991-2005 Kawahara Lab., Kyoto University
 * Copyright (c) 2000-2005 Shikano Lab., Nara Institute of Science and Technology
 * Copyright (c) 2005      Julius project team, Nagoya Institute of Technology
 * All rights reserved
 */

/* wav2mfcc-pipe.c --- split Wav2MFCC to perform per-frame-basis,
   and also realtime CMN for 1st-pass pipe-lining */

/************************************************************************/
/*    wav2mfcc.c   Convert Speech file to MFCC_E_D_(Z) file             */
/*----------------------------------------------------------------------*/
/*    Author    : Yuichiro Nakano                                       */
/*                                                                      */
/*    Copyright(C) Yuichiro Nakano 1996-1998                            */
/*----------------------------------------------------------------------*/
/************************************************************************/


#include <sent/stddefs.h>
#include <sent/mfcc.h>

static double *fbank;                    ///< Local buffer to hold filterbank
static FBankInfo fb;		///< Local buffer to hold filterbank information

/** 
 * Initialize and setup buffers for a MFCC computataion.
 * 
 * @param para [in] configuration parameters
 * @param bf [out] pointer to the entry point of workspace for FFT
 * @param ssbuf [in] noise spectrum, or NULL if not using spectral subtraction
 * @param ssbuflen [in] length of above, ignoredwhen @a ssbuf is NULL
 */
void
WMP_init(Value para, float **bf, float *ssbuf, int ssbuflen)
{
  /* Get filterbank information and initialize tables */
  fb = InitFBank(para);
  
  if((fbank = (double *)mymalloc((para.fbank_num+1)*sizeof(double))) == NULL){
    j_error("WMP_init failed\n");
  }
  if((*bf = (float *)mymalloc(fb.fftN * sizeof(float))) == NULL){
    j_error("WMP_init failed\n");
  }

  if (ssbuf != NULL) {
    /* check ssbuf length */
    if (fb.fftN != ssbuflen) {
      j_error("Error: Wav2MFCC_E_D: noise spectrum length not match\n");
    }
  }
}

/** 
 * Calculate MFCC and log energy for one frame.  Perform spectral subtraction
 * if @a ssbuf is specified.
 * 
 * @param mfcc [out] buffer to hold the resulting MFCC vector
 * @param bf [i/o] work area for FFT
 * @param para [in] configuration parameters
 * @param ssbuf [in] noise spectrum, or NULL if not using spectral subtraction
 */
void
WMP_calc(float *mfcc, float *bf, Value para, float *ssbuf)
{
  float energy = 0.0;

  if (para.raw_e) {
    /* calculate log raw energy */
    energy = CalcLogRawE(bf, para.framesize);
    /* pre-emphasize */
    PreEmphasise(bf, para);
    /* hamming window */
    Hamming(bf, para.framesize);
  } else {
    /* pre-emphasize */
    PreEmphasise(bf, para);
    /* hamming window */
    Hamming(bf, para.framesize);
    /* calculate log energy */
    if (!para.c0) energy = CalcLogRawE(bf, para.framesize);
  }
  /* filterbank */
  MakeFBank(bf, fbank, fb, para, ssbuf);
  /* 0'th cepstral parameter */
  if(para.c0)energy = CalcC0(fbank, para);
  /* MFCC */
  MakeMFCC(fbank, mfcc, para);
  /* weight cepstrum */
  WeightCepstrum(mfcc, para);
  /* Normalise Log Energy is not implemented */
  if(para.enormal) {
    j_error("normalize log energy is not implemented in pipeline mode\n");
  }
  mfcc[para.mfcc_dim] = energy;
}

/** 
 * Calculate delta coefficients from base MFCC for one frame.
 * 
 * @param c [i/o] array of MFCC vector, the new delta coef. will be appended at frame @a t
 * @param t [in] the target frame where the delta coef. should be computed
 * @param frame [in] maximum frame number + 1 in which the base MFCC exists.
 * @param para [in] configuration parameter
 */
void WMP_Delta(float **c, int t, int frame, Value para)
{
  int theta, n, dim, B = 0;
  float A1, A2, sum;

  for(theta = 1; theta <= para.delWin; theta++)
    B += theta * theta;

  dim = para.vec_num / 2;

  for(n = 1; n <= dim; n++){
    sum = 0;
    for(theta = 1; theta <= para.delWin; theta++){
      /* Replicate the first or last vector */
      /* at the beginning and end of speech */
      if(t - theta < 0) A1 = c[0][n - 1];
      else A1 = c[t - theta][n - 1];
      if(t + theta >= frame) A2 = c[frame - 1][n - 1];
      else A2 = c[t + theta][n - 1];
      sum += theta * (A2 - A1);
    }
    c[t][n + para.mfcc_dim] = sum / (2 * B);
  }
}


/* real-time CMN */
/* 01/01/19 store last 500frame samples */

#define CPMAX 500		///< Maximum number of frames to store ceptral mean for CMN update
static float *lastcsum = NULL;	///< Accumulated MFCC values for the last CPMAX frames
static float **lastc = NULL;    ///< Array of MFCC for the last CPMAX frames
static int dim;			///< Local workarea to store the number of MFCC dimension.
static int cp = 0;		///< Current pointer to store MFCC in @a lastcsum, @a lastc
static int filled = 0;		///< Number of filled part in @a lastc.
static float *mfcc_ave = NULL;  ///< Current MFCC mean subtracting from input
static float *mfcc_ave_new = NULL; ///< Next MFCC mean

static float *lastcsum_bak = NULL; ///< Local backup of @a lastcsum for input rejection
static float **lastc_bak = NULL;   ///< Local backup of @a lastc for input rejection
static int cp_bak;	  ///< Local backup of @a cp for input rejection
static int filled_bak; ///< Local backup of @a filled for input rejection

/** 
 * Update CMN by replacing the current cepstral mean by that of
 * last CPMAX frames.
 * 
 */
void
CMN_realtime_update()
{
  int i;
  static float *p;

  /* allocate needed buffer */
  if (mfcc_ave_new == NULL) {
    mfcc_ave_new = (float *)mymalloc(sizeof(float) * dim);
  }
  for(i=0;i<dim;i++) {
    mfcc_ave_new[i] = lastcsum[i] / (float)filled;
  }
  if (mfcc_ave == NULL) {
    mfcc_ave = (float *)mymalloc(sizeof(float) * dim);
  }
  /* swap current <-> new */
  p = mfcc_ave;
  mfcc_ave = mfcc_ave_new;
  mfcc_ave_new = p;
  /* backup current mfcc data and sum info, to prepare for the case
     in which this input is rejected for later processing */
  if (lastcsum_bak == NULL) {
    lastcsum_bak = (float *)mymalloc(sizeof(float) * dim);
    lastc_bak = (float **)mymalloc(sizeof(float*) * CPMAX);
    for(i=0;i<CPMAX;i++) {
      lastc_bak[i] = (float *)mymalloc(sizeof(float) * dim);
    }
  }
  memcpy(lastcsum_bak, lastcsum, sizeof(float) * dim);
  for(i=0;i<CPMAX;i++) {
    memcpy(lastc_bak[i], lastc[i], sizeof(float) * dim);
  }
  cp_bak = cp;
  filled_bak = filled;
}

/** 
 * Restore the old cepstral information, to cancel the last input.
 * The current cepstral mean will still be kept for next input.
 * 
 */
void
CMN_realtime_keep()
{
  int i;

  /* let mfcc_ave as is */
  
  /* copy the backup mfcc data to current */
  if (lastcsum != NULL && lastcsum_bak != NULL) {
    memcpy(lastcsum, lastcsum_bak, sizeof(float) * dim);
    for(i=0;i<CPMAX;i++) {
      memcpy(lastc[i], lastc_bak[i], sizeof(float) * dim);
    }
    cp = cp_bak;
    filled = filled_bak;
  }
}

/** 
 * Perform CMN on the incoming input, and store the MFCC to local
 * work area for laster update of cepstral mean.
 * 
 * @param mfcc [i/o] MFCC vector to perform CMN
 * @param cdim [in] length of above
 */
void
CMN_realtime(float *mfcc, int cdim)
{
  int t,i;

  /* store mfcc data and sum info to lastc and lastcsum */
  if (lastcsum == NULL) {	/* initialize */
    dim = cdim;
    lastcsum = (float *)mymalloc(sizeof(float) * dim);
    for(i=0;i<dim;i++) lastcsum[i] = 0.0;
    lastc = (float **)mymalloc(sizeof(float*) * CPMAX);
    for(t=0;t<CPMAX;t++) {
      lastc[t] = (float *)mymalloc(sizeof(float) * dim);
    }
    filled = 0;
    cp = 0;
  }
  
  if (filled < CPMAX) {
    filled++;
  } else {
    for(i=0;i<dim;i++) {
      lastcsum[i] -= lastc[cp][i];
    }
  }
  for(i=0;i<dim;i++) {
    lastcsum[i] += mfcc[i];
    lastc[cp][i] = mfcc[i];
  }
  cp++;
  if (cp >= CPMAX) cp = 0;
  
  /* do CMN */
  if (mfcc_ave != NULL) {
    for(i=0;i<dim;i++) {
      mfcc[i] -= mfcc_ave[i];
    }
  }
}

/** 
 * Read binary with byte swap (assume file is Big Endian)
 * 
 * @param buf [out] data buffer
 * @param unitbyte [in] size of unit in bytes
 * @param unitnum [in] number of units to be read
 * @param fp [in] file pointer
 * 
 * @return TRUE if required number of units are fully read, FALSE if failed.
 */
static boolean
myread(void *buf, size_t unitbyte, int unitnum, FILE *fp)
{
  if (myfread(buf, unitbyte, unitnum, fp) < (size_t)unitnum) {
    return(FALSE);
  }
#ifndef WORDS_BIGENDIAN
  swap_bytes(buf, unitbyte, unitnum);
#endif
  return(TRUE);
}

/** 
 * Write binary with byte swap (assume data is Big Endian)
 * 
 * @param buf [in] data buffer
 * @param unitbyte [in] size of unit in bytes
 * @param unitnum [in] number of units to write
 * @param fd [in] file descriptor
 * 
 * @return TRUE if required number of units are fully written, FALSE if failed.
 */
static boolean
mywrite(void *buf, size_t unitbyte, int unitnum, int fd)
{
#ifndef WORDS_BIGENDIAN
  swap_bytes(buf, unitbyte, unitnum);
#endif
  if (write(fd, buf, unitbyte * unitnum) < unitbyte * unitnum) {
    return(FALSE);
  }
#ifndef WORDS_BIGENDIAN
  swap_bytes(buf, unitbyte, unitnum);
#endif
  return(TRUE);
}

/** 
 * Load CMN parameter from file.  If the number of MFCC dimension in the
 * file does not match the specified one, an error will occur.
 * 
 * @param filename [in] file name
 * @param dim [in] required number of MFCC dimensions
 * 
 * @return TRUE on success, FALSE on failure.
 */
boolean
CMN_load_from_file(char *filename, int dim)
{
  FILE *fp;
  int veclen;
  if ((fp = fopen_readfile(filename)) == NULL) {
    j_printerr("Error: CMN_load_from_file: failed to open\n");
    return(FALSE);
  }
  /* read header */
  if (myread(&veclen, sizeof(int), 1, fp) == FALSE) {
    j_printerr("Error: CMN_load_from_file: failed to read header\n");
    fclose_readfile(fp);
    return(FALSE);
  }
  /* check length */
  if (veclen != dim) {
    j_printerr("Error: CMN_load_from_file: length mismatch\n");
    fclose_readfile(fp);
    return(FALSE);
  }
  /* read body */
  if (mfcc_ave == NULL) {
    mfcc_ave = (float *)mymalloc(sizeof(float) * dim);
  }
  if (myread(mfcc_ave, sizeof(float), dim, fp) == FALSE) {
    j_printerr("Error: CMN_load_from_file: failed to read\n");
    fclose_readfile(fp);
    return(FALSE);
  }
  if (fclose_readfile(fp) == -1) {
    j_printerr("Error: CMN_load_from_file: failed to close\n");
    return(FALSE);
  }

  return(TRUE);
}

/** 
 * Save the current CMN vector to a file.
 * 
 * @param filename [in] filename to save the data.
 * 
 * @return TRUE on success, FALSE on failure.
 */
boolean
CMN_save_to_file(char *filename)
{
  int fd;

  if ((fd = creat(filename, 0644)) == -1) {
    j_printerr("Error: CMN_save_to_file: failed to open\n");
    return(FALSE);
  }
  /* write header */
  if (mywrite(&dim, sizeof(int), 1, fd) == FALSE) {
    j_printerr("Error: CMN_save_to_file: failed to write header\n");
    close(fd);
    return(FALSE);
  }
  /* write body */
  if (mfcc_ave == NULL) {
    j_printerr("Error: no data to write?? should not happen!\n");
  }
  if (mywrite(mfcc_ave, sizeof(float), dim, fd) == FALSE) {
    j_printerr("Error: CMN_save_to_file: failed to write header\n");
    close(fd);
    return(FALSE);
  }
  close(fd);
  
  return(TRUE);
}
