/*
 * Java Template Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.web.tag;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.jsp.JspException;

import org.apache.commons.lang.StringUtils;
import org.jtp.web.bean.PagingBean;

/**
 * 改ページ情報設定カスタムタグクラス
 *
 * @author <a href="mailto:nagashiba.satoshi@opcybe.co.jp">Satoshi Nagashiba</a>
 * @version $Revision: 1.1 $ $Date: 2007/04/06 07:22:14 $
 * @since 2.0 (Oct,26,2006)
 */
public abstract class BasePagingTag extends BaseTag {

	/** コレクション（リスト） */
	private String collection = null;
	/** 現在ページ */
	private String page = null;
	/** １ページ単位の表示行数 */
	private String maxRows = null;
	/** 改ページ設定情報 */
	private PagingBean pagingBean = null;
	
	/** １ページ単位のデフォルト表示行数 */
	private static final int DEFAULT_MAXROWS = 10;
	
	/**
	 * 開始タグを処理するメソッド
	 * @return ボディをJSPとして評価しない
	 * @throws JspException JSP例外
	 */
	public final int doStartTag() throws JspException {
		//表示件数設定
		if (StringUtils.isEmpty(maxRows)) {
			maxRows = String.valueOf(DEFAULT_MAXROWS);
		}

		//改ページ情報を設定
		PagingBean pagingBean = createPagingBean();
		
		//改ページ設定情報をリクエストに設定する
		pageContext.getRequest().setAttribute(PagingBean.ID, pagingBean);
		
		return super.doStartTag();
	}

	/**
	 * 改ページ設定情報を生成する
	 * @return 生成した改ページ設定情報
	 * @throws JspException JSP例外
	 */
	private PagingBean createPagingBean() throws JspException {
		//１ページあたりの最大件数
		int rows = Integer.parseInt(maxRows);
		
		pagingBean = new PagingBean();
		
		//現在頁
		if (page == null) page = new String("1");
		pagingBean.setCurrent(java.lang.Integer.parseInt(page));

		//前頁
		pagingBean.setPrev(pagingBean.getCurrent() - 1);
		
		//次頁
		pagingBean.setNext(pagingBean.getCurrent() + 1);
		
		//データリスト取得
		List list = (List) pageContext.findAttribute(collection);
		//リストが取得できない場合、空のリストを作成
		if (list == null) list = new ArrayList();
		
		//データ件数
		pagingBean.setCount(list.size());

		//総頁
		int allPage = list.size() / rows;
		//総頁が0以下の場合、1頁を設定
		if (allPage <= 0) {
			allPage = 1;
		}
		else { 
			if ((list.size() % rows) > 0) allPage += 1;
		}
		
		pagingBean.setAll(allPage);
		
		//開始インデックス
		if (pagingBean.getCurrent() > 1) {
			pagingBean.setOffset((pagingBean.getCurrent() - 1) * rows);
		}
		
		//終了インデックス
		pagingBean.setLength(rows);
		
		return pagingBean;
	}
	
	/**
	 * 改ページ設定情報の取得
	 * @return 改ページ設定情報
	 */
	protected PagingBean getPagingBean() {
		return this.pagingBean;
	}
	
	/**
	 * コレクション（リスト）取得
	 * @return collection コレクション（リスト）
	 */
	public String getCollection() {
		return collection;
	}
	
	/**
	 * コレクション（リスト）設定
	 * @param collection コレクション（リスト）
	 */
	public void setCollection(String collection) {
		this.collection = collection;
	}
	
	/**
	 * 現在ページ取得
	 * @return page 現在ページ
	 */
	public String getPage() {
		return page;
	}
	
	/**
	 * 現在ページ設定
	 * @param page 現在ページ
	 */
	public void setPage(String page) {
		this.page = page;
	}
	
	/**
	 * １ページ単位の表示行数取得
	 * @return maxRows １ページ単位の表示行数
	 */
	public String getMaxRows() {
		return maxRows;
	}
	
	/**
	 * １ページ単位の表示行数設定
	 * @param maxRows １ページ単位の表示行数
	 */
	public void setMaxRows(String maxRows) {
		this.maxRows = maxRows;
	}
}
